/*
    GSrnd
    
    A control panel for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998,99 Oliver Gantz (o.gantz@tu-bs.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "resources.h"


typedef struct _rc_data rc_data;
typedef struct _rc_group rc_group;


struct _rc_data {
	char *name;
	char *value;
	rc_data *next;
};


struct _rc_group {
	char *name;
	rc_data *first, *last;
	rc_group *next;
};


static char path[1024];


static rc_group *groups_first = (rc_group *)NULL;
static rc_group *groups_last = (rc_group *)NULL;
static rc_group *current_group = (rc_group *)NULL;



int rc_load (const char *fname)
{
	FILE *f;
	char buffer[160], *c;
	int len;

	sprintf(path, "%s/%s", getenv("HOME"), fname);
	if ((f = fopen(path, "r")) == (FILE *)NULL)
		return -1;

	while (fgets(buffer, 160, f) != (char *)NULL) {
		if ((buffer[0] == '#') || (buffer[0] == '\n'))
			continue;
		if ((c = strchr(buffer, '\n')) != (char *)NULL)
			*c = '\0';
		len = strlen(buffer)-1;
		if ((len >= 0) && (buffer[0] == '[') && (buffer[len] == ']')) {
			buffer[len] = '\0';
			rc_set_group(buffer+1);
		}
		else if (current_group != (rc_group *)NULL) {
			if ((c = strchr(buffer, '=')) == (char *)NULL)
				c = buffer + len;
			else
				*c = '\0';
			c++;
			rc_write_str(buffer, c);
		}
	}

	return fclose(f);
}


int rc_save (void)
{
	FILE *f;
	rc_group *g = groups_first;
	rc_data *d;

	if ((f = fopen(path, "w")) == (FILE *)NULL)
		return -1;

	fprintf(f, "# This file is generated.\n# Do not edit!\n");
	while (g != (rc_group *)NULL) {
		fprintf(f, "[%s]\n", g->name);
		d = g->first;
		while (d != (rc_data *)NULL) {
			fprintf(f, "%s=%s\n", d->name, d->value);
			d = d->next;
		}
		g = g->next;
	}

	return fclose(f);
}


void rc_set_group (const char *group)
{
	rc_group *g = groups_first;

	while (g != (rc_group *)NULL) {
		if (strcasecmp(group, g->name) == 0) {
			current_group = g;
			return;
		}
		g = g->next;
	}

	g = (rc_group *)malloc(sizeof(rc_group));
	g->name  = strdup(group);
	g->first = (rc_data *)NULL;
	g->last  = (rc_data *)NULL;
	g->next  = (rc_group *)NULL;
	if (groups_first == (rc_group *)NULL)
		groups_first = g;
	else
		groups_last->next = g;
	groups_last = g;
	current_group = g;
}


char *rc_read_str (char *key, char *def)
{
	rc_data *d;

	if (current_group != (rc_group *)NULL) {
		d = current_group->first;
		while (d != (rc_data *)NULL) {
			if (strcasecmp(key, d->name) == 0)
				return d->value;
			d = d->next;
		}
	}
	
	return def;
}


void rc_write_str (char *key, char *value)
{
	rc_data *d;

	if (current_group == (rc_group *)NULL)
		return;

	d = current_group->first;
	while (d != (rc_data *)NULL) {
		if (strcasecmp(key, d->name) == 0) {
			free(d->value);
			d->value = strdup(value);
			return;
		}
		d = d->next;
	}
	
	d = (rc_data *)malloc(sizeof(rc_data));
	d->name = strdup(key);
	d->value = strdup(value);
	d->next = (rc_data *)NULL;
	if (current_group->first == (rc_data *)NULL)
		current_group->first = d;
	else
		current_group->last->next = d;
	current_group->last = d;
}


int rc_read_int (char *key, int def)
{
	char *c;
	
	return ((c = rc_read_str(key, (char *)NULL)) == (char *)NULL) ? def : atoi(c);
}


void rc_write_int (char *key, int value)
{
	char buffer[12];

	sprintf(buffer, "%d", value);
	rc_write_str(key, buffer);
}


int rc_read_bool (char *key, int def)
{
	char *c;

	return ((c = rc_read_str(key, (char *)NULL)) == (char *)NULL) ? def : (strcasecmp(c, "true") == 0);
}


void rc_write_bool (char *key, int value)
{
	if (value)
		rc_write_str(key, "true");
	else
		rc_write_str(key, "false");
}
