/*
    GSrnd
    
    A control panel for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998,99 Oliver Gantz (o.gantz@tu-bs.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/

#include <gtk/gtk.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <srnd.h>
#include "setup_dialog.h"
#include "resources.h"


extern GtkTooltips *gsrnd_tooltips;


GSrndSetupPreferences gsrnd_setup_preferences;

int srnd_fd;



static int current_mode;

static GtkWidget *dbspeaker[4];
static GtkWidget *dbvolcw, *dbvolsw;
static GtkObject *dbvollr, *dbvolc, *dbvols;
static GtkObject *dbdelay;
static GtkWidget *tdeffect[6];
static GtkObject *tdlevel;
static GtkWidget *nmode;
static GtkObject *nlevel;


static GtkWidget *gsrnd_setup_dialog = (GtkWidget *)NULL;


void gsrnd_setup_load_preferences (void)
{
	rc_set_group("Dolby Pro Logic");
	gsrnd_setup_preferences.dbspeaker = rc_read_int("Speaker", 0);
	gsrnd_setup_preferences.dbvollr = rc_read_int("Volume L/R", 100);
	gsrnd_setup_preferences.dbvolc = rc_read_int("Volume C", 100);
	gsrnd_setup_preferences.dbvols = rc_read_int("Volume S", 100);
	gsrnd_setup_preferences.dbdelay = rc_read_int("Delay", 21);
	rc_set_group("3D-Phonic");
	gsrnd_setup_preferences.tdeffect = rc_read_int("Effect", 0);
	gsrnd_setup_preferences.tdlevel = rc_read_int("Level", 50);
	rc_set_group("Noise Generator");
	gsrnd_setup_preferences.nmode = rc_read_int("Mode", 0);
	gsrnd_setup_preferences.nlevel = rc_read_int("Level", 50);
}


void gsrnd_setup_save_preferences (void)
{
	rc_set_group("Dolby Pro Logic");
	rc_write_int("Speaker", gsrnd_setup_preferences.dbspeaker);
	rc_write_int("Volume L/R", gsrnd_setup_preferences.dbvollr);
	rc_write_int("Volume C", gsrnd_setup_preferences.dbvolc);
	rc_write_int("Volume S", gsrnd_setup_preferences.dbvols);
	rc_write_int("Delay", gsrnd_setup_preferences.dbdelay);
	rc_set_group("3D-Phonic");
	rc_write_int("Effect", gsrnd_setup_preferences.tdeffect);
	rc_write_int("Level", gsrnd_setup_preferences.tdlevel);
	rc_set_group("Noise Generator");
	rc_write_int("Mode", gsrnd_setup_preferences.nmode);
	rc_write_int("Level", gsrnd_setup_preferences.nlevel);
	rc_save();
}


void gsrnd_setup_default (void)
{
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dbspeaker[gsrnd_setup_preferences.dbspeaker]), TRUE);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvollr), (gfloat)gsrnd_setup_preferences.dbvollr * 0.1);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvolc), (gfloat)gsrnd_setup_preferences.dbvolc * 0.1);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvols), (gfloat)gsrnd_setup_preferences.dbvols * 0.1);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(dbdelay), (gfloat)gsrnd_setup_preferences.dbdelay);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tdeffect[gsrnd_setup_preferences.tdeffect]), TRUE);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(tdlevel), (gfloat)gsrnd_setup_preferences.tdlevel * 0.1);
	gtk_option_menu_set_history(GTK_OPTION_MENU(nmode), gsrnd_setup_preferences.nmode);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(nlevel), (gfloat)gsrnd_setup_preferences.nlevel * 0.1);

	gtk_widget_set_sensitive(dbvolcw, gsrnd_setup_preferences.dbspeaker != 2);
	gtk_widget_set_sensitive(dbvolsw, gsrnd_setup_preferences.dbspeaker != 3);
}	
	

void gsrnd_setup_init_mode (void)
{
	switch (current_mode) {
		case 0:	ioctl(srnd_fd, SRNDDBPL_MODE, gsrnd_setup_preferences.dbspeaker);
			ioctl(srnd_fd, SRNDDBPL_VOL_LR, (int)((double)gsrnd_setup_preferences.dbvollr * 0.16 + 0.5));
			ioctl(srnd_fd, SRNDDBPL_VOL_C, (int)((double)gsrnd_setup_preferences.dbvolc * 0.16 + 0.5));
			ioctl(srnd_fd, SRNDDBPL_VOL_S, (int)((double)gsrnd_setup_preferences.dbvols * 0.16 + 0.5));
			ioctl(srnd_fd, SRNDDBPL_DELAY, gsrnd_setup_preferences.dbdelay - 15);
			break;
		case 1:	ioctl(srnd_fd, SRND3DPHON_MODE, gsrnd_setup_preferences.tdeffect);
			ioctl(srnd_fd, SRND3DPHON_LEVEL, (int)((double)gsrnd_setup_preferences.tdlevel * 0.15 + 0.5));
			break;
		case 3:	ioctl(srnd_fd, SRNDNOISE_MODE, gsrnd_setup_preferences.nmode);
			ioctl(srnd_fd, SRNDNOISE_LEVEL, (int)((double)gsrnd_setup_preferences.nlevel * 0.09 + 0.5));
	}
}


void gsrnd_setup_set_mode (int mode)
{
	static int mode_inits[4] = {
		SRNDDBPL_INIT,
		SRND3DPHON_INIT,
		SRNDTHROUGH_INIT,
		SRNDNOISE_INIT
	};

	current_mode = mode;
	
	ioctl(srnd_fd, mode_inits[current_mode]);
	gsrnd_setup_init_mode();
}


void gsrnd_setup_set_volume (int volume)
{
	ioctl(srnd_fd, SRNDDAC_VOL, volume);
}


void gsrnd_setup_set_bypass (int bypass)
{
	ioctl(srnd_fd, SRNDBYPASS, bypass);
}


char *gsrnd_setup_get_version (char *buffer)
{
	int ver;

	if (ioctl(srnd_fd, SRNDVERSION, &ver) == 0)
		sprintf(buffer, "%d.%d", ver >> 8, ver & 0xff);
	else
		sprintf(buffer, "not found");

	return buffer;
}


void setup_dolby_speaker_callback (GtkWidget *widget, gpointer data)
{
	gsrnd_setup_preferences.dbspeaker = (int)data;
	if (current_mode == 0)
		ioctl(srnd_fd, SRNDDBPL_MODE, gsrnd_setup_preferences.dbspeaker);

	gtk_widget_set_sensitive(dbvolcw, gsrnd_setup_preferences.dbspeaker != 2);
	gtk_widget_set_sensitive(dbvolsw, gsrnd_setup_preferences.dbspeaker != 3);
}
	

void setup_dolby_vollr_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.dbvollr = (int)(adjustment->value * 10.0 + 0.5);

	if (current_mode == 0)
		ioctl(srnd_fd, SRNDDBPL_VOL_LR, (int)((double)gsrnd_setup_preferences.dbvollr * 0.16 + 0.5));
}


void setup_dolby_volc_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.dbvolc = (int)(adjustment->value * 10.0 + 0.5);

	if (current_mode == 0)
		ioctl(srnd_fd, SRNDDBPL_VOL_C, (int)((double)gsrnd_setup_preferences.dbvolc * 0.16 + 0.5));
}


void setup_dolby_vols_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.dbvols = (int)(adjustment->value * 10.0 + 0.5);

	if (current_mode == 0)
		ioctl(srnd_fd, SRNDDBPL_VOL_S, (int)((double)gsrnd_setup_preferences.dbvols * 0.16 + 0.5));
}


void setup_dolby_delay_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.dbdelay = (int)(adjustment->value + 0.5);

	if (current_mode == 0)
		ioctl(srnd_fd, SRNDDBPL_DELAY, gsrnd_setup_preferences.dbdelay - 15);
}


void setup_dolby_optimize_callback (GtkWidget *widget, gpointer data)
{
	int vlr, vc, vs, max;

	vlr = gsrnd_setup_preferences.dbvollr;
	vc  = gsrnd_setup_preferences.dbvolc;
	vs  = gsrnd_setup_preferences.dbvols;
	max = vlr;
	if ((gsrnd_setup_preferences.dbspeaker != 2) && (vc > max))
		max = vc;
	if ((gsrnd_setup_preferences.dbspeaker != 3) && (vs > max))
		max = vs;
	if ((max > 0) && (max < 100)) {
		gsrnd_setup_preferences.dbvollr = vlr * 100 / max;
		gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvollr), (gfloat)gsrnd_setup_preferences.dbvollr * 0.1);
		ioctl(srnd_fd, SRNDDBPL_VOL_LR, (int)((double)gsrnd_setup_preferences.dbvollr * 0.16 + 0.5));
		if (gsrnd_setup_preferences.dbspeaker != 2) {
			gsrnd_setup_preferences.dbvolc = vc * 100 / max;
			gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvolc), (gfloat)gsrnd_setup_preferences.dbvolc * 0.1);
			ioctl(srnd_fd, SRNDDBPL_VOL_C, (int)((double)gsrnd_setup_preferences.dbvolc * 0.16 + 0.5));
		}
		if (gsrnd_setup_preferences.dbspeaker != 3) {
			gsrnd_setup_preferences.dbvols = vs * 100 / max;
			gtk_adjustment_set_value(GTK_ADJUSTMENT(dbvols), (gfloat)gsrnd_setup_preferences.dbvols * 0.1);
			ioctl(srnd_fd, SRNDDBPL_VOL_C, (int)((double)gsrnd_setup_preferences.dbvolc * 0.16 + 0.5));
		}
	}
}


GtkWidget *gsrnd_dolby_frame_new (void)
{
	static char *speakers[4] = {
		"Normal",
		"Wide",
		"Phantom",
		"Dolby 3 Stereo"
	};
	int i;
	char buffer[40];
	GtkWidget *vbox;
	GtkWidget *table;
	GtkWidget *frame1, *frame2;
	GtkWidget *volume;
	GtkWidget *label;
	GtkWidget *button;
	GSList *group;
	

	frame1 = gtk_frame_new("Dolby Pro Logic settings");
	gtk_container_set_border_width(GTK_CONTAINER(frame1), 6);

	table = gtk_table_new(3, 5, FALSE);

	frame2 = gtk_frame_new("Speaker");
	gtk_container_set_border_width(GTK_CONTAINER(frame2), 6);

	vbox = gtk_vbox_new(FALSE, 0);
	group = (GSList *)NULL;
	for (i=0; i < 4; i++) {
		dbspeaker[i] = gtk_radio_button_new_with_label(group, speakers[i]);
		gtk_signal_connect(GTK_OBJECT(dbspeaker[i]), "clicked", GTK_SIGNAL_FUNC(setup_dolby_speaker_callback), (gpointer)i);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON(dbspeaker[i]));
		gtk_box_pack_start(GTK_BOX(vbox), dbspeaker[i], TRUE, TRUE, 0);
		gtk_widget_show(dbspeaker[i]);
		sprintf(buffer, "%s Speaker Arrangement", speakers[i]);
		gtk_tooltips_set_tip(gsrnd_tooltips, dbspeaker[i], buffer, "");
	}
	gtk_container_add(GTK_CONTAINER(frame2), vbox);
	gtk_widget_show(vbox);
	gtk_table_attach(GTK_TABLE(table), frame2, 0, 1, 0, 3, GTK_FILL, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(frame2);

	label = gtk_label_new("Left/Right");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	dbvollr = gtk_adjustment_new(0.0, 0.0, 10.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(dbvollr, "value_changed", GTK_SIGNAL_FUNC(setup_dolby_vollr_callback), NULL);
	volume = gtk_hscale_new(GTK_ADJUSTMENT(dbvollr));
	gtk_table_attach(GTK_TABLE(table), volume, 2, 3, 0, 1, GTK_FILL | GTK_EXPAND, 0, 6, 2);
	gtk_widget_show(volume);
	gtk_tooltips_set_tip(gsrnd_tooltips, volume, "Left/Right Channel Volume", "");

	label = gtk_label_new("Center");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	dbvolc = gtk_adjustment_new(0.0, 0.0, 10.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(dbvolc, "value_changed", GTK_SIGNAL_FUNC(setup_dolby_volc_callback), NULL);
	dbvolcw = gtk_hscale_new(GTK_ADJUSTMENT(dbvolc));
	gtk_table_attach(GTK_TABLE(table), dbvolcw, 4, 5, 0, 1, GTK_FILL | GTK_EXPAND, 0, 6, 2);
	gtk_widget_show(dbvolcw);
	gtk_tooltips_set_tip(gsrnd_tooltips, dbvolcw, "Center Channel Volume", "");

	label = gtk_label_new("Surround");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	dbvols = gtk_adjustment_new(0.0, 0.0, 10.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(dbvols, "value_changed", GTK_SIGNAL_FUNC(setup_dolby_vols_callback), NULL);
	dbvolsw = gtk_hscale_new(GTK_ADJUSTMENT(dbvols));
	gtk_table_attach(GTK_TABLE(table), dbvolsw, 2, 3, 1, 2, GTK_FILL | GTK_EXPAND, 0, 6, 2);
	gtk_widget_show(dbvolsw);
	gtk_tooltips_set_tip(gsrnd_tooltips, dbvolsw, "Surround Channel Volume", "");

	label = gtk_label_new("Delay");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 3, 4, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	dbdelay = gtk_adjustment_new(15.0, 15.0, 30.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(dbdelay, "value_changed", GTK_SIGNAL_FUNC(setup_dolby_delay_callback), NULL);
	volume = gtk_hscale_new(GTK_ADJUSTMENT(dbdelay));
	gtk_table_attach(GTK_TABLE(table), volume, 4, 5, 1, 2, GTK_FILL | GTK_EXPAND, 0, 6, 2);
	gtk_widget_show(volume);
	gtk_tooltips_set_tip(gsrnd_tooltips, volume, "Surround Channel Delay", "");

	button = gtk_button_new_with_label("Optimize");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(setup_dolby_optimize_callback), NULL);
	gtk_table_attach(GTK_TABLE(table), button, 1, 3, 2, 3, GTK_FILL | GTK_EXPAND, 0, 2, 2);
	gtk_widget_show(button);
	gtk_tooltips_set_tip(gsrnd_tooltips, button, "Optimize Channel Volumes", "");

	gtk_container_add(GTK_CONTAINER(frame1), table);
	gtk_widget_show(table);

	return frame1;
}


void setup_3dphonic_effect_callback (GtkWidget *widget, gpointer data)
{
	gsrnd_setup_preferences.tdeffect = (int)data;

	if (current_mode == 1)
		ioctl(srnd_fd, SRND3DPHON_MODE, gsrnd_setup_preferences.tdeffect);
}


void setup_3dphonic_level_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.tdlevel = (int)(adjustment->value * 10.0 + 0.5);

	if (current_mode == 1)
		ioctl(srnd_fd, SRND3DPHON_LEVEL, (int)((double)gsrnd_setup_preferences.tdlevel * 0.09 + 0.5));
}


GtkWidget *gsrnd_3dphonic_frame_new (void)
{
	static char *effects[6] = {
		"Standard",
		"Music/Sports",
		"Theater",
		"Strong",
		"Strong Theater 1",
		"Strong Theater 2"
	};
	int i;
	char buffer[32];
	GtkWidget *hbox1, *hbox2, *vbox;
	GtkWidget *frame1, *frame2;
	GtkWidget *volume;
	GtkWidget *label;
	GSList *group;
	
	frame1 = gtk_frame_new("3D-Phonic settings");
	gtk_container_set_border_width(GTK_CONTAINER(frame1), 6);

	hbox1 = gtk_hbox_new(FALSE, 0);

	frame2 = gtk_frame_new("Effect");
	gtk_container_set_border_width(GTK_CONTAINER(frame2), 6);
	vbox = gtk_vbox_new(FALSE, 0);
	group = (GSList *)NULL;
	for (i=0; i < 6; i++) {
		tdeffect[i] = gtk_radio_button_new_with_label(group, effects[i]);
		gtk_signal_connect(GTK_OBJECT(tdeffect[i]), "clicked", GTK_SIGNAL_FUNC(setup_3dphonic_effect_callback), (gpointer)i);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON(tdeffect[i]));
		gtk_box_pack_start(GTK_BOX(vbox), tdeffect[i], TRUE, TRUE, 0);
		gtk_widget_show(tdeffect[i]);
		sprintf(buffer, "%s Effect", effects[i]);
		gtk_tooltips_set_tip(gsrnd_tooltips, tdeffect[i], buffer, "");
		
	}
	gtk_container_add(GTK_CONTAINER(frame2), vbox);
	gtk_widget_show(vbox);

	gtk_box_pack_start(GTK_BOX(hbox1), frame2, FALSE, FALSE, 0);
	gtk_widget_show(frame2);

	vbox = gtk_vbox_new(FALSE, 0);
	hbox2 = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new("Effect Level");
	gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 4);
	gtk_widget_show(label);

	tdlevel = gtk_adjustment_new(0.0, 0.0, 10.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(tdlevel, "value_changed", GTK_SIGNAL_FUNC(setup_3dphonic_level_callback), NULL);
	volume = gtk_hscale_new(GTK_ADJUSTMENT(tdlevel));
	gtk_box_pack_end(GTK_BOX(hbox2), volume, TRUE, TRUE, 4);
	gtk_widget_show(volume);
	gtk_tooltips_set_tip(gsrnd_tooltips, volume, "Effect Level", "");

	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 10);
	gtk_widget_show(hbox2);

	gtk_box_pack_end(GTK_BOX(hbox1), vbox, TRUE, TRUE, 4);
	gtk_widget_show(vbox);

	gtk_container_add(GTK_CONTAINER(frame1), hbox1);
	gtk_widget_show(hbox1);

	return frame1;
}


void setup_noise_mode_callback (GtkWidget *widget, gpointer data)
{
	gsrnd_setup_preferences.nmode = (int)data;

	if (current_mode == 3)
		ioctl(srnd_fd, SRNDNOISE_MODE, gsrnd_setup_preferences.nmode);
}


void setup_noise_level_callback (GtkAdjustment *adjustment, gpointer data)
{
	gsrnd_setup_preferences.nlevel = (int)(adjustment->value * 10.0 + 0.5);

	if (current_mode == 3)
		ioctl(srnd_fd, SRNDNOISE_LEVEL, (int)((double)gsrnd_setup_preferences.nlevel * 0.09 + 0.5));
}


GtkWidget *gsrnd_noise_frame_new (void)
{
	static char *modes[5] = {
		"Off",
		"Left Channel",
		"Center Channel",
		"Right Channel",
		"Surround Channel"
	};
	int i;
	GtkWidget *frame;
	GtkWidget *hbox1, *hbox2, *vbox;
	GtkWidget *label;
	GtkWidget *menu, *item;
	GtkWidget *volume;
	

	frame = gtk_frame_new("Noise Generator settings");
	gtk_container_set_border_width(GTK_CONTAINER(frame), 6);

	hbox1 = gtk_hbox_new(FALSE, 0);
	vbox = gtk_vbox_new(FALSE, 0);
	hbox2 = gtk_hbox_new(FALSE, 0);

	label = gtk_label_new("Mode");
	gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 10);
	gtk_widget_show(label);

	menu = gtk_menu_new();
	for (i=0; i < 5; i++) {
		item = gtk_menu_item_new_with_label(modes[i]);
		gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(setup_noise_mode_callback), (gpointer)i);
		gtk_container_add(GTK_CONTAINER(menu), item);
		gtk_widget_show(item);
	}
	nmode = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(nmode), menu);
	gtk_box_pack_start(GTK_BOX(hbox2), nmode, FALSE, FALSE, 4);
	gtk_widget_show(nmode);
	gtk_tooltips_set_tip(gsrnd_tooltips, nmode, "Noise Mode", "");

	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 10);
	gtk_widget_show(hbox2);

	hbox2 = gtk_hbox_new(FALSE, 0);

	label = gtk_label_new("Noise Level");
	gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 10);
	gtk_widget_show(label);

	nlevel = gtk_adjustment_new(0.0, 0.0, 10.0, 1.0, 1.0, 0.0);
	gtk_signal_connect(nlevel, "value_changed", GTK_SIGNAL_FUNC(setup_noise_level_callback), NULL);
	volume = gtk_hscale_new(GTK_ADJUSTMENT(nlevel));
	gtk_box_pack_start(GTK_BOX(hbox2), volume, TRUE, TRUE, 4);
	gtk_widget_show(volume);
	gtk_tooltips_set_tip(gsrnd_tooltips, volume, "Noise Level", "");

	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 10);
	gtk_widget_show(hbox2);

	gtk_box_pack_start(GTK_BOX(hbox1), vbox, FALSE, FALSE, 0);
	gtk_widget_show(vbox);

	gtk_container_add(GTK_CONTAINER(frame), hbox1);
	gtk_widget_show(hbox1);
	
	return frame;
}


void gsrnd_setup_dialog_destroy (GtkWidget *widget, gpointer *data)
{
	gtk_widget_destroy(gsrnd_setup_dialog);
	gsrnd_setup_dialog = NULL;
}


void gsrnd_setup_dialog_ok (GtkWidget *widget, gpointer *data)
{
	gsrnd_setup_save_preferences();
	gtk_widget_hide(gsrnd_setup_dialog);
}


void gsrnd_setup_dialog_default (GtkWidget *widget, gpointer *data)
{
	gsrnd_setup_load_preferences();
	gsrnd_setup_default();
	gsrnd_setup_init_mode();
}


void gsrnd_setup_dialog_apply (GtkWidget *widget, gpointer *data)
{
	gsrnd_setup_save_preferences();
}


void gsrnd_setup_dialog_cancel (GtkWidget *widget, gpointer *data)
{
	gtk_widget_hide(gsrnd_setup_dialog);
}


void gsrnd_setup_dialog_create (void)
{
	GtkWidget *notebook;
	GtkWidget *frame, *label;
	GtkWidget *button;
	
	if (!gsrnd_setup_dialog) {
		gsrnd_setup_dialog = gtk_dialog_new();
		gtk_window_set_wmclass(GTK_WINDOW(gsrnd_setup_dialog), "setup_dialog", "GSrnd");
		gtk_window_set_title(GTK_WINDOW(gsrnd_setup_dialog), "GSrnd Mode Settings");
		gtk_signal_connect(GTK_OBJECT(gsrnd_setup_dialog), "destroy", GTK_SIGNAL_FUNC(gsrnd_setup_dialog_destroy), NULL);

		notebook = gtk_notebook_new();
	
		frame = gsrnd_dolby_frame_new();
		gtk_widget_show(frame);
		label = gtk_label_new("Dolby Pro Logic");
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);
	
		frame = gsrnd_3dphonic_frame_new();
		gtk_widget_show(frame);
		label = gtk_label_new("3D-Phonic");
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

		frame = gsrnd_noise_frame_new();
		gtk_widget_show(frame);
		label = gtk_label_new("Noise Generator");
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gsrnd_setup_dialog)->vbox), notebook, TRUE, TRUE, 0);
		gtk_widget_show(notebook);
	
		gtk_widget_show(GTK_DIALOG(gsrnd_setup_dialog)->vbox);


		button = gtk_button_new_with_label("OK");
		gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gsrnd_setup_dialog_ok), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gsrnd_setup_dialog)->action_area), button, TRUE, TRUE, 0);
		gtk_widget_grab_default(button);
		gtk_widget_show(button);

		button = gtk_button_new_with_label("Default");
		gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gsrnd_setup_dialog_default), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gsrnd_setup_dialog)->action_area), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

		button = gtk_button_new_with_label("Apply");
		gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gsrnd_setup_dialog_apply), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gsrnd_setup_dialog)->action_area), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

		button = gtk_button_new_with_label("Cancel");
		gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gsrnd_setup_dialog_cancel), NULL);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gsrnd_setup_dialog)->action_area), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

		gtk_widget_show(GTK_DIALOG(gsrnd_setup_dialog)->action_area);

		gsrnd_setup_default();
	}

	if (!GTK_WIDGET_VISIBLE(gsrnd_setup_dialog))
		gtk_widget_show(gsrnd_setup_dialog);
	else
		gdk_window_raise(gsrnd_setup_dialog->window);
}
