/*
    KSrnd
    
    A control panel for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998 Oliver Gantz (o.gantz@tu-bs.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/


#include "ksrndsetup.h"
#include "ksrndsetup.moc"

#include <kapp.h>
#include <kmsgbox.h>
#include <kiconloader.h>

#include <qtabdlg.h> // <qtabdialog.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qlabel.h>
#include <qwidget.h>
#include <qbttngrp.h> // <qbuttongroup.h>
#include <qlayout.h>
#include <qradiobt.h> // <qradiobutton.h>
#include <qslider.h>
#include <qpushbt.h> // <qpushbutton.h>
#include <qcombo.h> // <qcombobox.h>
#include <qtooltip.h>

#include <unistd.h>


#include "srnddev.h"


extern KApplication * mykapp;



KSrndSetup::KSrndSetup( QWidget * parent, char * devname ) : QTabDialog ( parent )
{
	KIconLoader *KIL;
	QPixmap pix;
	QSize size;
	QLabel *lablr, *labd, *label;
	int i, version;

	gotToolTips = false;

	KsConfig = mykapp->getConfig();
	currentmode = 0;

	KIL = mykapp->getIconLoader();
	pix = KIL->loadIcon("volume.xpm");

	page1 = new QWidget( this );
	page2 = new QWidget( this );
	page3 = new QWidget( this );
	addTab( page1, i18n("Dolby Pro Logic") );
	addTab( page2, i18n("3D-Phonic") );
	addTab( page3, i18n("Noise Generator") );

	/* Page 1 setup */
	dbspeaker = new QButtonGroup( i18n("Speaker"), page1 );
	QVBoxLayout *vbox = new QVBoxLayout( dbspeaker, 4 );
	vbox->addSpacing( dbspeaker->fontMetrics().height() );
	dbspeakerb[0] = new QRadioButton( i18n("&Normal"), dbspeaker );
	dbspeakerb[1] = new QRadioButton( i18n("&Wide"), dbspeaker );
	dbspeakerb[2] = new QRadioButton( i18n("&Phantom"), dbspeaker );
	dbspeakerb[3] = new QRadioButton( i18n("Dolby &3 Stereo"), dbspeaker );
	for (i=0; i < 4; i++) {
		vbox->addWidget( dbspeakerb[i] );
		dbspeakerb[i]->setMinimumSize( dbspeakerb[i]->sizeHint() );
	}
	vbox->activate();
	dbspeaker->move(10, 10);
	connect(dbspeaker, SIGNAL(clicked(int)), SLOT(setDBPLMode(int)));

	lablr = new QLabel( i18n("&Left/Right"), page1 );
	lablr->adjustSize();
	lablr->move(dbspeaker->x()+dbspeaker->width()+20, 10);

	dbvollr = new QSlider( 0, 16, 1, 0, QSlider::Horizontal, page1 );
	dbvollr->setTickInterval( 2 );
	dbvollr->setTickmarks( QSlider::Right );
	size = dbvollr->sizeHint();
	dbvollr->setGeometry(lablr->x(), lablr->y()+lablr->height()+5, 120, size.height());
	lablr->setBuddy(dbvollr);
	label = new QLabel( page1 );
	label->setPixmap(pix);
	label->adjustSize();
	label->move(dbvollr->x()+dbvollr->width()-label->width(), dbvollr->y()-14);
	connect(dbvollr, SIGNAL(valueChanged(int)), SLOT(setDBPLVolLR(int)));

	dblabc = new QLabel( i18n("&Center"), page1 );
	dblabc->adjustSize();
	dblabc->move(dbvollr->x()+dbvollr->width()+10, 10);

	dbvolc = new QSlider( 0, 16, 1, 0, QSlider::Horizontal, page1 );
	dbvolc->setTickInterval( 2 );
	dbvolc->setTickmarks( QSlider::Right );
	size = dbvolc->sizeHint();
	dbvolc->setGeometry(dblabc->x(), dbvollr->y(), 120, size.height());
	dblabc->setBuddy(dbvolc);
	dbpixc = new QLabel( page1 );
	dbpixc->setPixmap(pix);
	dbpixc->adjustSize();
	dbpixc->move(dbvolc->x()+dbvolc->width()-dbpixc->width(), dbvolc->y()-14);
	connect(dbvolc, SIGNAL(valueChanged(int)), SLOT(setDBPLVolC(int)));

	dblabs = new QLabel( i18n("&Surround"), page1 );
	dblabs->adjustSize();
	dblabs->move(lablr->x(), dbvollr->y()+dbvollr->height()+15);

	dbvols = new QSlider( 0, 16, 1, 0, QSlider::Horizontal, page1 );
	dbvols->setTickInterval( 2 );
	dbvols->setTickmarks( QSlider::Right );
	size = dbvols->sizeHint();
	dbvols->setGeometry(dblabs->x(), dblabs->y()+dblabs->height()+5, 120, size.height());
	dblabs->setBuddy(dbvols);
	dbpixs = new QLabel( page1 );
	dbpixs->setPixmap(pix);
	dbpixs->adjustSize();
	dbpixs->move(dbvols->x()+dbvols->width()-dbpixs->width(), dbvols->y()-14);
	connect(dbvols, SIGNAL(valueChanged(int)), SLOT(setDBPLVolS(int)));

	labd = new QLabel( i18n("&Delay"), page1 );
	labd->adjustSize();
	labd->move(dblabc->x(), dblabs->y());

	dbdelay = new QSlider( 0, 15, 1, 0, QSlider::Horizontal, page1 );
	dbdelay->setTickInterval( 3 );
	dbdelay->setTickmarks( QSlider::Right );
	size = dbdelay->sizeHint();
	dbdelay->setGeometry(labd->x(), dbvols->y(), 120, size.height());
	labd->setBuddy(dbdelay);
	dbdelayms =  new QLabel( "15 ms", page1 );
	dbdelayms->adjustSize();
	dbdelayms->move(dbdelay->x()+dbdelay->width()-dbdelayms->width(), labd->y());
	connect(dbdelay, SIGNAL(valueChanged(int)), SLOT(setDBPLDelay(int)));

	dboptimize = new QPushButton( i18n("&Optimize"), page1 );
	dboptimize->adjustSize();
	dboptimize->move(dbvols->x(), dbvols->y() + dbvols->height() + 15);
	connect(dboptimize, SIGNAL(clicked()), SLOT(DBPLOptimize()));

	/* Page 2 setup */  
	tdeffect = new QButtonGroup( i18n("Effect"), page2 );
	vbox = new QVBoxLayout( tdeffect, 4 );
	vbox->addSpacing( tdeffect->fontMetrics().height() );

	tdeffectb[0] = new QRadioButton( i18n("&Standard"), tdeffect );
	tdeffectb[1] = new QRadioButton( i18n("&Music/Sports"), tdeffect );
	tdeffectb[2] = new QRadioButton( i18n("&Theater"), tdeffect );
	tdeffectb[3] = new QRadioButton( i18n("Strong"), tdeffect );
	tdeffectb[4] = new QRadioButton( i18n("Strong Theater 1"), tdeffect );
	tdeffectb[5] = new QRadioButton( i18n("Strong Theater 2"), tdeffect );
	for (i=0; i < 6; i++) {
		vbox->addWidget( tdeffectb[i] );
		tdeffectb[i]->setMinimumSize( tdeffectb[i]->sizeHint() );
	}
	vbox->activate();
	tdeffect->move(10, 10);
	connect(tdeffect, SIGNAL(clicked(int)), SLOT(set3DPhonMode(int)));

	label = new QLabel( i18n("Effect &Level"), page2 );
	label->adjustSize();
	label->move(tdeffect->x()+tdeffect->width()+20, 10);

	tdlevel = new QSlider( 0, 15, 1, 0, QSlider::Horizontal, page2 );
	tdlevel->setTickInterval( 3 );
	tdlevel->setTickmarks( QSlider::Right );
	size = tdlevel->sizeHint();
	tdlevel->setGeometry(label->x(), label->y()+label->height()+5, 140, size.height());
	label->setBuddy(tdlevel);
	label = new QLabel( page2 );
	label->setPixmap(pix);
	label->adjustSize();
	label->move(tdlevel->x()+tdlevel->width()-label->width(), tdlevel->y()-14);
	connect(tdlevel, SIGNAL(valueChanged(int)), SLOT(set3DPhonLevel(int)));


	/* Page 3 setup */
	label = new QLabel( i18n("&Mode"), page3 );
	label->setAlignment(AlignVCenter);
	nmode = new QComboBox( false, page3 );
	nmode->insertItem(i18n("Off"), 0);
	nmode->insertItem(i18n("Left Channel"), 1);
	nmode->insertItem(i18n("Center Channel"), 2);
	nmode->insertItem(i18n("Right Channel"), 3);
	nmode->insertItem(i18n("Surround Channel"), 4);
	nmode->adjustSize();

	size = label->sizeHint();
	label->setGeometry(10,10,size.width(),nmode->height());
	label->setBuddy(nmode);
	nmode->setGeometry(label->x()+label->width()+6, 10, nmode->width()+6, nmode->height());
	connect(nmode, SIGNAL(activated(int)), SLOT(setNoiseMode(int)));

	label = new QLabel( i18n("Noise &Level"), page3 );
	label->adjustSize();
	label->move(10, nmode->y()+nmode->height()+15);

	nlevel = new QSlider( 0, 9, 1, 0, QSlider::Horizontal, page3 );
	nlevel->setTickInterval( 1 );
	nlevel->setTickmarks( QSlider::Right );
	size = nlevel->sizeHint();
	nlevel->setGeometry(10, label->y()+label->height()+5, nmode->x()+nmode->width()-10, size.height());
	label->setBuddy(nlevel);
	label = new QLabel( page3 );
	label->setPixmap(pix);
	label->adjustSize();
	label->move(nlevel->x()+nlevel->width()-label->width(), nlevel->y()-14);
	connect(nlevel, SIGNAL(valueChanged(int)), SLOT(setNoiseLevel(int)));


	setOKButton(i18n("OK"));
	setDefaultButton(i18n("Default"));
	setApplyButton(i18n("Apply"));
	setCancelButton(i18n("Cancel"));

	connect( this, SIGNAL(defaultButtonPressed()), SLOT(reset()) );
	connect( this, SIGNAL(applyButtonPressed()), SLOT(apply()) );

	page1->setFixedSize(dbvolc->x()+dbvolc->width()+10, tdeffect->y()+tdeffect->height()+10);
	page2->setFixedSize(page1->width(), page1->height());
	page3->setFixedSize(page1->width(), page1->height());
	setFixedSize(page1->width()+20, page1->height()+20);

	setCaption(i18n("KSrnd Mode Settings"));

	srnddev = new SrndDev(devname);
	if (!srnddev->isOpen) {
		QString emessage = i18n("Could not open SRND device.");
		emessage += "\n";
		emessage += srnddev->errorStr();
		KMsgBox::message(0, i18n("KSrnd Error"), emessage, KMsgBox::EXCLAMATION);
	}
	else {
		srnddev->getVersion(&version);
		if ((version >= 144) && (version <= 147)) { /* prevent use of SRND version 0.90 to 0.93 */ 
			QString emessage = i18n("Incompatible SRND driver version.\nPlease install at least SRND Version 0.94.");
			KMsgBox::message(0, i18n("KSrnd Error"), emessage, KMsgBox::EXCLAMATION);
			exit(-1);
		}
	}
}


KSrndSetup::~KSrndSetup()
{
	delete srnddev;
}


void KSrndSetup::addToolTips( bool on )
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(dbspeaker, i18n("Speaker Arrangement"));
		QToolTip::add(dbvollr, i18n("Left/Right Channel Volume"));
		QToolTip::add(dbvolc, i18n("Center Channel Volume"));
		QToolTip::add(dbvols, i18n("Surround Channel Volume"));
		QToolTip::add(dbdelay, i18n("Surround Channel Delay"));
		QToolTip::add(dboptimize, i18n("Optimize Channel Volumes"));
		QToolTip::add(tdeffect, i18n("3D-Phonic Effect"));
		QToolTip::add(tdlevel, i18n("Effect Level"));
		QToolTip::add(nmode, i18n("Noise Mode"));
		QToolTip::add(nlevel, i18n("Noise Level"));
	}
	else {
		QToolTip::remove(dbspeaker);
		QToolTip::remove(dbvollr);
		QToolTip::remove(dbvolc);
		QToolTip::remove(dbvols);
		QToolTip::remove(dbdelay);
		QToolTip::remove(dboptimize);
		QToolTip::remove(tdeffect);
		QToolTip::remove(tdlevel);
		QToolTip::remove(nmode);
		QToolTip::remove(nlevel);
	}
}


void KSrndSetup::getSrndVersion( char * ver )
{
	int v;
	
	if (srnddev->getVersion(&v) == 0)
        	sprintf(ver, "%d.%d", v >> 8, v & 0xff);
        else
                sprintf(ver, i18n("not found"));
}

void KSrndSetup::setMode( int mode )
{
	currentmode = mode;

	srnddev->setMode(currentmode);
	initMode();
}

void KSrndSetup::setVolume( int vol )
{
	srnddev->setVolume(vol);
}

void KSrndSetup::setBypass( bool on )
{
	srnddev->setBypass(on);
}

void KSrndSetup::setDBPLMode( int mode )
{
	dbvolc->setEnabled(mode != 2);
	dblabc->setEnabled(mode != 2);
	dbpixc->setEnabled(mode != 2);
	dbvols->setEnabled(mode != 3);
	dblabs->setEnabled(mode != 3);
	dbpixs->setEnabled(mode != 3);

	if (currentmode == 0)
		srnddev->setDBPLMode(mode);
}

void KSrndSetup::setDBPLVolLR( int vol )
{
	if (currentmode == 0)
		srnddev->setDBPLVolLR(vol);
}

void KSrndSetup::setDBPLVolC( int vol )
{
	if (currentmode == 0)
		srnddev->setDBPLVolC(vol);
}

void KSrndSetup::setDBPLVolS( int vol )
{
	if (currentmode == 0)
		srnddev->setDBPLVolS(vol);
}

void KSrndSetup::setDBPLDelay( int delay )
{
	char buf[6];
	
	sprintf(buf, "%2d ms", 15+delay);
	dbdelayms->setText(buf);

	if (currentmode == 0)
		srnddev->setDBPLDelay(delay);
}

void KSrndSetup::DBPLOptimize()
{
	int vlr, vc, vs, max;

	vlr = dbvollr->value();
	vc  = dbvolc->value();
	vs  = dbvols->value();
	max = vlr;
	if (dbvolc->isEnabled() && (vc > max))
		max = vc;
	if (dbvols->isEnabled() && (vs > max))
		max = vs;
	if ((max > 0) && (max < 16)) {
		dbvollr->setValue(vlr * 16 / max);
		if (dbvolc->isEnabled())
			dbvolc->setValue(vc * 16 / max);
		if (dbvols->isEnabled())
			dbvols->setValue(vs * 16 / max);
	}
}

void KSrndSetup::set3DPhonMode( int mode )
{
	if (currentmode == 1)
		srnddev->set3DPhonMode(mode);
}

void KSrndSetup::set3DPhonLevel( int level )
{
	if (currentmode == 1)
		srnddev->set3DPhonLevel(level);
}

void KSrndSetup::setNoiseMode( int mode )
{
	if (currentmode == 3)
		srnddev->setNoiseMode(mode);
}

void KSrndSetup::setNoiseLevel( int level )
{
	if (currentmode == 3)
		srnddev->setNoiseLevel(level);
}


void KSrndSetup::initMode()
{
	int i;

	switch (currentmode) {
		case 0:
			for (i=0; i < 4; i++) {
				if (dbspeakerb[i]->isChecked())
					setDBPLMode(i);
			}
			setDBPLVolLR(dbvollr->value());
			setDBPLVolC(dbvolc->value());
			setDBPLVolS(dbvols->value());
			setDBPLDelay(dbdelay->value());
			break;
		case 1:
			for (i=0; i < 6; i++) {
				if (tdeffectb[i]->isChecked())
					set3DPhonMode(i);
			}
			set3DPhonLevel(tdlevel->value());
			break;
		case 3:
			setNoiseMode(nmode->currentItem());
			setNoiseLevel(nlevel->value());
	}
}

void KSrndSetup::reset()
{
	int s;

	KsConfig->setGroup("Dolby Pro Logic");
	s = KsConfig->readNumEntry("Speaker", 0);
	dbspeakerb[s]->setChecked(true);
	dbvolc->setEnabled(s != 2);
	dblabc->setEnabled(s != 2);
	dbpixc->setEnabled(s != 2);
	dbvols->setEnabled(s != 3);
	dblabs->setEnabled(s != 3);
	dbpixs->setEnabled(s != 3);
	dbvollr->setValue(KsConfig->readNumEntry("Volume L/R", 16));
	dbvolc->setValue(KsConfig->readNumEntry("Volume C", 16));
	dbvols->setValue(KsConfig->readNumEntry("Volume S", 16));
	dbdelay->setValue(KsConfig->readNumEntry("Delay", 8));

	KsConfig->setGroup("3D-Phonic");
	tdeffectb[KsConfig->readNumEntry("Effect", 0)]->setChecked(true);;
	tdlevel->setValue(KsConfig->readNumEntry("Level", 8));

	KsConfig->setGroup("Noise");
	nmode->setCurrentItem(KsConfig->readNumEntry("Mode", 0));
	nlevel->setValue(KsConfig->readNumEntry("Level", 5));
	
	initMode();
}

void KSrndSetup::apply()
{
	int i;

	KsConfig->setGroup("Dolby Pro Logic");
	for (i=0; i < 4; i++)
		if (dbspeakerb[i]->isChecked())
			KsConfig->writeEntry("Speaker", i);
	KsConfig->writeEntry("Volume L/R", dbvollr->value());
	KsConfig->writeEntry("Volume C", dbvolc->value());
	KsConfig->writeEntry("Volume S", dbvols->value());
	KsConfig->writeEntry("Delay", dbdelay->value());

	KsConfig->setGroup("3D-Phonic");
	for (i=0; i < 6; i++)
		if (tdeffectb[i]->isChecked())
			KsConfig->writeEntry("Effect", i);
	KsConfig->writeEntry("Level", tdlevel->value());

	KsConfig->setGroup("Noise");
	KsConfig->writeEntry("Mode", nmode->currentItem());
	KsConfig->writeEntry("Level", nlevel->value());

	KsConfig->sync();
}
