/****************************************************************************************
*  kmp3.cpp										*
*											*
*  Created by 										*
*	Cajus Pollmeier (Cajus.Pollmeier@mfh-iserlohn.de)				*
*	SMF aka Antoine Laydier (laydier@usa.net)					*
*											*
*  This program is free software; you can redistribute it and/or modify			*
*  it under the terms of the GNU General Public License as published by			*
*  the Free Software Foundation; either version 2 of the License, or			*
*  (at your option) any later version.							*
*											*
*  This program is distributed in the hope that it will be useful,			*
*  but WITHOUT ANY WARRANTY; without even the implied warranty of			*
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
*  GNU General Public License for more details.						*
*											*
*  You should have received a copy of the GNU General Public License			*
*  along with this program; if not, write to the Free Software				*
*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
****************************************************************************************/

// QT-Headers
#include <qapp.h>
#include <qslider.h>
#include <qobject.h>
#include "docking.h"
#include "kmp3.h"
#include "bwlednum.h"
#include "musicBrowser.h"

// Modified by SMF
#include "configPlayer.h"
#include "configDisplay.h"
// Added by CP
#include "configMisc.h"

// Added by ANL
#include "ObscureVisApp.h"

#include "player.h"
#include "playlist.h"
#include <stdio.h>

#include <X11/Xlib.h>

// Bitmaps
#include "bitmaps/playpaus.xbm"
#include "bitmaps/stop.xbm"
#include "bitmaps/repeat.xbm"
#include "bitmaps/repeat1.xbm"
#include "bitmaps/nexttrk.xbm"
#include "bitmaps/prevtrk.xbm"
#include "bitmaps/ff.xbm"
#include "bitmaps/rew.xbm"
#include "bitmaps/info.xbm"
#include "bitmaps/poweroff.xbm"
#include "bitmaps/eject.xbm"
#include "bitmaps/logo.xbm"
#include "bitmaps/shuffle.xbm"
#include "bitmaps/options.xbm"


/****************************************************************************************
* 					GLOBALS						*
****************************************************************************************/
DockWidget *dock_widget;

ObscureVisApp *a;
kmp3 *k;

cfg_struct cfgData;
QStrList *playlist;
bool commandLine;

char *errMessages[]={   "Socket communication failed",
                        "Cant communicate with player",
                        "Cant access audio device",
                        "Cant change priority",
                        "No mpeg input file exists",
                        "Internel player error",
                        "Player died, sorry",
                        "There is no song in playlist"
                      };


/* Implementation of class kmp3 *********************************************************
*  modified:	21Sep98 by Cp								*
*		30Sep98 by SMF								*
*		03Okt98 by Cp								*
*		22Okt98 by Cp                                                           *
*               22Dec98 by SMF                                                          *
****************************************************************************************/


void kmp3::stripNames(char *to, char *from ){
  char *temp= to;

  while( *from ){
    if( ( *from == '/' ) ){
      from++;
      temp= to;
    }
    else
      *temp++= *from++;
  }

  *temp= 0;
}


void kmp3::stripExtension( char *from ){
  int i;
  char *temp= from;

  if( strstr( (const char*)from, "." ) != NULL ){
    i= strlen( from )-1;
    while( temp[i] != '.' ){
      temp[i--]= 0;
    }
    temp[i]= 0;
  }

}


// Draw the panel
void kmp3::drawPanel(){
  int ix = 0;
  int iy = 0;
  const int WIDTH = 86;
  const int HEIGHT = 27;
  const int SBARWIDTH = 151;
  
  QBitmap playBmp( playpause_width, playpause_height, playpause_bits,TRUE );
  QBitmap stopBmp( stop_width, stop_height, stop_bits, TRUE );
  QBitmap prevBmp( prevtrk_width, prevtrk_height, prevtrk_bits, TRUE );
  QBitmap nextBmp( nexttrk_width, nexttrk_height, nexttrk_bits, TRUE );
  QBitmap repeatBmp( repeat_width, repeat_height, repeat_bits, TRUE );
  QBitmap repeat1Bmp( repeat1_width, repeat1_height, repeat1_bits, TRUE );
  QBitmap forwBmp( ff_width, ff_height, ff_bits, TRUE );
  QBitmap backBmp( rew_width, rew_height, rew_bits, TRUE );
  QBitmap ejectBmp( eject_width, eject_height, eject_bits, TRUE );
  QBitmap infoBmp( info_width, info_height,info_bits, TRUE );
  QBitmap powerBmp( poweroff_width, poweroff_height, poweroff_bits, TRUE );
  QBitmap shuffleBmp( shuffle_width, shuffle_height, shuffle_bits, TRUE );
  QBitmap logoBmp( logo_width, logo_height, logo_bits, TRUE );
  QBitmap configBmp( options_width, options_height, options_bits, TRUE );

  setCaption( APP_NAME );

  logoPB= makeButton( ix, iy, WIDTH+2, 2*HEIGHT );
  logoPB->setPixmap( logoBmp );
	
  iy+= 2*HEIGHT;
  ejectPB= makeButton( ix, iy, WIDTH/2+1, HEIGHT );
  ejectPB->setPixmap( ejectBmp );
  
  ix+= WIDTH/2;
  infoPB= makeButton( ix+1, iy, WIDTH/2+1, HEIGHT );
  infoPB->setPixmap( infoBmp );
	
  ix= 0; iy+= HEIGHT;
  powerPB= makeButton( ix, iy, WIDTH+2, HEIGHT );	
  powerPB->setPixmap( powerBmp );

  ix+= WIDTH+2;
  volSB = new QSlider( 0, 100, 5,  50, QSlider::Horizontal, this, "Slider" );
  volSB->setGeometry( ix, iy, SBARWIDTH, HEIGHT/2 );
  volSB->setValue(100);

  iy+= HEIGHT/2;
  blankPB= makeButton( ix, iy, SBARWIDTH, HEIGHT/2+1 );	
//  blankBar = new QWidget(this);
//  blankBar->setGeometry( ix, iy, SBARWIDTH, HEIGHT/2+1 );
	
  ix+= SBARWIDTH;
  iy-= HEIGHT/2;
  shufflePB= makeButton( ix, iy, 38, HEIGHT );
  shufflePB->setPixmap( shuffleBmp );

  ix+= 38;
  configPB= makeButton( ix, iy, 38, HEIGHT );
  configPB->setPixmap( configBmp );

  ix+= 38;
  prevPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  prevPB->setPixmap( prevBmp );

  iy-= HEIGHT;
  backPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  backPB->setPixmap( backBmp );

  iy-= HEIGHT;
  stopPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  stopPB->setPixmap( stopBmp );

  iy-= HEIGHT;
  playPB= makeButton( ix, iy, WIDTH, HEIGHT );
  playPB->setPixmap( playBmp );

  ix+= WIDTH/2; iy+= HEIGHT;
  repeatPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  repeatPB->setPixmap( repeatBmp ); 

  iy+= HEIGHT;
  forwPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  forwPB->setPixmap( forwBmp );

  iy+= HEIGHT;
  nextPB= makeButton( ix, iy, WIDTH/2, HEIGHT );
  nextPB->setPixmap( nextBmp );

  backdrop = new QFrame(this);
  backdrop->setGeometry( WIDTH+2, 0, SBARWIDTH + 2*38, 3* HEIGHT );

  for (int u = 0; u<5;u++){
    trackTimeLED[u] = new BW_LED_Number(this );
    trackTimeLED[u]->setGeometry( WIDTH + 6 + u*20, 10, 23, 6* HEIGHT /5);
    trackTimeLED[u]->setLEDoffColor(cfgData.background_color);
    trackTimeLED[u]->setLEDColor(cfgData.led_color,cfgData.background_color);
  }

  QString zeros("--:--");
  setLEDs(zeros);

  statusLabel = new QLabel( this );
  statusLabel->setGeometry( 2*SBARWIDTH - 96 , 5, 60, 15 );
  statusLabel->setAlignment( AlignLeft );
  statusLabel->setText(klocale->translate("Ready"));

  volumeLabel = new QLabel( this );
  volumeLabel->setGeometry( 2*SBARWIDTH - 96 , 33, 50, 15 );
  volumeLabel->setAlignment( AlignLeft );
  volumeLabel->setText("Vol:100%");

  bitrateLabel = new QLabel( this );
  bitrateLabel->setGeometry( 2*SBARWIDTH - 96 , 19, 50, 15 );
  bitrateLabel->setAlignment( AlignLeft );
  bitrateLabel->setText("- kBit/s");

  timeLabel = new QLabel( this );
  timeLabel->setGeometry( 2*SBARWIDTH - 34 , 5, 40, 15 );
  timeLabel->setAlignment( AlignLeft );
  timeLabel->setText("--:--");

  trackLabel = new QLabel( this );
  trackLabel->setGeometry( 2*SBARWIDTH - 34 , 33, 40, 15 );
  trackLabel->setAlignment( AlignLeft );
  trackLabel->setText("-/-");

  freqLabel = new QLabel( this );
  freqLabel->setGeometry( 2*SBARWIDTH - 34 , 19, 40, 15 );
  freqLabel->setAlignment( AlignLeft );
  freqLabel->setText("- kHz");

  artistLabel = new QScrLabel( this );
  artistLabel->setGeometry( WIDTH + 10 , 46, SBARWIDTH + 2*38 -20, 15 );
  artistLabel->setAlignment( AlignLeft );
  artistLabel->setLabel(klocale->translate("No title loaded"));

  titleLabel = new QScrLabel( this );
  titleLabel->setGeometry( WIDTH + 10 , 62, SBARWIDTH + 2*38 - 20, 15 );
  titleLabel->setAlignment( AlignLeft );
  titleLabel->setLabel("");

  repeatLED = new QLabel(this);
  repeatLED->hide();
  repeatLED->setPixmap(repeatBmp);
  repeatLED->setGeometry( 2*SBARWIDTH-repeatBmp.width()+8, 44, repeatBmp.width(),repeatBmp.height());

  repeat1LED = new QLabel(this);
  repeat1LED->hide();
  repeat1LED->setPixmap(repeat1Bmp);
  repeat1LED->setGeometry( 2*SBARWIDTH-repeat1Bmp.width()+8, 44, repeat1Bmp.width(),repeat1Bmp.height());

  shuffleLED = new QLabel(this);
  shuffleLED->hide();
  shuffleLED->setPixmap(shuffleBmp);
  shuffleLED->setGeometry( 2*SBARWIDTH-shuffleBmp.width()+8, 60, shuffleBmp.width(),shuffleBmp.height());
}



// Easy makeing of buttons
QPushButton *kmp3::makeButton( int x, int y, int w, int h ){

  QPushButton *pb = new QPushButton( this );
  pb->setGeometry( x, y, w, h );
	
  return pb;
}


// Sets the time LEDs to the value of "symbols"
void kmp3::setLEDs(QString symbols){

  if(symbols.length() < 5) return;

  for(int i=0;i<5;i++)
    trackTimeLED[i]->display(symbols.data()[i]);
}        


// Actually sets the colors of most widgets
void kmp3::setColors(){

  backdrop->setBackgroundColor(cfgData.background_color);
  QColorGroup colgrp( cfgData.led_color, cfgData.background_color, cfgData.led_color,
			    cfgData.led_color , cfgData.led_color, cfgData.led_color, white );

  titleLabel ->setPalette( QPalette(colgrp,colgrp,colgrp) );
  artistLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  volumeLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  statusLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  bitrateLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  freqLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  trackLabel ->setPalette( QPalette(colgrp,colgrp,colgrp) );
  timeLabel->setPalette( QPalette(colgrp,colgrp,colgrp) );
  repeatLED->setPalette( QPalette(colgrp,colgrp,colgrp) );
  repeat1LED->setPalette( QPalette(colgrp,colgrp,colgrp) );
  shuffleLED->setPalette( QPalette(colgrp,colgrp,colgrp) );

  for (int u = 0; u< 5;u++){
    trackTimeLED[u]->setLEDoffColor(cfgData.background_color);
    trackTimeLED[u]->setLEDColor(cfgData.led_color,cfgData.background_color);
  }

  titleLabel ->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  artistLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  volumeLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  statusLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  trackLabel ->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  timeLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  bitrateLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
  freqLabel->setFont( QFont( "Helvetica", 10, QFont::Bold) );
}
        

// Activates docking of applications
void kmp3::setDocking(){
  static int temp= WAIT;
  if( dock_widget != NULL ){
    temp= dock_widget->stat;
    delete dock_widget;
  }
	
  dock_widget = new DockWidget("dockw");
       
  if( cfgData.docking ){ 
    dock_widget->dock();
    dock_widget->stat= temp;
  }

  allreadySet= false;

}


// Shows the config box
void kmp3::showConfigBox(){

  QTabDialog *tabdialog;
  tabdialog = new QTabDialog(0,"tabdialog",TRUE);
  tabdialog->setCaption( klocale->translate("KMP3 Configuration") );
  tabdialog->resize( 400, 450 );
  tabdialog->setCancelButton();

  QWidget *about = new QWidget(tabdialog);
  QGroupBox *box = new QGroupBox(about,"box");
  QLabel *label = new QLabel(box,"label");
  QLabel *hometext= new QLabel(box,"hometext");
  QString labelstring, homestring;

  box->setGeometry(10,10,365,360);
  label->setGeometry(160,30,190,300);

  // Modified by SMF (21/09/98)
  // to change version modify configure.in and run autoconf
  labelstring.sprintf("%s v%s\n",APP_NAME, VERSION);
  labelstring +="Copyright (c) 1998"\
		 "\n\nAlphabetical list of authors:"\
                 "\n\nAntoine Laydier\n"\
                 "smf@mail.dotcom.fr"\
                 "\n\nAaron Levinson\n"\
		 "alevinsn@inetarena.com"\
                 "\n\nCajus Pollmeier\n"\
                 "i00160@urmel.mfh-iserlohn.de"\
		 "\n\nThierry Vergote\n"\
		 "Thierry.Vergote@rug.ac.be";

  label->setAlignment(AlignLeft|WordBreak);
  label->setText(labelstring.data());

  hometext->setGeometry(10,300,340,50);
  homestring = klocale->translate("Visit the KMP3 Homepage for bugfixes or new versions\nat http://elon05.mfh-iserlohn.de/Pollmeier/kmp3");

  hometext->setAlignment(AlignCenter|WordBreak);
  hometext->setText(homestring.data());

  QString pixdir = a->kde_datadir() + QString("/kmp3/pics/");

  QPixmap pm((pixdir + "logo.xpm").data());
  QLabel *logo = new QLabel(box);
  logo->setPixmap(pm);
  logo->setGeometry(15, 30, pm.width(), pm.height()); 

  // Modified by SMF
  configPlayer *cfg1= new configPlayer( tabdialog, &cfgData, "cfgPlayer" );
  configDisplay *cfg2= new configDisplay( tabdialog, &cfgData, "cfgDisplay" );
  // Modified by CP
  configMisc *cfg3= new configMisc( tabdialog, &cfgData, "cfgMisc" );

  tabdialog->addTab(cfg2,klocale->translate("Display") );
  tabdialog->addTab(cfg1,klocale->translate("MPEG Player") );
  tabdialog->addTab(cfg3,klocale->translate("Misc"));
  tabdialog->addTab(about,klocale->translate("About"));

  tabdialog->setMinimumSize( 400, 450 );
  
  if(tabdialog->exec() == QDialog::Accepted){
    cfgData.led_color= cfg2->led_color;
    cfgData.background_color= cfg2->background_color;
    cfgData.tooltips= cfg2->tooltips;
    cfgData.line[0].setStr(cfg2->line[0].data());
    cfgData.line[1].setStr(cfg2->line[1].data());
    cfgData.line[2].setStr(cfg2->line[2].data());
    cfgData.loadLast= cfg1->loadLast;
    cfgData.docking= cfg2->docking; 
    cfgData.stripname= cfg2->stripname;
    cfgData.stripend= cfg2->stripend;
    cfgData.fuzzy= cfg3->fuzzy;
    cfgData.sensitive= cfg3->sensitive;
    cfgData.step= cfg3->step;
    cfgData.autoStart= cfg1->autostart;
    cfgData.autoStartCMD= cfg1->autostartcmd;
    // Added by SMF (12/23/98)
    cfgData.startShuffle= cfg1->startShuffle;
    cfgData.startLoop= cfg1->startLoop;

    if( ( cfgData.buffers != cfg1->buffers ) ||
	( cfgData.stereo != cfg1->stereo )   ||
        ( cfgData.mixdown != cfg1->mixdown ) ){ 

      cfgData.buffers= cfg1->buffers;
      cfgData.stereo= cfg1->stereo;
      cfgData.mixdown= cfg1->mixdown;
      int temp= plr->getIndex() + 1;
      reinitializePlayer();
      setPlayer();
      plr->setSongNr( temp );
    }

    cfgData.buffers= cfg1->buffers;
    cfgData.stereo= cfg1->stereo;
    cfgData.mixdown= cfg1->mixdown;
    mb->setFuzzy( cfgData.fuzzy );
    mb->setSensitive( cfgData.sensitive );
    mb->setStep( cfgData.step ); 
    setColors();
    setToolTips();
    setDocking();
    updateSong();
  }
	
  delete cfg1;
  delete cfg2;
  delete cfg3;
  delete about;
  delete tabdialog;
}


// if the info button is beeing pressed
void kmp3::infoPressed() 
{ 
  if( errorFlag )
    return;

  if( playlist->count() != 0 ){
    // Modified by SMF
    dock_widget->stat= WAIT;
    tag->setFiles(*playlist);
    tag->at(plr->getSongNr() );
    tag->exec();
    updateSong();						
  }
}


// set the volume label
void kmp3::setVolume( int value ){
  QString volString;

  volString.sprintf( "Vol:%02d%%", value );
  volumeLabel->setText( volString );
  cfgData.volume= value;
}


// This happens when pressing the kmp3 logo
void kmp3::logoPressed(){
  if( errorFlag )
    return;

  mb->exec();
  plr->setBrowserMode( false );
}


// Launch the playlist
void kmp3::ejectPressed(){
  if( errorFlag )
    return;

  // Modified by SMF (22/09/98)
  if( playlistdlg->exec() ) {
    if( playlist != NULL ) delete playlist;
    playlist = new QStrList( *(playlistdlg->songlist) );
    reinitializePlayer();
    plr->initCurentSong();
    updateSong();
  }
}


// Leave the player
void kmp3::powerPressed(){
  statusLabel->setText( klocale->translate("Wait") );
	plr->stop();			
	a->processEvents();
	a->flushX();

	writeSettings();

        // Added by SMF (22/09/98)
        delete tag;
        playlistdlg->saveDefault();
        delete playlistdlg;

	plr->quit();
	a->quit();
}


// Handle if shuffle pressed
void kmp3::shufflePressed(){
  if( errorFlag )
    return;

  plr->setShuffleFlag( !plr->getShuffleFlag() );

  if( plr->getPauseFlag() )
    plr->stop();

  if( plr->getShuffleFlag() )
    shuffleLED->show();
  else
    shuffleLED->hide();

  if( plr->getPlayingFlag() )
    plr->play();
  else
    plr->initCurentSong();
  
}


// Show the config Box
void kmp3::configPressed(){
  if( errorFlag )
    return;

  showConfigBox();	
}


void kmp3::playPressed(){
  if( errorFlag ){
    errorFlag= FALSE;
    reinitializePlayer();
    return;
  }

  if( playlist->count() ){
    if( !plr->getPlayingFlag() ){
      plr->play();
      if( dock_widget != NULL ){
        dock_widget->stat= PLAY;
      }
      statusLabel->setText(klocale->translate("Play"));
    }
    else{
      plr->pause();
      if( plr->getPauseFlag() ){
        statusLabel->setText(klocale->translate("Pause"));
        if( dock_widget != NULL ){
          dock_widget->stat= WAIT;
        }
      }
      else{
        statusLabel->setText(klocale->translate("Play"));
        if( dock_widget != NULL ){
          dock_widget->stat= PLAY;
        }
      }
    }
  }
}


void kmp3::stopPressed(){
  if( errorFlag )
    return;

  if( dock_widget != NULL )
    dock_widget->stat= WAIT;

  plr->stop();
  setLEDs( "--:--" );
  statusLabel->setText(klocale->translate("Ready"));
}


void kmp3::nextPressed(){
  if( errorFlag )
    return;

  plr->nextSong();
}


void kmp3::prevPressed(){
  if( errorFlag )
    return;

  plr->prevSong();
}



void kmp3::repeatPressed(){
  if( errorFlag )
    return;

  plr->cycleUpRepeatFlag();

  switch (plr->getRepeatFlag()) {
  case REPEAT_NO:
    repeat1LED->hide();
    repeatLED->hide(); // just in case
    break;
  case REPEAT_ALL:
    repeatLED->show();
    repeat1LED->hide(); // just in case
    break;
  case REPEAT_ONE:
    repeat1LED->show();
    repeatLED->hide(); // just in case
    break;
  }
}



// Avoid terminating by the window close button
void kmp3::closeEvent( QCloseEvent *e ){
  /*

  if( cfgData.docking ){

    if(dock_widget)
      dock_widget->SaveKmp3Position();

    this->hide();
  }
  */
  statusLabel->setText(klocale->translate("Wait"));
  plr->stop();			
  a->processEvents();
  a->flushX();
  
  writeSettings();
  
  // Added by SMF (22/09/98)
  delete tag;
  playlistdlg->saveDefault();
  delete playlistdlg;
  
  e->accept();

  plr->quit();
  a->quit();
}


// Configures the player
void kmp3::setPlayer(){
  plr->initCurentSong();
}


void kmp3::readSettings(){
  config = a->getConfig();

  config->setGroup("GENERAL");
  cfgData.tooltips  = (bool) config->readNumEntry("ToolTips",1);

  QColor defaultback = black;
  QColor defaultled = QColor(226,224,255);
  cfgData.background_color = config->readColorEntry("BackColor",&defaultback);
  cfgData.led_color = config->readColorEntry("LEDColor",&defaultled);
  cfgData.stripname = config->readNumEntry( "StripFileName", 1 );
  cfgData.stripend = config->readNumEntry( "StripExtension", 1 );
  cfgData.docking = config->readNumEntry( "PanelDocking", 1 );

  // Added by SMF
  cfgData.line[0] = config->readEntry("Line1","%1");
  cfgData.line[1] = config->readEntry("Line2","%2");
  cfgData.line[2] = config->readEntry("DockLine","%1 - %2");

  config->setGroup("PLAYER");
  cfgData.volume = config->readNumEntry("Volume",80);
  cfgData.loadLast = config->readNumEntry("LoadLast",1);
  cfgData.stereo= (bool) config->readNumEntry("Stereo",1);
  cfgData.autoStart= (bool) config->readNumEntry("Autostart",0);
  cfgData.autoStartCMD= (bool) config->readNumEntry("AutostartCMD",1);
  cfgData.mixdown= !(bool)config->readNumEntry("FullMixing",1);
  cfgData.buffers= config->readNumEntry("Buffers",64);
  cfgData.LastFile = config->readEntry("LastFile","");
  // Added by SMF (12/23/98)
  cfgData.startShuffle= (bool)config->readNumEntry("StartShuffle",0);
  cfgData.startLoop= config->readNumEntry("StartLoop",REPEAT_NO);

  if( cfgData.buffers < 64 )
    cfgData.buffers= 64;

  config->setGroup("MISC");
  cfgData.fuzzy = config->readNumEntry("FuzzySearch",1);
  cfgData.sensitive= config->readNumEntry("CaseSensitive",0);;
  cfgData.step= config->readNumEntry("SearchMode", 1 );

  // Added by SMF (22/12/98)
  config->setGroup("PLAYLIST");
  cfgData.pl_hidden = config->readNumEntry("Hidden",0);
  cfgData.pl_path = config->readNumEntry("SavePath",1);;
  cfgData.pl_m3u= config->readNumEntry("AppendM3U",1);
}

void kmp3::writeSettings(){
  config = a->getConfig();

  config->setGroup("GENERAL");
  writeBool( "ToolTips", cfgData.tooltips );
  writeBool( "PanelDocking", cfgData.docking );	
  writeBool( "StripFileName", cfgData.stripname ); 
  writeBool( "StripExtension", cfgData.stripend );
  config->writeEntry("BackColor",cfgData.background_color);
  config->writeEntry("LEDColor",cfgData.led_color);
  config->writeEntry("Line1",cfgData.line[0]);
  config->writeEntry("Line2",cfgData.line[1]);
  config->writeEntry("DockLine",cfgData.line[2]);
	
  config->setGroup("PLAYER");
  config->writeEntry("Volume", cfgData.volume);
  writeBool( "Stereo", cfgData.stereo );
  writeBool( "LoadLast", cfgData.loadLast );
  writeBool( "FullMixing", cfgData.mixdown );
  writeBool( "Autostart", cfgData.autoStart );
  writeBool( "AutostartCMD", cfgData.autoStartCMD );

  config->writeEntry("Buffers", cfgData.buffers);	
  config->writeEntry("LastFile", cfgData.LastFile);
  writeBool( "StartShuffle", cfgData.startShuffle );
  config->writeEntry("StartLoop", cfgData.startLoop);

  config->setGroup("MISC");
  writeBool( "FuzzySearch", cfgData.fuzzy );
  writeBool( "CaseSensitive", cfgData.sensitive );
  config->writeEntry( "SearchMode", cfgData.step );

  // Added by SMF (22/12/98)
  config->setGroup("PLAYLIST");
  writeBool( "Hidden", cfgData.pl_hidden );
  writeBool( "SavePath", cfgData.pl_path );
  writeBool( "AppendM3U", cfgData.pl_m3u );

  config->sync();
}


void kmp3::writeBool( const char text[], bool value ){
  if( value )
    config->writeEntry( text, 1 );
  else
    config->writeEntry( text, 0 ); 
}


QString kmp3::getHomeDir() {

  struct passwd *pwd;
  pwd = getpwuid(getuid());

  if(pwd == NULL)
    return QString("/");

  QString s = pwd->pw_dir;

  if(s.right(1) != "/")
    s += "/";

  return s;
}


void kmp3::makedirs(){
  QDir dir;
  QString d;

  d = getHomeDir();

  d += "/.kde";
  dir.setPath(d.data());

  if(!dir.exists()){
    dir.mkdir(d.data());
    chown(d.data(),getuid(),getgid());
    chmod(d.data(),S_IRUSR | S_IWUSR | S_IXUSR);
  }

  d += "/share";
  dir.setPath(d.data());

  if(!dir.exists()){
    dir.mkdir(d.data());
    chown(d.data(),getuid(),getgid());
    chmod(d.data(),S_IRUSR | S_IWUSR | S_IXUSR);
  }

  d += "/apps";
  dir.setPath(d.data());

  if(!dir.exists()){
    dir.mkdir(d.data());
    chown(d.data(),getuid(),getgid());
    chmod(d.data(),S_IRUSR | S_IWUSR | S_IXUSR);
  }

  d += "/kmp3" ;

  dir.setPath(d.data());

  if(!dir.exists()){
    dir.mkdir(d.data());
    chown(d.data(),getuid(),getgid());
    chmod(d.data(),S_IRUSR | S_IWUSR | S_IXUSR);
  } 

}



kmp3::kmp3( QWidget *parent, const char *name ) : QDialog( parent, name ){
  makedirs();
  drawPanel();
  readSettings();
  setColors();
  setToolTips();
  
  connect( logoPB,   SIGNAL(clicked()), SLOT(logoPressed()) );
  connect( ejectPB,  SIGNAL(clicked()), SLOT(ejectPressed()) );
  connect( infoPB,   SIGNAL(clicked()), SLOT(infoPressed()) );
  connect( powerPB,  SIGNAL(clicked()), SLOT(powerPressed()) );
  connect( shufflePB,SIGNAL(clicked()), SLOT(shufflePressed()) );
  connect( configPB, SIGNAL(clicked()), SLOT(configPressed()) );
  connect( repeatPB, SIGNAL(clicked()), SLOT(repeatPressed()) );
  connect( playPB, SIGNAL(clicked()), SLOT(playPressed()) );
  connect( stopPB, SIGNAL(clicked()), SLOT(stopPressed()) );

  connect( a, SIGNAL(kdisplayPaletteChanged()), SLOT(setColors()) );
 
  adjustSize();
  setFixedSize( width(), height() );

  errorFlag= FALSE;

  // Added by SMF
  tag = new TagEditor( this, "TagEditor");
  tag->hide();

  // Added by CP
  connect( tag, SIGNAL( tagChanged() ), SLOT( updateSong() ) );
  setDocking();
	
  // Modified by SMF (22/09/98)
  playlistdlg = new PlaylistDialog(this,"Playlist_Editor");
  playlistdlg->hide();

  if( !commandLine ){
    if( cfgData.loadLast ) 
      playlistdlg->loadDefault();
  }
  else 
    playlistdlg->setPlaylist( playlist );

  delete playlist;
  playlist = new QStrList( *(playlistdlg->songlist) );

  mb= new musicBrowser( this, "Music_Browser" );
  mb->hide();
  mb->setFuzzy( cfgData.fuzzy );
  mb->setSensitive( cfgData.sensitive );
  mb->setStep( cfgData.step );

  plr= NULL;          
  reinitializePlayer();

  // Added by SMF (12/23/98)
  if(cfgData.startShuffle) shufflePressed();
  for(int i=1;i<cfgData.startLoop;i++) repeatPressed();

  setPlayer();

  DropZone = new KDNDDropZone( this, DndURL );
  connect(DropZone,SIGNAL(dropAction(KDNDDropZone*)), SLOT(drop(KDNDDropZone *)));

  checkPlayer= new QTimer( this );
  connect( checkPlayer, SIGNAL( timeout() ), SLOT( playerReady() ) );

  if( cfgData.autoStartCMD && commandLine ){
    checkPlayer->start( 500, true );
  }
  else{
    if( cfgData.autoStart )
      checkPlayer->start( 500, true );
  }

}


void kmp3::playerReady(){
  plr->play();
}


void kmp3::drop( KDNDDropZone *zone ){
  unsigned int num, i;
  char *  file, * tmp;

  if( playlist == NULL )
    playlist= new QStrList();

  num = (zone->getURLList()).count();
  for(i=0;i<num;i++) {                         // process dropped objects
    tmp = (i==0 ? (zone->getURLList()).first()
           : (zone->getURLList()).next());
    if(!strncmp(tmp,"file:",5)) {              // is this a local object ?
      file = new char[strlen(tmp)-4];
      strcpy(file, tmp+5);
      playlist->append(file);
      delete[] file;
    }
    else {
      QMessageBox::critical(this,
                            klocale->translate("Drop Error"),
        klocale->translate("Only files and Directories are accepted !!!"));
    }
  }

  reinitializePlayer();
  plr->initCurentSong();
  commandLine= false;
  updateSong();
}


void kmp3::setStatus( int status ){
  if( dock_widget != NULL )
      dock_widget->stat= status;     
  if( status == PLAY )
    statusLabel->setText( klocale->translate("Play"));
}


void kmp3::updateTime(){
  QString theTime;

  if( playlist->count() ){
    theTime.sprintf( "%2.2d:%2.2d", plr->getMinutes(), plr->getSeconds() );
    setLEDs(theTime);
  }
  else{
    setLEDs( "--:--" );
    statusLabel->setText( klocale->translate("Ready") );
    dock_widget->stat= WAIT;
  }

}


void kmp3::showError( int Nr ){
  statusLabel->setText( klocale->translate("Error") );
  setLEDs( "--:--" );
  errorFlag= TRUE;

  titleLabel->setText( klocale->translate( errMessages[ Nr ] ));
  artistLabel->setText( klocale->translate("Play to retry or power to quit") );
}


void kmp3::updateSong(){
  QString updateStr("--:--");
  QString temp;
  unsigned frame;

  if (!plr->getPlayingFlag() && !plr->getPauseFlag())
    setLEDs(updateStr);
	
  if( playlist->count() ){
    if (!plr->getPauseFlag()) {
      frame= plr->getTotalLength();
      updateStr.sprintf( "%2.2d:%2.2d", frame/60, frame%60 );   	 
      timeLabel->setText(updateStr);
    }

    // Modified by SMF
    tag->setFiles( *playlist );
    tag->at( plr->getSongNr() );

    temp.sprintf("%d kBit/s", plr->getBitrate() );
    bitrateLabel->setText( temp );

    temp.sprintf("%d kHz", plr->getFrequency() / 1000 );
    freqLabel->setText( temp );

    if( tag->hasValidTag() ){
      makeLine(updateStr, 0);
      artistLabel->setLabel( updateStr );

      makeLine(updateStr, 1);
      titleLabel->setLabel( updateStr );

      makeLine(updateStr, 2);
    }
    else{
      if( cfgData.stripname ){
	char *temp= new char[ strlen( playlist->at( plr->getSongNr() ) ) ];
        stripNames( temp, playlist->at( plr->getSongNr() ) );
	
	if( cfgData.stripend )
	  stripExtension( temp );
	
	artistLabel->setLabel( temp );
        delete temp;
      }
      else
	artistLabel->setLabel( playlist->at( plr->getSongNr() ) );

      titleLabel->setLabel( "" );
      updateStr.sprintf( playlist->at( plr->getSongNr() ) );
    }

    // Added by CP 04.10.1998
    if( cfgData.docking ){
      if( allreadySet )
         QToolTip::remove( dock_widget );

      // check if it's "-", if so, replace with Unknown
      updateStr = updateStr.simplifyWhiteSpace();
      if (updateStr.isEmpty() || (strcmp(updateStr, "-") == 0))
	updateStr.setStr(klocale->translate("Unknown"));

      QToolTip::add( dock_widget, updateStr );
      allreadySet= true;
    }

    if (!plr->getPlayingFlag() && !plr->getPauseFlag())
      statusLabel->setText(klocale->translate("Ready"));
    updateStr.sprintf( "%d/%d", plr->getIndex()+1, playlist->count() );
    trackLabel->setText( updateStr );

    // set title (caption)
    QString fileNameStr(tag->getFilename());
    // get rid of beginning of path
    fileNameStr = fileNameStr.mid(fileNameStr.findRev('/') + 1,
				  fileNameStr.length());
    // get rid of file name extension
    fileNameStr = fileNameStr.left(fileNameStr.findRev('.'));
    updateStr.sprintf("%s: %s", APP_NAME, (const char *)fileNameStr);
    setCaption(updateStr);
  }
  else{
    artistLabel->setText( klocale->translate("No title loaded") );
    titleLabel->setText( "" );
    statusLabel->setText( klocale->translate("Ready") );

    // Added by CP 04.10.1998
    if( cfgData.docking ){
      if( allreadySet )
         QToolTip::remove( dock_widget );

      QToolTip::add( dock_widget, klocale->translate("No title loaded") );
      allreadySet= true;
    }

    timeLabel->setText( "--:--" );
    trackLabel->setText("-/-");
    freqLabel->setText("- kHz");
    bitrateLabel->setText("- kBit/s");
    setCaption(APP_NAME);
  }
}


void kmp3::setToolTips()
{
  if(cfgData.tooltips){
    QToolTip::add( playPB,       klocale->translate("Play/Pause"));
    QToolTip::add( stopPB,       klocale->translate("Stop"));
    QToolTip::add( repeatPB,     klocale->translate("Loop"));
    QToolTip::add( forwPB,       klocale->translate("Forward"));
    QToolTip::add( backPB,       klocale->translate("Backward"));
    QToolTip::add( nextPB,       klocale->translate("Next Song"));
    QToolTip::add( prevPB,       klocale->translate("Previous Song"));
    QToolTip::add( powerPB,      klocale->translate("Exit Kmp3"));
    QToolTip::add( logoPB,       klocale->translate("Music Browser"));
    QToolTip::add( configPB,     klocale->translate("Configure Kmp3"));
    QToolTip::add( ejectPB,      klocale->translate("Files/Songlist"));
    QToolTip::add( infoPB,       klocale->translate("Edit File Info"));
    QToolTip::add( shufflePB,    klocale->translate("Random Play"));
    QToolTip::add( volSB,        klocale->translate("Volume Control"));
  }
  else{
    QToolTip::remove( playPB);
    QToolTip::remove( stopPB);
    QToolTip::remove( repeatPB);
    QToolTip::remove( forwPB );
    QToolTip::remove( backPB);
    QToolTip::remove( nextPB );
    QToolTip::remove( prevPB );
    QToolTip::remove( powerPB );
    QToolTip::remove( configPB );
    QToolTip::remove( logoPB );
    QToolTip::remove( ejectPB );
    QToolTip::remove( infoPB );
    QToolTip::remove( shufflePB );
    QToolTip::remove( volSB );
  }
}

void kmp3::reinitializePlayer(){
  repeatLED->hide();
  repeat1LED->hide();
  shuffleLED->hide();

  if( plr != NULL ){
    disconnect( plr, SIGNAL( timeChanged() ));
    disconnect( plr, SIGNAL( songChanged() ));
    disconnect( plr, SIGNAL( songComplete() ));
    disconnect( nextPB, SIGNAL(clicked() ) );
    disconnect( prevPB, SIGNAL(clicked() ) );
    disconnect(  plr, SIGNAL( status( int ) ) );
    disconnect( plr, SIGNAL( listFinished()) );  
    disconnect( backPB, SIGNAL( pressed() ) );
    disconnect( backPB, SIGNAL( released() ) );  
    disconnect( forwPB, SIGNAL( pressed() ) );
    disconnect( forwPB, SIGNAL( released() ) );
    disconnect( volSB, SIGNAL(valueChanged(int)) );
    disconnect( plr, SIGNAL( mpegError( int ) ) );
    plr->quit();
    delete plr;
  }

  plr= new Player( cfgData.buffers );
  connect( plr, SIGNAL( timeChanged() ), SLOT( updateTime() ) );
  connect( plr, SIGNAL( songChanged() ), SLOT( updateSong() ) );
  connect( plr, SIGNAL( songComplete() ), mb, SLOT( nextSong() ) );
  connect( plr, SIGNAL( mpegError( int ) ),SLOT( showError( int )));
  connect( prevPB,   SIGNAL(clicked()), plr,SLOT(prevSong()) );
  connect( nextPB,   SIGNAL(clicked()), plr,SLOT(nextSong()) );
  connect( plr, SIGNAL( status( int ) ), SLOT( setStatus( int ) ) );
  connect( plr, SIGNAL( listFinished()), SLOT( updateSong() ) );
  connect( volSB, SIGNAL(valueChanged(int)), plr, SLOT(setVolume(int)));
  connect( volSB, SIGNAL(valueChanged(int)), SLOT(setVolume(int)));
  connect( backPB,   SIGNAL(pressed()), plr, SLOT(rewind()) );
  connect( backPB,   SIGNAL(released()), plr, SLOT( rewind() ) );
  connect( forwPB,   SIGNAL(pressed()), plr, SLOT( forward()) );
  connect( forwPB,   SIGNAL(released()), plr, SLOT( forward() ) ); 
  volSB->setValue(cfgData.volume);
  plr->setVolume( cfgData.volume );                                 
 }


// Added by SMF
// create text of line from tags
void kmp3::makeLine(QString & toMake, int lineNum)
{
  int pos =0;
  toMake.setStr(cfgData.line[lineNum].data());
  pos = toMake.find('%',pos);

  while( pos != -1 ) {
    switch (toMake[pos+1]) {
    case '%':
      toMake.replace(pos, 2, "%");
      pos++;
      break;
    case '1':
      toMake.replace(pos, 2, tag->getArtist());
      pos+=strlen(tag->getArtist());
      break;
    case '2':
      toMake.replace(pos, 2, tag->getTitle());
      pos+=strlen(tag->getTitle());
      break;
    case '3':
      toMake.replace(pos, 2, tag->getAlbum());
      pos+=strlen(tag->getAlbum());
      break;
    case '4':
      toMake.replace(pos, 2, tag->getYear());
      pos+=strlen(tag->getYear());
      break;
    case '5':
      toMake.replace(pos, 2, tag->getGenre());
      pos+=strlen(tag->getGenre());
      break;
    case '6':
      toMake.replace(pos, 2, tag->getComment());
      pos+=strlen(tag->getComment());
      break;
    case '7':
      if( cfgData.stripname ){
        char *temp= new char[ strlen( tag->getFilename() )];
        stripNames( temp, tag->getFilename() );

        if( cfgData.stripend )
          stripExtension( temp );

        toMake.replace(pos, 2, temp );
        delete temp;
      }
      else
        toMake.replace(pos, 2, tag->getFilename());

      pos+=strlen(tag->getFilename());
      break;
     default:   
      // do nothing
      pos++;
    }  
    pos = toMake.find('%',pos);  
  }
}


bool loadM3u( char *name) {
  QFile f(name);
  QFileInfo info(f);
  int i = 0;
  QString lerror;

  if(f.open( IO_ReadOnly | IO_Translate)) {
    char buffer[1024];
    QString tempstring;
    
    while(!f.atEnd()){
      buffer[0] = (char) 0;
      f.readLine(buffer,511);

      tempstring = buffer;
      tempstring = tempstring.stripWhiteSpace();
      if (!tempstring.isEmpty()){
	info.setFile(tempstring.data());
	if( (info.dir()).isRelative() ) {

	  info.setFile(f);
	  tempstring = info.dirPath() +"/"+ tempstring;
	  info.setFile(tempstring.data());
	  (info.dir()).convertToAbs();
	}
	if(info.exists()) {
	  playlist->append(info.filePath());
	}
	else i++;
      }
    }
    f.close();
  } 
  if(i) return false;
  return true;
}


bool isM3u( char *name ){
  if( strstr( name, ".m3u" ) != NULL ){

    if( strlen( strstr( name, ".m3u" ) ) > 4 )
      return false;

    return true;
  }

  if( strstr( name, ".M3U" ) != NULL ){

    if( strlen( strstr( name, ".M3U" ) ) > 4 )
      return false;

    return true;
  }

  return false;
}


int main( int argc, char *argv[] ){
  int i;

  a= new ObscureVisApp( argc, argv, "kmp3" );

  commandLine= false;

  if( argc > 1 ){
    playlist= new QStrList();

    for( i= 1; i < argc; i++ ){
      if( isM3u( argv[i] ) )
        loadM3u( argv[i] );
      else
        playlist->append( argv[i] );
    }
    commandLine= true;
  }

  k= new kmp3;

  a->setTopWidget( k );
  a->setMainWidget( k );
  k->show();

  XWindowAttributes wGetAttr;
  // get the window's attributes
  XGetWindowAttributes(qt_xdisplay(), k->winId(), &wGetAttr);
  // set event mask so that visibility events are sent to app
  wGetAttr.your_event_mask |= VisibilityChangeMask;
  XSelectInput(qt_xdisplay(), k->winId(), wGetAttr.your_event_mask);

  return a->exec();
}


