#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <controldata.h>
#include <strings.h>
#include <time.h>
#include <errno.h>

// Added by SMF 09/30/1998
#include <sys/uio.h>

#include "player.h"


// extern pointer to the KApplication
extern KApplication *a;
extern QStrList *playlist;


// Constructor of class Player
Player::Player( int buffers ){
  char tempStr[6];
  responseFlag= TRUE;
  isLocked= FALSE;
  list= NULL;
  browserMode= FALSE;

  if( ( buffers < 64 ) || ( buffers >= 1000 ) )
    buffers= 64;

  sprintf( tempStr, "%i", buffers ); 
  // get the executable, which is either KMP3PLAYER or amp

  if( getenv( "KMP3PLAYER" ) == NULL )
    amp << "kmp3player" << "-gui" << "-b" << tempStr;
  else
    amp << getenv( "KMP3PLAYER" );

  // Start the player process
  if( amp.start( KProcess::NotifyOnExit, KProcess::All ) == FALSE ){
    printf( "Position 1: Invoking amp\n" );
    emit mpegError( ERR_PLAYER );
  }

  // initialize SMF's Layer class
  layer= new Layer();
  CHECK_PTR( layer );

  // initialize some vars...
  index= 0;
  oldSeconds= seconds= minutes= 0;
  repeatFlag = REPEAT_NO;
  setShuffleFlag( FALSE );
  setPriority( PRIORITY_NORMAL );
  setPreLoadBuffer( 10 );
  setPreDecodeBuffer( 5 );
  dontTouchFlag= TRUE;
  pauseFlag= FALSE;
  playingFlag= FALSE;

  // initialize step timer
  sTimer = new QTimer();
  CHECK_PTR(sTimer);
  connect( sTimer, SIGNAL( timeout() ), SLOT( stepTimer() ) );
  
  // connect some SIGNALS & SLOTS
  connect( &amp, SIGNAL( processExited( KProcess * ) ), \
	   this, SLOT( notifyExit( KProcess * ) ) );   
  connect( &amp, SIGNAL( receivedStdout(KProcess *, char *, int) ), \
           this, SLOT( parseMessage(KProcess *, char *, int ) ) );
  connect( this, SIGNAL( next() ), this, SLOT( playerNextSong() ) );
}


// Destructor of class Player
Player::~Player(){
  // delete the player
  amp.kill();
}


// sendMessage sends a message to the player
void Player::sendMessage( int cmd, int data ){
  TControlMsg msg;

  msg.type = cmd;
  msg.data = data;

  amp.writeStdin( (char *) &msg, sizeof( TControlMsg ) ); 
}


// Opens the Song, tests, wether it is mpeg, and initializes the layer class
int Player::initCurentSong(){
  if( playlist->count() ){

    if( list == NULL ) prepareList();

    fd2= fopen( playlist->at( list[index] ), "r" );
    if( !fd2 ){
      emit mpegError( ERR_NOFILE );
      sendMessage( MSG_RELEASE, TRUE );
      return FALSE;
    }

    if( !layer->get( fd2 ) ){
      emit mpegError( ERR_NOFILE );
      sendMessage( MSG_RELEASE, TRUE );
      fclose( fd2 );
      return FALSE;
    }

    fclose( fd2 );
    pcmperfrm=layer->pcmPerFrame();
    freq= layer->sfreq();    
    emit songChanged();

    return TRUE;
  }

  return FALSE;
}    


// Tells the player to play the current file
void Player::play(){

  struct msghdr hdr;
  struct m_cmsghdr fdhdr;
  struct iovec iov[1];
  char data[2]; 

  if( playlist->count() ){

    if( list == NULL )
      prepareList();

    if( playlist->at( list[index] ) != NULL ){
      fd= open( playlist->at( list[index] ), O_RDONLY);

      if( !fd ){
        emit mpegError( ERR_NOFILE );
        sendMessage( MSG_RELEASE, TRUE );
        return;
      }
    
      if( !initCurentSong() ){
        sendMessage( MSG_RELEASE, TRUE );
        close( fd );
        return;
      }

      emit songChanged();

      sendMessage( MSG_SONG, 0 );

      iov[0].iov_base = data;
      iov[0].iov_len = 2;

      hdr.msg_iov = iov;
      hdr.msg_iovlen = 1; 
      hdr.msg_name = NULL;
      hdr.msg_namelen = 0;
         
      fdhdr.cmsg_len = sizeof(struct m_cmsghdr);
      fdhdr.cmsg_level = SOL_SOCKET;
      fdhdr.cmsg_type = SCM_RIGHTS;
      fdhdr.fd = fd;

      hdr.msg_control = &fdhdr;
//      hdr.msg_control = (char *) &fdhdr;
      hdr.msg_controllen = sizeof(struct m_cmsghdr);

      if( amp.sendMsg( &hdr, 0) < 0 ){
        printf( "Position 2: Transfering the file descriptor\n" ); 
	printf( "Error: %d\nMessage: %s\n", errno, strerror( errno ) );
        emit mpegError( ERR_SOCKET );
      }
      close( fd );
      dontTouchFlag= FALSE;
      playingFlag= TRUE;
      pauseFlag= FALSE;
      emit status( PLAY );
    }
    else{
      emit mpegError( ERR_NOFILE );
      sendMessage( MSG_RELEASE, TRUE );
      emit status( WAIT );
    }
  }

}


// Pauses the player
void Player::pause(){
  pauseFlag= !pauseFlag;
  sendMessage( MSG_CTRL, PLAY_PAUSE );
}


// Stops the player
void Player::stop(){
  pauseFlag= FALSE;
  playingFlag= FALSE;
  sendMessage( MSG_CTRL, PLAY_STOP );
  emit status( WAIT );
  minutes= seconds= 0;
}




// Slot parseMessage
void Player::parseMessage( KProcess *, char *buffer, int ){
  TControlMsg *msg= ( TControlMsg * ) buffer;

  switch( msg->type ){
    case MSG_FRAMES:	emit frames( msg->data ); 
			seconds= ( msg->data *pcmperfrm / 
				 freq ) % 60;	
			minutes= ( msg->data * pcmperfrm /
				 freq ) / 60;
			if( seconds != oldSeconds ){
			  oldSeconds= seconds;
			  emit timeChanged();
			}

			break;

    case MSG_POSITION:	emit position( msg->data );
			break;

    case MSG_INFO:	break;

    case MSG_NEXT:      if( browserMode )
		          emit songComplete();
			else
	                  emit next();

			break;

    case MSG_RESPONSE:	if( responseFlag == FALSE ){
			  response= msg->data;
			  responseFlag= TRUE;
			  a->exit_loop();
			}
			switch( msg->data ){
		          case FORWARD_BEGIN:
		          case FORWARD_STEP:
		          case FORWARD_END:
		               curState &= ~forwardStep;
		               break;
		          case REWIND_BEGIN:
		          case REWIND_STEP:
		          case REWIND_END:
		               curState &= ~rewindStep;
		               break;
       			}
			break;

    case MSG_AUDIOFAILURE:	emit mpegError( ERR_AUDIO );
			break;

    case MSG_PRIORITY:	emit mpegError( ERR_PRIORITY );
			break;

  }

}


// Step timer function / taken partly from jukebox
void Player::stepTimer(){
  int data;
  responseFlag= TRUE;

  if((curState & forwardState) && !(curState & forwardStep)){
    data = FORWARD_STEP;
    curState |= forwardStep;
  }
  else if((curState & rewindState) && !(curState & rewindStep)){
    data = REWIND_STEP;
    curState |= rewindStep;
  }
  else
    return;

  sendMessage( MSG_CTRL, data );
}


void Player::setBrowserMode( int mode ){
  browserMode= mode;
}


// For rewinding in the mpeg stream
void Player::rewind(){
  if( !dontTouchFlag && !pauseFlag ){
    if(!(curState & rewindState)){
      responseFlag= TRUE;
      sTimer->start(10);
      sendMessage( MSG_CTRL, REWIND_BEGIN);

      curState |= rewindState | rewindStep;
    }
    else{
      sTimer->stop();
      sendMessage( MSG_CTRL, REWIND_END );

      curState &= ~rewindState;
    }
  }
}


// For forwarding in the mpeg stream
void Player::forward(){
  if( !dontTouchFlag && !pauseFlag ){
    if(!(curState & forwardState)){
      responseFlag= TRUE;
      sTimer->start(10);
      sendMessage( MSG_CTRL, FORWARD_BEGIN);

      curState |= forwardState | forwardStep;
    }
    else{
      sTimer->stop();
      sendMessage( MSG_CTRL, FORWARD_END );

      curState &= ~forwardState;
    }
  }
} 


// Slot notifyExit
void Player::notifyExit( KProcess * ){
  if( !amp.normalExit() )
    emit mpegError( ERR_PLAYER_DIED );
}



// function setPreLoadBuffer
void Player::setPreLoadBuffer( int buffer ){
  // Zero Buffers?
  if( buffer <= 0 )
    buffer= 10;
   
  // Buffers smaller than buffers to predecode?
  if( buffer <= preDecodeBuffer )
    buffer= preDecodeBuffer + 5;

  preLoadBuffer= buffer;
  sendMessage( MSG_BUFFER, buffer );  
}


// function getPreloadBuffer
int Player::getPreLoadBuffer(){
  return preLoadBuffer;
}


// function getPreDecodeBuffer
int Player::getPreDecodeBuffer(){
  return preDecodeBuffer;
}


// function setPreDecodeBuffer
void Player::setPreDecodeBuffer( int buffer ){
  if( buffer <= 0 )
    buffer= 5;

  if( buffer >= preLoadBuffer ){
    buffer= preLoadBuffer - 5;
    if( buffer <= 0 ){
      buffer= preDecodeBuffer= 5;
      setPreLoadBuffer( 10 );
    }
  }

  preDecodeBuffer= buffer;
  sendMessage( MSG_BUFAHEAD, buffer );  
}


// function setPriority
void Player::setPriority( int prio ){
  switch( prio ){
    case PRIORITY_NORMAL:	
    case PRIORITY_REALTIME:
    case PRIORITY_NICE:		priority= prio;
				break;

    default:			priority= PRIORITY_NORMAL;
  }

  sendMessage( MSG_PRIORITY, priority );  
}


// function getPriority
int Player::getPriority(){
  return priority;
}


// function to wait for an event
int Player::gotResponse(){
  if( responseFlag ) 
    return TRUE;

  // start the "no response" timer ...
  timer = new QTimer();
  connect( timer, SIGNAL(timeout()), this, SLOT( noResponse() ) ); 
  timer->start( 1000, TRUE );

  // Go for an event Loop...
  a->enter_loop();

  delete timer;

  return responseFlag;
}


// function to exit the eventLoop
void Player::noResponse(){
  a->exit_loop();
}


// function isPaused
int Player::isPaused(){
  responseFlag= FALSE;
  sendMessage( MSG_QUERY, QUERY_PAUSED );

  if( gotResponse() ){
    pauseFlag= response;
    return response;
  }
  return true;
}


// function isPlaying
int Player::isPlaying(){
  responseFlag= FALSE;
  sendMessage( MSG_QUERY, QUERY_PLAYING );

  if( gotResponse() ){
    playingFlag= response;
    return response;
  }

  return true;
}



// set Song Number for actual Song
void Player::setSongNr( unsigned Nr ){
  if( Nr > playlist->count() ){
    index= 0;
    emit mpegError( ERR_NOSONG ); 
  }
  else{
    index= Nr-1;
    if( !playingFlag )
      initCurentSong();
    else
      play();
  }
  emit songChanged();
}


// set Song Number for actual Song
void Player::setIndex( unsigned Nr ){
  unsigned i= 0;

  if( Nr > playlist->count() ){
    index= 0;
    emit mpegError( ERR_NOSONG );
  }
  else{
    if( shuffleFlag ){
      while( i< playlist->count() )
        if( list[ i ] == Nr ){
          index= i;
          break;
        }
        else
          i++;
    }
    else
      index= Nr;

    if( !playingFlag )
      initCurentSong();
    else
      play();
  }
  emit songChanged();
}


void Player::quit(){
  sendMessage( MSG_QUIT, 0 );
}


void Player::cycleUpRepeatFlag(){
  switch (repeatFlag) {
  case REPEAT_NO:
    repeatFlag = REPEAT_ALL;
    break;
  case REPEAT_ALL:
    repeatFlag = REPEAT_ONE;
    break;
  case REPEAT_ONE:
    repeatFlag = REPEAT_NO;
    break;
  }
}


int Player::getRepeatFlag(){
  return repeatFlag;
}

// Added by SMF (22/12/98)
void Player::setRepeatFlag(int flag){
  repeatFlag = flag;
}

void Player::setShuffleFlag( int flag ){
  shuffleFlag= flag ? TRUE : FALSE;
  if( playlist->count() > 0 )
    prepareList();
}


int Player::getShuffleFlag(){
  return shuffleFlag;
}

void Player::nextSong() {
  nextSong(true);
}

void Player::playerNextSong() {
  nextSong(false);
}

void Player::nextSong(bool bUserPressed){
  if( !isLocked ){
    isLocked= TRUE; 
    index++;

    if( pauseFlag ) stop();

    if( index >= playlist->count() ){

      index= 0;

      if (bUserPressed) {
	// user pressed next button, so go first entry
	prepareList();
	index= 0;
	if( playingFlag ) play();
	else initCurentSong();
      }
      else {
	switch (repeatFlag) {
	case REPEAT_ALL:
	  prepareList();
	  index= 0;
	  if( playingFlag ) play();
	  else initCurentSong();
	  break;
	case REPEAT_ONE:
	  index = playlist->count() - 1;
	  if (playingFlag) play();
	  else initCurentSong();
	  break;
	default:
	  emit listFinished();
	  stop();
	  initCurentSong();
	}
      }
    }
    else{
      if( playingFlag ) {
	if ((repeatFlag == REPEAT_ONE) && !bUserPressed)
	  index--;
	play();
      }
      else initCurentSong();
    }

    isLocked= FALSE;
  }
}


void Player::prevSong(){
  if( !isLocked ){
    if( ( getSeconds() < 3 ) && ( getMinutes() == 0 ) ){
      isLocked= TRUE; 

      if( index > 0 )
        index--;
      else if (index == 0)
	index = playlist->count() - 1;

      if( playingFlag ) play();
      else initCurentSong();

      isLocked= FALSE;
    }
    else{
      if( playingFlag )
        play();
    }
  }
}


void Player::prepareList(){
  unsigned m, i, swap, z;
  time_t t;

  m= playlist->count();
  if( list != NULL )
    delete list;

  list= (unsigned *)malloc(  sizeof(unsigned) * ( m + 1 ) );
//  list= new sizeof(unsigned) * ( m + 1 );

  for( i=0; i<m; i++)
    list[i]=i;

  if( shuffleFlag ){
    srandom( time(&t) );
    for( i=0; i<m; i++ ){
      z= random() % m;
      swap= list[i];
      list[ i ]= list[ z ];
      list[ z ]= swap;
    }
  }
}


unsigned Player::getBitrate(){
  return layer->bitrate();
}


unsigned Player::getFrequency(){
  return layer->sfreq();
}


unsigned long Player::getTotalLength(){
  return layer->length();
}


unsigned Player::getSongNr(){
  return list[ index ];
}


unsigned Player::getIndex(){
  return index;
}


int Player::getSeconds(){
  return seconds;
}


int Player::getMinutes(){
  return minutes;
}

void Player::setVolume( int value ){
  vol.setVolume( value ); 
}


int Player::getVolume(){
  return vol.getVolume();
}

//#include "player.moc"
