/* tagEditor.cpp

  Created by SMF aka Antoine Laydier (laydier@usa.net)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*=============================================================================
  HEADERs
 =============================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include <qmsgbox.h>

#include <klocale.h>
#include <kapp.h>
#
#include "tagEditor.h"
//#include "tagEditor.moc"

/*=============================================================================
  #DEFINEs
=============================================================================*/
#define ktr           klocale->translate

/*=============================================================================
  GLOBALs
=============================================================================*/
extern int      errno;

/*=============================================================================
 Class : TagEditor (methods)
=============================================================================*/
// Const added by tvergote
const char  * const TagEditor::formats[] = 
{  
        // %a : Artist
        // %t : Title
        // %b : alBum
        // %y : Year
        // %g : Genre
        // %o : Other (path etc.)
        // %n : track Nr.
	// %c : comment
        // %% : %
        "%o/%a -#- %b#%t -#- -#- %c.mp3", // MP3stock format (undefined genre)
        "%o/%a -#- %t -#- -#- %c.mp3", // MP3stock format (undefined genre&album)
        "%o/%a -#- %b#%t -#- %g -#- %c.mp3", // MP3stock format   
        "%o/%a/%a - %b/%n %t.mp3",
        "%o/%a/%a - %b/%a - %t.mp3",
        "%o/%a - %b/%a - %n - %t.mp3",
        "%o/%a - %b/%a - %t.mp3",
        "%o/%a - %b/%n_%a_%t.mp3",
        "%o/%a - %b/%a_%t.mp3",
        "%o/%n - %a - %t.mp3",
        "%o/%a - %b - %t(%c).mp3", // old mp3stock format   
        "%o/%n_%a_%t.mp3",
        "%o/%a/%a - %t.mp3",
        "%o/%a/%a_-_%t.mp3",
        "%o/%g/%a-%t.mp3",
        "%o/%a - %b/(%a) - %t.mp3",
        "%o/(%a) - %t.mp3",
        "%o/(%a) - %b - %t.mp3",
        "%o/(%a)-%t.mp3",
        "%o/(%a)_%t.mp3",
        "%o/(%a)%t.mp3",
        "%o/(%a) %t.mp3",
        "%o/%a - %b/[%a - %n] - %t.mp3",
        "%o/%a - %b/[%a - %n]  - %t.mp3",
        "%o/[%a]-%t.mp3",
        "%o/[%a] %t.mp3",
        "%o/[%a] - %t.mp3",
        "%o/[%a] - %b - %t.mp3",
        "%o/%a - %t.mp3",
        "%o/%a-%t.mp3",
        "%o/%a_%t.mp3",
        "%o/%a - %b/%n - %t.mp3",
        "%o/%a/%b/%n %t.mp3",
        "%o/%a/%b/%n - %t.mp3",
        "%o/%a/%a - %b/%n %t.mp3",
        "%o/%a/%a - %b/%t.mp3",
        "%o/%a/%b/%t.mp3",
        "%o/%a/%t.mp3",
        "0"     // end with 0
};                               
/*-----------------------------------------------------------------------------
  Routine : TagEditor::TagEditor (constructor)
-----------------------------------------------------------------------------*/
TagEditor::TagEditor(QWidget *parent = 0, const char *name = 0)
        :QDialog( parent, name, TRUE)
{
  int i;
  fd = 0;
  layer =new Layer;
  tag = new Tag;
  fileList = new QStrList(TRUE);
   
  QString tmp;

  setStyle(WindowsStyle);
  setCaption( ktr("MP3 Tag Editor") );

  // Modified by SMF 10/17/98
  setFixedSize(465,300);

  // Added by SMF 10/17/98
  lFilename = new QLabel( ktr("File"), this);
  lFilename->setGeometry(15,10,100,25);
  slFilename = new QScrLabel(this);
  slFilename->setGeometry(115,10,340,25);

  // Modified by SMF 10/17/98
  CBtitle = new QCheckBox(ktr("Title"), this);
  CBtitle->setGeometry(15,50,100,25);
  leTitle = new QLineEdit(this,"leTitle");
  leTitle->setGeometry(115,50,345,25); 
  leTitle->setMaxLength(30);

  CBartist = new QCheckBox(ktr("Artist"), this);
  CBartist->setGeometry(15,90,100,25); 
  leArtist = new QLineEdit(this,"leArtist");
  leArtist->setGeometry(115,90,345,25); 
  leArtist->setMaxLength(30);

  CBalbum = new QCheckBox(ktr("Album"), this);
  CBalbum->setGeometry(15,130,100,25);  
  leAlbum = new QLineEdit(this,"leAlbum");
  leAlbum->setGeometry(115,130,345,25); 
  leAlbum->setMaxLength(30);
 
  CByear = new QCheckBox(ktr("Year"), this);
  CByear->setGeometry(15,170,100,25);  
  leYear = new QLineEdit(this,"leYear");
  leYear->setGeometry(115,170,60,25); 
  leYear->setMaxLength(4);

  CBgenre = new QCheckBox(ktr("Genre"), this);
  CBgenre->setGeometry(210,170,100,25);  
  cbGenre = new QComboBox(FALSE, this, "cbGenre");
  cbGenre->setGeometry(305,170,150,25); 

  CBcomment = new QCheckBox(ktr("Comment"), this);
  CBcomment->setGeometry(15,210,100,25);  
  leComment = new QLineEdit(this,"leComment");
  leComment->setGeometry(115,210,345,25); 
  leComment->setMaxLength(30);

  // init GENRE Combox Box ...
  // Modified by SMF 17/10/98
  for(i=0;i<= Tag::genre_largest;i++) 
    cbGenre->insertItem(Tag::genres[i],i);
  
  // Buttons ...
  bPrev = new QPushButton(ktr("&Previous"), this, "bPrev");
  connect(bPrev, SIGNAL(clicked()), this, SLOT(bPrevActivated()));

  bNext = new QPushButton(ktr("&Next"), this, "bNext");
  connect(bNext, SIGNAL(clicked()), this, SLOT(bNextActivated()));

  bAuto = new QPushButton(ktr("&Auto"), this,"bAuto");
  connect(bAuto,SIGNAL(clicked()), this, SLOT(bAutoActivated()));

  bReset = new QPushButton(ktr("&Reset"), this,"bReset");
  connect(bReset,SIGNAL(clicked()), this, SLOT(bResetActivated()));

  bSet = new QPushButton(ktr("&Set"), this, "bSet");
  connect(bSet,SIGNAL(clicked()), this, SLOT(bSetActivated()));

  bClose = new QPushButton(ktr("&Close"), this, "bClose");
  connect(bClose, SIGNAL(clicked()), this, SLOT(accept()));

  // disable all widgets...
  cbGenre->setEnabled(FALSE);
  leTitle->setEnabled(FALSE);
  leArtist->setEnabled(FALSE);
  leAlbum->setEnabled(FALSE);
  leYear->setEnabled(FALSE);
  leComment->setEnabled(FALSE);
  bSet->setEnabled(FALSE);
  bReset->setEnabled(FALSE);
  bAuto->setEnabled(TRUE); // This is probably not correct, but seems to work

  bPrev->setEnabled(FALSE);
  bPrev->hide();
  bNext->setEnabled(FALSE);
  bNext->hide();

  installEventFilter(this);
  placeWidgets();
}


/*-----------------------------------------------------------------------------
  Routine : TagEditor::~TagEditor() (destructor)
	    writes back config entries (current)
-----------------------------------------------------------------------------*/
TagEditor::~TagEditor()
{  
  closeFile();
  delete lFilename;
  delete slFilename;
  delete bAuto;
  delete bReset;
  delete bSet;
  delete bClose;
  delete bNext;
  delete bPrev;
  delete CBtitle;
  delete CBartist;
  delete CBalbum;
  delete CByear;
  delete CBgenre;
  delete CBcomment;
  delete cbGenre;
  delete leTitle;
  delete leArtist;
  delete leAlbum;
  delete leComment;
  delete leYear;
  delete fileList;
  delete layer;
  delete tag;
}

/*-----------------------------------------------------------------------------
  Routine : TagEditor::placeWidgets
-----------------------------------------------------------------------------*/
void TagEditor::placeWidgets( void )
{
  int w = width();
  int h = height();

  // Button Open, Prev, Next, Auto
  bClose->setGeometry(w - 85, h -41, 75, 26);
  bSet->setGeometry(w - 165, h -41, 75, 26);
  bReset->setGeometry(w - 245, h -41, 75, 26);
  bAuto->setGeometry(w - 300, h -41, 50, 26);

  bPrev->setGeometry(10, h -41, 75, 26);
  bNext->setGeometry(85, h -41, 75, 26);

  // Tag Editor
  leTitle->resize(w-15-110,25);
  leArtist->resize(w-15-110,25);
  leAlbum->resize(w-15-110,25);
  leComment->resize(w-15-110,25);
}


/*-----------------------------------------------------------------------------
  Routine : TagEditor::resizeEvent
-----------------------------------------------------------------------------*/
void TagEditor::resizeEvent(QResizeEvent *ev)
{
  QDialog::resizeEvent(ev);

  placeWidgets();
}

/*-----------------------------------------------------------------------------
  Routine : TagEditor::closeFile(void)
-----------------------------------------------------------------------------*/
void TagEditor::closeFile()
{
 if(fd) fclose(fd);
 fd = 0;

}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::bNextActivated
-----------------------------------------------------------------------------*/
void TagEditor::bNextActivated( void )
{ 
 fileList->at(fileList->at()+1);
 bPrev->setEnabled(TRUE);
 if(fileList->at() == (int)(fileList->count()-1) )
   bNext->setEnabled(FALSE);
 open();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::bPrevActivated
-----------------------------------------------------------------------------*/
void TagEditor::bPrevActivated( void )
{ 
 fileList->at(fileList->at()-1);
 bNext->setEnabled(TRUE);
 if(fileList->at() == 0) bPrev->setEnabled(FALSE);
 open();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::bAutoActivated
-----------------------------------------------------------------------------*/
void TagEditor::bAutoActivated( void )
{ 
 Auto();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::bResetActivated
-----------------------------------------------------------------------------*/
void TagEditor::bResetActivated( void )
{ 
 Reset();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::bSetActivated
-----------------------------------------------------------------------------*/
void TagEditor::bSetActivated( void )
{ 
 Set();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::Auto
  Added by Thierry Vergote (thierry.vergote@rug.ac.be)
-----------------------------------------------------------------------------*/

void TagEditor::Auto( void )
// TODO :  %%  
// list of not artist-names (mp3, root, mnt, dos, music, ...) (dynamic ?)
// move most of this to tag.cpp (Tag::AutoFill(char* filename))...
// MP3 instead of mp3 ???
// dynamic list of formats 
{
 int i,j,t,g,x; 
 int token_found;
 int format_ok = TRUE;
 int fn_pos, new_fn_pos;
 char buf[256];
 struct tag tmp_tag;
 char *  fileName;

 fileName = fileList->current();

 for (i=0;formats[i][0] != '0';i++) {
  format_ok = TRUE;
  fn_pos=strlen(fileName)-1;
  tmp_tag.title[0]  = '\0';
  tmp_tag.artist[0] = '\0'; 
  tmp_tag.album[0]  = '\0';
  tmp_tag.year[0]   = '\0';
  tmp_tag.genre     = tag->genre_largest ;
  for (j=strlen(formats[i])-1;j>0;j--) {
   if (formats[i][j-1] == '%' && formats[i][j] != '%') {
    if (j>1) {
     new_fn_pos=fn_pos;
     token_found = FALSE;
     while (!token_found) { 
      new_fn_pos--;
      if (new_fn_pos == 0) {
        format_ok = FALSE ;
        token_found = TRUE ;
      } else {
       if (fileName[new_fn_pos] == formats[i][j-2]) { 
        // check if complete token fits ...
        t=1;
        while ((fileName[new_fn_pos-t] == formats[i][j-2-t]) 
               && (formats[i][j-3-t] != '%')
               && (new_fn_pos > t) && (j-3 >t))
          t++;
        if (formats[i][j-3-t] == '%') {
         if ((formats[i][j-4-t] == 'y') || (formats[i][j-4-t] == 'n')) {
          if ((fileName[new_fn_pos-t] >= '0') && 
              (fileName[new_fn_pos-t] >= '9'))
             token_found = TRUE;
         } else 
          token_found = TRUE;
        }
       }
      }
     }
    } else {
     new_fn_pos=0;
    }
    // TODO : remove <space> at start of buf if any
    for (x=new_fn_pos;x<fn_pos;x++) {
     if (*(fileName+x+1) == '_' )
      buf[x-new_fn_pos]=' ';
     else 
      buf[x-new_fn_pos]=*(fileName+x+1); 
    }
    buf[fn_pos-new_fn_pos]='\0';
    
    if ((strchr(buf, '/')) && (formats[i][j] != 'o')) { 
      format_ok = FALSE;
    }
    switch (formats[i][j]) {
      case 'a' : // artist 
	if (atoi(buf)) { // if artist is number : something wrong !
          format_ok = FALSE;
          break;
        }
        if (strlen(buf) > 30)
          buf[30]='\0';
	if (tmp_tag.artist[0] == '\0') 
	 strcpy(tmp_tag.artist, buf);
	else 
         if (strcasecmp(tmp_tag.artist, buf) != 0)
          format_ok = FALSE ;
       break;
      case 't' : // title 
        if (strlen(buf) > 30)
          buf[30]='\0';
	if (tmp_tag.title[0] == '\0') 
	 strcpy(tmp_tag.title, buf);
	else 
         if (strcasecmp(tmp_tag.title, buf) != 0)
          format_ok = FALSE ;
       break;
      case 'b' : // album
        if (strlen(buf) > 30)
          buf[30]='\0';
	if (tmp_tag.album[0] == '\0') 
	 strcpy(tmp_tag.album, buf);
	else 
         if (strcasecmp(tmp_tag.album, buf) != 0)
          format_ok = FALSE ;
       break;
      case 'y' : // year
        if ((strlen(buf) != 4) ||(atoi(buf) == 0)) {
          format_ok = FALSE;
          break;
        }
	if (tmp_tag.year[0] == '\0') 
	 strcpy(tmp_tag.year, buf);
	else 
         if (strcmp(tmp_tag.year, buf) != 0)
          format_ok = FALSE ;
       break;
      case 'g' : // genre 
        for (g=0;g<=tag->genre_largest;g++) {
         if (!(strcasecmp(buf, tag->genres[g]))) {
           tmp_tag.genre = g;
           break;
         }
        }
        if (g > tag->genre_largest) 
         format_ok = FALSE; // unknown genre or no genre at all
       break;
      case 'o' : // other : do nothing with it :)
       break;
      case 'n' : // track nr
        if ((strlen(buf) != 2) ||(atoi(buf) == 0)) {
          format_ok = FALSE;
          break;
        }
       break;
      case 'c' : // comment TODO
       break;
      default :
       break;
    } // end of switch
   fn_pos = new_fn_pos;
   j--;
   if (!format_ok) 
    break; // stop checking format i further
   } else {
    if (formats[i][j] == fileName[fn_pos]) 
     fn_pos--;
    else { 
     format_ok = FALSE;
     break;
    } 
   }
  } // next char of format
  if (format_ok) 
   break; // we have match, stop looking
 } // formats for-loop
 if (format_ok) {
   tag->setTitle(tmp_tag.title); 
   leTitle->setText(tag->getTitle());
   tag->setAlbum(tmp_tag.album); 
   leAlbum->setText(tag->getAlbum());
   tag->setArtist(tmp_tag.artist); 
   leArtist->setText(tag->getArtist());
   tag->setYear(tmp_tag.year); 
   leYear->setText(tag->getYear());
   tag->setGenre(tmp_tag.genre); 
   cbGenre->setCurrentItem(tmp_tag.genre);

   tag->setComment(formats[i]); // QUICK DEBUG INFO
   leComment->setText(tag->getComment());
 }
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::Reset
-----------------------------------------------------------------------------*/
void TagEditor::Reset( void )
{ 
 readTag(bSet->isEnabled());
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::Set
-----------------------------------------------------------------------------*/
int TagEditor::Set( void )
{ 
  tag->setTitle(leTitle->text());
  tag->setAlbum(leAlbum->text());
  tag->setArtist(leArtist->text());
  tag->setYear(leYear->text());
  tag->setComment(leComment->text());
  tag->setGenre(cbGenre->currentItem());
  tag->set(fd);
 
  // Added by CP 10.10.1998
  if( playIndex == (uint) fileList->at() )
    emit tagChanged();

  return 0;
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::Open
-----------------------------------------------------------------------------*/
int TagEditor::open( void )
{ 
  char *  fileName;
  FILE *  fd2;
  bool    rw = TRUE;
  
  if(! fileList->isEmpty());

//  fileName = (char *)fileList->text();
  fileName = fileList->current();

  if(!fileName) return 1; 

  fd2 = 0;
  fd2 = fopen(fileName, "r+");
  if(!fd2) {
    rw = FALSE;
    fd2 = fopen(fileName, "r");
    if(!fd2) {
      QMessageBox::critical(this,
			  ktr("Error"), 
	    ktr("The file specified cannot be open\n\nSomething goes wrong !!!"));
    
      return 1;
    }
  }
  if(layer->get(fd2) == true) {
    // read Informations and Tag ...
    closeFile();
    fd = fd2;
    readTag(rw);
    // Added by SMF 10/17/98
    slFilename->setLabel(fileName);
    bSet->setEnabled(rw);
    bAuto->setEnabled(rw);
    bReset->setEnabled(TRUE);
  }
  else {   
    QMessageBox::critical(this,
		  ktr("Error"), 
		  ktr("The file specified \nis not a valid MP3 file\n\n"));
    fclose(fd2);
  }
  return 0;
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::readTag
-----------------------------------------------------------------------------*/
void  TagEditor::readTag ( bool rw )
{
  hasTag= tag->get(fd);
  cbGenre->setEnabled(rw);
  if(!CBgenre->isChecked())
    // Modified by SMF 17/10/98
    cbGenre->setCurrentItem(
      tag->getGenreNum()>=Tag::genre_largest ? 
      Tag::genre_largest : tag->getGenreNum());
  leTitle->setEnabled(TRUE);
  if(!CBtitle->isChecked())
    leTitle->setText(tag->getTitle());
  leArtist->setEnabled(TRUE);
  if(!CBartist->isChecked())
    leArtist->setText(tag->getArtist());
  leAlbum->setEnabled(TRUE);
  if(!CBalbum->isChecked())
    leAlbum->setText(tag->getAlbum());
  leYear->setEnabled(TRUE);
  if(!CByear->isChecked())
    leYear->setText(tag->getYear());
  leComment->setEnabled(TRUE);
  if(!CBcomment->isChecked())
    leComment->setText(tag->getComment());
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::setFile
-----------------------------------------------------------------------------*/
void TagEditor::setFile(const char * filename) 
{
  delete fileList;
  fileList = new QStrList(TRUE);
  fileList->append(filename);
}


/*-----------------------------------------------------------------------------
  Routine :  TagEditor::setFile
-----------------------------------------------------------------------------*/
void TagEditor::setFiles(const QStrList & filenames)
{
  delete fileList;
  fileList = new QStrList(filenames);
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::at
-----------------------------------------------------------------------------*/
void TagEditor::at(uint index)
{
  playIndex= index;
  fileList->at(index);
  open();
}

/*-----------------------------------------------------------------------------
  Routine :  TagEditor::show
-----------------------------------------------------------------------------*/
void TagEditor::show ()
{
  uint memo = fileList->at();
  if(fileList->count() > 1) {
    bPrev->show();
    if(memo > 0) bPrev->setEnabled(TRUE);
    bNext->show();
    if(memo < (fileList->count()-1)) bNext->setEnabled(TRUE);
  }
  QDialog::show();
  at(memo);
}


int TagEditor::hasValidTag(){
  return hasTag;
}
