##############################################################################
##     Filename:  file.tcl  
##
## Descripition:  Defines Window for File Loading
##
##   Procedures:  load_file
##                refresh
##                build_list
##                build_refresh
##                display_list
##                save_playlist
##                save_refresh
##
##---------------------------------------------------------------------------#
##
## Revision 1.1  1998/20/04  01:00:00  Mark C. Mains
## Initial revision
##
##############################################################################
proc load_file {directory filter} \
{
	global status

	set status(fileSelect) {}

	toplevel .load
	wm title .load "MP3Play Loader"

	frame .load.dircontrols -borderwidth 2 -relief groove
	button .load.dircontrols.parent -text "Parent" -highlightthickness 0 \
		-command {
			set dir $directory/..
			refresh $dir $filter
		}
	button .load.dircontrols.home -text "Home" -highlightthickness 0 \
		-command {
			set dir $env(HOME)
			refresh $dir $filter
		}
	button .load.dircontrols.root -text "Root" -highlightthickness 0 \
		-command {
			set dir /
			refresh $dir $filter
		}

	frame .load.currdir -borderwidth 2 -relief groove
	label .load.currdir.label -text Directory:
	entry .load.currdir.entry -textvariable directory -highlightthickness 0

	frame .load.list
	listbox .load.list.file -width 50 -height 15 -highlightthickness 0 \
		-yscrollcommand { .load.list.sbar set }
	scrollbar .load.list.sbar -orient vertical -highlightthickness 0 \
		-command { .load.list.file yview }

	frame .load.filter -borderwidth 2 -relief groove
	label .load.filter.label -text Filter:
	entry .load.filter.entry -textvariable filter -highlightthickness 0

	frame .load.controls -borderwidth 2 -relief groove
	button .load.controls.ok -text "OK" -highlightthickness 0 -command \
		{
			if {![catch {set file [.load.list.file get [.load.list.file \
				curselection]]}]} {

				set status(fileSelect) "$directory/$file"
			}
			destroy .load
		}
	button .load.controls.mp3cancel -text "Cancel" -highlightthickness 0 \
		-command { destroy .load }

	#
	# Pack sub-assemblies of widgets:
	#
	pack .load.dircontrols.parent -side left -expand yes -padx 8 -pady 4
	pack .load.dircontrols.home -side left -expand yes -padx 8 -pady 4
	pack .load.dircontrols.root -side left -expand yes -padx 8 -pady 4
	pack .load.currdir.label -side left -padx 4 -pady 4
	pack .load.currdir.entry -side left -padx 4 -pady 4 -expand yes -fill x
	pack .load.list.sbar -side right -fill y
	pack .load.list.file -side right -expand yes -fill both
	pack .load.filter.label -side left -padx 4 -pady 4
	pack .load.filter.entry -side left -expand yes -padx 4 -pady 4 -fill x
	pack .load.controls.ok -side left -expand yes -padx 8 -pady 4
	pack .load.controls.mp3cancel -side left -expand yes -padx 8 -pady 4

	#
	# Pack widgets into main window:
	#
	pack .load.dircontrols -fill x -padx 3 -pady 3
	pack .load.currdir -fill x -padx 3
	pack .load.list -padx 4 -pady 4 -expand yes -fill both
	pack .load.filter -fill x -padx 3
	pack .load.controls -fill x -padx 3 -pady 3


	#
	# Set Bindings
	#
	bind .load.list.file <Double-ButtonPress-1> {
		catch {set dir [.load.list.file get [.load.list.file curselection]]}
		refresh $dir $filter
	}
	bind .load.currdir.entry <Return> {
		set dir $directory
		refresh $dir $filter
	}
	bind .load.filter.entry <Return> {
		set dir $directory
		refresh $dir $filter
	}

	refresh $directory $filter

	# Set Minimium Size
	after idle {
		update idletasks
		wm minsize .load [winfo reqwidth .load] [winfo reqheight .load]
	}

    catch {grab .load}
    tkwait window .load
}
##############################################################################
proc refresh {dir mask} \
{
	global status
	global directory
	global filter

	if {[file isdirectory $dir]} {
		catch {[cd $dir]}
		set directory [pwd]
		set filter $mask

		.load.list.file delete 0 end

		set dirlist [exec ls -ab $directory]

		# Load Directories
		foreach i $dirlist {
			if {[file isdirectory $i]} {
				.load.list.file insert end "$i/"
			}
		}

		# Load Files that Match FILTER
		foreach i $dirlist {
			if {[lsearch $i $filter]!=-1 && [file isfile $i]} {
				.load.list.file insert end $i
			}
		}

	} elseif {[file isfile $dir]} {
		set status(fileSelect) "$directory/$dir"
		destroy .load
	}
}
##############################################################################
proc build_list {Bdirectory Bfilter} \
{
	global defaults
	global status

	set status(fileSelect) {}
	set status(templist) {}
	foreach song $status(playlist) {
		add_file $song templist
	}

	toplevel .build
	wm title .build "MP3Play Builder"

	frame .build.dircontrols -borderwidth 2 -relief groove
	button .build.dircontrols.parent -text "Parent" -highlightthickness 0 \
		-command {
			set dir $Bdirectory/..
			build_refresh $dir $Bfilter
		}
	button .build.dircontrols.home -text "Home" -highlightthickness 0 \
		-command {
			set dir $env(HOME)
			build_refresh $dir $Bfilter
		}
	button .build.dircontrols.root -text "Root" -highlightthickness 0 \
		-command {
			set dir /
			build_refresh $dir $Bfilter
		}

	frame .build.currdir -borderwidth 2 -relief groove
	label .build.currdir.label -text Directory:
	entry .build.currdir.entry -textvariable Bdirectory -highlightthickness 0


	frame .build.dirs
	frame .build.dirs.list
	listbox .build.dirs.list.file -width 30 -height 15 \
		-highlightthickness 0 -selectmode extended \
		-yscrollcommand { .build.dirs.list.sbar set }
	scrollbar .build.dirs.list.sbar -orient vertical -highlightthickness 0 \
		-command { .build.dirs.list.file yview }

	frame .build.dirs.playlist
	listbox .build.dirs.playlist.file -width 30 -height 15 \
		-highlightthickness 0 -selectmode extended\
		-yscrollcommand { .build.dirs.playlist.sbar set }
	scrollbar .build.dirs.playlist.sbar -orient vertical -highlightthickness 0 \
		-command { .build.dirs.playlist.file yview }

	frame .build.dirs.list_controls
	button .build.dirs.list_controls.add -highlightthickness 0 -bitmap \
		@$defaults(bmppath)/add.xbm -command \
		{
			set filelist [.build.dirs.list.file curselection]

			if {[llength $filelist] >= 1} {
				foreach index $filelist {
					set file [.build.dirs.list.file get $index]
					if {[file isfile $file]} {
						add_file "$Bdirectory/$file" templist
					}
				}
				display_list
			}

		}
	button .build.dirs.list_controls.remove -highlightthickness 0 -bitmap \
		@$defaults(bmppath)/remove.xbm -command \
		{
			set filelist [.build.dirs.playlist.file curselection]

			if {[llength $filelist] >= 1} {
				foreach index [lsort -integer -decreasing $filelist] {
					remove_file $index templist
				}
				display_list
			}
		}

	button .build.dirs.list_controls.moveup -highlightthickness 0 -bitmap \
		@$defaults(bmppath)/up.xbm -command \
		{
			set filelist [.build.dirs.playlist.file curselection]

			if {[llength $filelist] >= 1 && [lsearch $filelist 0] == -1} \
			{
				foreach index $filelist {
					set file [.build.dirs.playlist.file get $index]
					if {$index != 0} {
						swap_songs $index [expr $index-1] templist
					}
				}
				display_list

				foreach index $filelist {
					highlight [expr $index-1]
				}
			}

		}
	button .build.dirs.list_controls.movedown -highlightthickness 0 -bitmap \
		@$defaults(bmppath)/down.xbm -command \
		{
			set filelist [.build.dirs.playlist.file curselection]

			if {[llength $filelist] >= 1 && \
				[lsearch $filelist [expr [llength $status(templist)]-1]] == -1}\
			{
				foreach index [lsort -integer -decreasing $filelist] {
					set file [.build.dirs.playlist.file get $index]
					if {$index != [llength $status(templist)]} {
						swap_songs $index [expr $index+1] templist
					}
				}
				display_list

				foreach index $filelist {
					highlight [expr $index+1]
				}
			}

		}

	frame .build.filter -borderwidth 2 -relief groove
	label .build.filter.label -text Filter:
	entry .build.filter.entry -textvariable Bfilter -highlightthickness 0

	frame .build.controls -borderwidth 2 -relief groove
	button .build.controls.ok -text "OK" -highlightthickness 0 -command \
		{
			add_file clear playlist
			set status(list_index) 0
			foreach song $status(templist) {
				add_file $song playlist
			}
			set status(fileSelect) "NEWLIST"
			destroy .build
		}
	button .build.controls.mp3cancel -text "Cancel" -highlightthickness 0 \
		-command { destroy .build }
	button .build.controls.load -text "Load" -highlightthickness 0 -command \
		{
			load_file $defaults(list_dir) "*.lst"
			if {$status(fileSelect) != ""} {
				load_list $status(fileSelect) templist
				set $status(fileSelect) {}
				display_list
			}
			build_refresh $Bdirectory $Bfilter
		}
	button .build.controls.save -text "Save" -highlightthickness 0 -command \
		{
			if {[llength $status(templist)] == 0} {
				displayError "Play List is Empty"
			} else {
				save_playlist $defaults(list_dir)
			}
			build_refresh $Bdirectory $Bfilter
		}
	button .build.controls.mp3clear -text "Clear" -highlightthickness 0 \
		-command {
			add_file clear templist
			display_list
		}

	#
	# Pack sub-sub-assemblies of widgets:
	#
	pack .build.dirs.list.sbar -side right -fill y
	pack .build.dirs.list.file -side right -expand yes -fill both
	pack .build.dirs.list_controls.moveup -side top -padx 8 -pady 4
	pack .build.dirs.list_controls.add -side top -padx 8 -pady 4
	pack .build.dirs.list_controls.remove -side top -padx 8 -pady 4
	pack .build.dirs.list_controls.movedown -side top -padx 8 -pady 4
	pack .build.dirs.playlist.sbar -side right -fill y
	pack .build.dirs.playlist.file -side right -expand yes -fill both

	#
	# Pack sub-assemblies of widgets:
	#
	pack .build.dircontrols.parent -side left -expand yes -padx 8 -pady 4
	pack .build.dircontrols.home -side left -expand yes -padx 8 -pady 4
	pack .build.dircontrols.root -side left -expand yes -padx 8 -pady 4
	pack .build.currdir.label -side left -padx 4 -pady 4
	pack .build.currdir.entry -side left -padx 4 -pady 4 -fill x -expand yes

	pack .build.dirs.list -side right -fill both -expand yes 
	pack .build.dirs.list_controls -side right -padx 8 -pady 4
	pack .build.dirs.playlist -side right -fill both -expand yes

	pack .build.filter.label -side left -padx 4 -pady 4
	pack .build.filter.entry -side left -expand yes -padx 4 -pady 4 -fill x
	pack .build.controls.ok -side left -expand yes -padx 8 -pady 4
	pack .build.controls.load -side left -expand yes -padx 8 -pady 4
	pack .build.controls.save -side left -expand yes -padx 8 -pady 4
	pack .build.controls.mp3clear -side left -expand yes -padx 8 -pady 4
	pack .build.controls.mp3cancel -side left -expand yes -padx 8 -pady 4

	#
	# Pack widgets into main window:
	#
	pack .build.dircontrols -fill x -padx 3 -pady 3
	pack .build.currdir -fill x -padx 3
	pack .build.dirs -padx 4 -pady 4 -expand yes -fill both
	pack .build.filter -fill x -padx 3
	pack .build.controls -fill x -padx 3 -pady 3


	#
	# Set Bindings
	#
	bind .build.dirs.list.file <Double-ButtonPress-1> {
		catch {set dir [.build.dirs.list.file get \
			[.build.dirs.list.file curselection]]}
		build_refresh $dir $Bfilter
	}
	bind .build.dirs.playlist.file <Double-ButtonPress-1> {
		catch {set index [.build.dirs.playlist.file curselection]}
		if {$index != ""} {
			remove_file $index templist
		}
		display_list
	}
	bind .build.currdir.entry <Return> {
		set dir $Bdirectory
		build_refresh $dir $Bfilter
	}
	bind .build.filter.entry <Return> {
		set dir $Bdirectory
		build_refresh $dir $Bfilter
	}

	build_refresh $Bdirectory $Bfilter
	display_list

	# Set Minimium Size
	after idle {
		update idletasks
		wm minsize .build [winfo reqwidth .build] [winfo reqheight .build]
	}

    catch {grab .build}
    tkwait window .build
}
##############################################################################
proc build_refresh {dir mask} \
{
	global status
	global Bdirectory
	global Bfilter

	if {[file isdirectory $dir]} {
		catch {[cd $dir]}
		set Bdirectory [pwd]
		set Bfilter $mask

		.build.dirs.list.file delete 0 end

		set dirlist [exec ls -ab $Bdirectory]

		# Load Directories
		foreach i $dirlist {
			if {[file isdirectory $i]} {
				.build.dirs.list.file insert end "$i/"
			}
		}

		# Load Files that Match FILTER
		foreach i $dirlist {
			if {[lsearch $i $Bfilter]!=-1 && [file isfile $i]} {
					.build.dirs.list.file insert end $i
			}
		}

	} elseif {[file isfile $dir]} {
		add_file "$Bdirectory/$dir" templist
		display_list
	}
}
##############################################################################
proc display_list {} \
{
	global status

	.build.dirs.playlist.file delete 0 end

	foreach i $status(templist) {
		set title "[file rootname [file tail $i]]"
		.build.dirs.playlist.file insert end $title
	}
}
##############################################################################
proc highlight {index} \
{
	.build.dirs.playlist.file selection set $index $index
}
##############################################################################
proc save_playlist {Sdirectory} \
{
	global status

	toplevel .save
	wm title .save "Save Play List"

	frame .save.currdir -borderwidth 2 -relief groove
	label .save.currdir.label -text Directory:
	entry .save.currdir.entry -textvariable Sdirectory -highlightthickness 0

	frame .save.list
	listbox .save.list.file -width 30 -height 10 -highlightthickness 0 \
		-yscrollcommand { .save.list.sbar set }
	scrollbar .save.list.sbar -orient vertical -highlightthickness 0 \
		-command { .save.list.file yview }

	frame .save.filename -borderwidth 2 -relief groove
	label .save.filename.label -text Filename:
	entry .save.filename.entry -textvariable Sfilename -highlightthickness 0

	frame .save.controls -borderwidth 2 -relief groove
	button .save.controls.save -text "Save" -highlightthickness 0 -command \
		{
			if {$Sfilename != ""} {
				save_list "$Sdirectory/$Sfilename"
			}
			destroy .save
		}
	button .save.controls.mp3cancel -text "Cancel" -highlightthickness 0 \
		-command { destroy .save }

	#
	# Pack sub-assemblies of widgets:
	#
	pack .save.currdir.label -side left -padx 4 -pady 4
	pack .save.currdir.entry -side left -padx 4 -pady 4 -expand yes -fill x
	pack .save.list.sbar -side right -fill y
	pack .save.list.file -side right -expand yes -fill both
	pack .save.filename.label -side left -padx 4 -pady 4
	pack .save.filename.entry -side left -expand yes -padx 4 -pady 4 -fill x
	pack .save.controls.save -side left -expand yes -padx 8 -pady 4
	pack .save.controls.mp3cancel -side left -expand yes -padx 8 -pady 4

	#
	# Pack widgets into main window:
	#
	pack .save.currdir -fill x -padx 3
	pack .save.list -padx 4 -pady 4 -expand yes -fill both
	pack .save.filename -fill x -padx 3
	pack .save.controls -fill x -padx 3 -pady 3


	#
	# Set Bindings
	#
	bind .save.list.file <Double-ButtonPress-1> {
		catch {set dir [.save.list.file get [.save.list.file curselection]]}
		save_refresh $dir
	}
	bind .save.currdir.entry <Return> {
		set dir $Sdirectory
		save_refresh $dir
	}
	bind .save.filename.entry <Return> {
		if {$Sfilename != ""} {
			save_list "$Sdirectory/$Sfilename"
		}
		destroy .save
	}

	save_refresh $Sdirectory

	# Set Minimium Size
	after idle {
		update idletasks
		wm minsize .save [winfo reqwidth .save] [winfo reqheight .save]
	}

    catch {grab .save}
    tkwait window .save
}
##############################################################################
proc save_refresh {dir} \
{
	global Sdirectory
	global Sfilename

	if {[file isdirectory $dir]} {
		catch {[cd $dir]}
		set Sdirectory [pwd]

		.save.list.file delete 0 end

		set dirlist [exec ls -ab $Sdirectory]

		# Load Directories
		foreach i $dirlist {
			if {[file isdirectory $i]} {
				.save.list.file insert end "$i/"
			}
		}

		# Load Files that Match FILTER
		foreach i $dirlist {
			if {[lsearch $i "*.lst"]!=-1 && [file isfile $i]} {
				.save.list.file insert end $i
			}
		}

	} elseif {[file isfile $dir]} {
		set Sfilename $dir
	}
}
##############################################################################
