/**********************************************************************
 playmp3 - Copyright (C) 1997 - Hans Petter Bieker <zerium@pe049.persbraten.vgs.no>
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/

#ifdef HAVE_LIBPTHREAD
#  define PTHREADEDMPEG 1
#endif

#include <stdlib.h>
#include <stdio.h>

#include <time.h>
#include <getopt.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <ftw.h>

#include "playmp3.h"
#include "mp3info.h"
#include "play.h"

struct progoptions_t progoptions;
static playlist_t * public_playlist;
const char defaultaudiodev[] = DEFAUDIODEV;
  
char * program_name;

void parsecfgfile(FILE *File);
void option_req_argument (void);
void version(void);;
void usage(int error);

void print_mp3info(char * filename)
{
  int  file;
  tune song;
  char format[FORMAT_LEN+1];
  mp3file mp3;

  memset(&song,0,sizeof(song));
  song.gennum=-1;
  strcpy(format,"Playing %f:\n%IT%It%t%!a, %IE%IE%Ia(%a), %IE%IE%b kbit/s %M %ICCRC %IE%(%mm %ss%)");
  if (!((file=mp3open(&mp3,filename,0))==-1 ||
     !(mp3.hashdr=gethdr(file,&mp3.hdr)))) {
    mp3calc(&mp3);
    
    memset(&mp3.tag,0,sizeof(tag));
    mp3.hastag=gettag(file,&mp3.tag); 
    printstd(&mp3,format,filename,NULL);
  }
  close(file);
}

void option_req_argument(void)
{
  fprintf(stderr,"%s: option requires an argument -- \n", program_name);
  exit (1);
}

void player_crash(void)
{
  fprintf(stderr,"\n%s: An error accour in player\n",program_name);
  exit (2);
}

void usage(int status)
{
  if (status != 0)
    fprintf (stderr, "Try %s --help' for more information.\n",
             program_name);
  else {
    printf("usage: %s [ -options ] file1 [file2 ...]\n",program_name);
    printf(
           "  -t, --threadstore x     number of frames as buffer\n"
           "  -m, --forcetomono       force to mono\n"
           "  -r, --random            shuffle play\n"
           "  -R, --recursive         play subdirectories recursively\n"
           "  -d, --defaultplaypath   play files in default play path\n"
           "  -q, --quiet             quiet mode; don't display any info about the sample\n"
           "  -v, --verbose           explain what is being done\n"
           "      --help              display this help and exit\n"
           "      --version           output version information and exit\n");
  }
  exit(status);
}

void version(void)
{
  printf("%s: " PACKAGE_VERSION "\n",program_name);
  exit(EXIT_SUCCESS);
}

char * prepfilename ( char * filename)
{
  char * file = NULL;
  char * c;
  int i = 0;

  if (!(file = malloc(1+strlen(filename)*2)))
    err(1,"Unable to allocate filename string");

  c = filename;
  while (*c) {
    if (*c == '\'')
      file[i++] = '\\';
    file[i++] = *c;
    c++;
  }
  file[i++] = *c;
  return file;
}

void parsecfgfile(FILE *File)
{
	char Buffer[256];	
        unsigned int i;
	char c;

	while(!feof(File))
	{
		/* Read string */
		fscanf(File, "%s", Buffer);      		
	
		/* Comment */
		if(Buffer[0] == '#')
		{
			/* Got to next line */
			fgets(Buffer, 255, File);
		} 

#ifdef PTHREADEDMPEG
		/* number of frames */
		else if(!strcasecmp(Buffer, "ThreadStore"))
		{			
			fscanf(File, "%d ", &progoptions.threadstore);
		} 
#endif		
		/* recursive play */
			else if(!strcasecmp(Buffer, "Recursive"))
		{			
			fscanf(File, "%d ", &progoptions.recursive);
		} 
		
		/* shuffle play */
		else if(!strcasecmp(Buffer, "Random"))
		{			
			fscanf(File, "%d ", &progoptions.shuffle);
		} 
		
		/* force mono */
		else if(!strcasecmp(Buffer, "ForceToMono"))
		{			
			fscanf(File, "%d ", &progoptions.forcetomonoflag);
		} 
		
		/* defaultplaypath */
		else if(!strcasecmp(Buffer, "DefaultPlayPath"))
		{			
			if (fscanf(File, "%s ", Buffer)) {
				progoptions.defaultplaypath = realloc(progoptions.defaultplaypath, strlen(Buffer)+1);
				strcpy(progoptions.defaultplaypath,Buffer);
			}
		} 
		
		/* Unknown keyword */
		else
		{
			for(i=0, c=0; i < strlen(Buffer); i++)
				c += isgraph(Buffer[i]);
			if(c != 0)
			{
				fprintf(stderr, "%s: unknown keyword '%s'\n", program_name,Buffer);
				exit(1);
			}
		}	      
	}
}

/* playlist functions... this would look a lot better in c++ .. :) */
playlist_t * playlist_init(void)
{
  playlist_t * playlist;

  playlist = calloc(sizeof(playlist_t),1);
  playlist->numfiles = 0;
  
  return playlist;
}

/* adds filename to end of playlist */
void playlist_add(playlist_t * playlist, char * filename)
{
  playlist->numfiles++;
  if (!(playlist->files = realloc(playlist->files, playlist->numfiles * sizeof(void *))) && playlist->files)
    err(1,"Unable to reallocate playlist");
    
  playlist->files[playlist->numfiles-1] = filename;
}

/* removes entry #n from playlist. 0 == first */
void playlist_delete(playlist_t * playlist, int n)
{
  playlist->numfiles--;
  playlist->files[n]=playlist->files[playlist->numfiles];
  playlist->files = realloc(playlist->files, playlist->numfiles * sizeof(void *));
}

int playlist_recursive_adder(const char *file, struct stat *sb, int flag)
{
  char * filename = malloc(1+strlen(file));
  
  if (!filename)
    err(1,"Unable to allocate filename");
    
  strcpy(filename, file);

  if (flag != FTW_F) return 0;

  playlist_add(public_playlist, filename);
  return 0;
}

/* expand all directory names to filenames */
void playlist_recursive(playlist_t * playlist)
{
  char ** files = playlist->files;
  int numfiles = playlist->numfiles;
  struct stat stats; 

  /* remove dirs..*/
  playlist->files = NULL;
  playlist->numfiles = 0;

  while (numfiles) {
    --numfiles;
    if (stat(files[numfiles], &stats) == -1) err(1,files[numfiles]);
    if (ftw(files[numfiles], &playlist_recursive_adder, 0)) {
      fprintf(stderr,"%s: Err..\n",program_name);
      exit (1);
    }
  }
}

/* read config file */
void handle_cfg(void)
{
  FILE * file;
  char cfgfile[256];
  const char readmode [] = "r";
            
  if ((file = fopen(GLOBCFGFILE, readmode))) {
    parsecfgfile(file);
    fclose(file);
  }

  strcpy(cfgfile, (char*)getenv("HOME"));
  strcat(cfgfile, LOCCFGFILE);
  if ((file = fopen(cfgfile, readmode))) {
    parsecfgfile(file);
    fclose(file);
  }
}

/* check if they looks okay.. */
void chk_settings(void)
{
#ifdef HAVE_LIBPTHREAD
  if(progoptions.threadstore < MINTHREADSTORE)
  {
    fprintf(stderr, "%s: invalid number of frames specified, using default\n",program_name);
    progoptions.threadstore = DEFTHREADSTORE;  /* Use default */
  }
#endif
}    

void init(void)
{
  progoptions.verbose = DEFVERBOSE;
  progoptions.recursive = DEFRECURSIVE;
  progoptions.quiet = DEFQUIET;
  progoptions.shuffle = DEFSHUFFLE;
  progoptions.show_help = 0;
  progoptions.show_version = 0;
#ifdef HAVE_LIBPTHREAD
  progoptions.threadstore = DEFTHREADSTORE;
#endif
  progoptions.defaultplaypathflag = 0;
  progoptions.defaultplaypath = NULL;
  progoptions.audiodev = (char *)defaultaudiodev;
}

int main (int argc, char **argv)
{
  int c, i;

  program_name = argv[0];
  init();

  handle_cfg();
  
  while (1)
    {
      int option_index = 0;
      static const struct option long_options[] =
      {
#ifdef HAVE_LIBPTHREAD
        {"threadstore",   1, 0,                          't'},
#endif
        {"quiet",         0, 0,	                         'q'},
        {"recursive",     0, 0,                          'R'},
        {"forcetomono",   0, 0,                          'm'},
        {"defaultplaypath",0,0,                          'd'},
        {"help",          0, &progoptions.show_help,      1},
        {"version",       0, &progoptions.show_version,   1},
        {"verbose",       0, 0, 'v'},
        {"random",        0, 0, 'r'},
        {0, 0, 0, 0}
      };

      c = getopt_long (argc, argv, "thvrRmdq",
        long_options, &option_index);
      if (c == -1)
        break;

      switch (c)
        {
        case 'd': 
          progoptions.defaultplaypathflag = !progoptions.defaultplaypathflag; 
          break;

        case 'R': 
          progoptions.recursive = !progoptions.recursive; 
          break;

        case 'r': 
          progoptions.shuffle = !progoptions.shuffle; 
          break;

        case 'q': 
          progoptions.quiet = !progoptions.quiet; 
          break;

        case 'v':
          progoptions.verbose = !progoptions.verbose;
          break;

        case 'm':
          progoptions.forcetomonoflag = !progoptions.forcetomonoflag;
          break;

#ifdef HAVE_LIBPTHREAD
	case 't':
          if (sscanf(optarg,"%d",&progoptions.threadstore) != 1)
            option_req_argument();
          break;
#endif
        }
    }

  chk_settings();

  if (progoptions.show_help) usage(0);
  if (progoptions.show_version) version();

  if (optind == argc && !progoptions.defaultplaypath && progoptions.defaultplaypathflag) {
    fprintf(stderr, "%s: default play path not set\n", program_name);
    exit (1);
  }
  if (argc == optind && !(progoptions.defaultplaypath && progoptions.defaultplaypathflag)) {
    fprintf(stderr, "%s: missing file argument\n", program_name);
    usage (1);
  }

  public_playlist = playlist_init();

  /* using default play path */
  if (optind == argc && progoptions.defaultplaypath && progoptions.defaultplaypathflag) {
    if (progoptions.verbose) printf("Using %s as default play path.\n",progoptions.defaultplaypath);
    playlist_add(public_playlist, progoptions.defaultplaypath);
    progoptions.recursive = 1; /* default play path mode requires recursive mode */
  } else {
    /* rewrite playlist */
    while (optind < argc)
      playlist_add(public_playlist, argv[optind++]);
  }

  /* recursive play */
  if (progoptions.recursive) {
    if (progoptions.verbose) printf("Recursive play\n");
    playlist_recursive(public_playlist);
  }

  /* play 'em */
  if (progoptions.shuffle) {
    if (progoptions.verbose) printf("Shuffle play\n");
    srandom(time(NULL));
  }

  while(public_playlist->numfiles) {
    i=progoptions.shuffle?random()%public_playlist->numfiles:0;
    play(public_playlist->files[i]);
    playlist_delete(public_playlist, i);
  }
  return EXIT_SUCCESS;
}
