/********************************************************
 * acp -- Copy files to and from the rio.		*
 *							*
 * Usage:						*
 *	acp [opts] <file> [file....] i:			*
 *		Copy file to internal memory		*
 *	acp [opts] <file> [file....] x:			*
 *		Copy file to external memory		*
 *	acp [opts] i:<file> x:<file> <dir>		*
 *		Copy files from rio to given directory	*
 *							*
 *  Options						*
 *	-v -- verbose output				*
 *	-p <port> -- specify the address of the lpt	*
 *		port to use.				*
 *							*
 *  Rio files may be globbed if the arguements are	*
 *  quoted.						*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include "rio.h"
#include "common.h"
#include "progress.h"
#include "set_ix.h"
#include "rio_glob.h"

// The verbose flag
static bool verbose_flag = false;	


/********************************************************
 * copy_from_rio -- copy a file from the rio		*
 ********************************************************/
static void copy_from_rio(
    rio_class &rio_info,	// The RIO informaiton
    const int argc, 		// Arguement count
    char *argv[] 		// Arguements
)
{
    // The list of the files with wildcards expanded
    string_list file_list = rio_glob(rio_info, argc, argv, optind);

    // The file we are looking at now
    string_list::iterator cur_file;

    for (cur_file = file_list.begin(); cur_file != file_list.end();
	++cur_file) {
	if (verbose_flag)
	    printf( "Downloading %s\n", (*cur_file).c_str() );
	set_ix(rio_info, verbose_flag, (*cur_file).c_str());

	rio_info.read_rio_file(&(*cur_file).c_str()[2], 
		    progress_cb(verbose_flag ? progress_callback : NULL));
    }
}

/********************************************************
 * copy_to_rio -- copy a file to the rio		*
 ********************************************************/
static void copy_to_rio(
    rio_class &rio_info,// The RIO informaiton
    int argc, 		// Arguement count
    char *argv[] 	// Arguements
)
{
    if ((strcmp(argv[argc-1], "i:") != 0) &&
        (strcmp(argv[argc-1], "x:") != 0)) {
	fprintf(stderr, 
	    "Error: First or last parameter must be a rio \"drive\"\n");
	exit (EXIT_USAGE);
    }
    set_ix(rio_info, verbose_flag, argv[argc-1]);

    /* Remember the argc-1 -- -1 because the last parameter is 
       the rio memory 
    */
    for (/* optind set */; optind < argc-1; ++optind) {
	// upload
	if (verbose_flag)
	    printf( "uploading %s\n", argv[optind] );

	rio_info.send_rio_file(argv[optind], 
		    progress_cb(verbose_flag ? progress_callback : NULL));
    }
}

/********************************************************
 * usage -- tell the user how to use us			*
 ********************************************************/
static void usage(void)
{
    fprintf(stderr, "Usage is:\n");
    fprintf(stderr, " 	acp i|x:<files> <dir> -- copy from rio to disk\n");
    fprintf(stderr, "	acp <files> {i: | x:} -- copy to rio\n");
    fprintf(stderr, " Options:\n");
    fprintf(stderr, "	-v -- verbose\n");
    fprintf(stderr, " 	-p <port> -- Set I/O port\n");
    exit (EXIT_USAGE);
}

int main( int argc, char* argv[] )
{
    try {
	int port_base = PORT_BASE_DEFAULT;	// The I/O port base
	rio_class rio_info;			// The RIO informaiton

	while (1) {
	    int opt = getopt(argc, argv, "vp:");
	    if (opt == EOF)
		break;

	    switch (opt) {
		case 'v':
		    verbose_flag = true;
		    break;
		case 'p':
		    sscanf(optarg, "%x", &port_base);
		    break;
		default:
		    usage();
	    }
	}

	// setup
	rio_info.set_io_address(port_base);

	if ((strncmp(argv[optind], "i:", 2) == 0) ||
	    (strncmp(argv[optind], "x:", 2) == 0)) 
	{
	    copy_from_rio(rio_info, argc, argv);
	} else {
	    copy_to_rio(rio_info, argc, argv);
	}

	if (verbose_flag)
	    printf( "updating directory\n" );

	rio_info.flush();
    }

    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (EXIT_IO_ERROR);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (EXIT_IO_ERROR);
    }
    return(0);
}
