/********************************************************
 * adump [-v] [-p port] {i|x} file			*
 *							*
 * Dump all of the memory to a file.			*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include "rio.h"
#include "common.h"
#include "progress.h"
#include "set_ix.h"

// The verbose flag
static bool verbose_flag = false;	

/********************************************************
 * usage -- tell the user how to use us			*
 ********************************************************/
static void usage(void)
{
    fprintf(stderr, "Usage is:\n");
    fprintf(stderr, "	adump [-p port] {i|x} <file>\n");
    exit (EXIT_USAGE);
}

int main( int argc, char* argv[] )
{
    try {
	int port_base = PORT_BASE_DEFAULT;	// I/O port to use
	rio_class rio_info;			// The RIO informaiton

	while (1) {
	    int opt = getopt(argc, argv, "vp:");
	    if (opt == EOF)
		break;

	    switch (opt) {
		case 'v':
		    verbose_flag = true;
		    break;
		case 'p':
		    sscanf(optarg, "%x", &port_base);
		    break;
		default:
		    usage();
	    }
	}

	// setup
	rio_info.set_io_address(port_base);

	if ((argc - optind) != 2) 
	    usage();

	if (strcmp(argv[optind], "i") == 0) {
	    rio_info.set_mem_type(MEM_INTERNAL);
	} else if (strcmp(argv[optind], "x") == 0) {
	    rio_info.set_mem_type(MEM_EXTERNAL);
	} else
	    usage();

	// The name of the output file
	const char *const out_file_name = argv[optind+1];

	unsigned long int mem_size;	// Size of the memory in 32K blocks

	if (rio_info.get_mem_type() == MEM_INTERNAL) 
	    mem_size = C32M / C32K;
	else
	    mem_size = rio_info.external_memory_size_32K;

	FILE *out_file = fopen(out_file_name, "w");
	if (out_file == NULL) {
	    fprintf(stderr, "ERROR: Unable to open: %s\n", out_file_name);
	    exit (EXIT_IO_ERROR);
	}

	unsigned int cur_block;		// Current block number
	data_block block;	// The data block we are working on

	for (cur_block = 0; cur_block < mem_size; ++cur_block) {
	    rio_info.io.get_block_from_rio(block, cur_block);

	    if (fwrite(block.ptr(), 1, C32K, out_file) != C32K) {
		fprintf(stderr, "ERROR: Unable to write %s\n", out_file_name);
		exit (EXIT_IO_ERROR);
	    }
	}
	fclose(out_file);
	rio_info.flush();
    }

    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (EXIT_IO_ERROR);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (EXIT_IO_ERROR);
    }
    return(0);
}
