/********************************************************
 * aformat -- format the rio memory			*
 *							*
 *   Usage is aformat [-f] [-v] [-p <port>] (i:|x:)	*
 *   	-f -- fast format (do not check for bad blocks	*
 *   	-v -- turn on verbose messages			*
 *   	-p -- Set the portbase for the device		*
 *   	i -- Initialize internal ram			*
 *   	x -- Initialize external ram			*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "common.h"
#include "rio.h"
#include "progress.h"
#include "set_ix.h"


/********************************************************
 * usage -- Tell the user what to do.			*
 ********************************************************/
static void usage()
{
    fprintf(stderr, "Usage is aformat [-f] [-v] [-p <port>] (i:|x:)\n");
    fprintf(stderr, "	-f -- fast format (do not check for bad blocks\n");
    fprintf(stderr, "	-v -- turn on verbose messages\n");
    fprintf(stderr, "	-p -- Set the portbase for the device\n");
    fprintf(stderr, "	i -- Initialize internal ram\n");
    fprintf(stderr, "	x -- Initialize external ram\n");
    exit (EXIT_USAGE);
}

int main( int argc, char* argv[] )
{
    try {
	// The port to use
	int port_base = PORT_BASE_DEFAULT;

	rio_class rio_info;	// The RIO informaiton
	bool verbose = false;	// True if we need to chatter
	bool fast = false;	// Fast format without bad block check

	// Options (-v -- verbose) (-f fast) (-p port)
	while (1) {
	    int opt = getopt(argc, argv, "vfp:");	// Get options

	    if (opt == EOF)
		break;

	    switch (opt) {
		case 'v':
		    verbose = true;
		    break;
		case 'p':
		    sscanf(optarg, "%x", &port_base);
		    break;
		case 'f':
		    fast = true;
		    break;
		default:
		    usage();
		    break;
	    }
	}
	if ((argc - optind) != 1) {
	    usage();
	}

	// setup
	rio_info.set_io_address(port_base);

	if (!set_ix(rio_info, verbose, argv[optind]))
	    usage();

	if ( verbose )
		printf( "Initializing with bad block check %s.\n", 
			(fast == false) ? "enabled" : "disabled" );

	rio_info.initialize(!fast, 
		    progress_cb(verbose ? progress_callback : NULL));

	if (verbose)
	    printf( "updating directory\n" );

	rio_info.flush();
    }

    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (EXIT_IO_ERROR);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (EXIT_IO_ERROR);
    }
    return(0);
}

