/********************************************************
 * als -- list the memory in the rio			*
 *							*
 * Usage is: als [-v] [-l] [-p <base>] 			*
 * 	-v -- Verbose					*
 *	-l -- Long 					*
 * 	-p <base>	-- Set I/O port base		*
 *							*
 * Todo: External only flag.  Internal only flag.	*
 *		gob matching.				*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include "rio.h"
#include "common.h"
#include "time.h"


// The type of listing
enum LIST_TYPE {LIST_SHORT, LIST_LONG, LIST_VERY_LONG};

// The variable controling the list type
static enum LIST_TYPE list_type = LIST_SHORT;

/********************************************************
 * do_entry -- write out a single entry			*
 ********************************************************/
static void do_entry(
    const int index,		 // Entry's index
    const rio_dir_entry cur_entry // The entry to write
)
{
    switch (list_type) {
	case LIST_SHORT:
	    printf( "%s\n", cur_entry.name);
	    break;
	case LIST_LONG:
	    printf( "%02u  %-8ld  %s  %-40.40s\n",
		    index+1,
		    cur_entry.file_size_rem,
		    time_to_string(cur_entry.upload_time),
		    cur_entry.name
	    );
	    break;
	case LIST_VERY_LONG:
	    printf( "%02u 0x%04hx %-4hu     0x%04hx %-8ld %s %-24.24s\n",
		    index+1,
		    cur_entry.file_position,
		    cur_entry.file_size_in_32k,
		    cur_entry.file_size_blocks,
		    cur_entry.file_size_rem,
		    time_to_string(cur_entry.upload_time),
		    cur_entry.name
	    );
	    break;
	default:
	    assert(0);
    }
}
   
/********************************************************
 * do_header -- write out a heading for the entry list	*
 ********************************************************/
static void do_header(void)
{
    switch (list_type) {
	case LIST_SHORT:
	    break;
	case LIST_LONG:
	    printf( "No  Size      Upload Date/Time   Title\n" );
	    printf( "-----------------------------------------------------------------------------\n" );
	    break;
	case LIST_VERY_LONG:
	    printf( "No 32KPos 32KCount Mod32K Size     Upload Date/Time  Title\n" );
	    printf( "-----------------------------------------------------------------------------\n" );
	    break;
	default:
	    assert(0);
    }
}
/********************************************************
 * do_directory -- Display a directory			*
 ********************************************************/
static void do_directory(
    rio_class& rio_info,		// The RIO information
    const enum MEM_TYPE external_flag	// Display external memory
)
{
    rio_info.set_mem_type(external_flag);

    // The current directory block
    const rio_dir_struct& dir_block = rio_info.get_directory();
    // The current directory block header
    const rio_dir_header& dir_header = dir_block.header;


    u_int entry_count = dir_header.entry_count;
    if ( entry_count ) {

	if ( entry_count > CRIO_MAX_DIRENTRY )
	    entry_count = CRIO_MAX_DIRENTRY;

	do_header();
	// extended output
	for( u_int index=0; index<entry_count; ++index) {
	    // The current directory entry
	    const rio_dir_entry cur_entry = rio_info.dir_entry(index);

	    do_entry(index, cur_entry);
	}
    }

    switch (list_type) {
	case LIST_VERY_LONG:
	    printf( "flash ram type: %s\n", 
		    rio_info.get_mem_type() ? 
		    "external" : "internal" );
	    printf( "bad 32K blocks: %hu\n", dir_header.bad_block_count);
	    printf( "   last update: %s\n", 
		    time_to_string(dir_header.update_time) );
	    printf( "     checksum1: 0x%04hx\n", dir_header.check_sum1);
	    printf( "     checksum2: 0x%04hx\n", dir_header.check_sum2);
	    printf( "  host version: 0x%04hx\n", dir_header.version);
	    // Fall through
	case LIST_LONG:
	    if (list_type == LIST_LONG) {
		printf( "flash ram type: %s\n", 
			rio_info.get_mem_type() ? 
			"external" : "internal" );
	    }
	    printf( "   entry count: %hu\n", dir_header.entry_count );
	    printf( "  total memory: %ld KB\n", 
		    ((long)dir_header.blocks_available * 
		     CRIO_SIZE_32KBLOCK) / 1024 );

	    printf( "   used memory: %ld KB\n", 
		((long)dir_header.blocks_used * 
		 CRIO_SIZE_32KBLOCK) / 1024 );

	    printf( " unused memory: %ld KB\n", 
		    ((long)dir_header.blocks_remaining * 
		     CRIO_SIZE_32KBLOCK) / 1024 );
	    break;
	case LIST_SHORT:
	    break;
	default:
	    assert(0);
    }
}

/********************************************************
 * usage -- Tell the user what to do.			*
 ********************************************************/
static void usage(void)
{
    fprintf(stderr,"Usage is: als [-v] [-p <base>] \n");
    fprintf(stderr,"	-v -- Verbose\n");
    fprintf(stderr,"	-p <base>	-- Set I/O port base\n");
    exit (8);
}
/*------------------------------------------------------*/
int main(int argc, char* argv[])
{
    try {
	// The port base for the ls
	int port_base = PORT_BASE_DEFAULT;
	rio_class rio_info;				// The RIO informaiton

	while (1) {
	    // The current option
	    int opt = getopt(argc, argv, "lvp:");

	    if (opt == EOF)
		break;

	    switch (opt) {
		case 'l':
		    list_type = LIST_LONG;
		    break;
		case 'v':
		    list_type = LIST_VERY_LONG;
		    break;
		case 'p':
		    sscanf(optarg, "%x", &port_base);
		    break;
		default:
		    usage();
		    break;
	    }
	}

	// setup
	rio_info.set_io_address(port_base);

	do_directory(rio_info, MEM_INTERNAL);
	do_directory(rio_info, MEM_EXTERNAL);
    }

    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (EXIT_IO_ERROR);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (EXIT_IO_ERROR);
    }
    return(EXIT_GOOD);
}

