/********************************************************
 * amem -- Display memory size				*
 *							*
 * Usage is: amem [-v] [-p <base>] 			*
 *		[-f] [-b] [i|x]				*
 * 	-v -- Verbose					*
 * 	-p <base>	-- Set I/O port base		*
 *	-f 		-- Display free space only	*
 *	-b		-- Display size in bytes	*
 *	i|x		-- Display internal or		*
 *				external data		*
 *							*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include "rio.h"
#include "common.h"
#include "time.h"

/********************************************************
 * usage -- Tell the user what to do.			*
 ********************************************************/
static void usage(void)
{
    fprintf(stderr, "Usage is: amem [-v] [-p <base>]\n");
    fprintf(stderr, "	[-f] [-b] [i|x]\n");
    fprintf(stderr, "	-v -- Verbose\n");
    fprintf(stderr, "	-p <base>	-- Set I/O port base\n");
    fprintf(stderr, "	-f 		-- Display free space only\n");
    fprintf(stderr, "	-b		-- Display size in bytes\n");
    fprintf(stderr, "	i|x		-- Display internal or\n");
    fprintf(stderr, "			   external data\n");
    exit (8);
}

static bool bytes = false;	// Display in bytes
static bool free_flag = false;	// Display free space only
/********************************************************
 * display_mem -- display memory a directory		*
 ********************************************************/
static void display_mem(
    rio_class& rio_info, 		// The RIO information
    const bool verbose_flag, 		// The verbose flag
    const enum MEM_TYPE external_flag	// Display external memory
)
{
    rio_info.set_mem_type(external_flag);

    // The current directory block
    const rio_dir_struct& dir_block = rio_info.get_directory();
    // The current directory block header
    const rio_dir_header& dir_header = dir_block.header;

    if (free_flag) {
	if (bytes) {
	    printf( "%ld\n", 
		    ((long)dir_header.blocks_remaining * 
		     CRIO_SIZE_32KBLOCK));
	} else {
	    printf( "%ld\n", 
		    ((long)dir_header.blocks_remaining * 
		     CRIO_SIZE_32KBLOCK) / 1024 );
	}
	return;
    }

    printf( "   entry count: %hu\n", dir_header.entry_count );
    printf( "  total memory: %ld KB\n", 
	    ((long)dir_header.blocks_available * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( "   used memory: %ld KB\n", 
	    ((long)dir_header.blocks_used * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( " unused memory: %ld KB\n", 
	    ((long)dir_header.blocks_remaining * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( "flash ram type: %s\n", 
	    rio_info.get_mem_type() ? "external" : "internal" );

    if ( verbose_flag ) {
	printf( "bad 32K blocks: %hu\n", dir_header.bad_block_count);
	printf( "   last update: %s\n", 
		time_to_string(dir_header.update_time));
	printf( "     checksum1: 0x%04hx\n", dir_header.check_sum1);
	printf( "     checksum2: 0x%04hx\n", dir_header.check_sum2);
	printf( "  host version: 0x%04hx\n", dir_header.version);
    }
}

/*------------------------------------------------------*/
int main(int argc, char* argv[])
{
    try {
	// Tell lots of stuff
	bool verbose_flag = false;

	// The port base for the ls
	int port_base = PORT_BASE_DEFAULT;

	rio_class rio_info;				// The RIO informaiton


	while (1) {
	    // The current option
	    int opt = getopt(argc, argv, "fbvp:");

	    if (opt == EOF)
		break;

	    switch (opt) {
		case 'v':
		    verbose_flag = true;
		    break;
		case 'p':
		    sscanf(optarg, "%x", &port_base);
		    break;
		case 'f':
		    free_flag = true;
		    break;
		case 'b':
		    bytes = true;
		    break;
		default:
		    usage();
		    break;
	    }
	}

	// setup
	rio_info.set_io_address(port_base);

	switch (argc - optind) {
	    case 0:
		display_mem(rio_info, verbose_flag, MEM_INTERNAL);
		display_mem(rio_info, verbose_flag, MEM_EXTERNAL);
		break;
	    case 1:
		switch (argv[optind][0]) {
		    case 'i':
			display_mem(rio_info, verbose_flag, MEM_INTERNAL);
			break;
		    case 'x':
			display_mem(rio_info, verbose_flag, MEM_EXTERNAL);
			break;
		    default:
			fprintf(stderr, "Unknown rio drive %s\n", argv[optind]);
			usage();
			break;
		}
		break;
	    default:
		usage();
		break;
	}
    }

    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (EXIT_IO_ERROR);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (EXIT_IO_ERROR);
    }
    return(EXIT_GOOD);
}

