///////////////////////////////////////////////////////////////////////////////
//
//	app.cpp
//
//	see RIO.CPP for version history
//
///////////////////////////////////////////////////////////////////////////////
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<time.h>
#include	<sys/stat.h>
#include 	"string"
#include	"rio.h"

// platform dependencies
#if defined(_WINNT)
	// MS VC++ v5.0 for WinNT v4
	#include	<stdlib.h>
	#define		SIZE_MAXPATH			_MAX_PATH
	#define		DELETEARRAY				delete[]

#elif defined(_WIN32)
	// MS VC++ v5.0 for Win9x
	#include	<stdlib.h>
	#define		SIZE_MAXPATH			_MAX_PATH
	#define		DELETEARRAY				delete[]

#elif defined(__linux__)
	// linux g++
	#include	<unistd.h>
	#include	<values.h>
	#if defined(PATH_MAX)
		#define		SIZE_MAXPATH			PATH_MAX
	#else
		#define		SIZE_MAXPATH			256
	#endif
	#define		DELETEARRAY				delete[]

#elif defined(__TURBOC__)
	// turboc v1.01
	#include	<dir.h>
	#define		SIZE_MAXPATH			MAXPATH
	#define		DELETEARRAY				delete

#else
	// not supported
	#error ! ! compiler/platform not supported ! !
#endif

// default port base
#if defined(__alpha)
	#define		PORT_BASE_DEFAULT		0x3bc
#else
	#define		PORT_BASE_DEFAULT		0x378
#endif

// return code

/********************************************************
 * TimeStr -- Turn time into a string			*
 ********************************************************/
static char* TimeStr(
    const long time_value 	// Time in UNIX standard format
)
{
    static char result[64];	// Result in string format
    struct tm* tm_value;	// Time value split apart
    tm_value = localtime(&time_value);

    if ( !tm_value )
	strcpy( result, "INVALID DATE/TIME" );
    else {
	sprintf(
		result,
		"%02u/%02u/%02u %02u:%02u:%02u",
		(u_int)tm_value->tm_mday % 100,
		(u_int)tm_value->tm_mon+1 % 100,
		(u_int)tm_value->tm_year % 100,
		(u_int)tm_value->tm_hour % 100,
		(u_int)tm_value->tm_min % 100,
		(u_int)tm_value->tm_sec % 100
	);
    }

    return result;
}

///////////////////////////////////////////////////////////////////////////////
// return file size
static long GetFileSize( char* pszPathFile )
{
	long lReturn = 0;

	FILE* fpFile = fopen( pszPathFile, "rb" );
	if ( fpFile )
	{
		struct stat sStat;
		if ( !stat(pszPathFile, &sStat) )
			lReturn = sStat.st_size;

		fclose( fpFile );
	}

	return lReturn;
}

///////////////////////////////////////////////////////////////////////////////
// progress callback
static void progress_callback( const int iPos, const int iCount )
{
    printf("blocks %-5hd\r", iCount-iPos );
    fflush(stdout);
}

///////////////////////////////////////////////////////////////////////////////
// display directory
static void DisplayDirectory(
    rio_class& rio_info, 		// The RIO information
    const bool verbose_flag 	// The verbose flag
)
{
    // The current directory block
    const rio_dir_struct& cDirBlock = rio_info.get_directory();
    const rio_dir_header& cDirHeader = cDirBlock.header;

    printf( "\n" );
    printf( "   entry count: %hu\n", cDirHeader.entry_count );
    printf( "  total memory: %ld KB\n", 
	    ((long)cDirHeader.blocks_available * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( "   used memory: %ld KB\n", 
	    ((long)cDirHeader.blocks_used * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( " unused memory: %ld KB\n", 
	    ((long)cDirHeader.blocks_remaining * 
	     CRIO_SIZE_32KBLOCK) / 1024 );

    printf( "flash ram type: %s\n", 
	    rio_info.get_mem_type() ? "external" : "internal" );

    if ( verbose_flag ) {
	printf( "bad 32K blocks: %hu\n", cDirHeader.bad_block_count);
	printf( "   last update: %s\n", 
		TimeStr(cDirHeader.update_time) );
	printf( "     checksum1: 0x%04hx\n", cDirHeader.check_sum1);
	printf( "     checksum2: 0x%04hx\n", cDirHeader.check_sum2);
	printf( "  host version: 0x%04hx\n", cDirHeader.version);
    }

    u_int uientry_count = cDirHeader.entry_count;
    if ( uientry_count ) {
	// The current directory entry
	const rio_dir_entry* pDirEntry = cDirBlock.dir_entry;

	if ( uientry_count > CRIO_MAX_DIRENTRY )
	    uientry_count = CRIO_MAX_DIRENTRY;

	// extended output
	if ( verbose_flag ) {
	    printf( "\n" );
	    printf( "No 32KPos 32KCount Mod32K Size     Upload Date/Time  Title\n" );
	    printf( "-----------------------------------------------------------------------------\n" );

	    for( u_int uiA=0; uiA<uientry_count; ++uiA, ++pDirEntry ) {
		printf( "%02u 0x%04hx %-4hu     0x%04hx %-8ld %s %-24.24s\n",
			uiA+1,
			pDirEntry->file_position,
			pDirEntry->file_size_in_32k,
			pDirEntry->file_size_blocks,
			pDirEntry->file_size_rem,
			TimeStr(pDirEntry->upload_time),
			pDirEntry->name
		);
	    }
	} else { // normal output
	    printf( "\n" );
	    printf( "No  Size      Upload Date/Time   Title\n" );
	    printf( "-----------------------------------------------------------------------------\n" );

	    for( u_int uiA=0; uiA<uientry_count; ++uiA, ++pDirEntry ) {
		printf( "%02u  %-8ld  %s  %-40.40s\n",
			uiA+1,
			pDirEntry->file_size_rem,
			TimeStr(pDirEntry->upload_time),
			pDirEntry->name
		);
	    }
	}
    }
}

///////////////////////////////////////////////////////////////////////////////
// hex dump directory block
static void DumpDirectory( rio_class& rio_info, int iPos, int iSize )
{
    long int iA;
    int iB;

    const rio_dir_struct& cDirBlock = rio_info.get_directory();
    u_char* pucS = ((u_char*)&cDirBlock) + iPos;

    if ( iSize % 16 )
	iSize = ((iSize / 16)+1) * 16;

    for( iA=0; iA<(iSize/16); ++iA, pucS+=16 ) {
	printf( "%08lx  ", iA*16 );

	for( iB=0; iB<16; ++iB ) {
	    int iC = *(pucS+iB);
	    printf( "%02x ", iC );
	    if ( (iB%8) == 7 )
		printf( " " );
	}

	for( iB=0; iB<16; ++iB ) {
	    int iC = *(pucS+iB);
	    printf( "%c", iC > 32 ? iC : '.' );
	    if ( (iB%8) == 7 )
		printf( " " );
	}

	printf( "\n" );

	if ( (iA%8) == 7 )
	    printf( "\n" );
    }
}

///////////////////////////////////////////////////////////////////////////////
// process playlist file
static bool ProcessPlaylist(
    rio_class& rio_info, 
    const char pszFile[], 
    bool verbose_flag 
)
{
    char szBuf[ SIZE_MAXPATH ];

    // open playlist for read
    FILE* fpFile = fopen( pszFile, "r" );
    if ( !fpFile ) {
	fprintf(stderr,  "unable to open '%s' for read\n", pszFile );
	return false;
    }

    // check if device can accommodate all files in playlist
    const rio_dir_struct& cDirBlock = rio_info.get_directory();
    const rio_dir_header& cDirHeader = cDirBlock.header;

    long lSizeAvailable = 
	(long)cDirHeader.blocks_available * CRIO_SIZE_32KBLOCK;

    long lSizeCurrent = 
	(long)cDirHeader.blocks_used * CRIO_SIZE_32KBLOCK;

    unsigned int entry_countCurrent = cDirHeader.entry_count;

    while( fgets(szBuf, sizeof(szBuf), fpFile) ) {
	// strip 'new line' char
	szBuf[ strlen(szBuf)-1 ] = 0;

	// get file size
	long lSize = GetFileSize( szBuf );
	if ( !lSize ) {
	    fprintf(stderr,  "unable to open '%s' for read\n", szBuf );
	    fclose( fpFile );
	    return false;
	}

	// check space
	lSizeCurrent += lSize;
	if ( lSizeCurrent > lSizeAvailable ) {
	    fprintf(stderr,  "entry '%s' exceed's memory capacity of device\n", 
		    szBuf );
	    fclose( fpFile );
	    return false;
	}

	// check enough entries
	++entry_countCurrent;
	if ( entry_countCurrent > CRIO_MAX_DIRENTRY ) {
	    fprintf(stderr,  
		"entry '%s' exceed's maximum directory entry count for device\n", 
		szBuf );
	    fclose( fpFile );
	    return false;
	}
    }

    // rewind playlist
    rewind( fpFile );

    // process entries
    while( fgets(szBuf, sizeof(szBuf), fpFile) ) {
	// strip 'new line' char
	szBuf[ strlen(szBuf)-1 ] = 0;

	// upload
	if ( verbose_flag )
	    printf( "uploading %s\n", szBuf );

	rio_info.send_rio_file(szBuf, 
		progress_cb(verbose_flag ? progress_callback : NULL));
    }

    // close playlist file
    fclose( fpFile );

    return true;
}

///////////////////////////////////////////////////////////////////////////////
// change playlist order
static void ChangePlaylistOrder(
    rio_class& rio_info, 
    char* pszPlaylistOrder 
)
{
    u_int auiPosNew[ CRIO_MAX_DIRENTRY ];

    u_int uiPosNew = 0;
    char* pszEntry;
    do
    {
	pszEntry = strtok( (uiPosNew) ? (char *)NULL : pszPlaylistOrder, " " );
	if ( pszEntry )
		auiPosNew[ uiPosNew++ ] = atoi( pszEntry ) - 1;
    } while( pszEntry && (uiPosNew < CRIO_MAX_DIRENTRY) );

    rio_info.set_file_order(auiPosNew, uiPosNew);
}

///////////////////////////////////////////////////////////////////////////////
// display help
static void Help( void )
{
    printf( "\nRio utility v%d.%02d - Ashpool Systems (c) 1999\n", CRIO_ID_VERSION/100, CRIO_ID_VERSION%100 );
    printf( "--------------------------------------------\n" );
    printf( "command line switches available :-\n" );
    printf( "	-d  display directory\n" );
    printf( "	-iy initialize with check for bad blocks\n" );
    printf( "	-in initialize without check for bad blocks\n" );
    printf( "	-x  perform operations on external flash ram\n" );
    printf( "	-u  specify file(s) to upload\n" );
    printf( "	-g  specify file to download\n" );
    printf( "	-f  specify text based playlist file which contains files to be upload\n" );
    printf( "	-z  specify file to delete\n" );
    printf( "	-za delete all files\n" );
    printf( "	-o  specify new playlist order in quotes\n" );
    printf( "	-p  specify parallel port base IO address, default=0x%x\n", PORT_BASE_DEFAULT );
    printf( "	-v  enable verbose mode\n" );

    printf( "\nexamples...\n" );
    printf( "	; display directory using parallel port at 0x278\n" );
    printf( "		rio -p 0x278 -d\n" );
    printf( "	; initialize (with bad block check) and upload files\n" );
    printf( "		rio -iy -u *.mp3\n" );
    printf( "	; delete existing files and upload playlist onto external flash ram\n" );
    printf( "		rio -za -f playlist.txt -x\n" );
    printf( "	; initialize, upload files in playlist and then display directory\n" );
    printf( "		rio -d -in -f playlist.txt\n" );
    printf( "	; download file then delete it and finally display directory\n" );
    printf( "		rio -d -g mp3Files/song.mp3 -z song.mp3\n" );
    printf( "	; reverse playlist order and then display directory\n" );
    printf( "		rio -o \"5 4 3 2 1\" -d\n" );
}

///////////////////////////////////////////////////////////////////////////////
int main( int argc, char* argv[] )
{
    // default settings
    bool display_dir_flag = false;
    bool bInit = false;
    bool bInitMarkBadBlock = false;
    enum MEM_TYPE bUseExternalFlash = MEM_INTERNAL;
    bool bDeleteAll = false;
    bool verbose_flag = false;
    char* pszFileDelete = NULL;
    char* pszFileDownload = NULL;
    char* playlist_file = NULL;
    string playlist_tmp_file;		// Name of temp file for playlist
    char* pszPlaylistOrder = NULL;
    int port_base = PORT_BASE_DEFAULT;
    int iPosDumpDirectory = 0;
    int iSizeDumpDirectory = 0;
    rio_class rio_info;				// The RIO informaiton

    try {
	// process command line args
	if ( argc < 2 ) {
	    Help();
	    return( false );
	}
	for( int iA=1; iA<argc; ++iA ) {
	    // check for display directory request
	    if ( !strcmp(argv[iA], "-d") )
		display_dir_flag = true;
	    // check for init with mark bad block request
	    else if ( !strcmp(argv[iA], "-iy") )
	    {
		bInit = true;
		bInitMarkBadBlock = true;
	    }
	    // check for init without mark bad block request
	    else if ( !strcmp(argv[iA], "-in") )
		bInit = true;
	    // check for delete all file request
	    else if ( !strcmp(argv[iA], "-za") )
		bDeleteAll = true;
	    // check for request to use external flash memory
	    else if ( !strcmp(argv[iA], "-x") )
		bUseExternalFlash = MEM_EXTERNAL;
	    // check for verbose mode request
	    else if ( !strcmp(argv[iA], "-v") )
		verbose_flag = true;
	    // check for dump directory request
	    else if ( !strcmp(argv[iA], "-h") ) {
		if ( (iA+2) < argc ) {
		    ++iA;
		    iPosDumpDirectory = atoi( argv[iA] );
		    ++iA;
		    iSizeDumpDirectory = atoi( argv[iA] );
		}
	    }
	    // check for delete file request
	    else if ( !strcmp(argv[iA], "-z") ) {
		if ( (iA+1) < argc ) {
		    ++iA;
		    pszFileDelete = argv[iA];
		}
	    }
	    // check for upload file request
	    else if ( !strcmp(argv[iA], "-u") ) {
		if ( (iA+1) < argc ) {
		    ++iA;

		    // append all command line upload requests to temp playlist
		    if (playlist_tmp_file == "")
			playlist_tmp_file = tmpnam( NULL );

		    if ( playlist_tmp_file != "") {
			FILE* fpFile = fopen(playlist_tmp_file.c_str(), "a" );
			if ( !fpFile ) {
			    fprintf(stderr,  
				    "unable to open '%s' for append\n", 
				    playlist_tmp_file.c_str());
			    return( false );
			}
			while( iA < argc ) {
			    fprintf( fpFile, "%s\n", argv[iA] );
			    if ( (iA+1) < argc ) {
				if ( argv[iA+1][0] == '-' )
				    break;
			    }
			    ++iA;
			}
			fclose( fpFile );
		    }
		}
	    }
	    // check for download file request
	    else if ( !strcmp(argv[iA], "-g") ) {
		if ( (iA+1) < argc ) {
		    ++iA;
		    pszFileDownload = argv[iA];
		}
	    }
	    // check for playlist file request
	    else if ( !strcmp(argv[iA], "-f") ) {
		if ( (iA+1) < argc ) {
		    ++iA;
		    playlist_file = argv[iA];
		}
	    }
	    // check for playlist order request
	    else if ( !strcmp(argv[iA], "-o") ) {
		if ( (iA+1) < argc ) {
		    ++iA;
		    pszPlaylistOrder = argv[iA];
		}
	    }
	    // check for port base request
	    else if ( !strcmp(argv[iA], "-p") ) {
		if ( (iA+1) < argc ) {
		    ++iA;
		    sscanf( argv[iA], "%x", &port_base );
		}
	    }
	    // else help
	    else {
		Help();
		return( false );
	    }
	}

	// setup
	rio_info.set_io_address(port_base);

	// select internal or external flash memory
	if ( verbose_flag )
	    printf( "performing operations on %s flash ram\n", 
		    bUseExternalFlash == MEM_EXTERNAL ? "external" : "internal" );
	rio_info.set_mem_type( bUseExternalFlash );

	// update directory flag
	bool bUpdateDirectory = false;

	// get current directory
	if ( verbose_flag )
	    printf( "downloading directory\n" );

	rio_info.flush();

	// if dump directory request
	if ( iSizeDumpDirectory )
	    DumpDirectory( rio_info, iPosDumpDirectory, iSizeDumpDirectory );

	// if init request
	if ( bInit ) {
	    if ( verbose_flag )
		    printf( "initializing with bad block check %s\n", 
			    bInitMarkBadBlock ? "enabled" : "disabled" );

	    rio_info.initialize(bool(bInitMarkBadBlock), 
			(const progress_cb) 
		      	(verbose_flag ? progress_callback : NULL) );

	    bUpdateDirectory = true;
	}

	// if playlist request
	if ( playlist_file ) {
	    if ( !ProcessPlaylist(rio_info, playlist_file, verbose_flag) )
		return( false );

	    bUpdateDirectory = true;
	}

	// if temp playlist request
	if (playlist_tmp_file != "") {
	    if ( !ProcessPlaylist(rio_info, playlist_tmp_file.c_str(), verbose_flag) )
		return( false );

	    bUpdateDirectory = true;
	}

	// if download request
	if ( pszFileDownload ) {
	    if ( verbose_flag )
		printf( "downloading %s\n", pszFileDownload );

	    rio_info.read_rio_file(pszFileDownload, 
			progress_cb(verbose_flag ? progress_callback : NULL));
	}

	// if delete request
	if ( pszFileDelete ) {
	    if ( verbose_flag )
		printf( "deleting %s\n", pszFileDelete );

	    rio_info.remove_file(pszFileDelete);

	    bUpdateDirectory = true;
	}

	// if delete all request
	if ( bDeleteAll ) {
	    if ( verbose_flag )
		printf( "deleting all files\n" );

	    rio_info.remove_all_files();
	    bUpdateDirectory = true;
	}

	// if playlist order request
	if ( pszPlaylistOrder ) {
	    if ( verbose_flag )
		printf( "changing playlist order\n" );

	    ChangePlaylistOrder(rio_info, pszPlaylistOrder);
	    bUpdateDirectory = true;
	}

	// if directory update required
	if ( verbose_flag )
	    printf( "updating directory\n" );

	rio_info.flush();

	// if display directory request
	if (display_dir_flag)
	    DisplayDirectory(rio_info, verbose_flag);
    } 
    catch (rio_error &error) {
	fprintf(stderr, "ERROR: %s (%d)\n", error.msg, error.error_code);
	exit (8);
    }
    catch (...) {
	fprintf(stderr, "ERROR: Unknown exception\n");
	exit (8);
    }

    // restore
    return(0);
}

