/********************************************************
 * rio.cpp -- Rio manipulation classes.			*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>

#include "common.h"
#include "binary.h"
#include "rio.h"
#include "rio_error.h"

// platform dependencies
#if defined(_WINNT)
	// MS VC++ v5.0 for WinNT v4
	#include	<windows.h>
	#include	<winioctl.h>
	#include	"rioioctl.h"
	#define		OUTPORT( p, v )			WinNTOutPort( p, v )
	#define		INPORT( p )			WinNTInPort( p )
	#define		CLOCK_SECOND			1000
	#define		DELETEARRAY			delete[]
	#define		ID_DRIVER_VERSION		101

#elif defined(_WIN32)
	// MS VC++ v5.0 for Win9x
	#include	<conio.h>
	#define		OUTPORT( p, v )			_outp( p, v )
	#define		INPORT( p )			_inp( p )
	#define		CLOCK_SECOND			1000
	#define		DELETEARRAY			delete[]

#elif defined(__linux__)
	// linux g++
	#include	<unistd.h>
	#if defined(__alpha)
		#include	<sys/io.h>
	#else
		#include	<sys/perm.h>
	#endif
	#include	<asm/io.h>
	#define		OUTPORT(p,v)			outb( v, p )
	#define		INPORT(p)			inb( p )
	static const unsigned long int	CLOCK_SECOND =	1000000;
	#define		DELETEARRAY			delete[]

#elif defined(__TURBOC__)
	// turboc v1.01
	#include	<dos.h>
	#define		OUTPORT( p, v )			outp( p, v )
	#define		INPORT( p )			inp( p )
	#define		CLOCK_SECOND			18
	#define		DELETEARRAY			delete

#else
	// not supported
	#error ! ! compiler/platform not supported ! !
#endif

const int MAX_RETRY = 3; // max tx/rx block retry
const int BLOCK_SIZE = 512;	// Size of a block

// block to use to check for presense
static const unsigned int ID_32KBLOCK_CHECKPRESENT = 0;

// io delay constants
/********************************************************
 * io delay constants.					*
 *	These constants appear to be based on nothing	*
 *	more than guesswork.  We should replace them	*
 *	with something more precise.			*
 ********************************************************/
#if defined(_WINNT)
	const unsigned int IODELAY_1 = 2000;	// Used during initialization
	const unsigned int IODELAY_2 = 2;	// For getting data
	const unsigned int IODELAY_3 = 10;	// Sending data
#else	/* _WINNT */
	const unsigned int IODELAY_1 = 20000;	// Used during initialization
	const unsigned int IODELAY_2 = 2;	// For getting data
	const unsigned int IODELAY_3 = 100;	// Sending data
#endif	/* _WINNT */

///////////////////////////////////////////////////////////////////////////////
// if WinNT
#if defined(_WINNT)

	// handle
	static HANDLE m_hDriver;

	// WinNT out port
	void WinNTOutPort( int iPort, int iValue )
	{
		DWORD dwSizeReturn;
		u_long ulInput = ((u_long)iPort-0x378) | ((ULONG)iValue << 16);
		DeviceIoControl( m_hDriver, RIOIO_IOCTL_WRITE, &ulInput,
			sizeof(long), NULL, 0, &dwSizeReturn, NULL );
	}

	// WinNT in port
	int WinNTInPort( int iPort )
	{
		DWORD dwSizeReturn;
		u_long ulPort = iPort - 0x378;
		u_long ulData = 0;
		DeviceIoControl( m_hDriver, RIOIO_IOCTL_READ, &ulPort, sizeof(ulPort),
			&ulData, sizeof(char), &dwSizeReturn, NULL );
		return (int)ulData;
	}

#endif

///////////////////////////////////////////////////////////////////////////////
// if DOS
#if defined(__TURBOC__)
	// get clock ticks
	long clock( void )
	{
		return (long) (*(int far*)MK_FP( 0x40, 0x6c ));
	}
#endif

///////////////////////////////////////////////////////////////////////////////
/********************************************************
 * get_file -- return the file name only of a file	*
 ********************************************************/
static const char *const get_file(
    const char file_with_path[]	// File name to get the base of
)
{
    // Length of the file name
    int length = strlen( file_with_path );

    // File=="", real easy
    if ( length == 0 )
	return file_with_path;

    // Pointer to the current character in the file
    const char* cur_char = file_with_path + length - 1;

    while( *cur_char != '\\' && *cur_char != '/' && *cur_char != ':' )
    {
	// If we backup to the front of the path
	if ( cur_char == file_with_path )
	    return cur_char;
	--cur_char;
    }
    ++cur_char;

    return cur_char;
}

/********************************************************
 * unset_io_address -- clear the use of an i/o address	*
 ********************************************************/
void rio_class::unset_io_address( void )
{
#if defined(_WINNT) /* if WinNT */
    // close device file
    if ( m_hDriver ) {
	CloseHandle( m_hDriver );
	m_hDriver = NULL;
    }
#endif
}

/********************************************************
 * rio_class::rio_class -- Initialize the class		*
 ********************************************************/
rio_class::rio_class(): dir(*this, io)
{
    // if WinNT
#if defined(_WINNT)
    m_hDriver = NULL;
#endif

    // default to use internal flash
    mem_type = MEM_INTERNAL;
    present = false;	// Check present has not run
}

/********************************************************
 * rio_class::~rio_class -- destroy class, unuse addr.	*
 ********************************************************/
rio_class::~rio_class()
{
    unset_io_address();
}

/*------------------------------------------------------*/

/********************************************************
 * find_first_free -- find the first free dir		*
 *	entry.						*
 ********************************************************/
u_int rio_class::find_first_free( void )
{
    u_int index;	// index of the free directory block
    // Pointer to the directory entry
    u_char* dir_ptr = dir.blocks_used();

    for(index = 0; index < CRIO_MAX_32KBLOCK; ++index, ++dir_ptr ) {
	if ( *dir_ptr == CRIO_ID_32KBLOCK_FREE )
	    return index;
    }
    return 0xffff;
}

/********************************************************
 * mark_bad_blocks -- Scan the memory for bad blocks	*
 * 	and mark any problems.				*
 ********************************************************/
void rio_class::mark_bad_blocks(
    const progress_cb progress_function	// Tells how far we are getting
)
{
    // create temp block
    data_block cur_block;

    // block count
    u_short end_block_pos;

    if (mem_type == MEM_EXTERNAL)
	end_block_pos = external_memory_size_32K;
    else
	end_block_pos = CRIO_COUNT_32KBLOCKIN32M;

    u_char *blocks_used = dir.blocks_used();

    // Block 0 is marked bad, but actually used
    blocks_used[0] = CRIO_ID_32KBLOCK_USED;

    // process all blocks (except directory block)
    int bad_block_count = 0;	// # bad blocks
    u_short pos_block;		// Position of the current 32K block

    for(pos_block = 1; pos_block < end_block_pos; ++pos_block )
    {
	u_int block_index;	// Index into the current block

	// progress callback
	if (progress_function) {
	    progress_function(pos_block, end_block_pos);
	}

	// default to bad block
	u_char state = CRIO_ID_32KBLOCK_BAD;

	// create and send test block 1
	memset(cur_block.ptr(), B_10101010, CRIO_SIZE_32KBLOCK);
	io.send_block_to_rio(cur_block, pos_block, 0, 0);

	// get test block 1
        io.get_block_from_rio(cur_block, pos_block);

	// check test block 1
	for(block_index = 0; block_index < CRIO_SIZE_32KBLOCK; ++block_index) {
	    if (cur_block[block_index] != B_10101010 )
		break;
	}
	// if ok
	if (block_index == CRIO_SIZE_32KBLOCK) {
	    // create and send test block 2
	    memset(cur_block.ptr(), B_01010101, CRIO_SIZE_32KBLOCK);
	    io.send_block_to_rio(cur_block, pos_block, 0, 0);

	    // get test block 2
	    io.get_block_from_rio(cur_block, pos_block);

	    // check test block 2
	    for(block_index = 0; block_index < CRIO_SIZE_32KBLOCK; 
		    ++block_index) {
		if (cur_block[block_index] != B_01010101)
		    break;
	    }
	    // if ok, block ok
	    if (block_index == CRIO_SIZE_32KBLOCK)
		state = CRIO_ID_32KBLOCK_FREE;
	}

	// store block state
	blocks_used[pos_block] = state;

	// adjust bad block count
	if ( state == CRIO_ID_32KBLOCK_BAD ) {
	    ++bad_block_count;
	    --dir.header().blocks_remaining;
	}
    }
    // if tx,rx block error or interrupted
    if (pos_block < end_block_pos)
	throw (rio_error(CRIO_ERROR_INTERRUPTED, 
		    "Unable to complete bad block scan"));

    // store blocks used and bad block count
    dir.header().blocks_used = bad_block_count;
    dir.header().bad_block_count = bad_block_count;
}

/********************************************************
 * set_mem_type -- Decide which memory to use		*
 ********************************************************/
void rio_class::set_mem_type(
    const enum MEM_TYPE new_mem_type	// Type of memory to use
)
{
    if (new_mem_type == mem_type)
	return;

    dir.dir_write();			// Write out any old dir
    io.set_mem_type(new_mem_type);
    mem_type = new_mem_type;
    dir.clobber();			// Directory is invalid
    check_present();
}

/********************************************************
 * rio_class::check_present -- Check to see if a RIO is	*
 *	present.					*
 *							*
 * This program:					*
 *	1. Reads the ID block.				*
 *	2. Writes a test pattern to the ID block	*
 *	3. Reads the test pattern back in		*
 *	4. Verifies that the test pattern worked	*
 *	5. Restores the original ID block.		*
 *							*
 * Return:						*
 *	true -- Rio present.				*
 *	false -- No Rio found.				*
 ********************************************************/
void rio_class::check_present(void)
{
    // If we've already checked, return
    if (present)
	return;

    // The index into the data we are looking at
    unsigned long byte_index;

    data_block id_block;		// Place to save the original id block

    io.get_block_from_rio(id_block, ID_32KBLOCK_CHECKPRESENT);

    try {
	// make copy of block
	data_block test_block;

	// Generate the test block
	for( byte_index=0; byte_index<CRIO_SIZE_32KBLOCK; ++byte_index )
	    test_block[byte_index] = (u_char)byte_index;

	// Send the test block
	    io.send_block_to_rio(test_block, ID_32KBLOCK_CHECKPRESENT, 0, 0 );

	/* Get the block a second time */
	memset(test_block.ptr(), 0, CRIO_SIZE_32KBLOCK );
	io.get_block_from_rio(test_block, ID_32KBLOCK_CHECKPRESENT );

	// compare test block read back against sent
	for(byte_index=0; 
	    byte_index < CRIO_SIZE_32KBLOCK;
	    ++byte_index ) {
		if ( test_block[byte_index] != (u_char)byte_index )
			break;
	}

	// if block different
	if ( byte_index < (24*1024) )
	    throw (rio_error(CRIO_ERROR_DEVICENOTFOUND, 
			"device not found (miscompare)"));

	//restore original block
	io.send_block_to_rio(id_block, ID_32KBLOCK_CHECKPRESENT, 0, 0);
    }
    catch (rio_error &)
    {
	throw (rio_error(CRIO_ERROR_DEVICENOTFOUND, "device not found"));
    }


    // The code indicating the size of the external memory (code)
    u_char memory_size_code = io.get_memory_size();
    // The code indicating the size of the external memory (bytes)
    u_long memory_size_512;

    switch( memory_size_code ) {
	case 0xEA:
	case 0x64:	
	    memory_size_512 = 4096;
	    break;

	case 0xE3:
	case 0xE5:	
	    memory_size_512 = 8192;
	    break;

	case 0xE6:	
	    memory_size_512 = 16384;	
	    break;

	case 0x73:	
	    memory_size_512 = C32K;	
	    break;

	case 0x75:	
	    memory_size_512 = 65536;	
	    break;

	default:	
	    memory_size_512 = 0;		
	    break;
    };

    // convert page count to 32K block count
    external_memory_size_32K = 
	(memory_size_512 * BLOCK_SIZE) / CRIO_SIZE_32KBLOCK;
    present = true;
}


/********************************************************
 * initialize -- initialize the memory			*
 ********************************************************/
void rio_class::initialize(
    const bool mark_bad_blocks_flag, 	// If true, mark bad blocks

    // Display progress
    const progress_cb progress_function
)
{
    // available blocks
    if (mem_type == MEM_EXTERNAL)
	dir.initialize(external_memory_size_32K);
    else
	dir.initialize(CRIO_COUNT_32KBLOCKIN32M);

    u_char *blocks_used = dir.blocks_used();
    // assume directory block is ok
    blocks_used[0] = CRIO_ID_32KBLOCK_BAD;
    dir.header().blocks_used = 1;

    // if mark bad block request
    if (mark_bad_blocks_flag) {
	mark_bad_blocks(progress_function);
    } else {
	/*
	 * Mark the blocks that are known to be used and bad
	 *
	 * Note: I've only tested the bad block list for the 
	 * external cards I have (8 and 32MB).
	 */
	// The used array
	u_char *blocks_used = dir.blocks_used();

	if (mem_type == MEM_EXTERNAL) {
	    blocks_used[0] = CRIO_ID_32KBLOCK_USED;
            blocks_used[138] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[394] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[395] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[431] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[443] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[475] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[540] = CRIO_ID_32KBLOCK_BAD;
	    blocks_used[761] = CRIO_ID_32KBLOCK_BAD;
	} else {
	    // Internal memory 
	    blocks_used[0] = CRIO_ID_32KBLOCK_USED;
	}
    }
}



/********************************************************
 * set_file_order -- Specify the play order.		*
 ********************************************************/
void rio_class::set_file_order(
    const u_int play_order[], 	// The order to play
    const u_int entry_count 	// Number of entries
)
{
    // create copy of directory entries
    rio_dir_entry* dir_entry = dir.get_entry_array();
    rio_dir_entry* new_dir_entry = new rio_dir_entry[CRIO_MAX_DIRENTRY];
    if (!new_dir_entry)
    {
	throw (rio_error(CRIO_ERROR_ALLOC, "new failed"));
    }
    memcpy(new_dir_entry, dir_entry, 
	    CRIO_MAX_DIRENTRY * sizeof(rio_dir_entry));

    // current entry count
    u_int uientry_count = dir.header().entry_count;

    u_int new_pos_index;	// Index for the entry we are moving

    // move entries
    for(new_pos_index=0; new_pos_index < entry_count; ++new_pos_index) {
	// Index of the entry from the old dir
	u_int old_dir_index = play_order[new_pos_index];

	if ((old_dir_index >= uientry_count) || 
	    (new_pos_index >= uientry_count)) {
	    DELETEARRAY new_dir_entry;
	    throw (rio_error(CRIO_ERROR_INVALIDFILEPOSITION, 
			"invalid file position"));
	}

	memcpy(&new_dir_entry[new_pos_index], 
	       &dir_entry[old_dir_index], sizeof(rio_dir_entry));
    }

    // update current directory entries
    memcpy(dir_entry, new_dir_entry, 
	    CRIO_MAX_DIRENTRY * sizeof(rio_dir_entry));

    DELETEARRAY new_dir_entry;
}

/********************************************************
 * send_rio_file -- Tranmit a file to the rio.		*
********************************************************/
void rio_class::send_rio_file(
    const char file_name_path[], 	// The path name

    // The progress report
    const progress_cb progress_function
)
{
    dir.dir_read();	// Make sure that the directory is in memory

    // directory header
    rio_dir_header& dir_header = dir.header();

    // if no enough room in directory for entry
    if (dir_header.entry_count >= CRIO_MAX_DIRENTRY) {
	throw(rio_error(CRIO_ERROR_MAXDIRENTRY, 
		error_str(
		    "max number of directory entries (%d) already in use", 
		CRIO_MAX_DIRENTRY)));
    }

    // open file for read
    FILE* in_file = fopen(file_name_path, "rb");
    if (!in_file) {
	throw (rio_error(CRIO_ERROR_OPEN, 
		error_str("unable to open '%s' for read", file_name_path)));
    }

    // get fileinfo
    struct stat file_info;

    if (stat(file_name_path, &file_info)) {
	fclose(in_file);
	throw (rio_error(CRIO_ERROR_STAT, 
		    error_str("stat() failed for '%s'", file_name_path)));
    }

    // determine if enough room to store file
    u_short file_size_in_32k = (file_info.st_size/CRIO_SIZE_32KBLOCK) +
	    (file_info.st_size % CRIO_SIZE_32KBLOCK ? 1 : 0);

    if ( file_size_in_32k > dir_header.blocks_remaining ) {
	fclose(in_file);
	throw (rio_error(CRIO_ERROR_MEMORY, 
		error_str("not enough memory on device to upload '%s'", 
		    file_name_path)));
    }

    // get first four bytes of file which determine mp3 properties
    u_char mp3_property[4];

    // How far we've searched for properities
    long int first_byte;

    for (first_byte = 0; first_byte < 1024; ++first_byte) {
	fseek(in_file, first_byte, SEEK_SET);
	if (fread(mp3_property, sizeof(mp3_property), 1, in_file) < 1 ) {
	    fclose(in_file);
	    throw (rio_error(CRIO_ERROR_READ, 
		    error_str("error reading from file '%s'", 
			file_name_path)));
	}
	// We are looking for
	//	fff3 or fff2
	//	|||+----------- 001x
	//	|||		|||+----- Error Checking bit (don't care)
	//	|||		|++------ 01 -- Layer 3
	//	|||		+-------- ISO Audio flag
	//	+++------------- 12 bits of sync
	if ((mp3_property[0] == 0xFF) && ((mp3_property[1] & 0xFE) == 0xF2)) {
	    break;
	}
	mp3_property[0] = 0;	// Signal error
    }
    // When we get here mp3_property[0] == ff -- OK
    // 				     0 -- error
    if (mp3_property[0] == 0) {
	throw (rio_error(CRIO_ERROR_BAD_FILE, "Unknown file format"));
    }
    fseek(in_file, first_byte, SEEK_SET);

    // point to directory entry that will be used
    rio_dir_entry& dir_entry = dir.allocate_entry(file_size_in_32k);

    // find first free 32K block
    u_short free_pos = find_first_free();
    if (free_pos == 0xffff) {
	fclose(in_file);
	throw (rio_error(CRIO_ERROR_CORRUPT, 
		"no free 32K blocks, initialization recommended"));
    }

    // update directory entry
    memset(&dir_entry, 0, sizeof(rio_dir_entry));
    dir_entry.file_position = free_pos;
    dir_entry.file_size_in_32k = file_size_in_32k;
    dir_entry.file_size_blocks = (file_info.st_size - first_byte) % 
			CRIO_SIZE_32KBLOCK;
    dir_entry.file_size_rem = file_info.st_size - first_byte;
    dir_entry.upload_time = time(NULL);
    dir_entry.mp3_property[0] = mp3_property[3];
    dir_entry.mp3_property[1] = mp3_property[2];
    dir_entry.mp3_property[2] = mp3_property[1];
    dir_entry.mp3_property[3] = mp3_property[0];
    strncpy(dir_entry.name, 
	    get_file(file_name_path), sizeof(dir_entry.name) );

    // create zero initialized temp block
    data_block cur_block;	

    cur_block.zero();

    // process all 32K blocks of file
    u_short cur_block_pos = dir_entry.file_position;

    // Location of the previous block
    u_short prev_block_pos = 0xffff;

    // Location of the end block in the file
    u_short end_block_pos = dir_entry.file_size_in_32k;

    // Location of the next block
    u_short next_block_pos;

    // The index of the current block
    u_short cur_block_index;

    for( cur_block_index=0; cur_block_index<end_block_pos; ++cur_block_index ) {
	// progress callback
	if ( progress_function ) {
	    progress_function(cur_block_index, end_block_pos);
	}

	u_char *blocks = dir.blocks_used();
	// mark as block used
	blocks[cur_block_pos] = CRIO_ID_32KBLOCK_USED;

	// get next block and update FAT
	if (cur_block_index == (end_block_pos-1) ) {
	    next_block_pos = 0xffff;
	    dir.FAT(cur_block_pos) = 0;
	} else {
	    next_block_pos = find_first_free();
	    if (next_block_pos == 0xffff ) {
		throw(rio_error(CRIO_ERROR_CORRUPT, 
			"no free 32K blocks, initialization recommended"));
	    }
	    dir.FAT(cur_block_pos) = next_block_pos;
	}

	// determine read size
	u_int read_size;

	if ((cur_block_index == (end_block_pos-1)) && 
		dir_entry.file_size_blocks) {
	    read_size = dir_entry.file_size_blocks;
	    memset(cur_block.ptr()+read_size, 0, CRIO_SIZE_32KBLOCK-read_size );
	} else
	    read_size = CRIO_SIZE_32KBLOCK;

	// read block
	if (fread(cur_block.ptr(), read_size, 1, in_file) < 1 ) {
	    throw (rio_error(CRIO_ERROR_READ, 
		    error_str("error reading from file '%s'", 
			file_name_path)));
	}

	// tx block
	io.send_block_to_rio(cur_block, 
		cur_block_pos, prev_block_pos,
		next_block_pos);

	// update current and prev block markers
	prev_block_pos = cur_block_pos;
	cur_block_pos = next_block_pos;
    }

    // IO outro
    io.finish();

    // close file
    fclose(in_file);
}

/********************************************************
 * read_rio_file -- Read a file from the rio and 	*
 *		write to disk				*
 ********************************************************/
void rio_class::read_rio_file(
    const char file_name_path[], 	// The file name to write

    // The progress function
    const progress_cb progress_function
)
{
    // get directory entry for file
    const char* File = get_file(file_name_path);

    // The directory entry to look at
    rio_dir_entry* dir_entry_ptr = dir.find_file(File);

    if (dir_entry_ptr == NULL) {
	throw (rio_error(CRIO_ERROR_FILENOTFOUND, 
		error_str("file '%s' not present on device", File)));
    }

    // open file for write
    FILE* out_file = fopen(file_name_path, "wb" );

    if (out_file == NULL) {
	throw (rio_error(CRIO_ERROR_OPEN, 
		error_str("unable to open '%s' for write", 
		    file_name_path)));
    }

    data_block cur_block; // create temp block

    // The current block we are reading
    u_short cur_block_pos = dir_entry_ptr->file_position;

    // The last block 
    u_short end_block_pos = dir_entry_ptr->file_size_in_32k;

    u_short block_index;	// Block counter

    for(block_index=0; block_index<end_block_pos; ++block_index) {
	// progress callback
	if (progress_function) {
	    progress_function(block_index, end_block_pos);
	}

	// rx block
	io.get_block_from_rio(cur_block, cur_block_pos);

	// determine write size
	int write_size;

	if ((block_index == (end_block_pos-1)) && dir_entry_ptr->file_size_blocks)
	    write_size = dir_entry_ptr->file_size_blocks;
	else
	    write_size = CRIO_SIZE_32KBLOCK;

	// save block
	if (fwrite(cur_block.ptr(), write_size, 1, out_file) < 1 ) {
	    throw (rio_error( CRIO_ERROR_WRITE, 
		    error_str("error writing to file '%s'", 
			file_name_path)));
	}

	// next block
	cur_block_pos = dir.FAT(cur_block_pos);
    }

    io.finish();
    fclose(out_file);
}
/********************************************************
 * rio_dir::~rio_dir -- Destroy the directory		*
 ********************************************************/
rio_dir::~rio_dir(void)
{
    dir_write();
}
/********************************************************
 * dir_read -- Read in a directory			*
 ********************************************************/
void rio_dir::dir_read(void)
{
    if (valid) return;

    // create temp block
    data_block raw_dir_block;
    io.get_block_from_rio(raw_dir_block, 0);

    // io outro
    io.finish();

    // store directory
    memcpy(&dir_block, raw_dir_block.ptr(), sizeof(dir_block) );

    // validate checksums
    is_valid();
    valid = true;
}
/********************************************************
 * rio_dir::dir_write -- Write out the directory	*
 *	and reset the dirty flag.			*
 ********************************************************/ 
void rio_dir::dir_write(void)
{
    if (!dirty) return;

    // create zero initialized temp block
    data_block cur_block;
    cur_block.zero();

    // directory header
    rio_dir_header& dir_header = dir_block.header;

    // update directory header time
    dir_header.update_time = time(NULL);
    /*
     * This hack makes it easy to compare directories entries
     * written by the rio software and written by us.
     */
    if (getenv("DEBUG_TIME") != NULL)
	dir_header.update_time = atol(getenv("DEBUG_TIME"));

    // create checksums (note: second checksum 
    // 	needs to be calculated first as
    //	this makes up part of the first checksum)
    dir_header.check_sum2 = calculate_check_sum2();
    dir_header.check_sum1 = calculate_check_sum1();

    // copy directory block to temp block
    memcpy(cur_block.ptr(), &dir_block, sizeof(dir_block));

    // send block
    io.send_block_to_rio(cur_block, 0, 0, 0 );
    io.finish();

    dirty = false;
}

/********************************************************
 * is_valid -- Return true if the directory is valid	*
 ********************************************************/
void rio_dir::is_valid(void)
{
    u_short check_sum1 = calculate_check_sum1();
    u_short check_sum2 = calculate_check_sum2();
    if (
	check_sum1 != dir_block.header.check_sum1 ||
	check_sum2 != dir_block.header.check_sum2
    ) {
	throw(rio_error(CRIO_ERROR_CORRUPT, 
		"invalid directory checksum, initialization recommended"));
    }
}

/********************************************************
 * calculate checksum1 (checksum of directory header)	*
 ********************************************************/
u_short rio_dir::calculate_check_sum1(void)
{
    // The checksum itself
    u_short check_sum = dir_block.header.check_sum1;

    // Pointer to the current sum
    u_short* data_ptr = (u_short*)&dir_block.header;

    unsigned int index;	// Index for counting the sum

    for(index=0; index<(sizeof(rio_dir_header)/sizeof(short)); ++index )
	    check_sum -= *data_ptr++;

    return check_sum;
}

/********************************************************
 * calculate_check_sum2 -- Calculate second checksum	*
 *  (checksum of directory entries, used flags and FAT) *
 ********************************************************/
u_short rio_dir::calculate_check_sum2(void)
{
    u_short check_sum = 0;	// The calculated checksum

    // Pointer to the current data we are checksumming
    u_short* cur_short = (u_short*)dir_block.dir_entry;

    // The size of what to checksum
    int size = ( sizeof(rio_dir_struct) - sizeof(rio_dir_header) ) / 
		sizeof(short);

    int index;	// Index for checksumming

    for(index = 0; index < size; ++index) {
	check_sum -= *cur_short;
	cur_short++;
    }
    return check_sum;
}
/********************************************************
 * get_dir_entry -- Get a directory entry		*
 *							*
 * Returns -- the entry from the list			*
 ********************************************************/
const rio_dir_entry &rio_dir::get_dir_entry(
    const unsigned int index	// Index of the entry to get
)
{
    if (!valid) {
	dir_read();
    }

    assert(valid);
    assert(index <= dir_block.header.entry_count);
    return (dir_block.dir_entry[index]);
}
/********************************************************
 * initialize -- initialize a directory.		*
 *							*
 * Used during the formatting of an entry		*
 ********************************************************/
void rio_dir::initialize(
    const int size		// Size of the memory
)
{
    // The directory header
    rio_dir_header& dir_header = dir_block.header;

    // init directory header
    memset(&dir_header, 0, sizeof(dir_header) );

    //	set version (so compatible with Rio Manager v1.01)
    dir_header.version = 0x0100;

    dir_header.one = 0xFFFF;	// A version number (?)

    // init directory entries
    memset(&dir_block.dir_entry, 0, sizeof(dir_block.dir_entry) );

    // init block used flags
    memset(&dir_block.block_used, CRIO_ID_32KBLOCK_FREE, 
	    sizeof(dir_block.block_used));

    // init FAT
    memset(&dir_block.FAT, 0, sizeof(dir_block.FAT) );

    // available blocks
    dir_header.blocks_available = size;

    // blocks remaining (taking into account bad blocks)
    dir_header.blocks_remaining =
	    dir_header.blocks_available;

    dirty = true;
}

/********************************************************
 * find_file -- Locate a file name in a directory	*
 ********************************************************/
rio_dir_entry* rio_dir::find_file(
    const char File[]
)
{
    dir_read();

    // search directory entries for matching filename
    int entry_count = dir_block.header.entry_count;

    // The entry we are looking at
    rio_dir_entry* dir_entry_ptr = dir_block.dir_entry;

    for( int index=0; index < entry_count; ++index, ++dir_entry_ptr )
    {
	if ( !strcmp(File, dir_entry_ptr->name) )
		return dir_entry_ptr;
    }

    return NULL;
}

/********************************************************
 * remove_file -- Remove a file				*
 *							*
 * Returns:						*
 * 	True -- the file was removed			*
 ********************************************************/
void rio_dir::remove_file(
    const char File[]	// The name of the file
)
{
    if (!valid) {
	dir_read();
    }

    // get directory entry for file
    rio_dir_entry* dir_entry_ptr = find_file(File);

    if (dir_entry_ptr == NULL) {
	throw(rio_error(
		CRIO_ERROR_FILENOTFOUND, 
		error_str("file '%s' not present on device", File)));
    }

    dirty = true;

    // free FAT and blocks used
    u_short cur_block_index = dir_entry_ptr->file_position;
    while( cur_block_index )
    {
	dir_block.block_used[cur_block_index] = CRIO_ID_32KBLOCK_FREE;

	u_short Temp = dir_block.FAT[cur_block_index];
	dir_block.FAT[cur_block_index] = 0;
	cur_block_index = Temp;
    }

    // adjust directory header
    rio_dir_header& dir_header = dir_block.header;
    --dir_header.entry_count;
    dir_header.blocks_used -= dir_entry_ptr->file_size_in_32k;
    dir_header.blocks_remaining += dir_entry_ptr->file_size_in_32k;

    // clear directory entry
    memset( dir_entry_ptr, 0, sizeof(rio_dir_entry) );

    // shuffle directory entries
    int entry_index = dir_entry_ptr - dir_block.dir_entry;
    // The number of entry
    int entry_count = (int)dir_header.entry_count - entry_index;

    int cur_entry;	// Entry we are moving up
    for(cur_entry = 0; cur_entry < entry_count; ++cur_entry) {
	memcpy(
		&dir_block.dir_entry[entry_index+cur_entry],
		&dir_block.dir_entry[entry_index+cur_entry+1],
		sizeof(rio_dir_entry)
	);
    }
}

/********************************************************
 * allocate_entry -- Add a new entry to the director and*
 *		return a reference to it.		*
 ********************************************************/
rio_dir_entry &rio_dir::allocate_entry(
    unsigned const int file_size_in_32k	// Size of the file
)
{
    // The next entry to allocate
    rio_dir_entry& dir_entry = 
	dir_block.dir_entry[dir_block.header.entry_count];
    dirty = true;

    // update directory header
    if ( !dir_block.header.entry_count) {
	// adjust for dir block
	++dir_block.header.blocks_used;
	--dir_block.header.blocks_remaining;

	// mark first block as used by directory
	dir_block.block_used[0] = CRIO_ID_32KBLOCK_USED;

	// first entry in FAT used by directory
	dir_block.FAT[0] = 0;
    }
    ++dir_block.header.entry_count;
    dir_block.header.blocks_used += file_size_in_32k;
    dir_block.header.blocks_remaining -= file_size_in_32k;
    return (dir_entry);
}
/********************************************************
 * remove_all_files -- Zap all the files in internal or	*
 *	external memory.				*
 ********************************************************/
void rio_dir::remove_all_files(
    // Which memory to remove the files from
    const enum MEM_TYPE mem_type,

    // Size of external memory 
    const int external_memory_size_32K
)
{
    if (!valid) {
	dir_read();
    }
    dirty = true;

    // The header to be changed
    rio_dir_header& dir_header = dir_block.header;

    // init entries
    dir_header.entry_count = 0;

    // init block used flags
    u_char* block_ptr = dir_block.block_used;

    for( unsigned int block_count=0; 
	 block_count<CRIO_MAX_32KBLOCK; 
	 ++block_count, ++block_ptr)
    {
	if ( *block_ptr != CRIO_ID_32KBLOCK_BAD)
	    *block_ptr = CRIO_ID_32KBLOCK_FREE;
    }

    // init FAT
    memset(&dir_block.FAT, 0, sizeof(dir_block.FAT) );

    // available blocks
    if (mem_type == MEM_EXTERNAL)
	dir_header.blocks_available = external_memory_size_32K;
    else
	dir_header.blocks_available = CRIO_COUNT_32KBLOCKIN32M;

    // used
    dir_header.blocks_used = dir_header.bad_block_count;

    // blocks remaining
    dir_header.blocks_remaining =
	    dir_header.blocks_available -
	    dir_header.blocks_used;
}
