/********************************************************
 * rio -- Define the rio_class and rio_dir class	*
 ********************************************************/

#ifndef		_RIO_
#define		_RIO_

#include <sys/types.h>
#include <assert.h>
#include "rio_error.h"
#include "rio_io.h"

// version
static const unsigned int CRIO_ID_VERSION = 106;

// blocks available on 32M unit
static const unsigned int CRIO_COUNT_32KBLOCKIN32M = 1024;

// max available 32k blocks
static const unsigned int CRIO_MAX_32KBLOCK = 8192;

// size of 32k block
static const unsigned int CRIO_SIZE_32KBLOCK = 32768;

// max directory entries
static const unsigned int CRIO_MAX_DIRENTRY = 60;

// 32Kblock type
static const unsigned char CRIO_ID_32KBLOCK_USED	= 0x00;
static const unsigned char CRIO_ID_32KBLOCK_BAD		= 0x0f;
static const unsigned char CRIO_ID_32KBLOCK_FREE	= 0xff;

// 32 bit word
#if defined(__alpha)
	typedef int SIGNED32;
	typedef unsigned int UNSIGNED32;
#else
	typedef long SIGNED32;
	typedef unsigned long UNSIGNED32;
#endif

/*
 * data_block -- A 32K data block with automatic allocation and
 *		deletion.
 */
class data_block {
    private:
	u_char data[CRIO_SIZE_32KBLOCK];
    public:
	// data_block --	Constructor defaults
	// ~data_block -- 	Destructor defaults
	// data_block(data_block) 	-- Copy constructor default
	// operator =	-- Assignment operator defaults
	void zero(void) {
	    memset(data, '\0', sizeof(data));
	}
	u_char *ptr(void) {
	    return (data);
	}
	const u_char *const ptr(void) const {
	    return (data);
	}
	u_char &operator [](unsigned int index) {
	    assert (index < CRIO_SIZE_32KBLOCK);
	    return (data[index]);
	}
};

typedef const void (*progress_cb)(const int pos, const int count);

/*------------------------------------------------------*/

// directory header
struct rio_dir_header
{
    u_short entry_count;		// [0-1] Number of entries
    u_short blocks_available;		// [2-3] Blocks avaliable
    u_short blocks_used;		// [4-5] Blocks used
    u_short blocks_remaining;		// [6-7] Blocks remaining?
    u_short bad_block_count;		// [8-9] Bad blocks
    SIGNED32 update_time;		// [10-13] Last time updated
    u_short check_sum1;			// [14-15] Checksum part 1
    u_short check_sum2;			// [16-17] Checksum part 2
    u_short one;			// [18-19] A value of 1 (version?)
    u_short version;			// [20-21] Version number
    char not_used_3[512 - 22];		// Not used
} __attribute__((packed));

// directory entry
struct rio_dir_entry
{
    u_short file_position;		// File position (in 32 blocks)
    u_short file_size_in_32k;		// File size in 32k blocks
    u_short file_size_blocks;		// File size rounded to a block size
    SIGNED32 file_size_rem;		// Bytes left in the last block
    char not_used[5];			// Not used
    SIGNED32 upload_time;		// Time song upload
    u_char mp3_property[4];		// Properties of the mp3 sound
    char not_used_3[5];			// Not used
    char name[128 - 28];		// The name of the song
} __attribute__((packed));

// directory block
struct rio_dir_struct
{
    rio_dir_header header;	// The directory header

    // The directory entries
    rio_dir_entry dir_entry[CRIO_MAX_DIRENTRY];

    // The block used flags
    u_char block_used[CRIO_MAX_32KBLOCK];

    // The FAT
    u_short FAT[CRIO_MAX_32KBLOCK];
} __attribute__((packed));

class rio_class;
class rio_io;
class rio_dir {
    private:
	rio_class &rio_info;	// Information about the rio we are using
	rio_io &io;	// The I/O information
	bool dirty;	// True if the block is dirty
	bool valid;	// True if the data is valid

	// directory block
	rio_dir_struct dir_block;

    public:
	explicit rio_dir(rio_class &i_rio_info, rio_io &i_rio_io) :
	    rio_info(i_rio_info),
	    io(i_rio_io),
	    dirty(false),
	    valid(false)
	{}
	~rio_dir();
    private:
	// No copy constructor or assignment
	rio_dir(const rio_dir &);
	rio_dir &operator = (const rio_dir &);

    public:
	// Calculate the checksums
	u_short calculate_check_sum1(void);
	u_short calculate_check_sum2(void);
    public:

	void dir_read(void);	// Read the directory
	void dir_write(void);	// Write out the directory
	void is_valid(void);	// Return true if valid dir bloc

	unsigned int get_entry_count(void) {
	    if (!valid) {
		dir_read();
	    }
	    return (dir_block.header.entry_count);
	}

	const rio_dir_entry &get_dir_entry(
		const unsigned int index);

	void initialize(
	    const int size		// Size of the memory
	);
	// We are changing memory types.  Clobber the dir
	void clobber(void) {
	    valid = false;
	}
	// Locate a file in the current directory
	rio_dir_entry* find_file(const char file[]);

	void remove_file(
	    const char file[]
	);

	void remove_all_files(const enum MEM_TYPE use_exernal_flash,
			const int external_memory_size_32K);

	rio_dir_entry &allocate_entry(
	    unsigned const int file_size_in_32k	// Size of the file
	);

	const rio_dir_struct& get_directory(void) { 
	    dir_read();
	    return dir_block; 
	}
	// Return a pointer to the block used
	// Note: We let the user modify this without setting dirty
	// because he will change a dir entry too (and thus set dirty)
	u_char *blocks_used(void)
	{
	    return (dir_block.block_used);
	}
	// Return a reference to the fat
	// Note: We let the user modify this without setting dirty
	// because he will change a dir entry too (and thus set dirty)
	u_short &FAT(const unsigned int index) {
	    assert (index < sizeof(dir_block.FAT));
	    return (dir_block.FAT[index]);
	}
	rio_dir_header &header(void)
	{
	    return (dir_block.header);
	}

	// This entry is designed for use by debugging / dumping
	// programs.  It gives the caller information
	// that a normal user does not need.
	rio_dir_entry *get_entry_array(void)
	{
	    return (dir_block.dir_entry);
	}
};


/*------------------------------------------------------*/
class rio_class
{
    private:
	bool present;	// True if the device is present
    public: 		// This is public only for the adump procedure
	rio_io io;	// The I/O information
    private:

	void io_delay(
	    const unsigned int ticks	// Ticks to delay
	);
    public:		// This is public  for directory dumping only
	rio_dir dir;			// The rio directory

	// external flash 32K block count
	u_int external_memory_size_32K;
    protected:

	// internal/external flash ram flag
	enum MEM_TYPE mem_type;

	u_int find_first_free(void);

	void mark_bad_blocks(
	    const progress_cb progress_function
	);

	void check_present(void);

    public:
	// set/unset
	void unset_io_address(void);
	void set_io_address(const int port_base) {
	    io.set_io_address(port_base);
	    check_present();
	}

	// constructors/destructors
	rio_class();
	~rio_class();

	// retrieval
	const rio_dir_struct& get_directory(void) { 
	    return dir.get_directory();
	}

	const enum MEM_TYPE get_mem_type(void) const { 
	    return mem_type; 
	}

	// operations
	void set_mem_type(
	    const enum MEM_TYPE new_mem_type	// Memory type to use
	);

	void initialize(
	    const bool mark_bad_blocks_flag, 	// Mark bad blocks?
	    const progress_cb progreess_function // Display progress
	);
	void remove_file(
	    const char file[]		// File to remove
	) {
	    dir.remove_file(file);
	}
	void remove_all_files(void) {
	    dir.remove_all_files(mem_type, 
			external_memory_size_32K);
	}
	void set_file_order(
	    const u_int pos_order[], 	// The order
	    const u_int count 		// Number of elements in the order
	);
	void send_rio_file(
	    const char file_name[], 	// File to send
	    // Progress callback.
	    const progress_cb progress_function);

	void read_rio_file(
	    const char file_name[], 	// File to get
	    // Progress callback
	    const progress_cb progress_function);

	const rio_dir_entry &dir_entry(
	    const unsigned int index	// Index of entry to get
	) {
	    return (dir.get_dir_entry(index));
	}
	void flush(void) {
	    dir.dir_write();
	}
	// rio_dir can access the low level functions
	friend class rio_dir;
};

#endif		// _RIO_

