/********************************************************
 * rio_io -- Handle the actual reading / writing	*
 *	to the rio.					*
 ********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <assert.h>

#include "binary.h"
#include "rio.h"
#include "common.h"

// platform dependencies
#if defined(_WINNT)
	// MS VC++ v5.0 for WinNT v4
	#include	<windows.h>
	#include	<winioctl.h>
	#include	"rioioctl.h"
	#define		OUTPORT( p, v )			WinNTOutPort( p, v )
	#define		INPORT( p )			WinNTInPort( p )
	#define		CLOCK_SECOND			1000
	#define		DELETEARRAY			delete[]
	#define		ID_DRIVER_VERSION		101

#elif defined(_WIN32)
	// MS VC++ v5.0 for Win9x
	#include	<conio.h>
	#define		OUTPORT( p, v )			_outp( p, v )
	#define		INPORT( p )			_inp( p )
	#define		CLOCK_SECOND			1000
	#define		DELETEARRAY			delete[]

#elif defined(__linux__)
	// linux g++
	#include	<unistd.h>
	#if defined(__alpha)
		#include	<sys/io.h>
	#else
		#include	<sys/perm.h>
	#endif
	#include	<asm/io.h>
	#define		OUTPORT(p,v)			outb( v, p )
	#define		INPORT(p)			inb( p )
	static const unsigned long int	CLOCK_SECOND =	1000000;
	#define		DELETEARRAY			delete[]

#elif defined(__TURBOC__)
	// turboc v1.01
	#include	<dos.h>
	#define		OUTPORT( p, v )			outp( p, v )
	#define		INPORT( p )			inp( p )
	#define		CLOCK_SECOND			18
	#define		DELETEARRAY			delete

#else
	// not supported
	#error ! ! compiler/platform not supported ! !
#endif

// port offset constants
#define		OFFSET_PORT_DATA		0
#define		OFFSET_PORT_STATUS		1
#define		OFFSET_PORT_CONTROL		2

// max tx/rx block retry
#define		MAX_RETRY			3

static const unsigned int BLOCK_SIZE = 512;	// Size of a block

// block to use to check for presense
static const unsigned int ID_32KBLOCK_CHECKPRESENT = 0;

// end 512 byte block
struct end_block_struct
{
    UNSIGNED32 next_512_pos;	// Position of the next block
    u_char next_8k_pos1;	// Position again
    u_char next_8k_pos2;	

    UNSIGNED32 prev_251_pos;	// Position of the prev block
    u_char prev_8k_pos1;	
    u_char prev_8k_pos2;	

    u_short check_sum;		// Checksum
    u_short prev_32K_pos;	// Another block pointer
} __attribute__((packed));

// io delay constants
/********************************************************
 * io delay constants.					*
 *	These constants appear to be based on nothing	*
 *	more than guesswork.  We should replace them	*
 *	with something more precise.			*
 ********************************************************/
#if defined(_WINNT)
    const unsigned int IO_DELAY_INIT = 2000;	// Used during initialization
    const unsigned int IO_DELAY_GET = 2;	// For getting data
    const unsigned int IO_DELAY_SEND = 10;	// Sending data
#else	/* _WINNT */
    const unsigned int IO_DELAY_INIT = 20000;	// Used during initialization
    const unsigned int IO_DELAY_GET = 2;	// For getting data
    const unsigned int IO_DELAY_SEND = 100;	// Sending data
#endif	/* _WINNT */

/*------------------------------------------------------*/
#if defined(_WINNT)
    // handle
    static HANDLE m_hDriver;

    // WinNT out port
    void WinNTOutPort( int iPort, int iValue )
    {
	DWORD dwSizeReturn;
	u_long ulInput = ((u_long)iPort-0x378) | ((ULONG)iValue << 16);
	DeviceIoControl( m_hDriver, RIOIO_IOCTL_WRITE, &ulInput,
		sizeof(long), NULL, 0, &dwSizeReturn, NULL );
    }

    // WinNT in port
    int WinNTInPort( int iPort )
    {
	DWORD dwSizeReturn;
	u_long ulPort = iPort - 0x378;
	u_long ulData = 0;
	DeviceIoControl( m_hDriver, RIOIO_IOCTL_READ, &ulPort, sizeof(ulPort),
		&ulData, sizeof(char), &dwSizeReturn, NULL );
	return (int)ulData;
    }

#endif

/*------------------------------------------------------*/
// if DOS
#if defined(__TURBOC__)
    // get clock ticks
    long clock( void )
    {
	    return (long) (*(int far*)MK_FP( 0x40, 0x6c ));
    }
#endif
/********************************************************
 * error_str -- Print to a string and return the result	*
 ********************************************************/
const char *const error_str(
    const char* const format, 	// Format string
    ... 			// Format argument
)
{
    static char the_string[100];// The string we are making
    va_list arg_ptr;		// Pointer to the argument list

    va_start(arg_ptr, format );
    vsprintf(the_string, format, arg_ptr );
    va_end(arg_ptr);
    return (the_string);
}

/********************************************************
 * rio_io::command_out -- Output a command		*
 ********************************************************/
inline void rio_io::command_out(
    const u_short v1, 	// The data byte
    const u_short v2, 	// The first control byte
    const u_short v3	// The second control byte
) { 
    OUTPORT(port_data, v1); 
    OUTPORT(port_control, v2); 
    OUTPORT(port_control, v3); 
}
/********************************************************
 * io_delay -- Delay after an IO.  This is done in a	*
 * 	machine dependent manner based on the counter	*
 *	supplied.					*
 ********************************************************/
inline void rio_io::io_delay(
    const unsigned int ticks	// Ticks to delay
) {
    unsigned int delay_count;	// Number of ticks so far

    for(delay_count = 0; delay_count < ticks; ++delay_count) 
	INPORT(port_status); 
}
/********************************************************
 * wait_ack -- Wait for an Ack to come back from the	*
 *		device.					*
 ********************************************************/
void rio_io::wait_ack(void)
{
    // The time to wait until for the ack
    long stop_time = clock() + CLOCK_SECOND;

    while(stop_time > clock())
    {
	// The character from the port
	u_char rec_char = INPORT(port_status);

	// Check for a good status
	if (rec_char & 0x08) 
	    return;
    }
    throw rio_error(CRIO_ERROR_ACK, "Missing ACK from rio");
}
/********************************************************
 * get_data_byte -- Retreive a data byte from the 	*
 *		device					*
 *							*
 * Returns the data byte				*
 ********************************************************/
u_int rio_io::get_data_byte(void)
{
    // get hi nibble
    OUTPORT(port_control, B_00000000);
    io_delay(IO_DELAY_GET);

    u_char rec_char = INPORT( port_status );
    u_char in_byte = ((rec_char & 0xf0) ^ 0x80) >> 4;

    // get lo nibble and combine with previous nibble to make byte
    OUTPORT(port_control, B_00000100);
    io_delay(IO_DELAY_GET);

    rec_char = INPORT(port_status);
    in_byte |= (rec_char & 0xf0) ^ 0x80;

    // reverse bits in byte
    u_char reverse_byte = 0;
    for( int bit_count=0; bit_count<8; ++bit_count )
    {
	reverse_byte <<= 1;
	reverse_byte |= (in_byte & 1);
	in_byte >>= 1;
    }

    return reverse_byte;
}
/********************************************************
 * unset_io_address -- remove any previous I/O handler	*
 ********************************************************/
void rio_io::unset_io_address(void)
{
#if defined(_WINNT) /* if WinNT */
    // close device file
    if ( m_hDriver ) {
	CloseHandle( m_hDriver );
	m_hDriver = NULL;
    }
#endif
}
/********************************************************
 * set_io_address -- Set the I/O base			*
 ********************************************************/
void rio_io::set_io_address(
    const int init_port_base 	// The I/O base of the LPT
) {
    // cleanup any previous Set()
    unset_io_address();

    // determine ports
    port_base    = init_port_base;
    port_data    = port_base + OFFSET_PORT_DATA;
    port_status  = port_base + OFFSET_PORT_STATUS;
    port_control = port_base + OFFSET_PORT_CONTROL;

// if linux
#if defined(__linux__)
    // request access to required ports
    if ( ioperm(port_base, 3, 1) ) {
	throw(rio_error(
	    CRIO_ERROR_IOPRERM, 
		    error_str("ioperm() failed, reason '%s'", 
			sys_errlist[errno])
		));
    }
#endif /* __linux__ */

// if WinNT
#if defined(_WINNT)
    // open generic IO device
    m_hDriver = CreateFile( "\\\\.\\RioDev", GENERIC_READ | GENERIC_WRITE,
	    FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL );

    if ( m_hDriver == INVALID_HANDLE_VALUE ) {
	throw(rio_error(
	     CRIO_ERROR_CREATEFILE, 
	    error_str("CreateFile() failed, reason %ld\n", GetLastError())
	    ));
    }

    // check version
    DWORD dwSizeReturn;
    u_long ulVersion;

    DeviceIoControl( m_hDriver, RIOIO_IOCTL_GETVERSION, NULL,
	    0, &ulVersion, sizeof(ulVersion), &dwSizeReturn, NULL );

    if ( ulVersion != ID_DRIVER_VERSION ) {
	throw(rio_error(
		CRIO_ERROR_DRIVERVERSION, 
	    error_str(
		"incorrect RioIO driver version, "
		"v%d.%d loaded, v%d.%d expected\n",
		 ulVersion/100, ulVersion%100, ID_DRIVER_VERSION/100, 
		 ID_DRIVER_VERSION%100 )));
    }

#endif
}

/********************************************************
 * send_block_to_rio_once -- Send a block with one	*
 ********************************************************/
void rio_io::send_block_to_rio_once(
    const data_block &block,	// The data block to be sent
    const u_int block_pos, 	// The location of the data block
    const u_int prev_block_pos,	// The prev block in the file
    const u_int next_block_pos 	// The next block in the list
)
{
    u_long pos_all;		// Location of the block (all parts)
    u_long pos_hi;		// Location of the block (high byte)
    u_long pos_mid;		// Location of the block (high middle)
    u_long pos_low;		// Location of the block (high low)
    unsigned int block_index;	// Index of the current block
    unsigned int byte_index;	// Current byte to send

    io_begin();

    // prepare 32K block
    int iC = 1;
    if (mem_type == MEM_EXTERNAL)
	iC = 2;

    for(block_index = 0; block_index < 4; block_index += iC) {
	pos_all = 
	    ((u_long)block_pos * C32K + u_long(block_index) * 8192) / 
	    BLOCK_SIZE;

	/*
	 * Convert position into a byte stream
	 *
	 *	0100 0000 0000 <--- 0x4000
	 *	1234 5678 9ABC <--- pos_all
	 *	
	 *		Int 	Ext	(Memory)
	 *	pos_hi	12	0x04
	 *	pos_mid	5678	23 4678
	 *	pos_low 9ABC	9ABC
	 */
	pos_low = pos_all & 0xff;

	if (mem_type == MEM_EXTERNAL) {
	    pos_hi = 0x04;
	    pos_mid = (pos_all & 0xff00) >> 8;
	} else {
	    pos_hi = pos_all / 0x4000;
	    pos_mid = (pos_all & 0x3f00) >> 8;
	}

	command_out(B_10101011, B_00001100, B_00000100);
	command_out(pos_hi, B_00000000, B_00000100);
	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_01100000, B_00000000, B_00000100);
	command_out(B_10100010, B_00001100, B_00000100);
	command_out(pos_low, B_00000000, B_00000100);
	command_out(pos_mid, B_00000000, B_00000100);

	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_11010000, B_00000000, B_00000100);

	wait_ack();

	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_01110000, B_00000000, B_00000100);
	command_out(B_10100000, B_00001100, B_00000100);

	OUTPORT(port_control, B_00000000);
	io_delay(IO_DELAY_SEND);

	OUTPORT(port_control, B_00000100);
	io_delay(IO_DELAY_SEND);
    }

    // send 32K in 512 byte chunks
    const u_char* block_ptr = block.ptr();

    for(block_index = 0; 
	block_index < (C32K / BLOCK_SIZE); 
	++block_index, block_ptr += BLOCK_SIZE) {

	pos_all = ((u_long)block_pos * C32K + 
		(u_long)block_index * BLOCK_SIZE) / BLOCK_SIZE;
	pos_low = pos_all & 0xff;

	if (mem_type == MEM_EXTERNAL) {
	    pos_hi = 0x04;
	    pos_mid = (pos_all & 0xff00) >> 8;
	} else {
	    pos_hi = pos_all / 0x4000;
	    pos_mid = (pos_all & 0x3f00) >> 8;
	}

	// issue upload 512 byte block command
	command_out( B_10101011, B_00001100, B_00000100 );
	command_out( pos_hi, B_00000000, B_00000100 );
	command_out( B_10100001, B_00001100, B_00000100 );
	command_out( B_10000000, B_00000000, B_00000100 );
	command_out( B_10100010, B_00001100, B_00000100 );
	command_out( B_00000000, B_00000000, B_00000100 );
	command_out( pos_low, B_00000000, B_00000100 );
	command_out( pos_mid, B_00000000, B_00000100 );

	command_out( B_10100011, B_00001100, B_00000100 );

	// create checksum of 512 byte block
	u_short check_sum = 0;
	u_short* byte_ptr = (u_short*)block_ptr;
	for(byte_index = 0; 
	    byte_index < (BLOCK_SIZE/sizeof(short)); 
	    ++byte_index, ++byte_ptr)
	    check_sum -= *byte_ptr;

	// clock out data
#if defined(_WINNT)
	{
	    DWORD dwSizeReturn;
	    DeviceIoControl( m_hDriver, RIOIO_IOCTL_WRITEBLOCK, 
		    block_ptr, BLOCK_SIZE,
		    NULL, 0, &dwSizeReturn, NULL );
	}
#else
	// Pointer to the current character
	const u_char* cur_char_ptr = block_ptr;

	for(byte_index = 0; 
	    byte_index < BLOCK_SIZE; 
	    ++byte_index, ++cur_char_ptr) {

	    OUTPORT(port_data, (*cur_char_ptr));

	    if (!(byte_index & 1))
		OUTPORT(port_control, B_00000000);
	    else
		OUTPORT(port_control, B_00000100);

	    io_delay(1);
	}
#endif

	// prepare end of block
	end_block_struct end_block;

	memset( &end_block, 0, sizeof(end_block_struct) );
	end_block.check_sum = check_sum;

	if ( next_block_pos == 0xffff ) {
	    end_block.next_512_pos = 0xffffffff;
	    end_block.next_8k_pos1 = 0xff;
	    end_block.next_8k_pos2 = 0xff;
	} else {
	    end_block.next_512_pos = 
			((u_long)next_block_pos * 64) * 256;
	    end_block.next_8k_pos1 = next_block_pos / 256;
	    end_block.next_8k_pos2 = next_block_pos / 256;

	    if (mem_type == MEM_EXTERNAL) {
		end_block.next_512_pos	+= 0x01000000;
		end_block.next_8k_pos1 = 0;
		end_block.next_8k_pos2 = 0;
	    }
	}

	if ( prev_block_pos == 0xffff )
	{
	    end_block.prev_251_pos = 0xffffffff;
	    end_block.prev_8k_pos1 = 0xff;
	    end_block.prev_8k_pos2 = 0xff;
	    end_block.prev_32K_pos = 0xffff;
	} else {
	    end_block.prev_251_pos = 
		((u_long)prev_block_pos * 64) * 256;

	    end_block.prev_8k_pos1 = prev_block_pos / 256;
	    end_block.prev_8k_pos2 = prev_block_pos / 256;

	    if (mem_type == MEM_EXTERNAL) {
		end_block.prev_251_pos	+= 0x01000000;
		end_block.prev_8k_pos1 = 0;
		end_block.prev_8k_pos2 = 0;
	    }

	    end_block.prev_32K_pos = prev_block_pos * 256;
	}

	// output end of block
#if defined(_WINNT)
	{
	    DWORD dwSizeReturn;
	    DeviceIoControl( m_hDriver, RIOIO_IOCTL_WRITEBLOCK, 
		    &end_block,
		    sizeof(end_block_struct), NULL, 0, &dwSizeReturn, NULL );
	}
#else
	cur_char_ptr = (u_char*)&end_block;

	for(byte_index = 0; 
	    byte_index < sizeof(end_block_struct); 
	    ++byte_index, ++cur_char_ptr ) {

	    OUTPORT(port_data, (*cur_char_ptr));

	    if (!(byte_index & 1))
		OUTPORT(port_control, B_00000000);
	    else
		OUTPORT(port_control, B_00000100);

	    io_delay(1);
	}
#endif

	// end of tx
	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_00010000, B_00000000, B_00000100);

	wait_ack();

	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_01110000, B_00000000, B_00000100);
	command_out(B_10100000, B_00001100, B_00000100);

	OUTPORT(port_control, B_00000000);
	io_delay(IO_DELAY_SEND);

	OUTPORT(port_control, B_00000100);
	io_delay(IO_DELAY_SEND);
    }
}

/********************************************************
 * send_block_to_rio -- transmit a block		*
 ********************************************************/
void rio_io::send_block_to_rio(
    const data_block &block,	// The block to send
    const u_int block_pos,	// The location of the block
    const u_int prev_block_pos,	// The location of the previous block
    const u_int next_block_pos 	// The location of the next block
)
{
    int retry_count = 0;	// The number of time to try

    while (retry_count < MAX_RETRY) {
	try {
	    send_block_to_rio_once(block, block_pos, 
		    prev_block_pos, next_block_pos);
	    return;
	}
	catch (...) {
	    usleep(CLOCK_SECOND);
	    ++retry_count;
	}
    }

    throw(
	rio_error(CRIO_ERROR_TXBLOCKRETRY, 
	    "too many retries for tx block" ));
}

/********************************************************
 * get_block_from_rio_once -- Receive a 32k block	*
 ********************************************************/
void rio_io::get_block_from_rio_once(
    data_block &block, 		// The block to read
    const u_int block_pos 	// The position to read from
)
{
    u_long pos_all;	// Position of the current 512 byte block

    // Positioned divided into three parts
    u_long pos_hi;		// The high 8 bits
    u_long pos_mid;		// The middle 8 bits
    u_long pos_low;		// The low 8 bits
    unsigned long int cur_block;// Current block number
    unsigned int cur_byte;	// Current byte number

    /* Note=0x8000 = 32K */

    // io intro
    io_begin();

    // get 32K in 512 byte chunks
    u_char* cur_ptr = block.ptr();

    for(cur_block = 0; 
	cur_block < (C32K / BLOCK_SIZE); 
	++cur_block, cur_ptr += BLOCK_SIZE) {

	pos_all = 
	    ((u_long)block_pos * 0x8000 + 
	     (u_long)cur_block * BLOCK_SIZE) / BLOCK_SIZE;

	/*
	 * Take the position and turn it into high, low, mid
	 *
	 *	Block = aa|bb|cc
	 *
	 *			HH MM LL
	 * Internal memory	aa bb cc
	 * External memory	04 bb cc
	 */

	if (mem_type == MEM_EXTERNAL) {
	    pos_hi = 0x04;
	    pos_mid = (pos_all & 0xff00) >> 8;
	} else {
	    pos_hi = pos_all / 0x4000;
	    pos_mid = (pos_all & 0x3f00) >> 8;
	}
	pos_low = pos_all & 0xff;

	// issue download 512 byte block command
	command_out(B_10101011, B_00001100, B_00000100);
	command_out(pos_hi, B_00000000, B_00000100);
	command_out(B_10100001, B_00001100, B_00000100);
	command_out(B_00000000, B_00000000, B_00000100);
	command_out(B_10100010, B_00001100, B_00000100);
	command_out(B_00000000, B_00000000, B_00000100);
	command_out(pos_low, B_00000000, B_00000100);
	command_out(pos_mid, B_00000000, B_00000100);

	wait_ack();

	command_out( B_10100000, B_00001100, B_00000100 );

	// clock in data
#if defined(_WINNT)
	{
	    DWORD dwSizeReturn;
	    DeviceIoControl(m_hDriver, RIOIO_IOCTL_READBLOCK, 
		    	cur_ptr, BLOCK_SIZE,
		    	cur_ptr, BLOCK_SIZE, &dwSizeReturn, NULL );
	}
#else
	for(cur_byte = 0;cur_byte < BLOCK_SIZE; ++cur_byte)
		*(cur_ptr+cur_byte) = get_data_byte();
#endif

	unsigned char end_data[16];
	// clock in 16 bytes which are ignored
	for(cur_byte = 0; cur_byte < 16; ++cur_byte) {
	    end_data[cur_byte] = get_data_byte();
	}
	wait_ack();
    }
}

/********************************************************
 * get_block_from_rio -- Receive a 32k block		*
 ********************************************************/
void rio_io::get_block_from_rio(
    data_block &block, 		// The block to read
    const u_int block_pos 	// The block number 
)
{
    int cur_try;	// Current try 
	
    for (cur_try = 0; cur_try < MAX_RETRY; ++cur_try) {
	try {
	    get_block_from_rio_once(block, block_pos);
	    return;
	}
	catch (...) {
	    usleep(CLOCK_SECOND);
	    ++cur_try;
	}
    }
    throw rio_error(CRIO_ERROR_RXBLOCKRETRY, 
	"too many retries for rx block" );
}
/********************************************************
 * io_begin -- Initialize the I/O			*
 ********************************************************/
void rio_io::io_begin(void)
{
    OUTPORT(port_control, B_00000100);

    command_out(B_10101000, B_00001100, B_00000100);

    OUTPORT(port_control, B_00000000);
    io_delay(IO_DELAY_INIT);

    OUTPORT(port_control, B_00000100);
    io_delay(IO_DELAY_INIT);

    command_out(B_10101101, B_00001100, B_00000100);
    command_out(B_01010101, B_00000000, B_00000100);
    command_out(B_10101110, B_00001100, B_00000100);
    command_out(B_10101010, B_00000000, B_00000100);
    command_out(B_10101000, B_00001100, B_00000100);

    OUTPORT(port_control, B_00000000);
    io_delay(IO_DELAY_INIT);

    OUTPORT(port_control, B_00000100);
    io_delay(IO_DELAY_INIT);
}

/********************************************************
 * io_end -- end the io.  Return control to printer	*
 ********************************************************/
void rio_io::io_end(void)
{
    command_out(B_10101101, B_00001100, B_00000100);
    command_out(B_11111111, B_00000000, B_00000100);
    OUTPORT(port_data, B_00000000);
}
/********************************************************
 * get_memory_size -- Return the memory size code	*
 *	for the external memory				*
 *							*
 * Returns -- The size code.				*
 ********************************************************/
u_char rio_io::get_memory_size(void)
{
    io_begin();

    // issue read id command
    command_out(0xAB, B_00001100, B_00000100);
    command_out(0x04, B_00000000, B_00000100);
    command_out(0xA1, B_00001100, B_00000100);
    command_out(0x90, B_00000000, B_00000100);
    command_out(0xA2, B_00001100, B_00000100);
    command_out(0x00, B_00000000, B_00000100);
    command_out(0xA0, B_00001100, B_00000100);

    // determine mem size in no of 512 byte blocks
    get_data_byte();
    // The memory size code is the next byte
    u_char memory_size_code = get_data_byte();

    return (memory_size_code);
}
/********************************************************
 * rio_io -- initialize the rio and check for		*
 *	possible compiler packing problm (with struct	*
 *	end_block_struct)				*
 ********************************************************/
rio_io::rio_io(void): 
    port_base(-1), 
    port_data(-1), 
    port_status(-1), 
    port_control(-1)
{
    assert(sizeof(struct end_block_struct) == 16);
}
