/*
 *  ximp3  A simple mp3 player
 *  
 *  Copyright (C) 2001 Mats Peterson
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *  
 *  Please send any comments/bug reports to
 *  matsp888@yahoo.com  (Mats Peterson)
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "ximp3.h"
#include "audio.h"
#include "audiofork.h"


void quit(int retcode)
{
    int status;
    
    kill(v->pid, SIGTERM);
    waitpid(v->pid, &status, WUNTRACED);
    exit(retcode);
}


void sig_handler(int sig)
{
    switch (sig) {
	case SIGUSR1:
	    v->retval = 1;
	    break;
	case SIGUSR2:
	    v->retval = 0;
	    break;
	case SIGTERM:
	    quit(1);
    }
#if !defined(__FreeBSD__) && !defined(__NetBSD__) && !defined(__OpenBSD__)
    signal(sig, sig_handler);
#endif
}


static void usage(void)
{
    fprintf(stderr, "%s v%s\n", PACKAGE, VERSION);
    fprintf(stderr, "usage: %s [-d device] [-b buffers] [-s] [-l] [-v] [-r] file ...\n", PACKAGE);
    exit(1);
}


void init(int argc, char **argv)
{
    int c, tmp;

    if (! (v = malloc(sizeof(VARS)))) {
	fprintf(stderr, "Couldn't allocate global variables\n");
	exit(1);
    }

    v->flist = NULL;
    v->flist_size = 0;
    v->shuffle_ord = NULL;

    strcpy(v->device, "/dev/dsp");
    v->tot_playbufs = TOT_PLAYBUFS;
    v->shuffle = 0;
    v->loop = 0;
    v->verbose = 0;
    v->remote = 0;

    while ((c = getopt(argc, argv, "d:b:slvr")) != -1) {
	switch (c) {
	    case 'd':
		strcpy(v->device, optarg);
		break;
	    case 'b':
		tmp = atoi(optarg);
		if (tmp < 2) {
		    fprintf(stderr,
			    "Error: Number of buffers must be at least 2\n");
		    exit(1);
		}
		v->tot_playbufs = tmp;
		break;
	    case 's':
		v->shuffle = 1;
		break;
	    case 'l':
		v->loop = 1;
		break;
	    case 'v':
		v->verbose = 1;
		break;
	    case 'r':
		v->remote = 1;
		break;
	    case '?':
	    default:
		usage();
	}
    }

    if ((argc - optind) < 1)
	usage();

    if (v->remote) {
	setlinebuf(stdout);
	setlinebuf(stderr);
    }

    if (! fork_audio())
	exit(1);

    signal(SIGUSR1, sig_handler);
    signal(SIGUSR2, sig_handler);
    signal(SIGTERM, sig_handler);
    signal(SIGPIPE, SIG_IGN);
}
