#!/bin/bash
#
# Copyright (C) 1997 by John P. Weiss
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the Artistic License, included as the file
# "LICENSE" in the source code archive.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# You should have received a copy of the file "LICENSE", containing
# the License John Weiss originally placed this program under.



# This is the command to play a sound file.  It can be either a shell
# script interface to sox or some other command, like bplay.
#
PLAY="bplay"


# This command controls the mixer volume settings.  The variable
# "VOLPARM" is the parameter that "VOLCTRL" uses to set the volume.
# The default is the program "setmixer" - if you have some other
# program that controls the soundcard volume from the command line,
# change these two variables appropriately.  
#
# If you don't want volume control, just set "VOLCTRL" to some
# non-existent program.  
#
VOLCTRL="setmixer"
VOLPARM="vol"

# This is the default volume level.  The script resets the mixer to
# this level upon exit.
#
DEFLEVEL="70"



usage ()
{
    echo "usage: mplay [-l <file> [<file> ...] ] [<playlist> [<playlist> ...] ]"
    echo ""
    echo "There are two ways to use mplay.  If the \"-l\" option is present,"
    echo "the subsequent list of audio files is played in sequence."
    echo ""
    echo "You can also give the names of playlists to mplay as arguments."
    echo "A playlist is an ASCII text file with a sequence of commands:"
    echo ""
    echo "    -dir <dirname>"
    echo "    -basedir <dirname>    Sets the current directory to"
    echo "                          <dirname>.  This is used for both"
    echo "                          audio files and play lists."
    echo "    -v <num>"
    echo "    -vol <num>            Sets the volume to <num>."
    echo "    -f <sndfile>"
    echo "    -p <sndfile>"
    echo "    -play <sndfile>"
    echo "    -file <sndfile>       Play the audio file <sndfile>."
    echo "    -play_list <plist>    Change to the play list <plist>"
    echo "                          and execute it.  After mplay finishes,"
    echo "                          it continues executing the parent"
    echo "                          play list."
}

multiplay ()
{
    BASEDIR=""
    while [ "x$1" != "x" ]; do
	   case $1 in 
		  -dir|-basedir)
			 shift
			 BASEDIR=$1
			 # echo "Now Playing from: $BASEDIR"
			 shift
			 ;;
		  -v|-vol)
			 shift
			 if [ -x "${VC}" ] ; then 
				${VOLCTRL} ${VOLPARM} $1
				# echo "New Volume: $1"
			 fi
			 shift
			 ;;
		  -f|-file|-p|-play)
			 shift
			 echo -n "Now playing: $1"
			 echo -en "                                               \r"
			 eval "${PLAY} ${BASEDIR}${1} > /dev/null 2>&1"
			 shift
			 ;;
		  -play_list)
			 shift
			 echo -n "                              "
			 echo -n "                              "
			 echo ""
			 echo "Playing new playlist \"${1}\""
			 old_basedir=$BASEDIR
			 multiplay `eval "cat ${BASEDIR}${1}"`
			 BASEDIR=$old_basedir
			 shift
			 ;;
		  *)
			 echo ""
			 echo ""
			 echo "Unknown script file option: $1."
			 echo ""
			 shift
			 ;;
	   esac
    done
}


VC=`which ${VOLCTRL}`
# Process command line arguments
#
if [ "x$1" = "x-l" ] ; then
    shift
    plist=""
    for arg in $*; do
	   plist="${plist} -f ${arg}"
    done
    multiplay ${plist}
elif [ "x$1" = "x-h" ] ; then
    usage
else
    for file in $*; do
	   multiplay `cat ${file}`
    done
fi

echo -n "                                                                 "
echo ""

if [ -x "${VC}" ] ; then 
    ${VOLCTRL} ${VOLPARM} ${DEFLEVEL}
fi
