/*  =========================================================
    SING - ALONG DISK PLAYER.
    (C) 1998, 1999   Michael Glickman
    ----------------------------------------------------------
    NOTICE:
            Sing-Along Disk Player is copyrighted by the author.
            See GNU general public licence (GPL) available at
	    ftp://metalab.unc.edu/pub/gnu/COPYING for details
	    and legal issues.

            You are expected to provide appropriate references
            when using a part of the code in your software.

	    Author strongly advices against using this code, or
	    a part of it, in an application designed to run  on
	    any Microsoft(tm) platfrom.
    ========================================================= */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <linux/types.h>
#include "sad.h"
#include "sadp_curses.h"

extern char  CDDeviceName[];
extern char  AudioDeviceName[];
extern char  MixerDeviceName[];

extern short   FourInterlace;
extern short   AutoPlay;
extern short   ProbeCD;
extern short   LoopPlay;
extern short   SaveSet, SaveData;

extern u_int  SamplingRate;
extern short  RunBackground;
extern short  QuietMode;
extern short  BkgrColour, ShadColour, BottomColour;
extern short  FourWndCount, WaveObturate;
extern short  CharSet, OptionsChanged;

extern MOUSE_SUPPORT_OPTION MouseSupport;
extern PLAY_MODE_TYPE  PlayMode;
extern TIME_DISPLAY_MODE_TYPE TimeDisplayMode;
extern HEADER_MODE  HeaderMode;
extern OPERATION StartOper;
extern PLATFORM  Platform;
extern char  *ShortDBName;

extern short  HelpMode;
extern u_char track_cur;
extern u_char cd_status;

static char *OptOper = "APLDS";
static char *OptHeader = "NDT";
static short *OptVar[] = {&AutoPlay, &ProbeCD, &LoopPlay,
                          &SaveData, &SaveSet};
static short *PtrColours[3] = {&BkgrColour, &ShadColour, &BottomColour};
static MOUSE_SUPPORT_OPTION MouseSupportOrig;
static short  CharSetOrig;

static int  ProcessDeviceName(char *DeviceName, const char *option, const char *prefix)
{
     char s;

     if (option == NULL) return 0;

     s = *option;
     if (s == '*' )  DeviceName[0] = '\0';
     else
     if (s == '/')  strcpy(DeviceName, option);
     else
     if (isdigit(s))
     {  if (!prefix || *(option+1) )
        { fprintf (stderr, "Invalid device name %s\n", option);
          return 0;
        }
	  strcpy(DeviceName, prefix);
	  strcat(DeviceName, option);
     }
     else
     {
	  strcpy(DeviceName, "/dev/");
	  strcat(DeviceName, option);
     }

     return 1;
}


static int ProcessInteger(const char *option, const char *name, short *target,
                                        short lowval, short highval)
{
    long val;
    char *endptr;

    if (option == NULL) return 0;
    val = strtol(option, &endptr, 10);

    if (*endptr == '\0' && 
              val >=(long)lowval && val <= (long)highval )
    {  *target = (u_short)val;
       return 1;
    }

    fprintf (stderr, "%s can be a whole number between %d and %d\n",
                       name,  lowval, highval );
    return 0;
}

static int ProcessSamplingRate(char *option)
{
    long val;

     if (option == NULL) return 0;
    val = strtol(option, NULL, 10);

    if (val >= 8000  && val <= 44100)
    {  SamplingRate = val;
       return 1;
    }
    else
    if (val >=0 && val <= 2)
    {   SamplingRate = 44100 >> val;
       return 1;
    }

    fprintf (stderr,
         "To set sampling rate, use 0, 1, 2 or a whole number between 8000 and 44100\n");

    return 0;
}


static int  ProcessMouseSupport(char *option)
{

    switch (tolower(*option))
    {  case 'n':
		MouseSupport = MOUSE_NONE;
		break;

       case 's':
       case 'y':
		MouseSupport = MOUSE_NATIVE;
		break;

#if GPM_SUPPORT
       case 'g':
		MouseSupport = MOUSE_GPM;
		break;
#endif

	  default:
	     fprintf (stderr,
#if GPM_SUPPORT
              "Valid mouse support options: n, s, g \n");
#else
              "Valid mouse support options: n or y \n");
#endif
	    return 0;
    }

    return 1;
}


static int ProcessOptions(char *option)
{
     char *pos;
     char c;

     if (option == NULL) return 0;

     while ((c=*option++) != '\0')
     {
         pos = strchr(OptOper, c);
         if (pos)
         {  *(OptVar[pos-OptOper]) = 1;
            continue;
         }

         pos = strchr(OptOper, toupper(c));
         if (pos)
         {  *(OptVar[pos-OptOper]) = 0;
            continue;
         }

         fprintf (stderr, "Invalid option: %c\n", c);
         return 0;
      }



/*
      short bl;
      bl = 1;
 while ((c=*option++) != '\0')
{
	   switch(toupper(c))
	   {
 	        case '+':
			bl = 1; break;

	        case '-':
          		bl = 0; break;


	        case 'A':
			AutoPlay = bl;
			break;

	        case 'P':
			ProbeCD = bl;
       			break;
    }
}
*/
     return 1;
}


static int ProcessColours(char *option)
{  static char buf[2] = { '\0', '\0'};
   static char *name[3] = {NULL, NULL, NULL};
   short i;
   char c;

   if (option == NULL) return 0;
   if (toupper(*option) == 'M' && *(option+1) == '\0')
   {  BkgrColour = -10;
      return 1;
   }


   BkgrColour = ShadColour = BottomColour = -1;
   if (!name[0])
   {	if (Platform == PLTF_XFORMS)
	{  name[0] = "Background colour";
	   name[1] = "Top foreground colour";
	   name[2] = "Bottom foreground colour";
	}
	else
	{  name[0] = "Panel background colour";
	   name[1] = "Button background colour";
	   name[2] = "Button foreground colour";
	}
   }

   for (i=0; i<3 && (c=*option++) != '\0' ; i++)
   {  buf[0] = c;
      if (! ProcessInteger(buf, name[i], PtrColours[i], 0, 7))
      return 0;
   }

   if (i==0)
   {  fprintf (stderr,  "Missing colour specidication");
      return 0;
   }
   return 1;
}


static int ProcessTimeDisplayMode(char *optarg)
{

    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;

    switch(toupper(*optarg))
    { case 'N':
      case 'E':
          TimeDisplayMode = DM_ELAPSED;
          break;

      case 'R':
          TimeDisplayMode = DM_REMAINED;
          break;

      case 'P':
          TimeDisplayMode = DM_PROGRESS;
          break;

      case 'M':
          TimeDisplayMode = DM_PROGREM;
          break;

      default:
          goto BadArg;
    }

    return 1;

BadArg:
    fprintf (stderr, "Wrong time display mode, use single letter:\n");
    fprintf (stderr, "\tE - elapsed,   R - remained\n");
    fprintf (stderr, "\tP - %% elapsed, M - %% remained\n");
    return 0;

}

static int ProcessPlayMode(char *optarg)
{

    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;

    switch(toupper(*optarg))
    { case 'N':
	PlayMode = PLAY_NORMAL;
	break;

      case 'L':
	PlayMode = PLAY_LIST;
	break;

      case 'R':
	PlayMode = PLAY_RANDOM;
	break;

      case 'I':
	PlayMode = PLAY_INTRO;
	break;

      case 'S':
	PlayMode = PLAY_SINGLE;

      default:
	goto BadArg;
    }

    return 1;

BadArg:
    fprintf (stderr, "Wrong play mode, use single letter:\n");
    fprintf (stderr, "\tN - normal, L - loop, R - random, I - intro\n");
    return 0;

}
static int ProcessOper(char *optarg)
{
    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;
    switch(toupper(*optarg))
    {
      case 'P':
         StartOper = OPER_PLAY;
	 break;

      case 'S':
         StartOper = OPER_STOP;
	 break;

      case 'N':
         StartOper = OPER_NEXT;
	 break;

      case 'R':
         StartOper = OPER_PREV;
	 break;

      default:
	goto BadArg;
    }

    return 1;

BadArg:
    fprintf (stderr, "Wrong operation, use single letter:\n");
    fprintf (stderr, "\tP - playl, S - stop, N - next, R - previous\n");
    return 0;
}

static int ProcessHeaderMode(char *optarg)
{   char *ptr;

    if (strlen(optarg) !=1) goto BadArg;

    ptr = strchr(OptHeader, toupper(*optarg));
    if (ptr == NULL) goto BadArg;

    HeaderMode =  ptr - OptHeader;
    return 1;

BadArg:
    fprintf (stderr, "Wrong header mode, use single letter:\n");
    fprintf (stderr, "\tT - track name, D - disc info(title+artist), N - none\n");
    return 0;
}


static void	ShowCommandLineHelp()
{
  printf ("\nSing-Along Disk Player. Version %d.%d.%d. Command line options.\n\n",
		SADP_VERSION >> 16, (SADP_VERSION >> 8) & 0xFF, SADP_VERSION & 0xFF);

#ifdef DSP_SUPPORT
   printf ("\tA - audio (DSP) device path (default: /dev/dsp)\n");
#endif
   printf ("\tC - CD device path (default: /dev/cdrom)\n");
   printf ("\tD - data base path (default: ~/.xcddbase)\n");
#ifdef MIXER_SUPPORT
   printf ("\tM - mixer device path (default: /dev/mixer)\n");
#endif
   printf ("\tH - header mode: t-track name(default), d-disc info, n-none\n");
   printf ("\tO - options (use capital letter to turn feature on):\n");
   printf ("\t    P/p - auto-probe, A/a - auto-play L/l - loop play\n");
   printf ("\t    S/s - save settings at end, D/d - auto-save data\n");
   printf ("\tb - panel and button colours (1 to 3 digits, range 0-7)\n");
   printf ("\tc - character set (0-4)\n");
#ifdef DSP_SUPPORT
   printf ("\tf - sampling frequency: 0-44100, 1-22050, 2-11025\n");
#endif
   printf ("\th - show this page and exit\n");
#ifdef DSP_SUPPORT
   printf ("\ti - interlace value (0-5)\n");
#endif
   printf ("\tl - low profile\n");
#if GPM_SUPPORT
   printf ("\tm - mouse: n-none, s-curses, g-GPM\n");
#else
   printf ("\tm - mouse support: n-no, y-yes\n");
#endif


   printf ("\to - operation: p-play, s-stop/eject, n-next, r-prev\n");
   printf ("\tp - play mode: n-normal, l-list, r-random, i-intro, s-single\n");
#ifdef DSP_SUPPORT
   printf ("\ts - number of spectrum windows (0-2)\n");
#endif
   printf ("\tt - time display: e-elapsed, r-remain, p-%%elapsed, m-%%remain\n");

#ifdef DSP_SUPPORT
   printf ("\tw - wave obturation value (0-4) \n");
#endif
   printf ("\nSee man page for details.\n");

}

/* Returns:
        0  - Bad argument
        >0 - OK
        <0 - Unrecognised operation
*/
static short ProcessArgument(char oper, char *arg)
{   switch(oper)
    {
        case 'A':
           return ProcessDeviceName(AudioDeviceName, arg, "/dev/dsp");

        case 'C':
            return ProcessDeviceName(CDDeviceName, arg, "/dev/cdrom");

        case 'M':
            return ProcessDeviceName(MixerDeviceName, arg, "/dev/mixer");

        case 'S':
        case 's':
            return ProcessInteger(arg,
                         "Number of spectrum windows", &FourWndCount, 0, 2);
        case 'c':
        {   short res = ProcessInteger(arg, "Character set", &CharSet, 0, 4);
            return res;
        }


        case 'i':
            return   ProcessInteger(arg,
                         "Spectrum interlace", &FourInterlace, 0, 5);

        case 'w':
            return  ProcessInteger(arg,
                     "Wave obturation value", &WaveObturate, 0, 4);

        case 'O':
            return  ProcessOptions(arg);

        case 'H':
            return  ProcessHeaderMode(arg);

        case 'm':
            return  ProcessMouseSupport(arg);

        case 'r':
        case 'f':
            return  ProcessSamplingRate(arg);

        case 'b':
            return  ProcessColours(arg);

        case 'p':
            return   ProcessPlayMode(arg);

        case 't':
            return  ProcessTimeDisplayMode(arg);

        case 'D':
            ShortDBName = strdup(arg);
            return 1;
    }

    return -1;
}

int  process_arguments(int argc, char **argv)
{
    static char arglist[65] = "C:D:O:H:p:m:c:b:o:t:lh";
    double mips;
    int  rv;

    if (get_cpu_data(NULL, NULL, NULL, &mips) && mips > 250.0)
        FourInterlace = 0; else FourInterlace = 1;


    strcpy(CDDeviceName, "/dev/cdrom");

#ifdef DSP_SUPPORT
    strcat(arglist, "A:S:s:r:f:i:w:");
    strcpy(AudioDeviceName, "/dev/dsp");
#else
    AudioDeviceName[0] = '\0';
    FourWndCount = 0;
#endif

#ifdef MIXER_SUPPORT
    strcat(arglist, "M:");
    strcpy(MixerDeviceName, "/dev/mixer");
#else
    MixerDeviceName[0] = '\0';
#endif

    while ( (rv = getopt(argc, argv, arglist )) != -1)
    {
       switch (rv)
       { case 'h':
            ShowCommandLineHelp();
            return 0;

         case 'l':
            QuietMode = 1;
            return 1;

         case 'o':
            return  ProcessOper(optarg);

         default:
            if (ProcessArgument(rv, optarg) <= 0) return 0;
            OptionsChanged = 1;
       }
     }

    CharSetOrig = CharSet;
    MouseSupportOrig = MouseSupport;
    return 1;

}


static int SaveString(FILE *f, char oper, char *arg)
{
    if (arg == NULL) return -1;
    return fprintf (f, "%c %s\n", oper, arg);
}

static int SaveInteger(FILE *f, char oper, long val)
{
    return fprintf (f, "%c %ld\n", oper, val);
}

static int SaveOptions(FILE *f)
{    char c;
     char *inptr, *outptr;
     short i;
     char buf[21];

     inptr = OptOper; i = 0;
     outptr = buf;

     while ((c=*inptr++) != '\0')
         *outptr++ = *(OptVar[i++]) ? c : tolower(c);

     *outptr = '\0';
     return SaveString (f, 'O', buf);
}

/*
static int SaveMouseSupport(FILE *f)
{   static char arg[2] = {'\0', '\0'};
    char c;

    switch (MouseSupportOrig)
    {
#if GPM_SUPPORT
        case MOUSE_GPM:
            c = 'g'; break;
#endif

        case MOUSE_NATIVE:
            c = 's'; break;

        default:
            c = 'n';
    }
    arg[0] = c;
    return SaveString(f, 'm', arg);
}
*/

static int SavePlayMode(FILE *f)
{   const char *Modes = "NLRIS";
    static char arg[2] = {'\0', '\0'};

    arg[0] = Modes[PlayMode];
    return SaveString(f, 'p', arg);
}

static int SaveTimeDisplayMode(FILE *f)
{   const char *Modes = "ERPM";
    static char arg[2] = {'\0', '\0'};

    arg[0] = Modes[TimeDisplayMode];
    return SaveString(f, 't', arg);
}
static int SaveHeaderMode(FILE *f)
{  static char arg[2] = {'\0', '\0'};

    arg[0] = OptHeader[HeaderMode];
    return SaveString(f, 'H', arg);
}

static int SaveColours(FILE *f)
{   char arg[4];

    if (BkgrColour == -10)
       strcpy(arg, "m");
    else
    {  short i;
       char clr;
       for (i=0; i<3; i++)
       {  clr = *PtrColours[i];
          if (clr < 0 || clr > 7) break;
          arg[i] = clr + '0';
       }
       arg[i] = '\0';
    }
    return SaveString(f, 'b', arg);
}

void save_settings(void)
{
    char *fname = GetFullFileName("~/.sadprc");
    FILE *f;

    if (fname == NULL) return;
    f = fopen(fname, "w");
    free(fname);
    if (f == NULL) return;

    SaveString(f, 'A', AudioDeviceName);
    SaveString(f, 'C', CDDeviceName);
    SaveString(f, 'M', MixerDeviceName);
    SaveString(f, 'D', ShortDBName);

    SaveInteger(f, 's', FourWndCount);
    SaveInteger(f, 'c', CharSetOrig);
    SaveInteger(f, 'i', FourInterlace);
    SaveInteger(f, 'w', WaveObturate);
    SaveInteger(f, 'f', SamplingRate);

    SaveColours(f);
    SaveOptions(f);
/*    SaveMouseSupport(f); */
    SavePlayMode(f);
    SaveTimeDisplayMode(f);
    SaveHeaderMode(f);

    fclose(f);
    OptionsChanged = 0;

}

void load_settings(void)
{
    char *fname = GetFullFileName("~/.sadprc");
    char buf[121], *ptr, oper;
    short res;
    FILE *f;


    if (fname == NULL) return;
    f = fopen(fname, "r");
    free(fname);
    if (f == NULL) return;

    while(fgets(buf, 120, f) != NULL)
    {  ptr = strtok(buf, "#\r\n");
       ptr += strspn(ptr, " ");
       oper = *ptr++;
       if (oper == '\0') break;
       ptr += strspn(ptr, " ");
       ptr = strtok(ptr, " ");
       res = ProcessArgument(oper, ptr);
       if (res < 0)
          fprintf(stderr, "Invalid argument: %c\n", oper);
     }

     OptionsChanged = 0;
    fclose(f);
}

