/*  =========================================================
    SING - ALONG DISK PLAYER.
    (C) 1998, 1999   Michael Glickman
    ----------------------------------------------------------
    NOTICE:
            Sing-Along Disk Player is copyrighted by the author.
            See GNU general public licence (GPL) available at
            ftp://metalab.unc.edu/pub/gnu/COPYING for details
            and legal issues.

            You are expected to provide appropriate references
            when using a part of the code in your software.

            Author strongly advices against using this code, or
            a part of it, in an application designed to run  on
            any Microsoft(tm) platfrom.
    ========================================================= */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <mntent.h>
#include <string.h>
#include <unistd.h>

#include <errno.h>
#include <time.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <linux/cdrom.h>
#include <linux/major.h>
#include "sad.h"

#if defined (UCDROM_SUPPORT) && !defined(CDROM_DRIVE_STATUS)
#include <linux/ucdrom.h>
#endif

#if defined (UCDROM_SUPPORT) && !defined(CDSL_CURRENT)
/* #define CDSL_CURRENT    	((int) (~0U>>1)) */
#define CDSL_CURRENT  0    /* To cater for old UCDROM drivers */
#endif

#ifdef SCSI_SUPPORT
#include <scsi/scsi_ioctl.h>
#endif


/* #define INTRO_INTERVAL ( INTRO_DURATION_SEC * CD_FRAMES) */

extern char CDDeviceName[];
extern short ProbeCD, AutoPlay, LoopPlay;
extern PLAY_MODE_TYPE  PlayMode;
extern  int     TimeoutStart, TimeoutRepos, TimeoutLoad;
extern  int     DurationIntro, DurationAdvance;
extern  short   OptionsChanged, SaveData;

short  no_cd,  DataChanged = 0;
short  play_finished = 0;  /* Used in backgound play */

u_char CDDeviceType, CDNodeNumber;
u_char cd_status;
u_char dtrack_first, dtrack_last;   /* First & last tracks on disk */
u_char track_first, track_last;     /* First & last audio tracks on disk */
u_char track_cur;
u_long track_time_start, track_time_end, time_total, time_cur, time_req;
u_long time_start = 0;
u_long IntroInterval, AdvanceInterval;
u_char volume_left, volume_right;




ACCESS_TYPE         last_access;

static int cdrom_fd = -1;
static short new_message;
static short pos_forced;
static short new_play;
static char Message[120];
static char PidFileName[60];
static u_char track_old;

static int cd_next(void);
static int cd_stop(void);
static int insert_cd(void);


char  disc_name[61], disc_artist[61];
TRACK_INFO  *trk_info = NULL;

short   cd_playlist_size;
u_char  *cd_playlist = NULL;
extern  short  cur_plist_pos;

static short   cd_shuflist_size;
static u_char   *cd_shuflist = NULL;

short   cur_plist_size, cur_plist_pos;
u_char  *cur_plist = NULL;

/*=====================================================*/
char * cdaudio_get_track_name(u_char track)
{
    if (!trk_info) return NULL;
    return (trk_info + (track - dtrack_first)) -> name;
}

u_short cdaudio_get_track_audio(u_char track)
{
    if (!trk_info) return 0;
    return (trk_info + (track - dtrack_first))->audio;
}

/*=====================================================*/
static u_long msf_to_time(struct cdrom_msf0 *msf)
{
    return ((u_long)msf->minute * CD_SECS + (u_long)msf->second)
                                * CD_FRAMES + (u_long)msf->frame;
}

void time_to_msf(u_long timemsf,
                        u_char *min, u_char *sec, u_char *frm)
{
    *frm = (u_char) (timemsf % CD_FRAMES);
    timemsf /= CD_FRAMES;
    *min = (u_char) (timemsf / CD_SECS);
    *sec = (u_char) (timemsf % CD_SECS);
}
/*=====================================================*/
/*   Volume  operation                                           */

short cd_getvolume(void)
{  struct cdrom_volctrl vol_info;

    if (ioctl(cdrom_fd, CDROMVOLREAD, &vol_info) >=0)
    {   volume_left = vol_info.channel0;
        volume_right = vol_info.channel1;
        return 1;
    }
    else
         return 0;

}


short cd_setvolume(u_char vol_left, u_char vol_right)
{   struct cdrom_volctrl vol_info;

   if (ioctl(cdrom_fd, CDROMVOLREAD, &vol_info) <0)
                                                return 0;

    vol_info.channel0 = vol_left;
    vol_info.channel1 = vol_right;

    if (ioctl(cdrom_fd, CDROMVOLCTRL, &vol_info) >=0)
    { /*  cd_getvolume(); */
       return 1;
    }

    return 0;
}


/*=====================================================*/
/* Get current track start in msf */

static u_long get_track_start(int track)
{     TRACK_INFO *ti;

      if (!trk_info)
      {  strcpy(Message, "No track info");
         return ~0l;
      }

      ti = trk_info + (track - dtrack_first);
      if (ti->audio == 0) return ~1l;

      return ti->start;
}

static u_long get_track_end(int track)
{
      if (track >= track_last)  return time_total;
      return get_track_start(track + 1);
}

/*=====================================================*/
/*                 Disc  shuffle                      */
static void cd_shuflist_shuffle(short flag)
{    int i, r;
     int s1 = cd_shuflist_size - 1;
     u_char t;

     if (s1 < 0) return;

     i = 0;

     if (flag)
     {  /* Try to put current track at start */
        for (r=0; r<cd_shuflist_size; r++)
          if (cd_shuflist[r] == track_cur)
          {  if (r>0)
             {  cd_shuflist[r] = cd_shuflist[0];
                cd_shuflist[0] = track_cur;
             }
             i++; break;
          }
     }

     for(; i<s1; i++)
     {
        r = (int)((double)rand() * (cd_shuflist_size-i) / RAND_MAX) + i;

        if (r > i)
        {  t = cd_shuflist[i];
           cd_shuflist[i] = cd_shuflist[r];
           cd_shuflist[r] = t;
        }
     }
}


/*=====================================================*/
/*               New disc processing                  */
static int process_new_disc(void)
{
      /* Proceed new disc */
      struct cdrom_tochdr header;
      struct cdrom_tocentry toc_entry;
      TRACK_INFO *ti;
      u_char track, au, sz;
      int rc;

      rc = 0;
      audio_stop();

      /* Get number of tracks */
      if(ioctl(cdrom_fd, CDROMREADTOCHDR, &header) < 0)
      {  strcpy(Message, "CD header read error");
         goto BadLuck;
      }

      dtrack_first = header.cdth_trk0;
      dtrack_last = header.cdth_trk1;

      if (trk_info) free(trk_info);
      ti = trk_info =
         (TRACK_INFO *) calloc(dtrack_last - dtrack_first + 2, sizeof(TRACK_INFO));

      if (!trk_info)
      { strcpy(Message, "No memory to store track data");
        goto BadLuck;
      }

      track_first = 0xff;
      strcpy(disc_name, "** Unknown **");
      strcpy(disc_artist, "");

      if (cd_shuflist) free(cd_shuflist);
      cd_shuflist_size = 0;

      sz = dtrack_last - dtrack_first + 1;
      cd_shuflist = malloc(sz * sizeof(u_char));

      for (track= dtrack_first; track<= dtrack_last+1; track++)
      {

        toc_entry.cdte_track = (track > dtrack_last) ? CDROM_LEADOUT : track;
          toc_entry.cdte_format = CDROM_MSF;

          if(ioctl(cdrom_fd, CDROMREADTOCENTRY, &toc_entry) < 0)
          {  strcpy(Message, "CD TOC entry read error");
             goto BadLuck;
          }

          ti->audio = au = (track <= dtrack_last) &&
                              (toc_entry.cdte_ctrl != CDROM_DATA_TRACK);
          if (au)
          { if(track_first == 0xff) track_first = track;
            cd_shuflist[cd_shuflist_size++] = track;
            track_last = track;
          }

          sprintf (ti->name, "Track %u", (u_int) track);
                 /*  strcpy(ti->name, ""); */

          ti->start = msf_to_time(&(toc_entry.cdte_addr.msf));
          ti++;
      }

      if (cd_shuflist_size != sz)
          cd_shuflist = realloc(cd_shuflist, cd_shuflist_size * sizeof(u_char));

      /* Check for audio tracks */
      if (track_first == 0xff)
      {  strcpy(Message, "No audio tracks");
         goto BadLuck;
      }

      /* Get disc time */
      time_start = (trk_info+(track_first-dtrack_first))->start;
      time_total = (trk_info+(track_last-dtrack_first+1))->start;
      track_cur = track_first;
      time_cur = time_start;

      cddb_find();

      cd_accept_play_mode();

      rc = 1;

BadLuck:
      audio_reinitialize();
      if (rc == 0 && trk_info)
      {  free(trk_info); trk_info = NULL; }
      return rc;
}


/*=====================================================*/
/* Get pevious audio track number */

static u_char get_prev_track(u_char track_from)
{     u_long time_start;

        do
        {  time_start = get_track_start(track_from);
           if (time_start != ~1l) break;
        } while(--track_from >= track_first);

      if (time_start == ~0l) track_from = 0xff; /* Access error */
        return track_from;
}

/* Get next audio track number */
static u_char get_next_track(u_char track_from)
{     u_long time_start;

      do
      {  time_start = get_track_start(track_from);
           if (time_start != ~1l) break;
        }  while(++track_from <= track_last);

      if (time_start == ~0l) track_from = 0xff; /*  Access error  */
        return track_from;
}


int check_playlist(void)
{
   if (cur_plist_size <= 0)
      {  strcpy(Message, "No play list");
         new_message = 1;
         return 0;
      }

   return 1;
}

/* Get random track number */
/*
static u_char get_random_track(u_char track_from)
{   u_char track_to_play;

   if (track_first == track_last)
        track_to_play = track_first;
   do
   {       track_to_play = track_first +
                   (double)rand() * (track_last-track_first+1) / RAND_MAX;

           if (track_to_play > track_last)
               track_to_play = track_last;   // Just for safety
   } while (track_to_play == track_from);

   return get_next_track(track_to_play);    // Will pick up only audio track
}
*/

int check_cd(void)
{  u_char old_status;
   u_char  track_old, track_new;
   struct cdrom_subchnl subch;
   short no_cd_old, manual_skip;


   track_old = track_cur;
   old_status = cd_status;
   no_cd_old = no_cd;


   manual_skip = 0;  /* On if disk was manually advanced / retraid */
   cd_status = 0xff;
   no_cd = 0;

   if (cdrom_fd < 0 || (no_cd_old==1 && !ProbeCD))
   {
      no_cd = 1;
      if (no_cd_old != 1)
      {  strcpy(Message, "Disabled");
           goto BadLuck;
      }
      return 1;
   }


   /* Check if CD is still present */

#ifdef SCSI_SUPPORT
   /* This will avoid a stupid message from SCSI driver */
   if (CDDeviceType == SCSI_CDROM_MAJOR)
   { if (ioctl(cdrom_fd, SCSI_IOCTL_TEST_UNIT_READY) != 0 )
         no_cd = 1;
   }

#endif

#if defined(SCSI_SUPPORT) && defined(UCDROM_SUPPORT)
   else
#endif

#ifdef UCDROM_SUPPORT
   /* Try to use standard test disk routine (doesn't work in most cases) */
   { int res = ioctl(cdrom_fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
     if (res == CDS_NO_DISC || res == CDS_TRAY_OPEN
                            || res == CDS_DRIVE_NOT_READY)
             no_cd = 1;
   }
#endif


   /* In Non-SCSI case, attempt "read subchannel" straight away  */
   subch.cdsc_format = CDROM_MSF;   /* minutes-sec-frames */
   if(no_cd == 0 &&
            ioctl(cdrom_fd, CDROMSUBCHNL, &subch) != 0) no_cd =  1;

   if (no_cd)
   { if (no_cd_old==1) return 1;
     if (!ProbeCD)
     {  strcpy(Message, "Disabled");
        close(cdrom_fd); cdrom_fd = -1;
     }
     else
        strcpy(Message, "No CD in drive");
     goto BadLuck;

   }


   /* Ok, CD is there for sure. */
   cd_status = subch.cdsc_audiostatus;

   if (no_cd_old)
   {
      pos_forced = 0;
      new_play = 0;

      if (!process_new_disc()) goto BadLuck;

      if (AutoPlay && cd_status != CDROM_AUDIO_PLAY)
                cd_playtrack(track_cur, 1);
   }

   /* Check CDROM status */

   if (cd_status != old_status)
   {  new_message = 1;

      switch(cd_status)
      {
           case CDROM_AUDIO_INVALID:
/*                strcpy(Message, "No audio support"); */
                  strcpy(Message, "");

                    break;

           case CDROM_AUDIO_PLAY:
                strcpy(Message, "Playing");
                break;

           case CDROM_AUDIO_PAUSED:
                strcpy(Message, "Paused");
                break;

           case CDROM_AUDIO_ERROR:
                strcpy(Message, "Error");
                break;

           case CDROM_AUDIO_COMPLETED:
                strcpy(Message, "Stopped");
                break;

           case CDROM_AUDIO_NO_STATUS:
                strcpy(Message, "Ready");
                break;

           default:
                  sprintf(Message, "Unkn.stat. %d", cd_status);
                break;
        }


   }


   if (pos_forced)
       new_play = 0;
   else
   {
       u_long time_old;
       long   time_dif;

       track_new = subch.cdsc_trk;

       /* Needed to hide the bouncing effect */
       if (new_play)
       {  if (track_new < track_cur) return 1;
           new_play = 0;
       }

       track_cur = track_new;

       time_old = time_cur;
       time_cur = msf_to_time(&(subch.cdsc_absaddr.msf));

       if (old_status == CDROM_AUDIO_PLAY)
       {  time_dif = time_cur - time_old;
          if (time_dif < 0) time_dif = -time_dif;
          manual_skip = (time_dif > 2*CD_FRAMES);
       }

   }


   track_new = track_cur;

   if (last_access == ACCESS_PLAY && old_status == CDROM_AUDIO_PLAY)
   {

       switch(PlayMode)
       {
       /*
          case PLAY_REPEAT:
             if (manual_skip) break;
             if (cd_status != CDROM_AUDIO_PLAY) track_cur = 0xff;
             if (track_cur > track_old)
                  track_new = track_old;
             break;
       */
          case PLAY_RANDOM:
          case PLAY_LIST:
             if (manual_skip) break;
             if (cd_status != CDROM_AUDIO_PLAY || track_cur > track_old)
             {   if (++cur_plist_pos >= cur_plist_size)
                 {
                    track_new = 0xff;
                    if (!check_playlist() ) break;

                    if (PlayMode == PLAY_RANDOM)
                         cd_shuflist_shuffle(0);

                    if (!LoopPlay) break;

                    cur_plist_pos = 0;

                 }

                 track_new = cur_plist[cur_plist_pos];
             }
             break;
      /*
          case PLAY_LOOP:
             if (cd_status != CDROM_AUDIO_PLAY) track_cur = 0xff;
             if (track_cur > track_last)
                 track_new = track_first;
             break;
      */
          case PLAY_INTRO:
            if (manual_skip) break;
            if ((time_cur - track_time_start) > IntroInterval)
            {  /* printf ("%ld, %ld %d\n", time_cur, track_time_start, track_old); */
               track_new = track_old + 1;
               if (track_new > track_last)
               {  track_new = LoopPlay ? track_first : 0xff;
//                track_cur = track_new;
               }

            }
            break;

          case PLAY_SINGLE:
           if (cd_status != CDROM_AUDIO_PLAY || track_cur > track_old)
               track_new = LoopPlay ? track_old :  0xff;
               break;

          case PLAY_NORMAL:
              if (cd_status != CDROM_AUDIO_PLAY)
              {  if (LoopPlay)
                 {  track_new = track_first;
                    track_cur = 0xff;  /* To force play */
                 }
                 else
                 {  track_new = 0xff;
                    track_cur = track_first;
                    time_cur = time_start;
                    pos_forced = 1;
                 }
              }
              break;

        }

        if (track_new != track_cur)
        {  if (track_new == 0xff)
           {  if (cd_status == CDROM_AUDIO_PLAY)
            		  cd_stop();
  		  play_finished = 1;
           }
           else
               return cd_playtrack(track_new, 1);
        }
    }



   /* Just next track */
   if (track_cur < track_first || track_cur > track_last)
   {    track_cur = track_first;
        time_cur = time_start;
        pos_forced = 1;
   }


   if (track_cur != track_old)
   {
        track_time_start = get_track_start(track_cur);
        track_time_end = get_track_end(track_cur);
        if ( track_time_start == ~0l || track_time_end == ~0l)
        {  strcpy(Message, "CD error getting time");
           goto BadLuck;
        }
   }

   return 1;


BadLuck:
   if (no_cd)
   {     if (DataChanged)
         {   if (SaveData) cddb_write();
             DataChanged = 0;
         }
       track_cur = 0xff;
         track_first = 1; track_last = 0;
         strcpy(disc_name, "");
         strcpy(disc_artist, "");
   }

   new_message = 1;
   return 1;
}



/*=====================================================*/
int check_duplicate(void)
{
    int  PidFileFd;
    int  FilePid;
    int  retcd = 1;


    sprintf (PidFileName, "/tmp/sadp_pid_%u.%u",
                      (u_int)CDDeviceType, (u_int)CDNodeNumber);
    PidFileFd = open(PidFileName, O_RDONLY, 0);

    if (PidFileFd > 0)
    { if ( read( PidFileFd, &FilePid, sizeof(int)) == sizeof(int))
        {
           /* Check if the process exists */
           if (kill(FilePid, SIGTSTP) == 0)
           { fprintf (stderr, "SADP for %s is running with pid %d\n",
             CDDeviceName, FilePid);
             retcd = 0;
           }
        }

        close(PidFileFd);

        if (retcd == 0) return retcd;
    }

    PidFileFd = open(PidFileName, O_WRONLY | O_CREAT | O_TRUNC, 0666);

    if (PidFileFd > 0)
    {  FilePid = getpid();
       write(PidFileFd, &FilePid, sizeof(int));
       close(PidFileFd);
    }

    return retcd;       /* Will be 1 */

}
/*------------------------------------------------------*/
int cdaudio_initialize(void)
{
   FILE *mounts;
   struct mntent *mnt;
   struct stat disk_stat;
   dev_t rdev;

   cd_playlist_size = 0;
   cd_playlist = NULL;

   cd_shuflist_size = 0;
   cd_shuflist = NULL;

   stat(CDDeviceName, &disk_stat);
   rdev = disk_stat.st_rdev;

   /* Check if the dist has been already mounted */
   if((mounts = setmntent(MOUNTED, "r")))
   {
      while((mnt = getmntent(mounts)) != NULL)
      if (stat(mnt->mnt_fsname, &disk_stat) == 0 && 
            disk_stat.st_rdev == rdev) 
 /*   if(strcmp(mnt->mnt_fsname, CDDeviceName) == 0) */
      {
         fprintf(stderr, "%s is currently mounted under %s.\n",
                          mnt->mnt_fsname, mnt->mnt_dir);
         endmntent(mounts);
         return 0;
      }
   }

   endmntent(mounts);

   CDDeviceType = rdev >> 8;
   CDNodeNumber = rdev & 255;

   if (!check_duplicate()) return 0;

#ifndef SCSI_SUPPORT
   if (CDDeviceType == SCSI_CDROM_MAJOR && ProbeCD)
   {
      printf ("\n\aYou use a SCSI CDROM drive. To avoid 'junk messages'\n"
                  "enable SCSI support in sadp_config.h and recompile.\n\n"
                  "If you cannot get the source, disable auto-probing\n"
                  "in command line with  -Op parameter.\n\n"
                  "Press Enter to continue...");

        getchar();
   }
#endif

   cdrom_fd = open(CDDeviceName, O_RDONLY | O_NONBLOCK);

   if (cdrom_fd < 0)
   {    int error_number = errno;

       /* The following line avoids a bug found in some old CD modules
          (like sbpcd) that would not open a CD device without a disk
           in drive. Further open will be recalled with any CD command */

        if (error_number == ENXIO) goto Ok;

        {   char buf[257];
            int s = readlink(CDDeviceName, buf, 256);
            buf[s] = '\0';
            fprintf ( stderr,
               "Cound not open '%s'[%s]: %s\n", CDDeviceName, buf, strerror(error_number));
        }

        if (error_number == EACCES)
        {   fprintf(stderr, "%s%s%s",
              "\nTo get permissions, run, as root, 'chmod 666' ",
              "for your CD device, e.g.\n\n",
              "\t\tchmod 666  /dev/hdc\n");
        }
        else
        if (error_number == ENOENT)
        {
           fprintf(stderr,
             "\nPlease, link (as root) your CD device to '%s', e.g.\n\n", CDDeviceName);

           fprintf(stderr, "\t\tln -f /dev/hdc  %s\n", CDDeviceName);
        }

        return 0;

   }
Ok:

   srand(time(NULL));           /* In case random play is desired */
   return 1;
}

int cdaudio_start(void)
{  int ret_value;

   no_cd = 2;
   trk_info = NULL;
   pos_forced = 0;
   new_message = 1;
   new_play = 0;
   play_finished = 0;
   track_cur = 0xff;
   track_old = 0xff;
   cd_status = 0xff;

   last_access =  ACCESS_UNDEF;
   IntroInterval = (u_long) DurationIntro * CD_FRAMES;
   AdvanceInterval = (u_long) DurationAdvance * CD_FRAMES;

#ifdef SCSI_SUPPORT
   if (CDDeviceType == SCSI_CDROM_MAJOR)
                         ioctl(cdrom_fd, SCSI_IOCTL_DOORUNLOCK);
#endif

#if defined(SCSI_SUPPORT) && defined(UCDROM_SUPPORT)
   else
#endif

#ifdef UCDROM_SUPPORT
   {
      ioctl(cdrom_fd, CDROM_CLEAR_OPTIONS, CDO_LOCK);
   }
#endif

    cddb_initialize();

    ret_value = check_cd();

    if (ret_value)
    { if (cd_status == CDROM_AUDIO_PLAY)  last_access = ACCESS_PLAY;
    }

    return ret_value;
}

int cdaudio_deinitialize(void)
{
   if (DataChanged && SaveData) cddb_write();
   cddb_terminate();

   if (cd_playlist) free(cd_playlist);
   if (cd_shuflist) free(cd_shuflist);

   if (cdrom_fd >= 0)
   { // ioctl(cdrom_fd, CDROMSTOP);
     // ioctl(cdrom_fd, CDROMEJECT);
        close(cdrom_fd);
        remove(PidFileName);
   }


   if (trk_info)
   {  free(trk_info);
      trk_info = NULL;
   }

   return 1;
}


/*=====================================================*/
/* The following functions do not check the status,
   assuming that is has been done */

int cd_stop(void)
{  int rc;

   if (cdrom_fd < 0) return insert_cd();

   audio_stop();

   if (no_cd && last_access != ACCESS_LOAD && last_access != ACCESS_UNDEF)
   {   rc = (ioctl(cdrom_fd, CDROMCLOSETRAY) == 0);
       if(rc) sleep(TimeoutLoad);
       last_access = ACCESS_LOAD;
   }
   else
   if (cd_status == CDROM_AUDIO_PLAY)
   {   rc = (ioctl(cdrom_fd, CDROMSTOP, 0) == 0);
	 if (rc) play_finished = 1; 
       last_access = ACCESS_STOP;
   }
   else
   {   rc = (ioctl(cdrom_fd, CDROMEJECT) == 0);
       if(rc) sleep(TimeoutLoad);
       last_access = ACCESS_EJECT;
   }

   audio_reinitialize();
   new_message = 1;
   return rc;
}

static int insert_cd(void)
{  int ret_value;

   if (cdrom_fd >=0 && !no_cd) return 1;

   audio_stop();

   ret_value = 0;

   if (cdrom_fd < 0)
   {    cdrom_fd = open(CDDeviceName, O_RDONLY | O_NONBLOCK);
        if (cdrom_fd < 0) goto Out;
   }

   ioctl(cdrom_fd, CDROMCLOSETRAY);
   sleep(TimeoutLoad);

   no_cd = 2;
   check_cd();
   if (!no_cd) ret_value = 1;



Out:
   audio_reinitialize();
   return ret_value;
}





static int cd_startplay(int force)
{
     struct cdrom_msf   playmsf;
     int ret_value;


     if (cd_status != CDROM_AUDIO_PLAY)
     {

        if (force || AutoPlay)
             pos_forced = 0;
        else
        {
             pos_forced = 1;
             last_access = ACCESS_POSITION;
             return 1;
        }

     }

     audio_stop();
     time_to_msf (time_cur, &(playmsf.cdmsf_min0),
                            &(playmsf.cdmsf_sec0), &(playmsf.cdmsf_frame0));

     time_to_msf (time_total, &(playmsf.cdmsf_min1),
                            &(playmsf.cdmsf_sec1), &(playmsf.cdmsf_frame1));

     show_cd_time();
     show_cd_status("Starting play...", -4);
     refresh_output();
     if (cd_status == CDROM_AUDIO_PLAY)
     {     ret_value = ( ioctl(cdrom_fd, CDROMPAUSE) ==0 );
             if (ret_value >=0)
           {  cd_status = CDROM_AUDIO_PAUSED;
                if (TimeoutRepos > 0) sleep(TimeoutRepos);
           }
     }
     else
     if (cd_status == CDROM_AUDIO_PAUSED)
     {     ret_value = 1;
             if (TimeoutRepos > 0) sleep(TimeoutRepos);
             set_suspend_mode(0);
     }
     else
     {
             refresh_output();
             ret_value = ( ioctl(cdrom_fd, CDROMSTART) ==0);
             if (ret_value && TimeoutStart > 0) sleep(TimeoutStart);
             set_suspend_mode(0);
     }

     if (ret_value)
           ret_value = ( ioctl(cdrom_fd, CDROMPLAYMSF, &playmsf) ==0);
     if (ret_value)
      { new_play = 1;
        play_finished = 0;
      } 

     audio_reinitialize();
     last_access = ACCESS_PLAY;

     return ret_value;

}

int cd_pauseresume(void)
{  int ret_value;

   if (no_cd) return 0;

   if (pos_forced) return cd_startplay(1);

   switch(cd_status)
   { case CDROM_AUDIO_PAUSED:
        ret_value = (ioctl(cdrom_fd, CDROMRESUME) == 0) ;
          last_access = ACCESS_PLAY;
        return ret_value;

     case CDROM_AUDIO_PLAY:
          audio_stop();
        ret_value = (ioctl(cdrom_fd, CDROMPAUSE) == 0) ;
        audio_reinitialize();
          last_access = ACCESS_STOP;
        return ret_value;
   }

   return 0;
}

int cd_playtrack(int track, int force)
{   u_long new_time;


    if (track > track_last || track < track_first) return 0;

    new_time = get_track_start(track);
    if ((new_time & ~1l) == ~1l) return 0;

    track_cur = track;
    time_cur = new_time;
    track_time_start = new_time;
    track_time_end = get_track_end(track);

    return cd_startplay(force);

}


static int cd_play(void)
{

    if (!insert_cd()) return 0;

    if (PlayMode == PLAY_LIST || PlayMode == PLAY_RANDOM)
    { if (!check_playlist()) return 0;
      {  u_char new_trk = cur_plist[cur_plist_pos];
         if (new_trk != track_cur)
             return cd_playtrack(new_trk, 1);
      }
    }

    if (cdaudio_get_track_audio(track_cur) == 0) return 0;
    return  cd_startplay(1);
/*    return cd_playtrack(track_cur, 1); */
}

static int cd_back(void)
{
    if (no_cd) return 0;


    if (time_cur < AdvanceInterval)  /* (STEP_DISCRETE)*CD_FRAMES) */
          time_cur = 0;
    else
          time_cur =  time_cur -  AdvanceInterval; /* (STEP_DISCRETE)*CD_FRAMES;  */

    if (time_cur < track_time_start) time_cur = track_time_start;

    /*  Play  */

    return  cd_startplay( 0);

}



static int cd_forward(void)
{
    if (no_cd) return 0;

    time_cur =  time_cur + AdvanceInterval;  /* (STEP_DISCRETE)*CD_FRAMES; */
    if (time_cur > track_time_end) time_cur = track_time_end;

    /*  Play  */
    return  cd_startplay( 0);

}

static int cd_playfrom(short align)
{   u_char i;
    u_long prev_track_start, next_track_start = 0;

    if (no_cd) return 0;

    time_cur =  time_req;
    if (time_cur < time_start) time_cur = time_start;
    if (time_cur >= time_total) time_cur = time_total-1;

    prev_track_start = time_start;
    for (i=track_first+1; i<=track_last; i++)
    {  next_track_start = get_track_start(i);
         if (next_track_start > time_cur) break;
         prev_track_start = next_track_start;
    }

    track_cur = i-1;
    track_time_end = (i > track_last) ?  time_total : next_track_start;
    track_time_start = prev_track_start;
    if (align) time_cur = track_time_start;

    /*  Play  */
    return  cd_startplay( 0);

}

static int cd_prev(void)
{   u_char prev_track;

    if (no_cd) return 0;


   if (PlayMode == PLAY_RANDOM || PlayMode == PLAY_LIST)
   {/*     prev_track = track_old;
         if (prev_track == 0xff) prev_track = track_cur;
         track_old = track_cur;  */
         if (!check_playlist()) return 0;
         cur_plist_pos = (cur_plist_pos+cur_plist_size-1) % cur_plist_size;
         prev_track = cur_plist[cur_plist_pos];
   }
   else
   {     if (track_cur <= track_first) prev_track = track_last;
                           else  prev_track = track_cur-1;
   }

   prev_track = get_prev_track(prev_track);
   return  cd_playtrack(prev_track, 0);
}

static int cd_next(void)
{   u_char next_track;

    if (no_cd) return 0;

    if (PlayMode ==  PLAY_RANDOM || PlayMode == PLAY_LIST)
    {
        if (!check_playlist()) return 0;
         cur_plist_pos = (cur_plist_pos+1) % cur_plist_size;
         next_track = cur_plist[cur_plist_pos];
         /*et_random_track(track_cur);     random */
    }
    else
    {
         next_track = track_cur+1;
         if (next_track > track_last) next_track = track_first;
    }

    next_track = get_next_track(next_track);
    return  cd_playtrack(next_track, 0);
}

static int cd_next5(void)
{   u_char next_track;

    if (no_cd) return 0;

    if (PlayMode ==  PLAY_RANDOM || PlayMode == PLAY_LIST)
    {   short next_pos;

        if (!check_playlist()) return 0;
        next_pos = cur_plist_pos + 5;
        if (next_pos >= cur_plist_size) return 0;
        cur_plist_pos = next_pos;
        next_track = cur_plist[cur_plist_pos];
    }
    else
    {   next_track = track_cur + 5;
        if (next_track > track_last) return 0;
    }

    next_track = get_next_track(next_track);
    return  cd_playtrack(next_track, 0);
}

static int cd_prev5(void)
{   u_char prev_track;

    if (no_cd) return 0;

    if (PlayMode ==  PLAY_RANDOM || PlayMode == PLAY_LIST)
    {   short prev_pos;

        if (!check_playlist()) return 0;
        prev_pos = cur_plist_pos - 5;
        if (prev_pos < 0) return 0;
        cur_plist_pos = prev_pos;
        prev_track = cur_plist[cur_plist_pos];
    }
    else
    {   if (track_cur < track_first + 5) return 0;
         prev_track = track_cur - 5;
    }

    prev_track = get_prev_track(prev_track);
    return  cd_playtrack(prev_track, 0);
}

/*=====================================================*/
void cd_accept_play_mode(void)
{
    u_char track_new;

    if (PlayMode == PLAY_LIST)
    {  cur_plist_size = cd_playlist_size;
       cur_plist = cd_playlist;
    }
    else
    if (PlayMode == PLAY_RANDOM)
    {  cd_shuflist_shuffle(cd_status == CDROM_AUDIO_PLAY);
       cur_plist_size = cd_shuflist_size;
       cur_plist = cd_shuflist;
    }
    else return;

    cur_plist_pos = 0;

    if (!check_playlist() || cd_status == CDROM_AUDIO_PLAY) return;
    track_new = cur_plist[cur_plist_pos];
    if (track_new != track_cur)
    {  track_cur = track_new;
       time_cur = get_track_start (track_cur);
    }
}


int cdaudio_operate(OPERATION oper)
{  int ret;


   ret = check_cd();


   if (new_message)
   {

      show_cd_status(Message,
                      ( cdrom_fd < 0  ||
                        (no_cd &&
                         last_access != ACCESS_LOAD &&
                         last_access != ACCESS_UNDEF)
                      ) ? 0 :

                      (cd_status == CDROM_AUDIO_PLAY
                      ) ? 1 :

                          2
                    );

      if (no_cd) clear_cd_time();
      new_message = 0;
   }

   if (!ret) return ret;

/*   if (!no_cd && (cd_status == CDROM_AUDIO_PLAY ||
                  cd_status == CDROM_AUDIO_COMPLETED ||
                  cd_status == CDROM_AUDIO_PAUSED))
*/
   if (!no_cd)
   {   show_cd_time();
//       refresh_output();
   }


   if (oper == OPER_NONE) return 0;


   switch(oper)
   {
      case OPER_BACK:
         ret = cd_back();
           break;

      case OPER_PREV:
         ret = cd_prev();
           break;

      case OPER_STOP:
         ret = cd_stop();
           break;

        case OPER_STOPQUIT:
           ret = (cd_status == CDROM_AUDIO_PLAY) ? cd_stop() : 1;
           break;

      case OPER_NEXT:
         ret = cd_next();
           break;

      case OPER_FWD:
         ret = cd_forward();
           break;

      case OPER_PLAY:
           ret= cd_play();
           break;

      case OPER_AUTOPLAY:
           AutoPlay ^= 1;
           OptionsChanged = 1;
           show_mode(oper);
           ret = 1;
           break;

      case OPER_LOOPPLAY:
           LoopPlay ^= 1;
           OptionsChanged = 1;
           show_mode(oper);
           ret = 1;
           break;

      case OPER_PLAYMODE:
           PlayMode = (PlayMode+1) % 5;
           cd_accept_play_mode();
           OptionsChanged = 1;
           show_mode(oper);
           ret = 1;
           break;

      case OPER_PLAYMODE + 1000:
           PlayMode = (PlayMode+4) % 5;
           cd_accept_play_mode();
           OptionsChanged = 1;
           show_mode(OPER_PLAYMODE);
           ret = 1;
           break;

      case OPER_PAUSE:
           ret = cd_pauseresume();
           break;

      case OPER_FIRSTTRACK:
      {    u_char new_track;
           if ((PlayMode == PLAY_LIST || PlayMode == PLAY_RANDOM) && check_playlist())
           {   cur_plist_pos = 0;
               new_track = cur_plist[cur_plist_pos];
           }
           else
               new_track = track_first;
           ret = cd_playtrack(new_track, 0);
      }
           break;

      case OPER_PLAYFROM:
           ret= cd_playfrom(0);
           break;

      case OPER_PLAYFROMALIGNED:
         ret= cd_playfrom(1);
           break;

      case OPER_PREV5:
         ret = cd_prev5();
           break;

      case OPER_NEXT5:
         ret = cd_next5();
           break;

/*
        case OPER_MUTE:
         ret= cd_mute(0);
           break;

        case OPER_INCREASEVOLUME:
         ret= cd_increasevolume(1);
           break;

        case OPER_REDUCEVOLUME:
         ret= cd_increasevolume(-1);
           break;
*/
      default:
           ret = 0;
    }

    return ret;
}

/*
short cd_check_multisession(void)
{  struct cdrom_multisession cdi;

   if (cdrom_fd < 0
        || ioctl(cdrom_fd, CDROMMULTISESSION, &cdi) == 0) return -1;

   return (cdi.xa_flag == '1') ? 1 : 0;

}
*/
