/* sod2, a player for polychannel .csf music files.
 * Copyright (C) 1995, 1996 Russell Marks.
 *
 * Based on `sodman' by Graham Richards.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * sod2.c - the main player code.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <math.h>
#ifdef linux
#if 0
#include <sys/ioctl.h>     /* didn't exist in older kernels */
#endif
#include <sys/soundcard.h>
#endif
#ifdef _ALLOW_BUFFERING
#include <signal.h>
#endif
#include "sod2.h"
#include "grokfile.h"

#define SOD2_VER	"2.0"

/* remove 'inline' if the compiler doesn't like it */
#ifndef _ALLOW_INLINE
#define inline
#endif

/* djgpp doesn't appear to have optopt yet...? */
#ifdef MSDOS
int optopt='?';
#endif

/* global data */

int bsize=64;			/* pattern length being used */
int tempo=125;
int last_pattern,last_line;
int slotlen;			/* length of a notespace in samples */

char outfile[256];
FILE *out;

/* output environment */
/* output is generated internally at 'vsr', but output at 'psr'. */

int sample_maxval=255;		/* defaults to 8-bit output */
int sample_midval=128;		/* 'middle' value - approx half of above */
int sixteenbit=0;		/* guess :-) */
int widemag=256;		/* multiplier for 8-bit -> 16-bit */
int vsr=8000;			/* virtual (subsampled) sample rate */
int psr=8000;			/* physical (actual) sample rate */

/* flags */

int oversample=0;		/* if 1, do virtual->physical avg mapping */
int undersample=0;		/* if 1, add avgs between real samples */
#ifdef linux
int ioctlfile=1;
int firstioctl=1;
#else
int ioctlfile=0;
#endif
int clickdetect=0;		/* moan about possible clicks */
int buffer=0;			/* use buffering */
int preconstruct=0;		/* resample in advance, before playing */
int interpolate=0;		/* interpolate with fractional position */
int startline=1;
int pause_chk=1;
int pstereo=0;			/* pseudo-stereo */

int pstereobuf[1024];
int pstereobufsiz,pstereopos;
int psgap=250;

int realstereo=0;
int rstereobuf_l[1024],rstereobuf_r[1024];
int rstereopos;

int quiet=0;
int verbose=0;


int next_sample=0;
int sampletick=0,oldstick=0;
struct timeval oldtv,newtv;
int minhz=(1<<30);
int oldval;
int maxchan;

struct sample samples[MAX_SAMPLES];	/* ptrs to 8-bit unsigned samples */

struct pattern patterns[MAX_PATTERNS];

/* entries in line[x][] are indicies into patterns[]. */
int lines[MAX_LINES][MAX_LINE_DEPTH];

/* active notes array - an entry is NULL if unused, otherwise it points
 * to a struct note in a struct pattern somewhere.
 */
struct note *active[MAX_NOTES];
int last_active;

#ifdef _ALLOW_BUFFERING
unsigned char *buf_mem,*buf_head,*buf_tail;
struct itimerval itvslow,itvfast;
int out_fd;
int buf_in=0,buf_out=0;
int last_report=0;
int bufused=0;
int playdone=0;
int real_buf_write=BUF_WRITE;
#endif



/* function prototypes */
int main(int argc,char **argv);
void parseoptions(int argc,char **argv);
void usage_help(void);
void clear_data(void);
void runtime_fixup(void);
struct preconstruct *find_precon(struct note *nptr,
	struct preconstruct *preconarr,int preconlen);
void play(void);
void do_pause_chk(void);
int addnotes(int line,int noteslot);
static inline int gensample(int note,struct note *nptr);
static inline void gen_stereo(int pos,int val);
void output_sample(int count);
#ifdef linux
void ioctl_dsp(int fd);
#endif
#ifdef _ALLOW_BUFFERING
void init_buffer(void);
void kill_buffer(void);
void write_from_buffer(int n);
int write_to_buffer(int c,FILE *out);
#endif

int (*write_sample)(int c,FILE *out);


/* code starts here */

int main(int argc,char **argv)
{
int f;

#ifdef linux
strcpy(outfile,"/dev/dsp");
#else
strcpy(outfile,"output");
#endif

write_sample=fputc;		/* default to using stdio sample write */

parseoptions(argc,argv);

/* open output file */
if(strcmp(outfile,"-")==0)
  out=stdout;
else
  if((out=fopen(outfile,"wb"))==NULL)
    {
    fprintf(stderr,"Couldn't open output file '%s'.\n",outfile);
    exit(1);
    }

if(undersample) psr*=2;

if(realstereo) pstereo=0;

#ifdef linux
if(ioctlfile) ioctl_dsp(fileno(out));
#endif

if(pstereo)
  {
  for(f=0;f<1024;f++)
    pstereobuf[f]=0;
  pstereopos=0;
  oversample=undersample=0;	/* XXX output_pstereo() doesn't support 'em */
  }

if(realstereo)
  {
  for(f=0;f<1024;f++)
    rstereobuf_l[f]=rstereobuf_r[f]=0;
  rstereopos=0;
  oversample=undersample=0;
  }

vsr=psr;
if(oversample) vsr*=2;
if(undersample) vsr/=2;
if(pstereo) pstereobufsiz=(vsr*psgap)/22000;


clear_data();			/* zero out arrays and general init */
grokfile(argv[optind]);		/* read in file */

runtime_fixup();	/* fix runtime parts of arrays ready for playing */

#ifdef _ALLOW_BUFFERING
if(buffer) init_buffer();
#endif

play();			/* play the music */

#ifdef _ALLOW_BUFFERING
if(buffer)
  {
  playdone=1;
  while(bufused>0) pause();
  kill_buffer();
  }
#endif

if(strcmp(outfile,"-")!=0)
  fclose(out);

exit(0);
}


void parseoptions(int argc,char **argv)
{
int done=0,tmp,outchng=0;

done=0;
opterr=0;

do
  switch(getopt(argc,argv,"bcdg:hil:o:OpqRs:SUvwW:"))
    {
    case 'b':	/* use buffer process */
#ifdef _ALLOW_BUFFERING
      buffer=1;
#else
      fprintf(stderr,"Buffering was not enabled at compilation.\n");
      exit(1);
#endif
      break;
    case 'c':	/* click detect */
      clickdetect=1;
      break;
    case 'd':	/* disable pausing */
      pause_chk=0;
      break;
    case 'g':	/* ps gap */
      tmp=atoi(optarg);
      if(tmp==0 || tmp>1024)
        fprintf(stderr,"Strange gap length, ignoring\n");
      else
        psgap=tmp;
      break;
    case 'h':
      usage_help();
      exit(1);
    case 'i':	/* interpolate fractional position */
      interpolate=1;
      break;
    case 'l':	/* start line number */
      startline=atoi(optarg);
      if(startline==0)
        fprintf(stderr,"Strange start line, ignoring\n");
      break;
    case 'o':	/* output file */
      strcpy(outfile,optarg);
      outchng=1;
      ioctlfile=0;
      break;
    case 'O':	/* oversample */
      oversample=1;
      break;
    case 'p':	/* preconstruct */
      preconstruct=1;
      break;
    case 'q':	/* quiet, no messages at all */
      quiet=1;
      pause_chk=0;	/* disable pausing too */
      break;
    case 'U':	/* undersample */
      undersample=1;
      break;
    case 'v':	/* verbose, various extra messages */
      verbose=1;
      break;
    case 's':	/* sample rate */
      tmp=atoi(optarg);
      if(tmp==0)
        fprintf(stderr,"Strange sample rate, ignoring\n");
      else
        psr=tmp;
      break;
    case 'R':	/* real stereo */
      realstereo=1;
      break;
    case 'S':	/* pseudo-stereo */
      pstereo=1;
      break;
    case 'w':	/* wide - 16-bit */
      sample_maxval=65535;
      sample_midval=32768;
      sixteenbit=1;
      break;
    case 'W':	/* wide mag */
      tmp=atoi(optarg);
      if(tmp==0)
        fprintf(stderr,"Strange 16-bit multiplier, ignoring\n");
      else
        widemag=tmp;
      break;
    case '?':
      switch(optopt)
        {
        case 'g':
          fprintf(stderr,"The -g option needs a gap as argument.\n");
          break;
        case 'l':
          fprintf(stderr,"The -l option needs a starting line.\n");
          break;
        case 'o':
          fprintf(stderr,"The -o option takes a filename as an argument.\n");
          break;
        case 's':
          fprintf(stderr,"The -s option takes the output sample rate ");
          fprintf(stderr,"as an argument.\n");
          break;
        default:
          fprintf(stderr,"Option '%c' not recognised.\n",optopt);
        }
      exit(1);
    case -1:
      done=1;
    }
while(!done);

if(optind!=argc-1)	/* if no filename given */
  {
  usage_help();
  exit(1);
  }

if(undersample && oversample)
  {
  fprintf(stderr,"Oversampling and undersampling is stoopid, you eeediot!\n");
  exit(1);
  }

/* quiet overrides verbose always */
if(quiet) verbose=0;

if(outchng) buffer=0;	/* don't ever use buffering if output file changed */

/* pausing not supported if buffering, or if output is stdout */
if(buffer || strcmp(outfile,"-")==0) pause_chk=0;
}


void usage_help()
{
printf("Sod2 v%s - (c) 1995,1996 Russell Marks for improbabledesigns.\n",
	SOD2_VER);
printf(
"
usage: sod2 [-bcdhiOpqRSUvw] [-g gap] [-l start_line] [-o output_file]
		[-s sample_rate] [-W mult] [filename.csf | filename.tar]

	-b	use buffering (Linux only; speedy but uses more mem and CPU).
	-c	try to detect clicks - when a note stops playing and the
        	last sample was more than 10%% away from zero, it is
                reported with 'possible click from note on pattern N'.
                You can usually remove the click by using a tighter envelope.
                Note that this will not necessarily detect all clicks.
	-d	disable pausing (see man page for details).
        -g	set pseudo-stereo echo length ('gap'). The units are
        	number of samples at 22kHz and are scaled accordingly.
                The default is 250, and the maximum acceptable is 1024.
	-h	this usage help.
	-i	interpolate fractional sample position, for more accurate
        	resampling. Especially effective at low sample rates.
        -l	sets the starting line (usually 1). Good for starting halfway
        	through a track while developing it.
        -o	specify output file for music sample.
        -O	do 2x oversampling - note that this takes as much CPU time
        	as it would to play at *double* the output sample rate!
	-p	preconstruct - resample before playing (speedy but uses
        	more memory).
        -q	quiet - don't print any messages. Also, implies `-d'.
        -R	play in stereo as described by any '*s' or '*S' constructs
        	in the csf file (hence 'R' for 'real stereo'). If the file
                still seems to play in mono, try using '-S', which
                'stereoises' mono output.
        -s	specify sample rate to output at.
        -S	use pseudo-stereo.
        -U	do 2x undersampling - play at twice the specified sample
        	rate, and output an average value between every pair of
                output values.
        -v	verbose messages.
        -w	give 'wide' output, i.e. output a 16-bit sample.
        -W	set multiplier used to turn 8-bit -> 16-bit. Usually 256,
        	you may want to set this to smaller values to avoid
                output value clipping for files where the combined output
                is too loud. Not recommended generally, but can be useful.
                This flag is (obviously) ignored in normal (8-bit) mode.

With tar files, the *first* file in the archive must be the csf file,
and all samples used by the csf file must be in the tar archive.
");
}


void clear_data()
{
int f,g;

for(f=0;f<MAX_NOTES;f++)	active[f]=NULL;
for(f=0;f<MAX_SAMPLES;f++)	samples[f].data=NULL;
for(f=0;f<MAX_PATTERNS;f++)	patterns[f].mode=-1;
for(f=0;f<MAX_LINES;f++)
  for(g=0;g<MAX_LINE_DEPTH;g++)
    lines[f][g]=0;
}


void runtime_fixup()
{
int f,g,rate;
int maxgap=0,mingap=(1<<30);

/* the general idea here is to put in all the timing information
 * in the note structs.
 */

slotlen=15*vsr/tempo;

for(f=1;f<=last_pattern;f++) if(patterns[f].mode!=-1)
  for(g=0;g<bsize;g++) if(patterns[f].notes[g].vol>0)
    {
    rate=samples[patterns[f].notes[g].sample].rate;

    if(sixteenbit)
      patterns[f].notes[g].vol*=widemag;
    
    /* notenum -> next-sample gap */
    /* time to wheel out the magic numbers */
    patterns[f].notes[g].gap=FIX_DBL(((double)rate*
    	pow(1.059463,(double)patterns[f].notes[g].notenum)/(double)vsr));
    
    if(patterns[f].notes[g].gap>maxgap) maxgap=patterns[f].notes[g].gap;
    if(patterns[f].notes[g].gap<mingap) mingap=patterns[f].notes[g].gap;

    /* fix enveloping stuff */
    patterns[f].notes[g].sustain*=slotlen/8;
    patterns[f].notes[g].release*=slotlen/8;
    patterns[f].notes[g].relsus=patterns[f].notes[g].release;
    patterns[f].notes[g].release+=patterns[f].notes[g].sustain;

    patterns[f].notes[g].data=samples[patterns[f].notes[g].sample].data;
    patterns[f].notes[g].len=FIX_UP(samples[patterns[f].notes[g].sample].len);
    
    /* scale up (or down) stereo positioning
     * it's currently correct for 8kHz
     */
    patterns[f].notes[g].pos=(patterns[f].notes[g].pos*vsr)/8000;
    }


if(verbose)
  {
  double tmp;
  
  fprintf(stderr,
  	"max. gap %2d.%03d samples => lowest accurate sample rate %d Hz\n",
  	FIX_DOWN(maxgap),(FIX_FRAC(maxgap)*1000)/1024,FIX_DOWN(vsr*maxgap));
  
  /* there's a max. error of 1/1024 samples in mingap */
  tmp=60.0/((double)mingap)/1.059463;
  fprintf(stderr,
   "min. gap %2d.%03d samples => max. tuning variance %1.3f semitones/min\n",
  	FIX_DOWN(mingap),(FIX_FRAC(mingap)*1000)/1024,tmp);
  }


/* the remainder of the routine is for preconstruct mode only */

if(preconstruct)
  {
  /* ok, resample everything now rather than later */
  struct note *noteptr;
  int pattern,noteslot;
  int total_notes=0;
  int preconlen=0,preconsize=16;
  int mem=0;
  struct preconstruct *preptr;
  struct preconstruct *preconarr=
  		malloc(preconsize*sizeof(struct preconstruct));
  
  
  /* the theory behind this is that patterns and notes will almost
   * certainly be repeated many times, and if we generate all the notes
   * that will be needed *in advance*, this will probably save a
   * lot of CPU time when we start playing. The disadvantages are
   * that it delays the start of playback, and it takes a lot more
   * memory.
   */

  if(preconarr==NULL) die("allocate memory");

  if(verbose) fprintf(stderr,"finding unique notes...");
    
  /* we first find all the unique notes */
  for(pattern=1;pattern<=last_pattern;pattern++)
    for(noteslot=0;noteslot<bsize;noteslot++)
      {
      noteptr=&(patterns[pattern].notes[noteslot]);
      
      /* don't look at the rest of a pattern after ';' */
      if(noteptr->vol==-1) break;
      
      /* skip any notespaces without actual notes */
      if(noteptr->vol==0) continue;
      
      /* add the note if it hasn't been seen before */
      if(find_precon(noteptr,preconarr,preconlen)==NULL)
        {
        preconlen++;
        if(preconlen*sizeof(struct preconstruct)>preconsize)
          {
          preconsize+=128*sizeof(struct preconstruct);
          if((preconarr=realloc(preconarr,preconsize))==NULL)
            die("allocate memory");
          }
        
        preconarr[preconlen-1].notenum=noteptr->notenum;
        preconarr[preconlen-1].sustain=noteptr->sustain;
        preconarr[preconlen-1].release=noteptr->release;
        preconarr[preconlen-1].vol    =noteptr->vol;
        preconarr[preconlen-1].sample =noteptr->sample;
        preconarr[preconlen-1].notenum=noteptr->notenum;
        preconarr[preconlen-1].exnote =noteptr;
        }
      
      total_notes++;
      }
  
  if(!quiet)
    fprintf(stderr,"%d notes total, %d unique\n",total_notes,preconlen);
  if(verbose) fprintf(stderr,"generating resampled notes...\n");
  
  /* now do the resampling */
  last_active=0;
  for(f=0;f<preconlen;f++)
    {
    int onesec=vsr*(sixteenbit?2:1);
    int size=onesec;
    int len=0;
    
    if(verbose) fprintf(stderr,"\r%4d  of  %4d ",f+1,preconlen);
    
    active[0]=preconarr[f].exnote;
    active[0]->offset=0;
    active[0]->counter=0;
    
    if(!sixteenbit)
      {
      /* 8-bit */
      signed char *data=malloc(size);
      
      if(data==NULL) die("allocate memory");
      
      while(active[0]!=NULL)
        {
        g=gensample(0,active[0]);
        if(len>=size)
          if((data=realloc(data,size+=onesec))==NULL)
            die("allocate memory");
        if(g<-128) g=-128; if(g>127) g=127;
        data[len++]=g;
        }
      
      preconarr[f].data.p8=data;
      }
    else
      {
      /* 16-bit */
      signed short *data=malloc(size*sizeof(signed short));
      
      if(data==NULL) die("allocate memory");
      
      while(active[0]!=NULL)
        {
        g=gensample(0,active[0]);
        if(len>=size)
          {
          size+=onesec;
          if((data=realloc(data,size*sizeof(signed short)))==NULL)
            die("allocate memory");
          }
        if(g<-32768) g=-32768; if(g>32767) g=32767;
        data[len++]=g;
        }
      
      preconarr[f].data.p16=data;
      }

    preconarr[f].len=len;
    mem+=len;
    }

  if(verbose) fprintf(stderr,"\rfilling in note structs...\n");
  
  /* now go back and fill in the 'predata' and 'prelen' fields
   * in all the note structs.
   */
  for(pattern=1;pattern<=last_pattern;pattern++)
    for(noteslot=0;noteslot<bsize;noteslot++)
      {
      noteptr=&(patterns[pattern].notes[noteslot]);
      if(noteptr->vol==-1) break;
      if(noteptr->vol==0) continue;
      
      if((preptr=find_precon(noteptr,preconarr,preconlen))==NULL)
        die("find resampled note (can't happen!)");
      else
        {
        if(!sixteenbit)
          { /* 8-bit */
          noteptr->p.s8.predata=preptr->data.p8;
          noteptr->p.s8.premax =preptr->data.p8+preptr->len-1;
          }
        else
          { /* 16-bit */
          noteptr->p.s16.predata=preptr->data.p16;
          noteptr->p.s16.premax =preptr->data.p16+preptr->len-1;
          }
        }
      }
  
  if(!quiet)
    fprintf(stderr,"using %dk for preconstructed samples\n",(mem+1023)/1024);

  /* we don't need preconarr any more, so ditch it */
  free(preconarr);
  }	/* end of if(preconstruct) */
}


struct preconstruct *
find_precon(struct note *nptr,struct preconstruct *preconarr,int preconlen)
{
int f;

for(f=0;f<preconlen;f++)
  if(preconarr[f].notenum==nptr->notenum && 
     preconarr[f].sustain==nptr->sustain && 
     preconarr[f].release==nptr->release && 
     preconarr[f].vol    ==nptr->vol && 
     preconarr[f].sample ==nptr->sample) return(preconarr+f);

return(NULL);	/* couldn't find it */
}


/* finally, play the music.
 *
 * I originally had play(), addnotes(), gensample() and output_sample()
 * as one big function, but it became a bit confusing to work on.
 * (With the addition of the two stereo options, and preconstruct
 * supported in all cases, it's got worse again. :-/)
 *
 * Since gensample() is a stunningly major hotspot (tens of millions of
 * calls are likely), you should compile with -finline-functions (assuming
 * gcc is being used).
 */
void play()
{
int line,notesp,noteslot,count,note,pos;
int done=0;
struct note **aptr,*nptr;

last_active=-1;
oldval=sample_midval;
maxchan=0;

if(verbose)
  {
  gettimeofday(&oldtv,NULL);
  oldstick=sampletick;
  }

if(startline>last_line || startline<0) startline=last_line;
  
for(line=startline;line<=last_line+1 && done==0;line++)
  {
  for(noteslot=0;noteslot<bsize && done==0;noteslot++)
    {
#ifdef _ALLOW_PAUSE
    if(pause_chk && (noteslot&8)==0) do_pause_chk();
#endif
  
    if(addnotes(line,noteslot)) break;
    
    /* generate output for the duration of this notespace */
    /* this also kills off any notes which end during that time */
    for(notesp=0;notesp<slotlen;notesp++)
      {
      for(note=0;note<=last_active;)
        if(active[note]==NULL)
          memmove(active+note,active+note+1,
            sizeof(active[0])*(last_active---note));	/* wahay :-) */
        else
          note++;

      count=0;
      
      if(!realstereo)
        {
        /* mono or pseudostereo is "real-time", i.e. generate then play */
        if(preconstruct)
          {
          if(!sixteenbit)	/* 8-bit */
            for(note=0,aptr=active;note<=last_active;note++,aptr++)
              {
              count+=*((nptr=*aptr)->p.s8.preptr);
              if(++(nptr->p.s8.preptr)>nptr->p.s8.premax) *aptr=NULL;
              }
          else			/* 16-bit */
            for(note=0,aptr=active;note<=last_active;note++,aptr++)
              {
              count+=*((nptr=*aptr)->p.s16.preptr);
              if(++(nptr->p.s16.preptr)>nptr->p.s16.premax) *aptr=NULL;
              }
          }
        else
          {
          for(note=0,aptr=active;note<=last_active;note++,aptr++)
            count+=gensample(note,*aptr);
          }
  
        if(line==last_line+1 && last_active==-1) done=1;
        output_sample(count);
        }
      else
        {
        /* real stereo is tricky, as it works by delaying sounds
         * on the left or right channels to model the delay you get
         * in the real world when sounds originate at different places.
         * this delay is up to 16*vsr/8000. We need to incrementally write
         * (once for each gensample!) to a buffer at least this long.
         */
        if(preconstruct)
          {
          if(!sixteenbit)	/* 8-bit */
            for(note=0,aptr=active;note<=last_active;note++,aptr++)
              {
              pos=(nptr=*aptr)->pos;
              count=*(nptr->p.s8.preptr);
              gen_stereo(pos,count);
              if(++(nptr->p.s8.preptr)>nptr->p.s8.premax) *aptr=NULL;
              }
          else			/* 16-bit */
            for(note=0,aptr=active;note<=last_active;note++,aptr++)
              {
              pos=(nptr=*aptr)->pos;
              count=*(nptr->p.s16.preptr);
              gen_stereo(pos,count);
              if(++(nptr->p.s16.preptr)>nptr->p.s16.premax) *aptr=NULL;
              }
          }
        else
          {
          for(note=0,aptr=active;note<=last_active;note++,aptr++)
            {
            pos=(nptr=*aptr)->pos;
            count=gensample(note,nptr);
            gen_stereo(pos,count);
            }
          }
        
        if(line==last_line+1 && last_active==-1) done=1;
        
        output_sample(rstereobuf_l[rstereopos]);	/* left */
        output_sample(rstereobuf_r[rstereopos]);	/* right */
        
        rstereobuf_l[rstereopos]=rstereobuf_r[rstereopos]=0;
        rstereopos++; if(rstereopos>=1024) rstereopos=0;
        }
      
      sampletick++;
      if(last_active>maxchan) maxchan=last_active;
      }
    }
  
  if(!quiet)
    fprintf(stderr,"%swritten block %d\n",verbose?"\n":"",line);
  }

/* actually maxchan+1 as last_active is -1 for zero channels */
if(!quiet) fprintf(stderr,"done; max virtual channels %d\n",maxchan+1);

if(verbose) fprintf(stderr,"Lowest local sample rate %d Hz\n",minhz);
}


#ifdef _ALLOW_PAUSE

/* check if any keys are pending. Since we're reading line-by-line,
 * we're effectively only waiting for a CR; this doesn't matter,
 * as it's only a yes/no test.
 */
int input_pending()
{
struct timeval tv;
fd_set rfds;

FD_ZERO(&rfds); FD_SET(0,&rfds);
tv.tv_sec=tv.tv_usec=0;
select(1,&rfds,NULL,NULL,&tv);

if(!FD_ISSET(0,&rfds)) return(0);	/* no input */

/* so we're pausing. throw away all input. */
while(FD_ISSET(0,&rfds))
  {
  getchar();
  
  FD_ZERO(&rfds); FD_SET(0,&rfds);
  tv.tv_sec=tv.tv_usec=0;
  select(1,&rfds,NULL,NULL,&tv);
  }

return(1);	/* there was input (now ditched) */
}


/* re-open output file. */
int openout()
{
struct stat sbuf;

/* this *should* just be a matter of appending to the file in all
 * cases, but that seems problematic. So, we append if it's a file or
 * symlink, otherwise open with "wb".
 */

if(stat(outfile,&sbuf)<0) return(0);

if(S_ISREG(sbuf.st_mode) || S_ISLNK(sbuf.st_mode))
  {
  /* normal file, so append */
  if((out=fopen(outfile,"ab"))==NULL) return(0);
  }
else
  {
  /* device, fifo, etc. - just write */
  if((out=fopen(outfile,"wb"))==NULL) return(0);
  }

#ifdef linux
if(ioctlfile) ioctl_dsp(fileno(out));
#endif

return(1);	/* opened ok */
}


/* test for pause, and if paused, wait for continue.
 */
void do_pause_chk()
{
if(input_pending())
  {
  int openok;
  
  fclose(out);
  
  fprintf(stderr,"< paused - press return to continue > ");
  
  do
    {
    /* XXX messy, should have a non-timeout select to do this */
    while(!input_pending()) usleep(100000);
    
    if((openok=openout()))
      fprintf(stderr,"< continuing... >\n");
    else
      fprintf(stderr,"< still paused - couldn't re-open! > ");
    }
  while(!openok);
  }
}


#endif /* _ALLOW_PAUSE */


int addnotes(int line,int noteslot)
{
int lp,pat,nf,nf2;
int tmp,tmp2,linetime;
int *sublinep;
struct note *activenfp;

for(lp=0,sublinep=lines[line];lp<MAX_LINE_DEPTH;lp++,sublinep++)
  {
  if((*sublinep)==0) break;
  if(patterns[*sublinep].notes[noteslot].vol==-1)
    return(1);
  }

for(lp=0,sublinep=lines[line];lp<MAX_LINE_DEPTH;lp++,sublinep++)
  {
  if((pat=(*sublinep))==0) break;
  
  /* add all new notes to be played at this notespace */
  if(patterns[pat].notes[noteslot].vol!=0)
    {
    if(last_active>=MAX_NOTES)
      fprintf(stderr,"Too many simultaneous notes!\n");
    else
      {
      nf=++last_active;
      activenfp=active[nf]=&patterns[pat].notes[noteslot];
      activenfp->offset=0;
      activenfp->counter=0;
      if(preconstruct)
        if(!sixteenbit)
          activenfp->p.s8.preptr=activenfp->p.s8.predata;
        else
          activenfp->p.s16.preptr=activenfp->p.s16.predata;

      switch(active[nf]->mode)
        {
        case PIANO_MODE:
          /* turn off prev. notes of this pattern which have = notenum */
          tmp=activenfp->notenum;
          tmp2=activenfp->pattern;
          for(nf2=0;nf2<last_active;nf2++)
            if(active[nf2]!=NULL &&
               active[nf2]->notenum==tmp && active[nf2]->pattern==tmp2)
              active[nf2]=NULL;
          break;
        
        case RETRIGGER_MODE:
          /* turn off prev. notes of this pattern no matter what */
          tmp2=activenfp->pattern;
          for(nf2=0;nf2<last_active;nf2++)
            if(active[nf2]!=NULL && active[nf2]->pattern==tmp2)
              active[nf2]=NULL;

        default:
          /* nothing needs doing for multi mode */
        }
      }
    }
  }

if(verbose && noteslot%8==7)
  {
  fprintf(stderr,"%2d:",last_active+1);
  gettimeofday(&newtv,NULL);
  linetime=(newtv.tv_sec-oldtv.tv_sec)*100+
           (newtv.tv_usec-oldtv.tv_usec)/10000;
  fprintf(stderr,"%6d ",linetime=((sampletick-oldstick)*10000/linetime)/100);
  oldtv=newtv;
  oldstick=sampletick;
  if(linetime<minhz) minhz=linetime;
  }

return(0);
}


/* the big hotspot.
 * suggestions for improving the speed of this routine are most welcome.
 * note that this isn't really a hotspot when in preconstruct mode;
 * play() is then the biggest one.
 */
static inline int gensample(int note,struct note *nptr)
{
int val;

if(interpolate)
  {
  int sub=FIX_FRAC(nptr->offset);
  signed char *ptr=(nptr->data+FIX_DOWN(nptr->offset));
  val=((*ptr)*(1024-sub)+(*(ptr+1))*sub)*nptr->vol/102400;
  }
else
  val=(*(nptr->data+FIX_DOWN(nptr->offset)))*nptr->vol/100;

if(nptr->counter>=nptr->sustain)
  {
  /* scale to account for the decay */
  val*=1024-FIX_UP(nptr->counter-nptr->sustain)/nptr->relsus;
  val/=1024;
  }

nptr->offset+=nptr->gap;
nptr->counter++;

if(nptr->offset>=nptr->len || nptr->counter>=nptr->release)
  {
  /* this is considered an error, so we output it even if quiet==1 */
  if(clickdetect && (val<-sample_midval/10 || val>sample_midval/10))
    fprintf(stderr,"possible click from note on pattern %d\n",nptr->pattern);
  active[note]=NULL;
  }

return(val);
}


/* add val to stereo buffers */
static inline void gen_stereo(int pos,int val)
{
/* add val, correctly delayed on either left or right buffer,
 * to add the stereo positioning.
 */

if(pos<0)
  {
  rstereobuf_l[rstereopos]+=val;
  rstereobuf_r[(rstereopos-pos)%1024]+=val;
  }
else
  {
  rstereobuf_l[(rstereopos+pos)%1024]+=val;
  rstereobuf_r[rstereopos]+=val;
  }
}


void output_pstereo(int c,FILE *out)
{
int bl,br;

c-=sample_midval;

bl=(c-pstereobuf[pstereopos])/2;
br=(c+pstereobuf[pstereopos])/2;

if(sixteenbit)
  {
  write_sample(bl&0x00ff,out); write_sample((bl&0xff00)>>8,out);
  write_sample(br&0x00ff,out); write_sample((br&0xff00)>>8,out);
  }
else
  {
  write_sample(bl+128,out); write_sample(br+128,out);
  }
  
pstereobuf[pstereopos]=c;
pstereopos++; if(pstereopos>=pstereobufsiz) pstereopos=0;
}



void output_sample(int count)
{
int tmp;

count+=sample_midval;
if(count>sample_maxval) count=sample_maxval;
if(count<0) count=0;

if(pstereo)
  {
  output_pstereo(count,out);
  return;
  }

if(!sixteenbit)
  {
  /* 8-bit output */
  if(oversample)
    {
    if(sampletick%2)
      write_sample((count+oldval)>>1,out);
    else
      oldval=count;
    }
  else
    {
    if(undersample)
      {
      write_sample((count+oldval)>>1,out);
      oldval=count;
      }
    write_sample(count,out);
    }
  }
else
  {
  /* 16-bit output */
  if(oversample)
    {
    if(sampletick%2)
      {
      tmp=((count+oldval)>>1);
      tmp-=32768;
      write_sample(tmp&255,out);
      write_sample(tmp>>8,out);
      }
    else
      oldval=count;
    }
  else
    {
    if(undersample)
      {
      tmp=((count+oldval)>>1);
      tmp-=32768;
      write_sample(tmp&255,out);
      write_sample(tmp>>8,out);
      oldval=count;
      }
    count-=32768;
    write_sample(count&255,out);
    write_sample(count>>8,out);
    }
  }      
}


void die(char *str)
{
fprintf(stderr,"Couldn't %s.\n",str);
exit(1);
}


#ifdef linux
void ioctl_dsp(int fd)
{
int tmp;

if((ioctl(fd,SNDCTL_DSP_SPEED,&psr))==-1)
  fprintf(stderr,"Error setting frequency\n");
else
  if(!quiet && firstioctl) fprintf(stderr,"Playback frequency: %d Hz\n",psr);

firstioctl=0;

tmp=(!sixteenbit)?AFMT_U8:AFMT_S16_LE;
if((ioctl(fd,SNDCTL_DSP_SETFMT,&tmp))==-1)
  fprintf(stderr,"Error setting sample width\n");

if(pstereo || realstereo)
  {
  tmp=1;
  if((ioctl(fd,SNDCTL_DSP_STEREO,&tmp))==-1)
    fprintf(stderr,"Error enabling stereo\n");
  }
}
#endif


#ifdef _ALLOW_BUFFERING

/* Initialise buffer. This routine should be called just before
 * playing starts.
 */
void init_buffer()
{
int tmp;
int divby=psr;

out_fd=fileno(out);

tmp=BUF_FRAG_ARG;	/* two 16k fragments */
real_buf_write=BUF_WRITE;

/* yes, these *should* be separate like this */
if(sixteenbit) real_buf_write*=2,divby*=2,tmp++;
if(pstereo || realstereo) real_buf_write*=2,divby*=2,tmp++;

ioctl(out_fd,SNDCTL_DSP_SETFRAGMENT,&tmp);

if((buf_mem=malloc(BUFFER_SIZE))==NULL) die("malloc buffer");

buf_head=buf_tail=buf_mem;
bufused=0;

write_sample=write_to_buffer;

signal(SIGALRM,write_from_buffer);

tmp=((real_buf_write*2000)/divby)/10;
itvfast.it_value.tv_sec= tmp/1000;
itvfast.it_value.tv_usec=(tmp%1000)*1000;
itvfast.it_interval.tv_sec=itvfast.it_interval.tv_usec=0;

tmp=((real_buf_write*8000)/divby)/10;
itvslow.it_value.tv_sec= tmp/1000;
itvslow.it_value.tv_usec=(tmp%1000)*1000;
itvslow.it_interval.tv_sec=itvslow.it_interval.tv_usec=0;

setitimer(ITIMER_REAL,&itvslow,NULL);
}


void kill_buffer()
{
/* kill timer */
itvslow.it_value.tv_sec= 0;
itvslow.it_value.tv_usec=0;
setitimer(ITIMER_REAL,&itvslow,NULL);

free(buf_mem);
}


void write_from_buffer(int n)
{
int f;
struct timeval tv;

signal(SIGALRM,write_from_buffer);

if(bufused>=real_buf_write || (playdone && bufused>0))
  {
  f=write(out_fd,buf_head,(bufused<=real_buf_write)?bufused:real_buf_write);
  if(f!=-1)
    {
    bufused-=f; buf_out+=f;
    buf_head+=f; if(buf_head>=buf_mem+BUFFER_SIZE) buf_head-=BUFFER_SIZE;
    }

  gettimeofday(&tv,NULL);
  if(verbose && last_report+5<=tv.tv_sec)
    {
    int divby=psr*(sixteenbit?2:1)*((pstereo || realstereo)?2:1);
    
    fprintf(stderr,"%s[buf %7d:%7d",playdone?"":"\n",buf_in,buf_out);
    if(buf_out!=0)
      fprintf(stderr," = %d.%03d",buf_in/buf_out,(buf_in*1000/buf_out)%1000);
    fprintf(stderr,", %d ahead (%d.%03d sec)]\n",bufused,
    		bufused/divby,(bufused*1000/divby)%1000);
    last_report=tv.tv_sec;
    buf_in=buf_out=0;
    }
  setitimer(ITIMER_REAL,&itvslow,NULL);
  }
else
  {
  if(verbose) fprintf(stderr,"!");
  setitimer(ITIMER_REAL,&itvfast,NULL);
  }
}


int write_to_buffer(int c,FILE *out)
{
while(bufused>=BUFFER_SIZE-real_buf_write) pause();
*buf_tail++=c; bufused++; buf_in++;
if(buf_tail>=buf_mem+BUFFER_SIZE) buf_tail-=BUFFER_SIZE;
return(0);
}

#endif /* _ALLOW_BUFFERING */

