/*  ==========================================================
    SING - ALONG DISK PLAYER.
    (C) 1998, 1999   Michael Glickman
    ----------------------------------------------------------
    NOTICE:
            Sing-Along Disk Player is copyrighted by the author.
	    See COPYRIGHT regarding distribution policy and
	    conditions of use.
	
            You are expected to provide appropriate references
            when using a part of the code in your software.

	    Author strongly advices against using this code, or
	    a part of it, in an application designed to run  on
	    any Microsoft(tm) platfrom.
    ========================================================= */

#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <linux/types.h>
#include <time.h>
#include <curses.h>

#include "sad.h"
#include "sadp_curses.h"

#define  SADP_SETTINGS_FNAME  "~/.sadp/sadprc"

extern char    CDDeviceName[];
/*=============================*/
/*         DSP_SUPPORT         */
#ifdef DSP_SUPPORT
extern u_int  SamplingRate;
extern short  FourInterlace;
extern short  FourGain, WaveVScale;
extern short  AudioSampleSize;
/*  0 - don't use device number for OSS  */
/* -1 - no audio device */
/* -2 - path specified explicitly */
extern int    AudioDeviceNumber;
extern char   AudioDeviceName[];
#endif
/*                             */
/*=============================*/

/*=============================*/
/*        MIXER_SUPPORT        */
#ifdef MIXER_SUPPORT
/*  0 - don't use device number for OSS  */
/* -1 - no mixer device */
/* -2 - path specified explicitly */
extern int    MixerDeviceNumber;
extern char   MixerDeviceName[];
#endif

extern char   DefaultCategory[];
extern short  SaveMixer;

extern short   AutoPlay;
extern short   ProbeCD;
extern short   LoopPlay;
extern short   SaveSet, SaveData;

#ifdef RCDDB_SUPPORT
extern char  UserAddr[151];
extern char  MailServer[151];
extern char  UserFullName[151];
extern char  ProxyURL[151];
extern short UseProxy, ProtocolUsed;
extern short ForceReg, UseLogFile;
#endif

extern short  RunBackground;
extern short  QuietMode;
extern short  BkgrColour, ShadColour, BottomColour;
extern short  FourWndCount, WaveObturate, UseMMap;
extern short  CharSet, OptionsChanged;
extern short  UseMCN;

extern MOUSE_SUPPORT_OPTION MouseSupport, MouseSupportOrig;
extern PLAY_MODE_TYPE  PlayMode;
extern TIME_DISPLAY_MODE_TYPE TimeDisplayMode;
extern HEADER_MODE  HeaderMode;
extern OPERATION StartOper;
extern PLATFORM  Platform;

extern char *ShortDBPath;
extern char *ShortDBAltPath;
#ifdef DEFAULT_WDB_NAME
extern char  *ShortWDBName; 
#endif
#ifdef DEFAULT_XMCD_PATH
extern char *ShortXMCDPath;
#endif


extern short  HelpMode;
extern short  track_start;
extern u_char cd_status;

static char *OptHeader = "NDT";

static struct opt_info
{  	short *var_addr;	 
	char	opt_code;
	char	reserve;
} OptInfo[] =
  {  {&AutoPlay,   'A', '\0'}
   , {&ProbeCD,    'P', '\0'}	
   , {&LoopPlay,   'L', '\0'}
   , {&SaveData,   'D', '\0'}
   , {&SaveSet,    'S', '\0'}
   , {&SaveMixer,  'X', '\0'}
#ifdef MCN_SUPPORT
   , {&UseMCN,     'M', '\0'}
#endif
#if defined(DSP_SUPPORT) && (defined(MMAP_SUPPORT) || defined(ALSA_MMAP_SUPPORT))
   , {&UseMMap,    'M', '\0'}
#endif
#ifdef RCDDB_SUPPORT
   , {&ForceReg,   'R', '\0'}
   , {&UseLogFile, 'G', '\0'}
#endif
   };	

static short OptCount = sizeof(OptInfo) / sizeof(struct opt_info);

static short *PtrColours[3] = {&BkgrColour, &ShadColour, &BottomColour};
static short  FourWndCountOrig, CharSetOrig;



static int ProcessInteger(const char *option, const char *name, short *target,
                                        short lowval, short highval)
{
    long val;
    char *endptr;

    if (option == NULL) return 0;
    val = strtol(option, &endptr, 10);

    if (*endptr == '\0' && 
              val >=(long)lowval && val <= (long)highval )
    {  *target = (u_short)val;
       return 1;
    }

	{ char errmes[121];

	  sprintf (errmes, "%s can be a whole number between %d and %d",
                       name,  lowval, highval );
	  show_error_message(errmes);
	}	  					   
    return 0;
}

#ifdef DSP_SUPPORT
static int ProcessSamplingRate(char *option)
{
    long val;

     if (option == NULL) return 0;
    val = strtol(option, NULL, 10);

    if (val >= 8000  && val <= 96000)
    {  SamplingRate = val;
       return 1;
    }
    else
    if (val >=0 && val <= 2)
    {   SamplingRate = 44100 >> val;
       return 1;
    }

    show_error_message(
         "To set sampling rate, use 0, 1, 2\n"
		 "or a whole number between 8000 and 96000");

    return 0;
}

static int ProcessSampleSize(char *option)
{
    long val;

    if (option == NULL) return 0;
    val = strtol(option, NULL, 10);

    if (val == 8  || val == 16)
    {  AudioSampleSize = val;
       return 1;
    }

    show_error_message("Audio sample size can be either 8 or 16");

    return 0;
}

#endif

static int  ProcessMouseSupport(char *option)
{

    switch (tolower(*option))
    {  case 'n':
		MouseSupport = MOUSE_NONE;
		break;

       case 'c':
       case 'y':
		MouseSupport = MOUSE_NATIVE;
		break;

#if GPM_SUPPORT
       case 'g':
		MouseSupport = MOUSE_GPM;
		break;
#endif

	 case 'd' :
	 case 's' :
		MouseSupport = MOUSE_DEFAULT;	
		break;


	  default:
	     show_error_message(
#if GPM_SUPPORT
              "Valid mouse support options: d, n, c, g");
#else
              "Valid mouse support options: n or y");
#endif
	    return 0;
    }

    return 1;
}


static int ProcessOptions(char *option)
{
	short  i;	
	char c, c1;

	if (option == NULL) return 0;

	while ((c=*option++) != '\0')
	{  c1 = toupper(c);

	   for ( i=0; i<OptCount; i++)
	   {  if (OptInfo[i].opt_code == c1) break; }
	
	   if (i>= OptCount)
	   {  char err_mes[121];
	      sprintf (err_mes, "Invalid option: %c", c);
		  show_error_message(err_mes);
		  return 0;
  	   }
     	
	   *(OptInfo[i].var_addr) = (c==c1) ? 1: 0;
	}

      return 1;
}


static int ProcessColours(char *option)
{  static char buf[2] = { '\0', '\0'};
   static char *name[3] = {NULL, NULL, NULL};
   short i;
   char c;

   if (option == NULL) return 0;
   if (toupper(*option) == 'M' && *(option+1) == '\0')
   {  BkgrColour = -10;
      return 1;
   }


   BkgrColour = ShadColour = BottomColour = -1;
   if (!name[0])
   {	if (Platform == PLTF_XFORMS)
	{  name[0] = "Background colour";
	   name[1] = "Top foreground colour";
	   name[2] = "Bottom foreground colour";
	}
	else
	{  name[0] = "Panel background colour";
	   name[1] = "Button background colour";
	   name[2] = "Button foreground colour";
	}
   }

   for (i=0; i<3 && (c=*option++) != '\0' ; i++)
   {  buf[0] = c;
      if (! ProcessInteger(buf, name[i], PtrColours[i], 0, 7))
      return 0;
   }

   if (i==0)
   {  show_error_message("Missing colour specidication");
      return 0;
   }
   return 1;
}


static int ProcessTimeDisplayMode(char *optarg)
{

    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;

    switch(toupper(*optarg))
    { case 'N':
      case 'E':
          TimeDisplayMode = DM_ELAPSED;
          break;

      case 'R':
          TimeDisplayMode = DM_REMAINED;
          break;

      case 'P':
          TimeDisplayMode = DM_PROGRESS;
          break;

      case 'M':
          TimeDisplayMode = DM_PROGREM;
          break;

      default:
          goto BadArg;
    }

    return 1;

BadArg:
    show_error_message(
	    "Wrong time display mode, use single letter:\n"
	    "    E - elapsed,   R - remained\n"
	    "    P - % elapsed, M - % remained");
    return 0;

}

static int ProcessPlayMode(char *optarg)
{

    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;

    switch(toupper(*optarg))
    { case 'N':
	PlayMode = PLAY_NORMAL;
	break;

      case 'L':
	PlayMode = PLAY_LIST;
	break;

      case 'R':
	PlayMode = PLAY_RANDOM;
	break;

      case 'I':
	PlayMode = PLAY_INTRO;
	break;

      case 'S':
	PlayMode = PLAY_SINGLE;

      default:
	goto BadArg;
    }

    return 1;

BadArg:
    show_error_message("Wrong play mode, use single letter:\n"
                       "    N - normal, L - loop, R - random, I - intro");
    return 0;

}
static int ProcessOper(char *optarg)
{
    if (optarg == NULL || strlen(optarg) != 1) goto BadArg;
    switch(toupper(*optarg))
    {
      case 'P':
         StartOper = OPER_PLAY;
	 break;

      case 'S':
         StartOper = OPER_STOP;
	 break;

      case 'N':
         StartOper = OPER_NEXT;
	 break;

      case 'R':
         StartOper = OPER_PREV;
	 break;

      default:
	goto BadArg;
    }

    return 1;

BadArg:
    show_error_message("Wrong operation, use single letter:\n"
					   "    P - play, S - stop, N - next, R - previous");
    return 0;
}

static int ProcessHeaderMode(char *optarg)
{   char *ptr;

    if (strlen(optarg) !=1) goto BadArg;

    ptr = strchr(OptHeader, toupper(*optarg));
    if (ptr == NULL) goto BadArg;

    HeaderMode =  ptr - OptHeader;
    return 1;

BadArg:
    show_error_message("Wrong header mode, use single letter:\n"
                       "   T - track name, D - disc info(title+artist), N - none");
    return 0;
}


static void	ShowCommandLineHelp()
{
  printf ("\nSing-Along Disk Player. Version %s. Command line options:\n\n",
		get_sadp_version() );

#ifdef DSP_SUPPORT
   printf ("\tA - audio (DSP) device path (default: /dev/dsp)\n");
#endif
   printf ("\tC - CD device path (default: /dev/cdrom)\n");
#ifdef DEFAULT_WDB_NAME
   printf ("\tD - workman data base file name\n");  
#endif   
   printf ("\tH - header mode: t-track name(default), d-disc info, n-none\n");
#ifdef MIXER_SUPPORT
   printf ("\tM - mixer device path (default: /dev/mixer)\n");
#endif
   printf ("\tO - options (use capital letter to turn feature on):\n");
   printf ("\t    P/p - auto-probe, A/a - auto-play L/l - loop play\n");
   printf ("\t    S/s - save settings at end, D/d - auto-save data\n");
#if defined(DSP_SUPPORT) && (defined(MMAP_SUPPORT) || defined(ALSA_MMAP_SUPPORT))
   printf ("\t    M/m - enable/disable memory map for DSP\n");
#endif   
#ifdef MCN_SUPPORT
   printf ("\t    N/n - use/ignore MCN\n");
#endif
   printf ("\t    X/x - save/ignore mixer volumes\n");
#ifdef RCDDB_SUPPORT
   printf ("\t    G/g - use log for RCDDB connection\n");
#endif
#ifdef DSP_SUPPORT
   printf ("\tS - sample size (8 or 16, default: 16)\n");
#endif
#ifdef DEFAULT_XMCD_PATH
   printf ("\tX - xmcd data base root directory\n");  
#endif   
   printf ("\tb - panel and button colours (1 to 3 digits, range 0-7)\n");
   printf ("\tc - character set (0 to 4)\n");
#ifdef DSP_SUPPORT
   printf ("\tf - sampling frequency: 0-44100, 1-22050, 2-11025\n");
   printf ("\tg - spectrum analyser gain (%d to %d)\n",
		                  GAIN_MIN, GAIN_MAX) ;
#endif
   printf ("\th - show this page and exit\n");
#ifdef DSP_SUPPORT
   printf ("\ti - interlace value (%d to %d)\n", INTERLACE_MIN, INTERLACE_MAX);
#endif
   printf ("\tl - low profile\n");
#if GPM_SUPPORT
   printf ("\tm - mouse: n-none, d-default, c-curses, g-GPM\n");
#else
   printf ("\tm - mouse support: n-no, y-yes\n");
#endif
   printf ("\to - operation: p-play, s-stop/eject, n-next, r-prev\n");
   printf ("\tp - play mode: n-normal, l-list, r-random, i-intro, s-single\n");
#ifdef DSP_SUPPORT
   printf ("\ts - number of spectrum windows (1 or 2)\n");
#endif
   printf ("\tt - time display: e-elapsed, r-remain, p-%%elapsed, m-%%remain\n");
   printf ("\tT - starting track number\n");

#ifdef DSP_SUPPORT
   printf ("\tv - wave vertical scale (%d to %d)\n", VSCALE_MIN, VSCALE_MAX );
   printf ("\tw - wave obturation value (0 to 4) \n");
#endif
 
  printf("\nSee man page for details. SADP home: www.alphalink.com.au/~michg/ace/sadp/\n");

}

/* Returns:
        0  - Bad argument
        >0 - OK
        <0 - Unrecognised operation
*/
static short ProcessCharKey(char oper, char *arg)
{   switch(oper)
    {

        case 'C':
		return ProcessDeviceName(arg, "CD", "cdrom",
						CDDeviceName,  NULL);

        case 'c':
        {   short res = ProcessInteger(arg, "Character set", &CharSet, 0, 4);
            return res;
        }

        case 'O':
            return  ProcessOptions(arg);

        case 'H':
            return  ProcessHeaderMode(arg);

         case 'l':
            QuietMode = 1;
						return 1;

        case 'm':
            return  ProcessMouseSupport(arg);

        case 'b':
            return  ProcessColours(arg);

        case 'p':
            return   ProcessPlayMode(arg);

        case 't':
            return  ProcessTimeDisplayMode(arg);

		 	 	 case 'T':
					  return ProcessInteger(optarg, "Staring track: ",
																		&track_start, 1, 99);

       case 'D':
#ifdef DEFAULT_WDB_NAME
            ShortWDBName = strdup(arg);
#endif
            return 1;  

       case 'X':
#ifdef DEFAULT_XMCD_PATH
            ShortXMCDPath = strdup(arg);
#endif
            return 1;  

#ifdef DSP_SUPPORT
        case 'A':
           return ProcessDeviceName(arg, "audio", "dsp",
				 AudioDeviceName, &AudioDeviceNumber);

        case 'S':
            return ProcessSampleSize(arg);

        case 's':
            return ProcessInteger(arg,
                         "Number of spectrum windows", &FourWndCount, 1, 2);

        case 'r':
        case 'f':
            return ProcessSamplingRate(arg);

		case 'g':		
            return   ProcessInteger(arg,
                         "Spectrum gain", &FourGain, GAIN_MIN, GAIN_MAX);


        case 'i':
            return   ProcessInteger(arg,
                         "Spectrum interlace", &FourInterlace, INTERLACE_MIN, INTERLACE_MAX);

        case 'w':
            return  ProcessInteger(arg,
                     "Wave obturation value", &WaveObturate, OBTURATE_MIN, OBTURATE_MAX);

	case 'v':		
            return   ProcessInteger(arg,
                     "Wave vertical scale", &WaveVScale, VSCALE_MIN, VSCALE_MAX);
#endif

#ifdef MIXER_SUPPORT
        case 'M':
           return ProcessDeviceName(arg, "mixer", "mixer",
				 MixerDeviceName, &MixerDeviceNumber);
#endif
    }

    return -1;
}

static short ProcessStringKey(const char *oper, char *value)
{
/*    char c; */
    const char *arg_ptr;
    short ind, len;		

    char *text_args[] = {
#ifdef RCDDB_SUPPORT
                          "useraddr", "proxyaddr", "mailserver",
#endif
#ifdef DEFAULT_WDB_NAME
			   "wdbname", 
#endif
#ifdef DEFAULT_XMCD_PATH
			   "xmcdpath", 
#endif
                           "category", "dbpath", "dbalt",
			   NULL};	

    void *text_args_addr[] = {
#ifdef RCDDB_SUPPORT
			   UserAddr, ProxyURL, MailServer,
#endif
#ifdef DEFAULT_WDB_NAME
 		 	   &ShortWDBName,
#endif
#ifdef DEFAULT_XMCD_PATH
			   &ShortXMCDPath, 
#endif
                           DefaultCategory, &ShortDBPath, &ShortDBAltPath};

    short text_args_len[] = {
#ifdef RCDDB_SUPPORT
			    151, 151, 151, 
#endif			    
#ifdef DEFAULT_WDB_NAME
 		 	    0,
#endif
#ifdef DEFAULT_XMCD_PATH
			    0, 
#endif
                            15, 0, 0};

#ifdef RCDDB_SUPPORT
    char *int_args[] = {"proxy", "protocol", NULL};
    short *int_args_addr[] = {&UseProxy, &ProtocolUsed};
    char  *int_args_names[] = {"Use Proxy", "Used Protocol"};
    short int_args_min[] = {0, 1};
    short int_args_max[] = {1, 2};
#endif

    /* Convert to LC */
    arg_ptr = oper;	
/*    while ((c=*arg_ptr) != '\0')
         *arg_ptr++ = tolower(c); */

    /* Test for text argument */	 
    for (ind=0; (arg_ptr=text_args[ind]) != NULL; ind++)
    { if(strcasecmp(arg_ptr, oper) == 0) break; }

    if (arg_ptr != NULL)
    { len = text_args_len[ind];
	if (len == 0)
	{   char **arg_pptr = (char **)text_args_addr[ind];
	    *arg_pptr = strdup(value);
	}
	else
	{   char *arg_ptr1 = (char *)(text_args_addr[ind]);
 	    strncpy(arg_ptr1, value, len);
	}

  	return 1;
    }

#ifdef RCDDB_SUPPORT		
    /* Test for integer argument */	 
    for (ind=0; (arg_ptr=int_args[ind]) != NULL; ind++)
    { if(strcmp(arg_ptr, oper) == 0) break; }

    if (arg_ptr == NULL)
    {	char err_mes[121];
		sprintf(err_mes, "Invalid argument: %s", oper);
		show_error_message(err_mes);
  		return 0;
    }

    return ProcessInteger(value, int_args_names[ind], int_args_addr[ind],
                                 int_args_min[ind], int_args_max[ind]);
#else
    return 0;
#endif
}


int  process_arguments(int argc, char **argv)
{
    static char arglist[65] = 
#ifdef DSP_SUPPORT
        "A:S:r:f:i:w:g:s:v:"
#endif
#ifdef DEFAULT_WDB_NAME
	"D:"
#endif
#ifdef DEFAULT_XMCD_PATH
	"X:"
#endif
#ifdef MIXER_SUPPORT
        "M:"
#endif
	"C:D:O:H:p:m:c:b:o:t:T:lh";

    struct option long_options[] = {
		{"category", 1, NULL, 0x101},
#ifdef DEFAULT_WDB_NAME
 	 	{"wdbname", 1, NULL, 'D'},
#endif
#ifdef DEFAULT_XMCD_PATH
 	 	{"xmcdpath", 1, NULL, 'X'},
#endif
  		{"dbpath", 1, NULL, 0x110},
  		{"dbalt", 1, NULL, 0x111} };

    int  rv, long_index;

    while ( (rv = getopt_long(argc, argv, arglist, long_options, &long_index)) != -1)
    {
       switch (rv)
       { case 'h':
            ShowCommandLineHelp();
            return 0;

         case 'o':
            return  ProcessOper(optarg);

         default:
	    if (rv < 0x100)	
                rv = ProcessCharKey(rv, optarg);
	    else
                rv = ProcessStringKey(long_options[long_index].name,
                                         optarg);
	    if (rv <= 0) return 0;
       }
    }

    FourWndCountOrig = FourWndCount;
    CharSetOrig = CharSet;
    MouseSupportOrig = MouseSupport;
    return 1;

}


static int SaveString(FILE *f, char *oper, char *arg)
{
    if (arg == NULL) return -1;
    return fprintf (f, "%s %s\n", oper, arg);
}

static int SaveDeviceName(FILE *f, char *oper, char *name)
{
    if (*name == '\0')
	 return SaveString(f, oper, "*");	

    return SaveString(f, oper, name);	
}


static int SaveInteger(FILE *f, char *oper, long val)
{
    return fprintf (f, "%s %ld\n", oper, val);
}

static int SaveOptions(FILE *f)
{    char c;
     char *outptr;
     short i;
     char buf[21];

     outptr = buf;

     for (i=0; i<OptCount; i++)	
     {    
	   c = OptInfo[i].opt_code;	
	   *outptr++ = *(OptInfo[i].var_addr) ? c : tolower(c);
     }		

     *outptr = '\0';
     return SaveString (f, "O", buf);
}


static int SaveMouseSupport(FILE *f)
{   static char arg[2] = {'\0', '\0'};
    char c;

    switch (MouseSupportOrig)
    {
#if GPM_SUPPORT
        case MOUSE_GPM:
            c = 'g'; break;
#endif

        case MOUSE_DEFAULT:
            c = 'd'; break;

        case MOUSE_NATIVE:
            c = 'c'; break;

        default:
            c = 'n';
    }
    arg[0] = c;
    return SaveString(f, "m", arg);
}


static int SavePlayMode(FILE *f)
{   const char *Modes = "NLRIS";
    static char arg[2] = {'\0', '\0'};

    arg[0] = Modes[PlayMode];
    return SaveString(f, "p", arg);
}

static int SaveTimeDisplayMode(FILE *f)
{   const char *Modes = "ERPM";
    static char arg[2] = {'\0', '\0'};

    arg[0] = Modes[TimeDisplayMode];
    return SaveString(f, "t", arg);
}
static int SaveHeaderMode(FILE *f)
{  static char arg[2] = {'\0', '\0'};

    arg[0] = OptHeader[HeaderMode];
    return SaveString(f, "H", arg);
}

static int SaveColours(FILE *f)
{   char arg[4];

    if (BkgrColour == -10)
       strcpy(arg, "m");
    else
    {  short i;
       char clr;
       for (i=0; i<3; i++)
       {  clr = *PtrColours[i];
          if (clr < 0 || clr > 7) break;
          arg[i] = clr + '0';
       }
       arg[i] = '\0';
    }
    return SaveString(f, "b", arg);
}

void save_settings(void)
{
    char *fname = get_full_fname(SADP_SETTINGS_FNAME);
    char buf[81];	
    time_t t;	  	
    FILE *f;

    if (fname == NULL) return;
    f = fopen(fname, "w");
    free(fname);
    if (f == NULL) return;

    fprintf (f, "################################################\n");
    fprintf (f, "### This resource file has been generated by ###\n");
    fprintf (f, "### Sing-Along Disc Player, text vers. %-7s###\n", 
                                    get_sadp_version());
    fprintf (f, "###------------------------------------------###\n");
    time(&t);  strcpy(buf, ctime(&t));   strtokm(buf, "\n");
    fprintf (f, "### Created:    %-29s###\n", buf );
    fprintf (f, "################################################\n");

    SaveDeviceName(f, "C", CDDeviceName);

    SaveInteger(f, "c", CharSetOrig);

#ifdef MIXER_SUPPORT
    if (MixerDeviceNumber != DEV_EXPLICIT)	
	    SaveInteger(f, "M", MixerDeviceNumber);
    else	
	    SaveDeviceName(f, "M", MixerDeviceName);
#endif

#ifdef DSP_SUPPORT
    if (AudioDeviceNumber != DEV_EXPLICIT)	
	   SaveInteger(f, "A", AudioDeviceNumber);
    else	
 	   SaveDeviceName(f, "A", AudioDeviceName);

    SaveInteger(f, "S", AudioSampleSize);
    if (FourWndCount > 0 && FourWndCount <= 2)
           FourWndCountOrig = FourWndCount;	
    SaveInteger(f, "s", FourWndCountOrig);
    SaveInteger(f, "i", FourInterlace);
    SaveInteger(f, "w", WaveObturate);
    SaveInteger(f, "f", SamplingRate);
    SaveInteger(f, "g", FourGain);
    SaveInteger(f, "v", WaveVScale);
#endif

    SaveColours(f);
    SaveOptions(f);
    SaveMouseSupport(f); 
    SavePlayMode(f);
    SaveTimeDisplayMode(f);
    SaveHeaderMode(f);

    SaveString(f, "category", DefaultCategory);

    SaveString(f, "dbpath", ShortDBPath);	
    SaveString(f, "dbalt", ShortDBAltPath);	

#ifdef DEFAULT_WDB_NAME
/*    SaveString(f, "D", ShortWDBName);  */
    SaveString(f, "wdbname", ShortWDBName);	
#endif
#ifdef DEFAULT_XMCD_PATH
    SaveString(f, "xmcdpath", ShortXMCDPath);	
#endif

#ifdef RCDDB_SUPPORT
    SaveString(f, "useraddr",  UserAddr);
    SaveString(f, "proxyaddr", ProxyURL);
    SaveString(f, "mailserver", MailServer);

    SaveInteger(f, "proxy",  UseProxy);
    SaveInteger(f, "protocol", ProtocolUsed);
#endif

    fclose(f);
    OptionsChanged = 0;
    launch_message ("Settings are saved");

}

void load_settings(void)
{
    char *fname = get_full_fname(SADP_SETTINGS_FNAME);
    char buf[121], *ptr, *oper;
    short res;
    FILE *f;


    if (fname == NULL) return;
    f = fopen(fname, "r");
    free(fname);
    if (f == NULL) return;

    while(fgets(buf, 120, f) != NULL)
    {	ptr = strtokm(buf, "#\r\n");
	ptr += strspn(ptr, " ");
	if (*ptr == '\0') continue;
	oper = ptr;
	ptr = strchr(oper, ' ');
	if (!ptr) continue;
	*ptr++ = '\0';
	trim(oper);
	trim(ptr);
	
	if (strlen(oper) == 1)
	    res = ProcessCharKey(*oper, ptr);
	else
	    res = ProcessStringKey(oper, ptr); 
	    	
        if (res <= 0) {
          sprintf(buf, "Invalid argument: %s", oper);
		  show_error_message(buf);
		}		  
     }

    OptionsChanged = 0;
    fclose(f);
}

