/*  =========================================================
    SING - ALONG DISK PLAYER.
    (C) 1998, 1999   Michael Glickman  xsadp@yahoo.com
    ---------------------------------------------------------
    NOTICE:
          Sing-Along Disk Player is copyrighted by the author.
          See COPYRIGHT regarding distribution policy and
          conditions of use.

          You are expected to provide appropriate references
          when using a part of the code in your software.

	    Author strongly advices against using this code, or
	    a part of it, in an application designed to run  on
	    any Microsoft(tm) platfrom.

    ---------------------------------------------------------
    Started 06-Nov-99
    ========================================================= */

#include <sys/types.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <limits.h>
#include <curses.h>

#include "sad.h"
#include "sadp_curses.h"
#include "sadp_cursctrls.h"

extern chtype bkgr_attr, fill_attr, text_attr, btn_attr;
extern chtype brow_attr, brow_hilite_attr, dimmed_attr;
extern chtype help_hdr_attr, bar_attr, thumb_attr;
extern chtype side_attr, side_dim_attr, side_dim2_attr, side_text_attr;

extern chtype cCkboard, cBlock, cDiamond; /* cBar, cBullet, cBullet1; */
extern chtype cDownArrow, cUpArrow, cDiamond, cBullet;

#define MAX_MESSAGE_LINES  6
extern  int ScreenW, ScreenH;
extern  MOUSE_SUPPORT_OPTION MouseSupport;
extern  MEVENT  LastMouseEvent;
extern  short	IsX, AreColours;

int  scf_mouse_x, scf_mouse_y, scf_mouse_state;
int  scf_last_key;	

/*==============================================================*/
/*                         F O R M                 17-NOV-99    */
/*==============================================================*/
static void scf_set_form_defaults(SCF_FORM *frm)
{
	frm->event_handler = NULL;
	frm->caption = NULL;
      frm->focus_obj = NULL;
	frm->cancel_obj = NULL;

	frm->bkgr_attr = bkgr_attr;
	frm->frame_attr = fill_attr;
	frm->capt_attr = side_attr;
	frm->dim_attr =  side_dim2_attr;
	frm->frame_char = cBlock;

	frm->default_obj = NULL;
	frm->cur_browser = NULL;
	frm->alt_browser = NULL;
	frm->frame_drawer = NULL;
      
      frm->flags = 0;
}

static void  scf_accept_frm_overlap(SCF_FORM *frm)
{	SCF_OBJECT *obj;
      char *text;
      
      obj = frm->overlap_obj;
      if (!obj) return;
      
      text = (frm->flags & SCF_OVERLAP) ? "Ovl" : "Ins";
      scf_set_object_caption(obj, text);

}

void  scf_set_frm_overlap_object(SCF_FORM *frm, SCF_OBJECT *obj)
{
       frm->overlap_obj = obj; 
       scf_accept_frm_overlap(frm);
}

SCF_FORM *scf_create_form(WINDOW *parent, int hei, int wid, int y, int x,
   				const  char *caption)
{
	SCF_FORM *frm;
	int p_wid, p_hei;

	frm = (SCF_FORM *)malloc(sizeof(SCF_FORM));
	if (!frm) return NULL;

	frm->parent = parent;	

	if (parent) 
 	getmaxyx(parent, p_hei, p_wid);
	else
	{   p_hei = ScreenH; p_wid = ScreenW; }
    
	if (x == CENTRAL_LOCATION) x = (p_wid - wid)/2;	
	else
	if (x == BOTTOM_LOCATION) x = p_wid - wid;	
	else
	if (x<0) x += p_wid - wid ;	    	

	if (y == CENTRAL_LOCATION) y = (p_hei - hei)/2;
	else		    	
	if (y == BOTTOM_LOCATION) y = p_hei - hei;	
	else
	if (y<0) y += p_hei;	    	

	frm->wnd = NULL;	    
	frm->first = NULL;   	
	frm->last = NULL;

	frm->wid = wid;
	frm->hei = hei;
	frm->x = x;
	frm->y=  y;

	scf_set_form_defaults(frm);
	scf_set_form_caption(frm, caption);


	return frm;
}    

static void scf_draw_form_caption(SCF_FORM *frm)
{	const char *caption;	
	WINDOW *wnd;
	int ln, wid, x, cnt;

	wnd = frm->wnd;
	if (!wnd) return;
    
	caption = frm->caption;
	if (!caption) return;

	wid = frm->wid;
	ln = strlen(caption);
	x = wid-4;

	if (ln > x) ln = x;
   
	x = (wid-ln-2) /2;
   
 /* Draw left part of frame */
	wmove (wnd, 0, 1);
	cnt = x-1;
	wattrset(wnd, frm->frame_attr);
	while (--cnt >=0) waddch(wnd, frm->frame_char);

/* Draw caption */
	wattrset(wnd, frm->capt_attr);
	waddch(wnd, ' ');
	cnt = ln;
	while (--cnt>=0) waddch(wnd, *caption++ );
	waddch(wnd, ' ');

/* Draw right of frame */
	cnt = wid-ln-x-3; 
	wattrset(wnd, frm->frame_attr);
	while (--cnt >=0) waddch(wnd, frm->frame_char);
}

/*
SCF_FORM *scf_create_form_for_window(WINDOW *wnd)
{ 	
	SCF_FORM *frm;
	int x, y, wid, hei;

	frm = (SCF_FORM *)malloc(sizeof(SCF_FORM));
	if (!frm) return frm;

	getmaxyx(wnd, hei, wid);	
	getbegyx(wnd, y, x);	

	frm->wnd = wnd;
	frm->wid = wid;
	frm->hei = hei;
	frm->x = x;
	frm->y= y;

	scf_set_form_defaults(frm);
	frm->first = NULL;   	
  
	return frm;
}                        
*/

void scf_set_form_frame_attr(SCF_FORM *frm, 
                             chtype new_frame_attr, chtype new_frame_chr)
{
	frm->frame_attr = new_frame_attr;
	frm->frame_char = new_frame_chr;
}

void scf_set_form_attr(SCF_FORM *frm, 
                             chtype new_bkgr_attr, chtype new_capt_attr)
{
	frm->bkgr_attr = new_bkgr_attr;
	frm->capt_attr = new_capt_attr;
}

void scf_draw_form_frame(SCF_FORM *frm) 
{
	WINDOW *wnd;
	chtype fchr;	

	wnd = frm->wnd;
	if (!wnd) return;

	wbkgd(wnd, frm->bkgr_attr );
	werase(wnd);

	wattrset(wnd, frm->frame_attr);

	fchr = frm->frame_char;
      if (fchr)
      	wborder (wnd, fchr, fchr, fchr, fchr,
	            	  fchr, fchr, fchr, fchr);
}


void scf_set_form_caption(SCF_FORM *frm,
                          const char *caption)
{
	if (frm->caption) free(frm->caption);
     
	if (caption)
      	frm->caption = strdup(caption);
	else
      	frm->caption = NULL;
              
	scf_draw_form_caption(frm);
     
}


void scf_destroy_form(SCF_FORM *frm)
{ 
	SCF_OBJECT *obj, *next;
		
	obj = frm->first;
	  
	while(obj)
	{	next = obj->next;
		scf_destroy_object(obj);
		obj = next;
	}

	if (frm->wnd) delwin(frm->wnd);
	if (frm->caption) free(frm->caption);
	free(frm);
   
}   


static void scf_paint_form(SCF_FORM *frm)
{	SCF_OBJECT *obj;
      SCF_FRAME_DRAWER frame_draw;
	SCF_DRAW_ROUTINE draw_routine;

      frame_draw = frm->frame_drawer;
      if (frame_draw == NULL)
         frame_draw = scf_draw_form_frame;
      (*frame_draw)(frm);

	scf_draw_form_caption(frm);

	for (obj = frm->last; obj; obj = obj->prev)
	{
	   if ((obj->flags & SCF_HIDDEN) == 0) 
	   {  draw_routine = obj->draw_routine;
		if (draw_routine)
			(*draw_routine)(obj, 1);
   	   }
	}
}

static void scf_hide_form_objects(SCF_FORM *frm)
{	SCF_OBJECT *obj;
	SCF_DRAW_ROUTINE draw_routine;

	for (obj = frm->last; obj; obj = obj->prev)
	{
	   {  draw_routine = obj->draw_routine;
		if (draw_routine)
			(*draw_routine)(obj, 0);
   	   }
	}
}

void scf_refresh_form(SCF_FORM *frm)
{
	WINDOW *wnd;
    
	wnd = frm->wnd;
	if (wnd) wrefresh(wnd);	    	
}

void scf_redraw_form(SCF_FORM *frm)
{  
	WINDOW *wnd;
    
	wnd = frm->wnd;
	if (wnd)	
	{
            scf_invalidate_frames(frm);
		scf_paint_form(frm);
		wrefresh(wnd);
	}
}

void scf_show_form(SCF_FORM *frm, int status)
{	WINDOW *wnd;
     
	wnd = frm->wnd;

	if (status == 0)	
	{ 	if (wnd)
	 	{  delwin(wnd);
		   frm->wnd = NULL;
		}
		scf_hide_form_objects(frm);
		redraw_screen();		
		return;
	}

	if (!wnd) 
	{	WINDOW *parent = frm->parent;
		if (parent == NULL)
		   wnd = newwin(frm->hei, frm->wid, frm->y, frm->x);
		else
		   wnd = derwin(parent, frm->hei, frm->wid, frm->y, frm->x);

		frm->wnd = wnd;
		scf_paint_form(frm);

	}      

	if (wnd) wrefresh(wnd);
          	        
}


void scf_set_form_window(SCF_FORM *frm, WINDOW *wnd)
{ 
     frm->wnd = wnd;
     getbegyx(wnd, frm->y, frm->x);
     getbegyx(wnd, frm->hei, frm->wid);

     frm->hei -= frm->y;
     frm->wid -= frm->x;
     
     if (wnd)
     {  scf_invalidate_frames(frm);
	  scf_paint_form(frm);
        wrefresh(wnd);
     }   
}


int  scf_is_form_visible(SCF_FORM *frm)
{
	return  frm->wnd ? 1: 0;
}

void scf_invalidate_frames(SCF_FORM *frm)
{    SCF_OBJECT *obj;
     
	for (obj=frm->first; obj; obj = obj->next)
		obj->flags &= ~SCF_FRAME_DRAWN;
}            

void scf_show_tagged_objects(SCF_FORM *frm, int tag, int status)
{	SCF_OBJECT  *obj;
	int changed;
	
	obj = frm->first;
	changed = 0;
	  
	for (obj=frm->first; obj; obj = obj->next)
	{ 
		if (obj->tag != tag) continue;

		if (status) /* show */
		{  if ((obj->flags & SCF_HIDDEN) == 0)
                                        continue;
               obj->flags &= ~SCF_HIDDEN;
		} 
		else
		{  if ((obj->flags & SCF_HIDDEN) != 0)
                                        continue;
		   obj->flags |= SCF_HIDDEN;
		}

		changed = 1;
	}

	if (changed) scf_paint_form(frm);
}

int  scf_arrange_bottom_buttons(SCF_FORM *frm,
				chtype btn_norm_attr, chtype btn_spec_attr,
				int tag, int flags,
				int btn_count, int btnw, ...)
{
	int x, y, wid, hei, i, j;
	int  gap;
	char *btn_text, c;
	int *shortcuts;	

	SCF_BUTTON *bi;
	va_list arglist;	
    
	if (!frm) return 0;

	va_start (arglist, btnw);

	wid = frm->wid;
	hei = frm->hei;
	y = hei-3;

	gap = (wid-btn_count*btnw)/(btn_count+1);  /* gap between buttons */
	x = (wid+gap-btn_count*(btnw+gap))/2;
	gap += btnw;	    

    
	for (i=0; i<btn_count; i++)
	{  
	      	btn_text = va_arg(arglist, char *); 
       
		bi = scf_create_button(frm, 1, btnw, y, x, btn_text, flags);
	      	if (!bi) return 0;	
		if (i==0) frm->default_obj = (SCF_OBJECT *)bi;
		if (i==btn_count-1) frm->cancel_obj = (SCF_OBJECT *)bi;

		(bi->obj).oper = i+1;
	      	(bi->obj).tag = tag;

		(bi->obj).norm_attr = btn_norm_attr;
      		(bi->obj).spec_attr = btn_spec_attr;

		btn_text = va_arg(arglist, char *);
		shortcuts = (bi->obj).shortcuts;	
		for (j=0; j<SCF_MAX_SHORTCUTS && (c=*btn_text++) != '\0'; j++) 	
			shortcuts[j] = (int)c;		
		if (j<SCF_MAX_SHORTCUTS) shortcuts[j] = 0;	

		x += gap;
	}	
    
	va_end (arglist);
	return 1;
}

/* Returns 1 - shortcut, 0 -regular */    
static int  scf_check_for_shortcut(SCF_OBJECT *obj, int key)
{
	int *shortcuts, scut;
	int flags, i;
	int ukey, cntrl;	
	int ctrls_subst, case_sensitive;	
      		
	flags = obj->flags;
	shortcuts = obj->shortcuts;
	ukey = toupper(key);	
	cntrl = iscntrl(key);	
	ctrls_subst = cntrl && (flags & SCF_CTRLS_SUBST);
	case_sensitive = flags & SCF_CASE_SENSITIVE;	
	
	
      for (i=0; i< SCF_MAX_SHORTCUTS; i++)
      {
		 scut = shortcuts[i];
	 	 if (scut == 0) break;
	
		 if (scut == key) return 1;
		 if (ctrls_subst && (key + 'A'-1) == toupper(scut))
				return 1;
		 if (case_sensitive == 0 && toupper(scut) == ukey)  		
				return 1;
	
      }

      return 0;	
}

void scf_accept_active_browser(SCF_FORM *frm)
{	SCF_BROWSER  *brow;
	SCF_OBJECT   *obj;

      obj = frm-> cur_browser;
      if (obj && obj->type == SCF_BROWSER_TYPE)
	{
         brow = (SCF_BROWSER *) obj;
         scf_set_object_attr (obj, fill_attr, brow_hilite_attr);
         brow->capt_attr = side_attr;
         if (frm->wnd)
         {  scf_draw_brow_frame(obj);
            scf_draw_object(obj);
         }   
      }     

      obj = frm-> alt_browser;
      if (obj && obj->type == SCF_BROWSER_TYPE)
      {
         brow = (SCF_BROWSER *) obj;
         scf_set_object_attr (obj, fill_attr, brow_attr);
         brow->capt_attr = side_text_attr;
         if (frm->wnd)
         {  scf_draw_brow_frame(obj);
            scf_draw_object(obj);
         }   
      }       
	
}
		

static int  scf_accept_event(SCF_OBJECT *obj, SCF_EVENT evnt)
{ 
	SCF_HANDLER handler;	
	SCF_CALLBACK callback;	
      int rc;

	handler = obj->event_handler;
	if (handler)
	{  rc = (*handler)(obj, evnt);
	   if (rc != 0) goto OutOfHere;
      }

	rc =  obj->oper;

	callback = obj->callback_routine;
	if (callback &&
		 (*callback)(obj, evnt, obj->oper))
				rc =  0;
OutOfHere:
	return rc;
}

int  scf_check_mouse_event(SCF_FORM *frm)
{
	SCF_OBJECT  *obj;
	SCF_HANDLER handler;
	int  wnd_left, wnd_top;
	int  obj_x, obj_y;
	int  flags, rc;

	wnd_left = frm->x;
	wnd_top = frm->y; 

  	if( !get_mouse_event() ) return 0;

	scf_mouse_x = LastMouseEvent.x - wnd_left - 1;
	scf_mouse_y = LastMouseEvent.y - wnd_top - 1;
	scf_mouse_state = LastMouseEvent.bstate;

	handler = frm->event_handler;
	if (handler)
	{  rc = (*handler)(frm, SCF_CLICK_EVENT); 
	   if (rc) return 0;
	}

	if ((scf_mouse_state & BUTTON1_CLICKED) == 0)
			return 0;

	for(obj=frm->first; obj; obj = obj->next)
	{	if ((obj->flags & SCF_HIDDEN) != 0) continue;

		obj_x = scf_mouse_x - obj->x;
		if (obj_x < 0 || obj_x >= obj->wid) continue;

		obj_y = scf_mouse_y - obj->y;
		if (obj_y < 0 || obj_y >= obj->hei) continue;

            if (obj == frm->alt_browser)
            {  frm->alt_browser = frm->cur_browser;
               frm->cur_browser = obj;
               scf_accept_active_browser(frm);
            }   

            flags = obj->flags;
            if ((obj->flags & SCF_GETS_FOCUS) &&
                !(obj->flags & SCF_HIDDEN) &&
                !(obj->flags & SCF_INACTIVE) &&
                 (frm->focus_obj != obj)
               )  
            {  SCF_OBJECT  *old_focus_obj = frm->focus_obj;
               frm->focus_obj = obj;
               scf_draw_object(obj);
               if (old_focus_obj)
                     scf_draw_object(old_focus_obj);
            }      

		rc = scf_accept_event(obj, SCF_CLICK_EVENT);
            scf_refresh_form(frm);  
		if (rc != 0) return rc;
            
            
	}
     
	return 0;
}	

SCF_OBJECT *scf_get_next_focus_obj(SCF_FORM *frm, int shift) 
{	SCF_OBJECT *strt_obj, *cur_obj;
      int flags;
      
   	cur_obj = strt_obj = frm->focus_obj;

      do
      {  if (shift)
         {	cur_obj = cur_obj->next;
            if (!cur_obj) cur_obj = frm->first;
         }    
         else
         {	cur_obj = cur_obj->prev;
            if (!cur_obj) cur_obj = frm->last;
         }    
         flags = cur_obj->flags;
         if ( !(flags & SCF_HIDDEN)  &&
              !(flags & SCF_INACTIVE) &&
               (flags & SCF_GETS_FOCUS)
            ) break;
      } while (cur_obj != strt_obj);
          
      return cur_obj;
}   

void  scf_set_focus(SCF_FORM *frm, void *obj1)
{ 	SCF_OBJECT *old_focus, *new_focus;
      
      old_focus = frm->focus_obj;
      new_focus = (SCF_OBJECT *)obj1;

      frm->focus_obj = new_focus;
      
      if (new_focus) scf_draw_object(new_focus);
      if (old_focus && old_focus != new_focus)
         		 scf_draw_object(old_focus);
              
      if (old_focus == frm->cur_browser)
      {  frm->cur_browser = frm->alt_browser;
         frm->alt_browser = old_focus;
         scf_accept_active_browser(frm);
      }   
                 
      if (new_focus == frm->alt_browser)
      {  frm->alt_browser = frm->cur_browser;
         frm->cur_browser = new_focus;
         scf_accept_active_browser(frm);
      }   

}    
 
int  scf_check_key_event(SCF_FORM *frm, chtype key)
{	SCF_OBJECT  *obj;
	SCF_HANDLER handler;
      int shift;

	int rc = 0;
	
	handler = frm->event_handler;
	if (handler)
	{  rc =  (*handler)(frm, SCF_CLICK_EVENT);
	   if (rc != 0) goto OutOfHere;
	}

      shift = get_kbd_shift() & 1;   

      obj =  frm->cur_browser;
      if (obj != NULL)
      {  SCF_OBJECT *alt = frm->alt_browser;

         if (alt && alt->type == SCF_BROWSER_TYPE && scf_last_key == KEY_F(6))
         {	frm->cur_browser = alt;
            frm->alt_browser = obj;
            scf_accept_active_browser(frm);
            rc = -1;
            goto OutOfHere;
         }

         if (alt && shift) obj = alt;   
	
         if (obj->type == SCF_BROWSER_TYPE)
            rc = scf_browser_event_handler((SCF_BROWSER *)obj, SCF_CLICK_EVENT);
/*         else
              rc = scf_input_event_handler((SCF_INPUT *)obj, SCF_CLICK_EVENT);
*/
         if (rc != 0) goto OutOfHere;

      }

      obj = frm->focus_obj;
      if (obj != NULL)
      {  
          if (scf_last_key == 9 || scf_last_key == KEY_BTAB)
              frm->focus_obj =
                 scf_get_next_focus_obj(frm, shift || (scf_last_key==KEY_BTAB)); 
          if (obj == frm->focus_obj)                 
          {   rc = scf_accept_event (obj, SCF_CLICK_EVENT);
	        if (rc != 0) goto OutOfHere;
          }
          else
             scf_set_focus(frm, obj);
	}         
         
         
	for(obj=frm->first; obj; obj = obj->next)
	{ 
		if ((obj->flags & SCF_HIDDEN) != 0) continue;
		if ((obj->flags & SCF_INACTIVE) != 0) continue;
		if (scf_check_for_shortcut(obj, key))
            {  rc = scf_accept_event (obj, SCF_CLICK_EVENT);
		   if (rc != 0) goto OutOfHere;
		}         
	}

OutOfHere:     
      scf_refresh_form(frm);  
	return rc;
}	

int  scf_process_event(SCF_FORM *frm, chtype key)
{
	if(key == ERR) return 0;

	scf_last_key = key;

	if (key == KEY_MOUSE)
      	return scf_check_mouse_event(frm);		

	if ( (key == KEY_ENTER || key == 13) && frm->default_obj)
	 	return scf_accept_event (frm->default_obj, SCF_CLICK_EVENT);

	if ( key == 27 && frm->cancel_obj)
		return scf_accept_event (frm->cancel_obj, SCF_CLICK_EVENT);

	return scf_check_key_event(frm, key);
}      

int  scf_check_event(SCF_FORM *frm)
{	chtype key;
	
#if GPM_SUPPORT
	if (MouseSupport == MOUSE_GPM)
       	key = gpm_getch();
	else
#endif
	key = getch();
	
      return scf_process_event(frm, key);
}

int  scf_get_event(SCF_FORM *frm, int (*idle_func)(SCF_FORM *))
{	int rc;
     
	scf_show_form(frm, 1);

	while((rc=scf_check_event(frm)) <= 0) 
	{
		if (idle_func)
		{ rc = (*idle_func)(frm);
	        if (rc > 0) break;
      	}
		sleep(0);
	}

	return rc;
}
	

/*==============================================================*/
/*                       O B J E C T               17-NOV-99    */
/*==============================================================*/
/* Generic draw routine */
void scf_common_draw_routine(SCF_OBJECT *obj, int action)
{
	WINDOW *wnd;
	int count;
	const char *caption;
	int len, wid, offs;
	int highlight;
	char c;


	wnd = obj->frm->wnd;
	if (!wnd) return;

	if (action == 0) 
		caption = "";
	else
	{    caption = obj->caption;
		if (!caption) return;
	}
        
	wmove(wnd, obj->y, obj->x);

	wid = obj->wid;
	len = strlen(caption);
	if (len > wid) len = wid;

	switch(obj->align)
	{  	case SCF_LEFT_ALIGN:
			offs = 0;
			break;
		case SCF_RIGHT_ALIGN:
			offs = wid-len;
			break;
		default:
			offs = (wid-len)/2;
	}
	 
	wattrset(wnd, obj->norm_attr);
		
	count = offs;
	while (--count>=0) waddch(wnd, ' ');

	count = len;
	highlight = 0;
	while (--count>=0)
	{	c = *caption++;
		if (highlight == 0)
            {  highlight = scf_check_for_shortcut(obj, (int) c);
               if (highlight == 0 && isalpha(c))
                  highlight = scf_check_for_shortcut(obj, toupper(c)-'A'+1); 

               if (highlight) 
               {  wattrset(wnd, obj->spec_attr);
		      waddch(wnd, c);	 	
		      wattrset(wnd, obj->norm_attr);
                  continue;
               }
		}
   	     
            waddch(wnd, c);	 	
	}					

	count = wid - len - offs;
	while (--count>=0)
		waddch(wnd, ' ');
	return;

}                

void scf_draw_object(void *obj1)
{
	SCF_OBJECT *obj = (SCF_OBJECT *)obj1;
	SCF_DRAW_ROUTINE paint_routine;

	if ((obj->flags & SCF_HIDDEN) == 0)
	{  paint_routine = obj->draw_routine;
	   if (paint_routine) (*paint_routine) (obj, 1);
	}
}

void scf_set_object_caption(void *obj1, const char *caption)
{   
	SCF_OBJECT *obj;

	obj = (SCF_OBJECT *) obj1;
	if (obj->caption) free(obj->caption);
      if (caption)  obj->caption = strdup(caption);
      		 else obj->caption = NULL;

	scf_draw_object(obj1);
}	  	

void scf_set_object_align(void *obj1, SCF_ALIGN align)
{
	SCF_OBJECT *obj;


	obj = (SCF_OBJECT *) obj1;
	obj->align = align;	
}

void  scf_set_object_attr(void *obj1,
		chtype new_norm_attr, chtype new_spec_attr)
{  
	SCF_OBJECT *obj = (SCF_OBJECT *)obj1;

	if (new_norm_attr >=0)
		obj->norm_attr = new_norm_attr;

	if (new_spec_attr >=0)
		obj->spec_attr = new_spec_attr;

}

void  scf_set_object_callback(void *obj1,
		SCF_CALLBACK new_callback_routine, int new_oper)
{  
	SCF_OBJECT *obj = (SCF_OBJECT *)obj1;

	obj->callback_routine = new_callback_routine;
	obj->oper = new_oper;
}

void  scf_set_object_shortcuts(void *obj1, ...)
{
	SCF_OBJECT *obj;
	va_list  arglist;
	int  shortcut;
	int  i;

	va_start (arglist, obj1);
	obj = (SCF_OBJECT *) obj1;

	for (i=0; i<SCF_MAX_SHORTCUTS; i++)
	{    
		shortcut = va_arg(arglist, int); 
		obj->shortcuts[i] = shortcut;
		if (shortcut == 0) break;
	}	

	va_end(arglist);
}	

int  scf_add_object_shortcut(void *obj1, int new_shortcut)
{
	SCF_OBJECT *obj;
	int  i;

	obj = (SCF_OBJECT *)obj1;

	for (i=0; i<SCF_MAX_SHORTCUTS; i++)
	{    
		if (obj->shortcuts[i] == 0)
		{  obj->shortcuts[i] = new_shortcut;
	      	return 1;
	      }	
	}	

	return 0;
}	

void  scf_set_object_defaults(SCF_OBJECT *obj)
{
	obj->tag = 0;     	
	obj->oper = 0;	
	obj->flags = 0;	
	obj->norm_attr = text_attr;
	obj->spec_attr = help_hdr_attr;
	obj->shortcuts[0] = '\0';
	obj->event_handler = NULL;
	obj->callback_routine = NULL;
	obj->draw_routine = scf_common_draw_routine;
	obj->destroy_routine = NULL;
}

/* Generic create routine */
SCF_OBJECT *scf_create_object(SCF_FORM *frm, 
			int obj_size, int obj_type,
			int hei, int wid, int y, int x, 
			const char *caption)
{
	SCF_OBJECT  *obj, *next; 	

	obj = malloc(obj_size);
	if (!obj) return NULL;

	obj->frm = frm;
	next = frm->first;
 	frm->first = obj;

	obj->next = next;
	obj->prev = NULL;

	if (next) next->prev = obj;
	     else frm->last = obj;

	if (wid < 0) wid = strlen(caption);        
        
	if (x == CENTRAL_LOCATION)
		 x = (frm->wid - wid) >> 1;
	else 
 	if (x < 0) x += frm->wid;

	if (y == CENTRAL_LOCATION)
		 y = (frm->hei - hei) >> 1;
	else
	if (y < 0) y += frm->hei;

	obj->y = y;
	obj->x = x;
	obj->hei = hei; 	     	
	obj->wid = wid; 	
	obj->type = obj_type;     	
	scf_set_object_defaults(obj);
	obj->caption = NULL;

	scf_set_object_caption(obj, caption);

	return obj;
}


void scf_destroy_object(void *obj1)
{
	SCF_OBJECT *obj = (SCF_OBJECT *)obj1;
	SCF_ROUTINE kill_routine = obj->destroy_routine;	
	if (kill_routine) (*kill_routine)(obj);	

	if (obj->caption) free(obj->caption);	
	free(obj);
}

void scf_show_object(void *obj1, int action)
{ 
	SCF_OBJECT *obj = (SCF_OBJECT *)obj1;
	SCF_DRAW_ROUTINE draw_routine;

      if (action)  /* show */
      { 	if ((obj->flags & SCF_HIDDEN) == 0) 
			return;
		obj->flags &= ~SCF_HIDDEN;
      }
      else    /* hide */
	{	if ((obj->flags & SCF_HIDDEN) != 0)
			return;
		obj->flags |= SCF_HIDDEN;
	}	  		
	
	draw_routine = obj->draw_routine;
	if (draw_routine) (*draw_routine)(obj, action);
/*	scf_paint_form(obj->frm); */	
}


/*
void scf_draw_object_frame(SCF_OBJECT *obj) 
{   
	SCF_FORM  *frm;
	WINDOW    *wnd;	
	int  wid, hei, x, y;	
	chtype fchr;
    
	frm = obj->frm;	
	fchr = frm->frame_char;

	wnd = frm->wnd;	
	x = obj->x; y = obj->y;
	wid = obj->wid; hei = obj->hei;
     
	wattrset(wnd, frm->frame_attr);
	
	wmove(wnd, y, x);	     
	whline (wnd, fchr, wid);

	wmove(wnd, y+hei-1, x);
	whline (wnd, fchr, wid);
	     
	wmove(wnd, y+1, x);	     
	wvline (wnd, fchr, hei-2);
    
	wmove(wnd, y+1, x+wid-1);
	wvline (wnd, fchr, hei-2);
}
*/
/*==============================================================*/
/*                       T E X T                   17-NOV-99    */
/*==============================================================*/
SCF_TEXT *scf_create_text(SCF_FORM *frm,
                      int hei, int wid, int y, int x,
                      const char *caption)
{

	return  (SCF_TEXT *) scf_create_object(frm,
      		sizeof(SCF_TEXT), SCF_TEXT_TYPE,
			hei, wid, y, x, caption);
}

/*==============================================================*/
/*                     B U T T O N                 17-NOV-99    */
/*==============================================================*/
SCF_BUTTON *scf_create_button(SCF_FORM *frm,
			int hei, int wid, int y, int x,
			const char *caption, int flags)
{	
	SCF_OBJECT *obj;		

	obj = scf_create_object(frm,
			sizeof(SCF_BUTTON), SCF_BUTTON_TYPE,
			hei, wid, y, x, caption);
	if (!obj) return NULL;

	obj->flags = flags;
	obj->align = SCF_CENTRE_ALIGN;
	scf_set_object_attr(obj, btn_attr, thumb_attr);

	
	return (SCF_BUTTON *) obj;

}
 
/*==============================================================*/
/*                        T I C K                  20-NOV-99    */
/*==============================================================*/
void scf_draw_tick_routine(SCF_TICK *tick)
{
	SCF_OBJECT *obj;  
	WINDOW *wnd;
	chtype norm_attr;
	SCF_ALIGN align;
	int count, value;
	const char *caption;
	int len, wid, offs;
	int highlight;
	char c;

	obj = (SCF_OBJECT *)tick;

	wnd = obj->frm->wnd;
	if (!wnd) return;

 	caption = obj->caption;
	if (!caption) return;
        
	wmove(wnd, obj->y, obj->x);

	wid = obj->wid;
	len = strlen(caption);
	if (len > wid-3) len = wid-3;

	align = obj->align;

	switch(align)
	{  	case SCF_LEFT_ALIGN:
			offs = 0;
			break;
		case SCF_RIGHT_ALIGN:
			offs = wid-len-3;
			break;
		default:
			offs = (wid-len-3)/2;
	}

	value = tick->value;	 
	norm_attr = value ? tick->on_attr : obj->norm_attr;
	wattrset(wnd, norm_attr);
		
	count = offs;
	while (--count>=0) waddch(wnd, ' ');

	if (align != SCF_RIGHT_ALIGN)
	{  waddch(wnd, ' ');	
           waddch(wnd, value ? cDiamond : ' ');
	   waddch(wnd, ' ');	
	}

	count = len;
	highlight = 0;
	while (--count>=0)
	{	c = *caption++;
		if (highlight == 0)
            {  highlight = scf_check_for_shortcut(obj, (int) c);
               if (highlight == 0 && isalpha(c))
                  highlight = scf_check_for_shortcut(obj, toupper(c)-'A'+1); 

               if (highlight) 
               {  wattrset(wnd, obj->spec_attr);
		      waddch(wnd, c);	 	
		      wattrset(wnd, norm_attr);
                  continue;
               }
		}
   	     
            waddch(wnd, c);	 	
	}					

	if (align == SCF_RIGHT_ALIGN)
	{  waddch(wnd, ' ');	
	   waddch(wnd, value ? cDiamond : ' ');
	   waddch(wnd, ' ');	
	}

	count = wid - len - offs - 3;
	while (--count>=0)
		waddch(wnd, ' ');
                

}

int  scf_tick_event_hander(void *tick1, SCF_EVENT evnt)
{
	if (evnt == SCF_CLICK_EVENT)
	{   SCF_TICK *tick = (SCF_TICK *)tick1;
	    tick->value ^= 1;
	    scf_draw_tick_routine(tick);
	}			

	return 0;
}


SCF_TICK *scf_create_tick(SCF_FORM *frm, 
                      int hei, int wid, int y, int x,
                      const char *caption, int flags, int value)
{	SCF_OBJECT *obj;		
	SCF_TICK *tick;		

	obj = scf_create_object(frm,
			sizeof(SCF_TICK), SCF_TICK_TYPE,
			hei, wid, y, x, caption);
	if (!obj) return NULL;

	obj->flags = flags;
	scf_set_object_attr(obj, side_dim2_attr, thumb_attr);
	obj->draw_routine = (SCF_DRAW_ROUTINE) scf_draw_tick_routine;
	obj->event_handler = scf_tick_event_hander;

	tick = (SCF_TICK *) obj;
	tick->on_attr = side_attr;
	tick->value = value;


	return tick;
}

void scf_set_tick(SCF_TICK *tick, int new_value)
{	tick->value = new_value;

	scf_draw_tick_routine(tick);
}

int scf_get_tick_value(SCF_TICK *tick)
{
	return tick->value;
}
/*==============================================================*/
/*                         B R O W S E R S                      */
/*--------------------------------------------------------------*/
/* This type of browser will also store data.     15-NOV-99     */
/*==============================================================*/
static int scf_browser_mouse_handler(void *brow1)
{	int y, hei;
	SCF_OBJECT *obj;
	SCF_BROWSER *brow;

	obj = (SCF_OBJECT *)brow1;
	brow = (SCF_BROWSER *)brow1;

      if ((scf_mouse_state & BUTTON1_CLICKED) == 0) return 0;
	y = scf_mouse_y - obj->y;
	hei = obj->hei;

	if (y < 0 || y >= hei) return 0;

      if (y == 0 || y == hei-1)
      {	/* Scrolling */
		int x = scf_mouse_x - obj->x;
		int current;
		if (x != 2 && x != obj->wid-3) return 0;

		current = brow->current;
		if (y==0) scf_select_browser_line(brow, current-1);
		     else scf_select_browser_line(brow, current+1);
	
		scf_draw_browser_routine(brow, 1);
	} 

	else
	{	/* Changing current line */
	      scf_select_browser_line(brow, brow->top+y-1);
	}

	return  -1;
}

static int  scf_browser_key_handler(void *brow1)
{
	SCF_OBJECT *obj;
	SCF_BROWSER *brow;
	int current; 

	obj = (SCF_OBJECT *)brow1;
	brow = (SCF_BROWSER *)brow1;

	current = brow->current;

	switch(scf_last_key)
	{ case KEY_DOWN:
		if ( current < brow->nlines)
			scf_select_browser_line(brow, current+1);
		return -1;

	  case KEY_UP:
		if ( current > 0)
			scf_select_browser_line(brow, current-1);
		return -1;

	  case KEY_NPAGE:
	  {     int new_current = current + brow->eff_hei;
		if (new_current > brow->nlines)
			     new_current = brow->nlines;
              	if ( current != new_current)
		{   int new_top;
		    new_top = brow->top + brow->eff_hei;
		    if (new_top + brow->eff_hei > brow->nlines)
		    { new_top = brow->nlines - brow->eff_hei;
		      if (new_top < 0) new_top = 0;
		    }
		    brow->top = new_top;	

		    scf_select_browser_line(brow, new_current);
		}		
			
	  }
		return -1;

	  case KEY_PPAGE:
	  {     int new_current = current - brow->eff_hei;
		if (new_current < 0) new_current = 0;
              	if ( current != new_current)
			scf_select_browser_line(brow, new_current);
	  }
		return -1;

	  case KEY_HOME:
		scf_select_browser_line(brow, 0);
		return -1;

	  case KEY_END:
		scf_select_browser_line(brow, brow->nlines-1);
		return -1;


	}

	return 0;	

}

int scf_browser_event_handler(void *brow1, SCF_EVENT evnt)
{
	if (evnt != SCF_CLICK_EVENT) return 0;

      if (scf_last_key == KEY_MOUSE)
		return scf_browser_mouse_handler(brow1); 

	return scf_browser_key_handler(brow1);
}


static void scf_clear_browser_internal(SCF_BROWSER *brow)
{	SCF_BROWSER_LINE *cur, *next;
	char *text;	

	cur = brow->first;
	while (cur)
      {	next = cur->next;
		text = cur->text;
		if (text) free(text);
		free(cur);
		cur = next;
	}
      
	brow->first = NULL;	
	brow->current = -1;
	brow->top = 0;
	brow->nlines = 0;
}	      	
	

SCF_BROWSER *scf_create_browser(SCF_FORM *frm, 
			int hei, int wid, int y, int x,
			const char *caption)
{	SCF_BROWSER *brow;
	SCF_OBJECT  *obj;	

	obj = scf_create_object(frm,
      	sizeof(SCF_BROWSER), SCF_BROWSER_TYPE,
		hei, wid, y, x, caption);
	if (!obj) return NULL;
	obj->norm_attr = brow_attr;	
	obj->spec_attr = brow_hilite_attr;	

	obj->draw_routine = (SCF_DRAW_ROUTINE) scf_draw_browser_routine;
	obj->destroy_routine = (SCF_ROUTINE) scf_destroy_browser;
	obj->event_handler = scf_browser_event_handler;

	brow = (SCF_BROWSER *)obj;
	brow->first = NULL;
	brow->capt_attr = side_attr;
	scf_clear_browser_internal(brow);

	brow->eff_wid = wid-2;
	brow->eff_hei = hei-2;

	return brow;	
}      


/*  Line number starts from 0
    number < 0 - take line from last  */
SCF_BROWSER_LINE *scf_get_browser_line(SCF_BROWSER *brow, int number)
{	SCF_BROWSER_LINE *line;
                      
	line = brow->first;
	if (number < 0) number += (brow->nlines);	
	while (line && --number >= 0) line = line->next;

	return line;
}

const char *scf_get_browser_text(SCF_BROWSER *brow, int number)
{	SCF_BROWSER_LINE *bline;

	bline = scf_get_browser_line(brow, number);
	if (!bline) return NULL;
	return bline->text;
}


static void scf_draw_browser_text(SCF_BROWSER *brow, int number, const char *text)
{ 
	char c;
	int wid;
	WINDOW *wnd;
	SCF_OBJECT *obj;	
	
	obj = (SCF_OBJECT *)brow;	
	wnd = obj->frm->wnd;

	if (!wnd) return;
	
	wattrset(wnd,
		(number+brow->top==brow->current) ? obj->spec_attr : obj->norm_attr);
	wmove(wnd, obj->y+1 + number, obj->x+2);
	wid = obj->wid-4;
      
	while ((c=*text++)!='\0' && --wid>=0)
           		waddch(wnd, c);

	while (--wid>=0) waddch(wnd, ' ');	
}

void scf_set_browser_text(SCF_BROWSER *brow, int number, const char *text)
{	SCF_BROWSER_LINE *bline;
	char *old_text;

	bline = scf_get_browser_line(brow, number);
	if (!bline) return;

	old_text = bline->text;
	if (old_text) free(old_text);
	
	bline->text = strdup(text);
	number -= brow->top;
	if (number >= 0 && number <= brow->eff_hei)
	   scf_draw_browser_text(brow, number, text);
	
}

void scf_draw_brow_frame(void *obj1)
{

	WINDOW *wnd;
	SCF_OBJECT *obj;
	char *caption;
	int  len, pos, wid;

	obj = (SCF_OBJECT *)obj1;
	wnd = obj->frm->wnd;
	if (!wnd) return;

	wid = obj->wid;
	caption = obj->caption;
	if (!caption) return;
	len = strlen(caption);
	pos = wid-8;
	if (len > pos) len = pos;
	pos = (wid-len)/2;

	wmove(wnd, obj->y, obj->x+pos);
	wattrset(wnd, ((SCF_BROWSER *)obj1)->capt_attr);
	while (--len >= 0) waddch(wnd, *caption++);	
}

static void scf_decorate_object(SCF_OBJECT *obj)
{	
	SCF_FORM    *frm;
	WINDOW  *wnd;
	int x, y, wid, hei;
	chtype fchr;

	frm = obj->frm;
	wnd = frm->wnd;
	if (!wnd) return;

	x = obj->x;
	y = obj->y;
	wid = obj->wid;
	hei = obj->hei;		
	fchr = frm->frame_char;

	/* Top border */
	wattrset(wnd, frm->frame_attr);
	wmove(wnd, y, x);
	whline(wnd, fchr, wid);

	/* Bottom border */
	wmove(wnd, y+hei-1, x);
	whline(wnd, fchr, wid);

	/* Left border */
	wmove(wnd, y+1, x);
	wvline(wnd, fchr, hei-2);

	/* Right border */
	wmove(wnd, y+1, x+wid-1);
	wvline(wnd, fchr, hei-2);

	/* Caption */
	scf_draw_brow_frame(obj);

	/* Arrows */
	wattrset(wnd, frm->frame_attr);
	wmove(wnd, y, x+2);
	waddch(wnd, cUpArrow);

	wmove(wnd, y, x+wid-3);
	waddch(wnd, cUpArrow);

	y += hei-1;
	wmove(wnd, x, y);

	wmove(wnd, y, x+2);
	waddch(wnd, cDownArrow);

	wmove(wnd, y, x+wid-3);
	waddch(wnd, cDownArrow);

	obj->flags |= SCF_FRAME_DRAWN;
} 

void scf_draw_browser_routine(SCF_BROWSER *brow, int action) 
{	
	SCF_BROWSER_LINE *line;
	SCF_OBJECT *obj;
	int i, hei;   

	obj = (SCF_OBJECT *)brow;
	if (action == 0)
	{  obj->flags &= ~SCF_FRAME_DRAWN;
	   return;
	}

	if ((obj->flags & SCF_FRAME_DRAWN) == 0)
  	    scf_decorate_object(obj);	
              
	hei = brow->eff_hei;
	line = scf_get_browser_line(brow, brow->top);
      
	for (i=0; i<hei && line; i++)
	{	scf_draw_browser_text(brow, i, line->text);
		line = line->next;
	}   
       
	for (; i<hei; i++)
		scf_draw_browser_text(brow, i, "");
}

/* Line number < 0 - append */
SCF_BROWSER_LINE *scf_insert_browser_line(SCF_BROWSER *brow,
                     int line_number, char *text)
{ 
	SCF_BROWSER_LINE *new_line, *prev_line, *next_line;
	int nlines;	
	char *new_text;
      
	new_line = malloc(sizeof(SCF_BROWSER_LINE));
	new_text = strdup(text);
	if (!new_line || !new_text) return NULL;
	new_line-> text = new_text;
      
	prev_line = NULL;
	next_line = brow->first;

	nlines = brow->nlines;			
	if (line_number > nlines) line_number = nlines;	
	if (line_number < 0)  line_number += (nlines+1);

	while (line_number >= 0 && next_line)	
      {	prev_line = next_line;
		next_line = next_line->next;
		line_number--;  
      }
         
	new_line-> next = next_line;
      
	if (prev_line)
      	prev_line->next = new_line;
	else
		brow->first = new_line;
         
	brow->nlines = nlines+1;   
      return new_line;   
}      

SCF_BROWSER_LINE *scf_append_browser_line(SCF_BROWSER *brow, char *text)
{
	return  scf_insert_browser_line(brow, INT_MAX, text);
}
     
void   scf_make_browser_line_visible(SCF_BROWSER *brow, int number)
{	int tmp;

	tmp = brow->top;      /* First visible */
	if (number < tmp)
	{  brow->top = number;
	   return;
	}
	
	tmp += brow->eff_hei - 1;	/* Last visible */
	if (number > tmp)
	   brow->top += (number-tmp);

}

void   scf_delete_browser_line(SCF_BROWSER *brow, int number)
{	SCF_BROWSER_LINE *prev, *cur;

	/* Find line to be deleted */
	prev = NULL; cur = brow->first;
	while (cur && number)
	{	--number;
		prev = cur;
		cur = cur->next;
	}
	if (!cur) return;

	/* Shortcut link */
	if (prev) prev->next = cur->next;
	    else  brow->first = cur->next;

	/* Delete me */		
	if (cur->text) free(cur->text);
	free(cur);

	/* Adjust browser settings */
	number = brow->nlines - 1;
	brow->nlines = number;
	if (brow->current >= number)
	{	brow->current = --number;
		scf_make_browser_line_visible(brow, number);
	}

	/* Redraw */
	scf_draw_browser_routine(brow, 1);

}

void   scf_select_browser_line(SCF_BROWSER *brow, int number)
{  
	if (number < 0 || number >= brow->nlines) return;
	brow->current = number; 
	scf_make_browser_line_visible(brow, number);
	scf_draw_browser_routine(brow, 1);
}      


void scf_destroy_browser(SCF_BROWSER *brow)
{	
	scf_clear_browser_internal(brow);
}

void scf_clear_browser(SCF_BROWSER *brow)
{
	scf_clear_browser_internal(brow);
	scf_draw_browser_routine(brow, 1);
}

int  scf_get_browser_linecount(SCF_BROWSER *brow)
{
	return brow->nlines;
}

int  scf_get_browser_curline(SCF_BROWSER *brow)
{
	return brow->current;
}

/*==============================================================*/
/*                           I N P U T                          */
/*==============================================================*/
static char *scf_get_input_line_text(SCF_INPUT *inp, int lineno)
{
    if (lineno >= inp->line_count) return NULL;
    return  inp->content + lineno * (inp->col_count+1);
}                     

void scf_draw_input_line(SCF_INPUT *inp, int lineno)
{	SCF_OBJECT *obj;
	WINDOW *wnd;
	char *line_text;
	chtype t, nattr;
	int rowno, linelen, spacelen;
      int  x, y;

      if (lineno >= inp->line_count) return;
      rowno = lineno - inp->top_line;
	if (rowno < 0 || rowno > inp->eff_hei) return;
	
	obj = (SCF_OBJECT *)inp;
	wnd = obj->frm->wnd;
	if (wnd == NULL) return;

      line_text = scf_get_input_line_text(inp, lineno) +
                  inp->left_col;

	x = obj->x; y = obj->y + rowno;
	if (obj->caption != NULL) { x++; y++; }
	wmove(wnd, y, x);

	linelen = strlen(line_text);
	spacelen = inp->eff_wid - linelen;

	if (spacelen < 0) 
	{  linelen += spacelen; spacelen = 0; }
	
      if (obj  == obj->frm->focus_obj) 
      {  nattr = obj->norm_attr;

         if(lineno == inp->cur_line)
         {  wattrset(wnd, nattr);
	      x = inp->cur_pos;
            if (x > 0)
            {  waddnstr(wnd, line_text, x);
	         line_text += x;
            }
            linelen -= (x+1);	
            t = (chtype) *line_text++;
            if (t=='\0') { t = cDiamond; spacelen--; }
            else
            if (t==' ') t = cDiamond;
            
            wattrset(wnd, obj->spec_attr);
            waddch(wnd, t);
         }
      }   
      else
         nattr = (obj->flags & SCF_INACTIVE) ? obj->frm->dim_attr 
                                             : obj->norm_attr ;
      
      wattrset(wnd, nattr);
      if (linelen > 0)
   	   waddnstr(wnd, line_text, linelen);
       
      if (spacelen > 0)
	   whline(wnd, ' ', spacelen);

}

/* N.B. abs(step) should never exceed 1! 
static int scf_input_step(SCF_INPUT *inp, int step)
{	int cur_line, cur_pos;
      
      cur_line = inp->cur_line;
      cur_pos = inp->cur_pos + step;

      if (cur_pos < 0 && cur_line > 0)
      {   cur_line--;	
          cur_pos = strlen(scf_get_input_line_text(inp, cur_line));
      }    
      else
      if ( ( cur_pos >= inp->col_count || 
             cur_pos > strlen(scf_get_input_line_text(inp, cur_line)
           ) &&
           cur_line < inp->line_count-1
         )
      {  cur_line++;
         cur_pos = 0;
      }    
          
	scf_accept_input_pos(inp, cur_line, cur_pos);

}
*/

static int scf_accept_input_pos(SCF_INPUT *inp, int y, int x)
{    int   temp, temp1;
     short redraw;
   
     if (y < 0 || y >= inp->line_count) return 0;
     if (x < 0 || x >= inp->col_count) return 0;

     redraw = 0;

     /* Make line visible */
     temp = inp->top_line;
     if (y < temp)
     {  inp->top_line = y; redraw = 1; }
     temp1 = inp->eff_hei;
     if (y >= temp+temp1)
     { inp->top_line = y-temp1+1; redraw = 1; }

     /* Make column visible */
     temp = inp->left_col;
     if (x < temp)
     {  inp->left_col = x; redraw = 1; }
     temp1 = inp->eff_wid;
     if (x >= temp+temp1)
     { inp->left_col = x-temp1+1; redraw = 1; }
     
     /* Check column number */
     temp =strlen(scf_get_input_line_text(inp, y));
     if (x > temp) x = temp;

     /* Change position */
     temp = inp->cur_line;
     inp->cur_line = y;
     inp->cur_pos = x;
     
     /* Draw */
     if (redraw)
        scf_draw_object(inp);
     else
     {  scf_draw_input_line(inp, y);
        if (temp != y)
            scf_draw_input_line(inp, temp);
     }
     
     return 1;        	
     
}

static void scf_overlap_input_char(SCF_INPUT *inp, chtype c1)
{	char c;
      int cur_line, cur_pos;
      int linecount, colcount; 
      char *text;
      
      c = (char)c1 & 0xff;
      cur_line = inp->cur_line;
      cur_pos = inp->cur_pos;

      linecount = inp->line_count;
      colcount = inp->col_count; 
     
      text = scf_get_input_line_text(inp, cur_line);
      text[cur_pos] = c1;

      if (cur_pos < strlen(scf_get_input_line_text(inp, cur_line)) && 
                cur_pos < inp->col_count-1) 
     	     scf_accept_input_pos(inp, cur_line, cur_pos+1);
      else        
     	     scf_accept_input_pos(inp, cur_line+1, 0);
           
}

static void scf_insert_input_char(SCF_INPUT *inp, chtype c1)
{	char c, carry;
      int lineno, posno, i, l;
      int cur_line, cur_pos;
      int linecount, colcount;
      char *text;
      
      c = (char)c1 & 0xff;
      lineno = cur_line = inp->cur_line;
      posno = cur_pos = inp->cur_pos;

      linecount = inp->line_count;
      colcount = inp->col_count;
     
      text = scf_get_input_line_text(inp, lineno);
      l =  i = strlen(text);
      
      while (lineno < linecount)
      {  if (i >= colcount)
            carry = text[colcount-1];
         else   
         {  text[++i] = '\0'; carry = '\0'; }
            
         while (--i > posno) text[i] = text[i-1];   
         text[i] = c;

         if (lineno > cur_line)
            scf_draw_input_line(inp, lineno);
         
         if (carry == '\0') break;
         
         c = carry; lineno++; posno = 0;
         text = scf_get_input_line_text(inp, lineno);
             
         i = strlen(text);
      }  
            
      if (cur_pos >= colcount-1)  /*++l) */
      { /*  scf_draw_input_line(inp, cur_line); */
          cur_pos = 0; cur_line++;
      }    
      else cur_pos++;
      
      scf_accept_input_pos(inp, cur_line, cur_pos);   

}

static void delete_input_char(SCF_INPUT *inp)
{
      int cur_line, cur_pos;
      char *text;

      cur_line = inp->cur_line;
      cur_pos = inp->cur_pos;

      text = scf_get_input_line_text(inp, cur_line);
      if (text[cur_pos] == '\0')
      {  char *text1;
         int  next_line;
         int  l, l1, lmove;
         
         next_line = cur_line+1;
         if (next_line >= inp->line_count) return;

         text1 = scf_get_input_line_text(inp, next_line);

         l = strlen(text);
         l1 = strlen(text1);
         lmove = inp->col_count - l;
         if (lmove > l1) lmove = l1;
         
         /* Concatenate lines */
         memcpy(text+l, text1, lmove);
         
         /* Is line split ? */
         l1 -= lmove;
         if (l1 > 0)
         {  memcpy(text1, text1+lmove, l1);
            memset(text1+l1, '\0', lmove);
         }
         /* Not move rest lines up */
         else
         {  l = inp->col_count+1;
            l1 = (inp->line_count - next_line-1) * l;
            memcpy(text1, text1+l, l1);
            memset(text1+l1, '\0', l); 
         }
         
         scf_draw_object(inp);   
      }    
      else
      {  strcpy(text+cur_pos, text+cur_pos+1);
         scf_draw_input_line(inp, cur_line);
      }   
}

int scf_input_key_handler(void *inp1)
{
	SCF_OBJECT *obj;
	SCF_INPUT *inp;
      int cur_line, cur_pos;

	obj = (SCF_OBJECT *)inp1;
	inp = (SCF_INPUT *)inp1;

      cur_line = inp->cur_line;
      cur_pos = inp->cur_pos;

	switch(scf_last_key)
	{
        case KEY_LEFT:
            if (cur_pos > 0)
         	     scf_accept_input_pos(inp, cur_line, cur_pos-1);
            else
         	     scf_accept_input_pos(inp, cur_line-1, inp->col_count-1);
		return -1;
      
        case KEY_RIGHT:
            if (cur_pos < strlen(scf_get_input_line_text(inp, cur_line)) && 
                cur_pos < inp->col_count-1) 
         	     scf_accept_input_pos(inp, cur_line, cur_pos+1);
            else        
         	     scf_accept_input_pos(inp, cur_line+1, 0);
		return -1;
      
        case KEY_DOWN:
         	scf_accept_input_pos(inp, cur_line+1, cur_pos);
		return -1;

	  case KEY_UP:
         	scf_accept_input_pos(inp, cur_line-1, cur_pos);
		return -1;

	  case KEY_NPAGE:
         	scf_accept_input_pos(inp, cur_line + inp->eff_hei, cur_pos);
		return -1;

	  case KEY_PPAGE:
         	scf_accept_input_pos(inp, cur_line - inp->eff_hei, cur_pos);
		return -1;

	  case KEY_HOME:
         	scf_accept_input_pos(inp, cur_line, 0);
		return -1;

	  case KEY_END:
         	scf_accept_input_pos(inp, cur_line, inp->col_count-1);
		return -1;

        case KEY_ENTER:
        case 13:
         	scf_accept_input_pos(inp, cur_line+1, 0);
		return -1;

        case KEY_IC:    
        {	SCF_FORM *frm = obj->frm;
            frm->flags ^= SCF_OVERLAP;
            scf_accept_frm_overlap(frm);
        }     
		return -1;
            
        case KEY_DC:
            delete_input_char(inp);
		return -1;
            
        case KEY_BACKSPACE:
        case 8:
            if (cur_pos > 0)
         	     scf_accept_input_pos(inp, cur_line, cur_pos-1);
            else
         	     scf_accept_input_pos(inp, cur_line-1, inp->col_count-1);
            delete_input_char(inp);
            return -1;
	}

      if (scf_last_key >= ' ' && scf_last_key < 0x100)
      {  if (obj->frm->flags & SCF_OVERLAP)
             scf_overlap_input_char(inp, scf_last_key);
         else
             scf_insert_input_char(inp, scf_last_key);
         return -1;
      }            

	return 0;	

}

int scf_input_mouse_handler(void *inp1)
{        
      int x, y;
      SCF_OBJECT *obj;
      SCF_INPUT *inp;
      
     
      if ((scf_mouse_state & BUTTON1_CLICKED) == 0) return 0;

      obj = (SCF_OBJECT *)inp1;
      inp= (SCF_INPUT *)inp1;

	y = scf_mouse_y - obj->y;
	x = scf_mouse_x - obj->x;
      
      if (obj->caption)
      {  if (y == 0 || y == obj->hei-1)
         { 	/* Scrolling */
		if (x != 2 && x != obj->wid-3) return 0;
            if (--y >=0) y = 1; /* i.e. y = +1 or -1*/
            y += inp->cur_line;
            x = inp->cur_pos;
            goto AcceptIt;
         } 

         y--;
         x--;
      }   

      if (y < 0 || y >= inp->eff_hei)  return 0;
      if (x < 0 || x >= inp->eff_wid)  return 0;

      y += inp->top_line;
      x += inp->left_col;

AcceptIt:
   	scf_accept_input_pos(inp, y, x);
	return  -1;
   
}

int scf_input_event_handler(void *inp, SCF_EVENT evnt)
{	
	if (evnt != SCF_CLICK_EVENT) return 0;

      if (scf_last_key == KEY_MOUSE)
		return scf_input_mouse_handler(inp); 

	return scf_input_key_handler(inp);

}
void scf_destroy_input(SCF_INPUT *inp)
{
	if (inp->content)	free(inp->content);
}


void scf_draw_input_routine(SCF_INPUT *inp, int action) 
{	SCF_OBJECT *obj;
	int  hei;   
	int  i;

	obj = (SCF_OBJECT *)inp;
	if (action == 0)
	{  obj->flags &= ~SCF_FRAME_DRAWN;
	   return;
	}

	if (obj->caption != NULL && (obj->flags & SCF_FRAME_DRAWN) == 0)
  	    scf_decorate_object(obj);	
              
	hei = inp->eff_hei + inp->top_line;
      
	for (i=inp->top_line; i<hei; i++)
  	   scf_draw_input_line(inp, i);
}

/* 10-Dec-99 */
SCF_INPUT *scf_create_input(SCF_FORM *frm, 
                      int hei, int wid, int y, int x,
                      const char *caption, int maxlines, int maxcols)
{	SCF_INPUT  *inp;
	SCF_OBJECT  *obj;	
	int eff_wid, eff_hei;
	char *content;

	obj = scf_create_object(frm,
      	sizeof(SCF_INPUT), SCF_INPUT_TYPE,
		hei, wid, y, x, caption);

	if (!obj) return NULL;
	obj->norm_attr = bar_attr;	
	obj->spec_attr = thumb_attr;	

	obj->draw_routine = (SCF_DRAW_ROUTINE) scf_draw_input_routine;
	obj->destroy_routine = (SCF_ROUTINE) scf_destroy_input;
	obj->event_handler = scf_input_event_handler;
      obj->flags |= SCF_GETS_FOCUS;

	inp = (SCF_INPUT *)obj;
	inp->capt_attr = side_attr;

	if (caption)
      { eff_wid = wid-2; eff_hei = hei-2; }
	else
      { eff_wid = wid;  eff_hei = hei; }

      inp->eff_wid = eff_wid;
	inp->eff_hei = eff_hei;

	if (maxlines <= 0) maxlines = eff_hei;
	if (maxcols <= 0) maxcols = eff_wid;

	eff_wid = (maxcols+1)*maxlines; /* needed for trailing zero */
	content = (char *) malloc(eff_wid);
	if (!content) { free(inp); return NULL; }
	memset(content, '\0',  eff_wid); 
	inp->content = content;

	inp->top_line = 0;
	inp->left_col = 0;
	inp->cur_line = 0;
	inp->cur_pos = 0;
	inp->line_count = maxlines;	
	inp->col_count = maxcols;	

	return inp;	
}      

/* 11-Dec-99 */
void scf_set_input(SCF_INPUT *inp, char *value)
{	char *start_pos, *out_pos, c;
	int line_count, col_count, cur_col_count;

	out_pos = start_pos = inp->content;
	if (!start_pos) return;

	line_count = inp->line_count;
	cur_col_count = col_count = inp->col_count;

	for(;;)
	{ c = *value++;		
	  *out_pos++ = (c=='\n' ? '\0': c);
	  if (c=='\0') break;
	  if (--cur_col_count>0 && c != '\n') continue;
        while (--cur_col_count >= 0) *out_pos++ = '\0';
/*        *out_pos++ = '\0';    */
	  if (--line_count <= 0) break; 
        out_pos = start_pos += (col_count + 1);
	  cur_col_count = col_count;
      } 

      col_count = strlen(inp->content);
      if (inp->cur_pos > col_count) inp->cur_pos = col_count;

      scf_draw_object((SCF_OBJECT *)inp);
}

/* 11-Dec-99 */
char *scf_get_input(SCF_INPUT *inp)
{	char *line_pos, *content;
	int line_count, col_count, col_count_trail;
	int len;
	char *result, *res_pos;

	content = inp->content;
	if (content == NULL) return NULL;

	line_count = inp->line_count;
	col_count = inp->col_count;
	col_count_trail = col_count+1;

	line_pos = content + (line_count-1) * (col_count_trail);

	/* Find the last non_empty line */
	while (line_count > 0 && *line_pos == '\0')
	{  line_pos -= col_count_trail;
	   line_count--;
	}
	if (line_count <= 0) return "";

	result = malloc(line_count * col_count_trail);
	if (!result) return result;

	line_pos = content;
	res_pos = result;

	while(--line_count >= 0)
	{  len = strlen(line_pos);
	   memcpy(res_pos, line_pos, len);
	   res_pos += len;
/*	   if (line_count > 0 && len < col_count)	*res_pos++ = '\n'; */
	   if (line_count > 0) *res_pos++ = '\n';	
	   line_pos += col_count_trail;
	}

	*res_pos++ = '\0';
	return realloc(result, res_pos-result);	   	
}

void scf_activate_object(void *obj1, int status)
{    SCF_FORM *frm;
     SCF_OBJECT *obj;
     
     obj = (SCF_OBJECT *)obj1;
     frm = obj->frm;

     if (status)		/* activate */
     {  obj->flags &= ~SCF_INACTIVE;
        if (frm->focus_obj == NULL &&
                (obj->flags & SCF_GETS_FOCUS))
              frm->focus_obj = obj;
     }           
     else
     {  obj->flags |= SCF_INACTIVE;
        if (frm->focus_obj == obj)
        {  SCF_OBJECT *new_obj = scf_get_next_focus_obj(frm, 0);
           if (new_obj == obj) new_obj = NULL;
           frm->focus_obj = new_obj;
           if (new_obj) scf_draw_object(new_obj);
        }   
     }          

     scf_draw_object(obj);
}


/*==============================================================*/
/*                       D I A L O G U E S                      */
/*==============================================================*/
#define BOTTOM_BTN  0x8000

int curs_yesno(const char *message)
{	SCF_FORM *frm = NULL;
	int rc, wid;

	wid = strlen(message)+4;
	rc = -1;

	frm = scf_create_form(NULL, 8, wid,
                 CENTRAL_LOCATION, CENTRAL_LOCATION, NULL);
	if (!frm) return -1;
     
	scf_create_text(frm, 1, -1, 2, CENTRAL_LOCATION, message);
	
	if (!scf_arrange_bottom_buttons(frm,
      			btn_attr, thumb_attr,
				BOTTOM_BTN, SCF_CTRLS_SUBST,  
				2, 5, "Yes", "Y", "No", "N"))
					goto HastaPronto;
     	
	rc = 2 - scf_get_event(frm, NULL);
     
HastaPronto:
	scf_destroy_form(frm);
	return rc;
     
}     

void curs_message(int wid, int msgcount,  char *message, ...)
{	SCF_FORM *frm = NULL;
	char *btntext[] =
	{ "Agreed", "Go ahead", "I see", "Certainly", 
	  "Dismiss", "Finish"};
	int text_cnt = sizeof(btntext)/sizeof(char *);
	va_list arglist;	
	int  rnd, y;
	time_t next_time;
	char *txt;


	if (wid <= 0) wid = strlen(message) + 4;	


	frm = scf_create_form(NULL, msgcount+7, wid,
                 CENTRAL_LOCATION, CENTRAL_LOCATION, NULL);
	if (!frm) return;
     
	va_start (arglist, message);
	y = 2;
	scf_create_text(frm, 1, -1, y++, 2, message);


	while (--msgcount>0)
	{  
	   txt = va_arg(arglist, char *); 
	   scf_create_text(frm, 1, -1, y++, 2, txt);
	}
	
	va_end(arglist);

	next_time = time(NULL);
	srand(next_time);
	rnd = (int)((double)rand() * text_cnt / (double)RAND_MAX);
	txt = btntext[rnd];
	
	if (!scf_arrange_bottom_buttons(frm,
      			btn_attr, thumb_attr,
			BOTTOM_BTN, SCF_CTRLS_SUBST,  
			1, strlen(txt)+2, txt, ""))
					goto HastaPronto;
     	scf_show_form(frm, 1);

	next_time += 4;
	while (scf_check_event(frm) <= 0 &&
			time(NULL) < next_time);
     
HastaPronto:
	scf_destroy_form(frm);
	redraw_screen();	     
}     



