/*  =========================================================
    SING - ALONG DISK PLAYER.
    (C) 1998, 1999   Michael Glickman.  xsadp@yahoo.com
    ---------------------------------------------------------
	Sing-Along Disk Player is copyrighted by the author.
        See COPYRIGHT regarding distribution policy and
        conditions of use.

	You are expected to provide appropriate references
	when using a part of the code in your software.

	Author strongly advices against using this code, or
	a part of it, in an application designed to run  on
	any Microsoft(tm) platfrom.
    ========================================================= */
#define MAX_REC_LENGTH 1101
#define MAX_HDR_LENGTH 41
#define MAX_LIST_SIZE  256
#define TEMP_SHORT_NAME ".temp_db"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <linux/cdrom.h>
#include <dirent.h>
#include "sad.h"

/* ----------------------------------------------------------- */

/*
    To add a compressing utility just add elements to
    zip_extension, zip_apps and unzip_apps. 

    N.B. 'zip_apps' and 'unzip_apps' should pack/unpack given
    file to stdout. Avoid using command line arguments with
    'unzip_apps' because of apparent popen bug: 'bunzip2 -c'
    comes up with a rubbish at the end of file name, while
    'bzcat' works fine. 
*/ 	

static const char *zip_exts[] = {".bz2", ".gz", ".z"};
static const int zip_count = sizeof(zip_exts)/sizeof(char *);
static const char *zip_apps[] = {"bzip2 -c", "gzip -c", "compress -c"};
static const char *unzip_apps[] = {"bzcat", "zcat", "zcat"};

/* ----------------------------------------------------------- */
/* Type:  
		 0 - section header,
		 1 - section line,
		-1 - section footer (currently unused)
*/

typedef short (*section_reader) (short type, const char *title, char *options);
typedef short (*section_writer) (FILE *fo, short type, const char *title, char *options);

struct section_struct
{  	char  *name;
	section_reader rdr;
	section_writer wtr;
	short written;
};

static section_reader current_rdr;
static section_writer current_wtr;

static short read_disc_section(short type, const char *title, char *options);
static short read_tracks_section(short type, const char *title, char *options);
static short read_prefset_section(short type, const char *title, char *options);

static short write_general_section(FILE *fo, short type, const char *title, char *options);
static short write_disc_section(FILE *fo, short type, const char *title, char *options);
static short write_tracks_section(FILE *fo, short type, const char *title, char *options);
static short write_prefset_section(FILE *fo, short type, const char *title, char *options);

static short read_mixer_section(short type, const char *title, char *options);
static short write_mixer_section(FILE *fo, short type, const char *title, char *options);


static struct section_struct di_sections[] = 
{
  { "general",  NULL,  write_general_section, 0},
  { "disc",     read_disc_section, write_disc_section, 0},
  { "tracks",   read_tracks_section, write_tracks_section, 0},
  { "mixer",    read_mixer_section,  write_mixer_section,  0},
  { "prefset",  read_prefset_section,  write_prefset_section,  0}
};
static short di_section_count = sizeof(di_sections)/sizeof(struct section_struct);

/* ----------------------------------------------------------- */
static char  *cur_dir;		/* Directory where file was found */
static char  *cur_fname;	/* File name to update/create */
static short cur_zip;		/* Compression type */
static short di_newfile;	/* 1 - create, 0 - update */

/* ----------------------------------------------------------- */
extern  TRACK_INFO  *trk_info;
extern  u_char dtrack_first, dtrack_last, track_first, track_last;
extern  char disc_name[], disc_artist[];
extern  char disc_extra[];
extern  u_long dtime_total;
extern  u_long disc_revision;
extern  u_long cddb_disc_id;


extern short  DataChanged;
extern short  cd_playlist_size;
extern u_char *cd_playlist;
extern char  DiscCategory[15];  
extern char  DefaultCategory[15];  
extern PLATFORM  Platform;
extern short  cd_multi_artist;

extern u_int  MixerDeviceCount;
extern u_int  mixer_node;

extern void write_split_text(FILE *f, char *header,
                        char *text, int part_len);

#ifdef MCN_SUPPORT
const char *cd_upcode;   /* MCN (UPC) */
#endif

char  *cdix_index;
char  OwnerName[OWNER_NAME_SIZE];

PREFSET_INFO  *cd_prefsets;
PREFSET_INFO  *cd_current_prefset;
short	cd_prefset_count;
short UseMCN = 1;

#ifdef DEFAULT_WDB_NAME
char *ShortWDBName = DEFAULT_WDB_NAME;
static char *WDBName = NULL;
static int wcdb_find(void);
#endif

#ifdef DEFAULT_XMCD_PATH
char *ShortXMCDPath = DEFAULT_XMCD_PATH;
static char *XMCDPath = NULL;
static int xmcd_find(void);
#endif

char *ShortDBPath = DEFAULT_DB_PATH;
static char *DBPath = NULL;



char *ShortDBAltPath = DEFAULT_DB_ALTPATH;
static char *DBAltPath = NULL;
static short mixer_written = 0;

short  new_cd_in_db = 1;
short  SaveMixer = 0;

static void clear_cd_info(void);

static void free_name(char **name)
{
	if (*name)
	{ free(*name); *name = NULL; }
}

void scdb_initialize(void)
{
    
    DBPath = get_full_fname(ShortDBPath);

    if (strlen(ShortDBAltPath) > 0)
        DBAltPath = get_full_fname(ShortDBAltPath);

#ifdef  DEFAULT_WDB_NAME
    if (strlen(ShortWDBName) > 0)
        WDBName = get_full_fname(ShortWDBName); 
#endif

#ifdef  DEFAULT_XMCD_PATH
    if (strlen(ShortXMCDPath) > 0)
        XMCDPath = get_full_fname(ShortXMCDPath); 
#endif

    cd_prefset_count = 0;
    cd_prefsets = NULL;
    cdix_index = NULL;
    

    strncpy(OwnerName, get_owner_name(), OWNER_NAME_SIZE);
    

}


void scdb_terminate(void)
{
    free_name(&cur_fname);

#ifdef  DEFAULT_WDB_NAME
    free_name(&WDBName);
#endif
#ifdef  DEFAULT_XMCD_PATH
    free_name(&XMCDPath);
#endif
    free_name(&DBPath);
    free_name(&DBAltPath);

    free_name(&cdix_index);
    if (cd_prefsets) free(cd_prefsets);
}

/* =============================================================== */
static char *make_full_path(const char *dirpath, const char *fname)
{	char *res;

	res = (char *) malloc(strlen(dirpath) + strlen(fname) + 2);
	if (res)  sprintf(res, "%s/%s", dirpath, fname);
	return res;
}

static char *make_default_file_name(const char *pref, const char *middle,
                                    short fileno)
{
	char *res;
	int  zt = DEFAULT_ZIP_TYPE;
	const char *ext =  (zt < 0) ? "" : zip_exts[zt];

	res = (char *) malloc(strlen(pref) + strlen(middle) + strlen(ext) + 7);
	if (res)  sprintf (res, "%s%s.%03d%s",
			   pref, middle, fileno, ext);
				    
	return res;	

}

FILE *scddb_open_input_file(const char *dir, const char *fname, short zip_type)
{
	int  len;
	char *buf = NULL;
	FILE *f = NULL;

	len = strlen(dir)+strlen(fname)+2;

	if (zip_type < 0) 
	{  buf = (char *) malloc(len);
	   if (buf)
	   { sprintf (buf, "%s/%s", dir, fname);
	     f = fopen(buf, "r");
	   }
	}
	else
	{  const char *app = unzip_apps[zip_type];
	   buf = (char *) malloc(len + strlen(app) + 3);
	   if (buf)
	   { sprintf (buf, "%s \"%s/%s\"", app, dir, fname);
	     f = popen(buf, "r");
	   }
	}
	
	if(buf) free(buf);
	return f;		
}

void scddb_close_input_file(FILE *f, short zip_type)
{
	if (zip_type < 0) fclose(f);
	  	       else pclose(f);
}
/* ================================================================= */
/*                          READERS                                  */
static short read_disc_section(short type, const char *title, char *options)
{	
	u_long duration;

	if (type != 0) return 1;	/* Only lines are of interest */

	if (strcasecmp(title, "DURATION") == 0)
	{
	   if (sscanf(options, "%lu", &duration) != 1
      	     || duration != dtime_total) return -1;
	}
	else
	if (strcasecmp(title, "TITLE") == 0)
		strncpy(disc_name, options, DATA_NAME_SIZE);
	else
	if (strcasecmp(title, "ARTIST") == 0)
		strncpy(disc_artist, options, DATA_NAME_SIZE);
	else
	if (strcasecmp(title, "MULTI-ARTIST") == 0)
	    cd_multi_artist = strchr("0NF", toupper(*options)) ? 0 : 1;
	else
	if (strcasecmp(title, "EXTRA") == 0)
      {   if (memcmp(options, "**", 2) == 0) options+=2;
	    strncat(disc_extra, options, EXTRA_NAME_SIZE-strlen(disc_extra));
	}
	else
	if (strcasecmp(title, "CATEGORY") == 0)
		strncpy(DiscCategory, options, 15);
	else
	if (strcasecmp(title, "REVISION") == 0)
		sscanf(options, "%lu", &disc_revision);
	else
		return 0;

	return 1;
}

static short read_tracks_section(short type, const char *title, char *options)
{	
	char *linetail;
	int   trk_no;
	u_long start;
	TRACK_INFO *ti;	

/*	if (type == -1) return 1;	 No footer of interest */

	/* Process header */
	if (type == 1)		
	{ u_int trk_start, trk_end;

	  if (sscanf(options, "%u %u", &trk_start, &trk_end) != 2 ||
		trk_start != dtrack_first || trk_end != dtrack_last)
			return -1;
	  return 1;
	}

	/* Track information */
	if (options == NULL || *options == '\0') return 0;

	/* Get track info record */
	trk_no = strtol(options, &linetail, 0);

	if (linetail == NULL || 
              trk_no < dtrack_first || trk_no > dtrack_last) return 0;

	ti = trk_info + (trk_no - dtrack_first);
/*	linetail += strspn(linetail, " \t"); */
	linetail++;

	if (strcasecmp(title, "START") == 0)
	{
	   if (sscanf(linetail, "%lu", &start) != 1
      	     || start != ti->start) return -1;
	}
	else
	if (strcasecmp(title, "TITLE") == 0)
		strncpy(ti->name, linetail, DATA_NAME_SIZE);
	else
	if (strcasecmp(title, "ARTIST") == 0)
		strncpy(ti->artist, linetail, DATA_NAME_SIZE);
	else
	if (strcasecmp(title, "EXTRA") == 0)
		strncat(ti->extra, linetail, EXTRA_NAME_SIZE-strlen(ti->extra));
		

	return 1;
}

static short read_mixer_section(short type, const char *title, char *options)
{
	static short active = 0;
	int  device, node;
	u_int volume, balance;

	if (!SaveMixer) return 0;

	if (type == 1)		
	{ 
	  active = 0;
	  
	  if (sscanf(options, "%x", &node) == 1 &&
	        node == mixer_node) active = 1;

	  return active;
	}

	
	if (type == 0 && active && strcasecmp(title, "VOLUME")==0 &&
	     sscanf(options, "%d %u %u", &device, &volume, &balance) == 3)
	{	     
	    mixer_set_device_volume((short)device, volume, balance);
	    return 1;
	}	        
	
	return 0;
}

static short read_prefset_section(short type, const char *title, char *options)
{
	static PREFSET_INFO  *pi=NULL;
	u_char  *list_ptr;
	short   count, *countptr;
	char	  *txtptr;

	switch(type)
	{  case 1:    /* Header */
		if (options == NULL || strlen(options) == 0) break;
		trim(options);
		/* New prefset */
		cd_prefsets = realloc(cd_prefsets,
                               ++cd_prefset_count * sizeof(PREFSET_INFO));
		if (cd_prefsets == NULL) return -1;		
		pi = cd_prefsets + (cd_prefset_count-1);
		strncpy(pi->name, options, PREFSET_NAME_SIZE);
		strncpy(pi->owner, OwnerName, OWNER_NAME_SIZE);
		pi->playlist = malloc(MAX_LIST_SIZE * sizeof(u_char));
		pi->skiplist = malloc(MAX_LIST_SIZE * sizeof(u_char));
		pi->playlist_size = pi->skiplist_size = 0;
		pi->written = 0;
		break;

	   case 0:    /* Line */
	      if (!title || pi == NULL || options==NULL) return 0;
		if (strcasecmp(title, "owner") == 0)
		{ strncpy(pi->owner, options, OWNER_NAME_SIZE);
		  return 1;
		}
		else
		if (strcasecmp(title, "list") == 0)
		{ list_ptr = pi->playlist;
		  countptr = &(pi->playlist_size);		  
		}
		else
		if (strcasecmp(title, "skip") == 0)
		{ list_ptr = pi->skiplist;
		  countptr = &(pi->skiplist_size);		  
		}
		else  return 0;

		count = *countptr;
		list_ptr += count;
		txtptr = options;
		while(isdigit(*txtptr) && count < MAX_LIST_SIZE)
		{  *list_ptr++ = strtol(txtptr, &txtptr, 0);
		   txtptr += strspn(txtptr, " ,\t");
		   count++;	
		}
		*countptr = count;   		
		break;

	   case -1:    /* Footer */
		if (pi == NULL) return 0;
		pi->playlist = realloc(pi->playlist,
                                   pi->playlist_size * sizeof(u_char));
		pi->skiplist = realloc(pi->skiplist,
                                   pi->skiplist_size * sizeof(u_char));
		break;

	}
		
	return 1;
}

/* ================================================================= */
/*                       READ FRONT END                              */
/* ================================================================= */

static short process_input_header(char *frec)
{	const char *header;
      struct section_struct *secptr;
	short sec_count;
	
	header = strtokm(frec, ": \t");	/* Must exist */

/*	Footer call - currently unused
	if(current_rdr &&
	    (*current_rdr)(-1, header, NULL) < 0) return -1; */

	secptr = di_sections;
	sec_count = di_section_count;	
	while (--sec_count >=0)
	{  if (strcasecmp(header, secptr->name)==0) break;
	   secptr++;
	}

      current_rdr = (sec_count >= 0) ? secptr->rdr : NULL;

	if (current_rdr)
	{  char *options = strtokm(NULL, "");
	   if (options) options += strspn(options, " \t");	
 	   return (*current_rdr)(1, header, options);		
	}

	return  0;
}

static short process_input_line(char *recptr)
{	const char *title;
	char *options;

	if (!current_rdr) return 0;

	recptr += strspn(recptr, " \t");
	title = strtokm(recptr, ": \t");
	if (!title) return 0; 	
	
	options = strtokm(NULL, "");
	if (options) options += strspn(options, " \t");	
	return (*current_rdr)(0, title, options);
}

static short read_cd_info(const char *cdir,
				  const char *cfname, short ziptype)
{
	FILE *f;

	char frec[MAX_REC_LENGTH];
	short res, ret;

	f = scddb_open_input_file(cdir, cfname, ziptype);
	if (!f) return 0;

	ret = 1;
	current_rdr = NULL;

	while (fgets(frec, MAX_REC_LENGTH, f))
	{  strtokm(frec, "\r\n");
	   if (frec[0] == '#' || frec[0] == '\0')
				continue;	
	   /* res values:
		 1 - header/line OK
		 0 - unknown header/line
		-1 - wrong header/line
	   */
	   if (isalpha(*frec))
	 	 res = process_input_header(frec); 
	   else	
	       res = process_input_line(frec);

	   if (res < 0) { ret = 0; break; }
	}
	    	
	if (ret && current_rdr)
	{   res = (*current_rdr)(-1, NULL, NULL);
	    if (res < 0)  ret = 0;
      }

	scddb_close_input_file(f, ziptype);
	DataChanged = 0;
	return ret;
}
/* ================================================================= */
static short check_file_name(const char *fname, const char *index, short *file_no, short *zip_no)
{	const char *ptr;
	char  *ptr1;
	int  l;

	l = strlen(index);
	if (memcmp(fname, index, l)) return 0;

	ptr = fname+l;	
	if (*ptr++ != '.') return 0;

	*file_no = strtol(ptr, &ptr1, 10);
	if ((ptr1 - ptr) != 3) return 0;

	ptr += 3;
	if (*ptr == '\0') 
	{ *zip_no = -1; return 1; }
	
	for (l=0; l<zip_count && strcasecmp(ptr, zip_exts[l]); l++);
	if (l>=zip_count) return 0;
	*zip_no = l;	
	return 1;
}



static void scdb_process_path(const char *path,
			      const char *pref, const char *middle,
                              short *next_difno_ptr,  short *di_ziptype)
{
	DIR 	*dir_stream;
	struct dirent *dir_entry;
	short  di_zno;
	short  fno;
	int    next_difno, next_dpfno;
	char *ptr;

	dir_stream = opendir(path);
	if (!dir_stream) return;

	next_difno = next_dpfno = -1;


	while ((dir_entry = readdir(dir_stream)) != NULL)
	{
		ptr = dir_entry->d_name;	

		if ( memcmp(ptr, pref, 3) == 0 &&
                        check_file_name(ptr+3, middle, &fno, &di_zno))
		{   if (read_cd_info(path, ptr, di_zno))
		    {  cur_fname = strdup(ptr);
		       cur_dir = (char *) path;	
		       next_difno = fno;
		       break;
		    }
		    else			
  	            if (fno > next_difno) next_difno = fno;
		}

      }

	closedir(dir_stream);

	if (cur_fname == NULL)
	{  next_difno++;
         di_zno = DEFAULT_ZIP_TYPE;
	}
	if (next_difno_ptr) *next_difno_ptr = next_difno;
	if (di_ziptype) *di_ziptype = di_zno;

}

/* ================================================================= */
void  clear_disc_tracks(void)
{
	TRACK_INFO *ti;
	u_short track;

	cd_multi_artist = 0;
	ti = trk_info;      
	strcpy(disc_artist, "");	
	strcpy(disc_extra, "");	

	for (track= dtrack_first; track<= dtrack_last; track++)
	{	sprintf(ti->name, "Track %u", (u_int) track);
		strcpy(ti->artist, ""); 
		strcpy(ti->extra, ""); 
		ti++;
	}		 
}

static void clear_cd_info(void)
{ 
	PREFSET_INFO *pi;
	int count;
         
	disc_revision = 0;
	cddb_disc_id = 0;
	strcpy(disc_name, "** Unknown **");
	strcpy(DiscCategory, DefaultCategory);
      
	clear_disc_tracks();

	pi = cd_prefsets;

	count = pi==NULL ? 0 : cd_prefset_count;
	while (--count >= 0)
	{  free_name((char **)&(pi->playlist));
	   free_name((char **)&(pi->skiplist));
	   pi->written = 0;
	   pi++;
	}	

	if (cd_prefsets) free(cd_prefsets);
	cd_prefsets = NULL;
	cd_prefset_count = 0;
	cd_current_prefset = NULL;
	cd_playlist_size = 0;
	cd_playlist = NULL;
}

static void  process_alt_source(void)
{
#if defined(DEFAULT_WDB_NAME)
	if (wcdb_find()) return;
#endif

#if defined(DEFAULT_XMCD_PATH)
	if (xmcd_find()) return;
#endif

#if defined(CD_EXTENSION_SUPPORT)
	if (process_cdtext()) return;
#endif

#if  defined(CD_EXTRA_SUPPORT)
	process_cdextra();
#endif

	return;
}

short scdb_find(void)
{	short next_difn;
	const char *pref, *middle;

	if (cdix_index) free(cdix_index);
	cdix_index = generate_cdindex();

	free_name(&cur_fname);

#ifdef MCN_SUPPORT
	cd_upcode = NULL;

	if (UseMCN)
      {  cd_upcode = cd_get_mcn();

	   if (cd_upcode != NULL)
	   {  pref = cd_upcode + strspn(cd_upcode, " 0");
	      if (*pref == '\0') cd_upcode = NULL;
	   }
	}

	if (cd_upcode != NULL)
	{	pref = "DU_";
		middle = cd_upcode;
	}
	else
#endif /* def MCN_SUPPORT */
	{	pref = "DI_";
		middle = cdix_index;
	}

	new_cd_in_db = 1;	
	di_newfile = 1;
	clear_cd_info();

	scdb_process_path(DBPath, pref, middle, &next_difn, &cur_zip); 

	if (cur_fname == NULL && DBAltPath != NULL)
		scdb_process_path(DBAltPath, pref, middle, NULL, NULL);
	
	if (cur_fname == NULL)
	{
	   clear_cd_info();
	   process_alt_source();

	   cur_dir = DBPath;	
	   cur_fname = make_default_file_name(pref, middle, next_difn);
        }
	else
	{  new_cd_in_db = 0;
	   di_newfile = 0;
	}

	cd_create_prefset(NULL);
	cd_select_default_prefset();

	return 1;

}
/* ================================================================= */
/*                         WRITE UTILITIES                           */
/* ================================================================= */

static short check_handled_section(const char **handled_sections,
                                   const char *title)
{	const char *section;

	while ((section = *handled_sections++) != NULL) 
  	    if (strcasecmp(section, title) == 0) return 1;

	return 0;
}

static short write_starting_comment(FILE *fo, short flag)
{
	time_t time_now;

	fprintf(fo, "COMMENT\n");

	fprintf(fo, "\t-----------------------------------\n");
	if (flag)
	fprintf(fo, "\tsadp CD information and preferences\n"); 
	fprintf(fo, "\tcreated with %s %s\n", 
			(Platform == PLTF_CURSES) ? "sadp" : "xsadp" ,
			             get_sadp_version());
	time_now = time(NULL);
	fprintf(fo, "\t        %s", ctime(&time_now));
	fprintf(fo, "\t-----------------------------------\n");
	fprintf(fo, "\n");

	return 1;
}

static short write_general_section(FILE *fo, short type, const char *title, char *options)
{
	time_t time_now;

	static const char *handled_sections[] =
            {"application", "version", "written", NULL};

	switch(type)
	{
	   case 1:          /* Header */
		fprintf(fo, "GENERAL\n");
		fprintf(fo, "\tAPPLICATION: %s\n",
				(Platform == PLTF_CURSES) ? "sadp" : "xsadp" );
		fprintf(fo, "\tVERSION:  %06lx\n", (long)SADP_VERSION);
		time_now = time(NULL);
		fprintf(fo, "\tWRITTEN:  %s", ctime(&time_now));
		return 1;

/*	  case -1:
		fprintf(f, "\n");
		return 1;    */
	}


       /* Line */
	if (!title) return 0;

	return check_handled_section(handled_sections, title);
		


}


static short write_disc_section(FILE *fo, short type, const char *title, char *options)
{
	static const char *handled_sections[] =
            {"duration", "title", "artist", "multi-artist",
              "extra", "category", "revision", "index",  NULL};

	switch(type)
	{  case 1:    /* Header */
		fprintf(fo, "DISC\n");
		fprintf(fo, "\tDURATION: %lu\n", dtime_total);
		if (strlen(disc_name) > 0)
			fprintf(fo, "\tTITLE:    %s\n", disc_name);
		if (strlen(disc_artist) > 0)
			fprintf(fo, "\tARTIST:   %s\n", disc_artist);
		if (cd_multi_artist)
			fprintf(fo, "\tMULTI-ARTIST: Y\n");
		if (strlen(disc_extra) > 0)
			write_split_text(fo, "\tEXTRA:  **", disc_extra, 60);

#ifdef MCN_SUPPORT
		if (cd_upcode)
		        fprintf(fo, "\tMCN:      %s\n", cd_upcode);
#endif
		if (strlen(DiscCategory) > 0)
			fprintf(fo, "\tCATEGORY: %s\n", DiscCategory);
		fprintf(fo, "\tREVISION: %lu\n", ++disc_revision);
		fprintf(fo, "\tINDEX:    %s\n", cdix_index);
		break;

	   case 0:    /* Line */
	      if (!title) return 0;
		return check_handled_section(handled_sections, title);
	}
		
	return 1;

}

static void write_current_mixer_info(FILE *fo)
{
	u_int *mix_volumes, *mix_balances;

	
	mix_volumes = malloc(MixerDeviceCount * sizeof(u_int)); 
	mix_balances = malloc(MixerDeviceCount * sizeof(u_int)); 
    
	if (mix_volumes && mix_balances &&
		 mixer_get_volumes(mix_volumes, mix_balances))
	{	int i;

		fprintf (fo, "MIXER %x\n", mixer_node);
	    
	   for (i=0; i<MixerDeviceCount; i++)
	    	fprintf(fo, "\tVOLUME:  %d %u %u\n",
				         i, mix_volumes[i], mix_balances[i]);
	}

	if (mix_volumes) free(mix_volumes);
	if (mix_balances) free(mix_balances);

}



static short write_mixer_section(FILE *fo, short type, const char *title, char *options)
{	static short active = 0;
	static const char *handled_sections[] =   {"volume",  NULL};

	switch(type)
	{
	      case 1:	/* Header */
		  active = 0;
		  if (mixer_written || !SaveMixer)
			 return 0;
		  else
		  {   if (options != NULL &&
		          mixer_node != (int) strtol(options, NULL, 16))
		             return 0; 
		  
		      write_current_mixer_info(fo);
		      mixer_written = 1;
		  }		  
		  if (options == NULL)	fprintf(fo, "\n");
		  active = 1;
		  break;
		
	      case 0:    /* Line */
		  if (active == 0)   return 0;
		  return check_handled_section(handled_sections, title);

		case -1:	   
		  if (active == 0)   return 0;

	}
    
	return 1;

}


static short write_tracks_section(FILE *fo, short type, const char *title, char *options)
{
	TRACK_INFO  *ti;
	int   i;

	static const char *handled_sections[] =
            {"start", "title", "artist", "extra", NULL};

	switch(type)
	{  case 1:    /* Header */
		fprintf(fo, "TRACKS %u %u\n", dtrack_first, dtrack_last);
		ti = trk_info;
		for (i=dtrack_first; i<=dtrack_last; i++)	
		{
			fprintf(fo, "\tSTART:  %4u %lu\n", i, ti->start);

			if (strlen(ti->name) > 0)
				fprintf(fo, "\tTITLE:  %4u %s\n", i, ti->name);

			if (strlen(ti->artist) > 0)
				fprintf(fo, "\tARTIST: %4u %s\n", i, ti->artist);

			if (strlen(ti->extra) > 0)
			{  char ext_hdr[19];
			   sprintf(ext_hdr, "\tEXTRA   %4u ", i);	
			   write_split_text(fo, ext_hdr, ti->extra, 57);
			}

			ti++;
		}
		break;

	   case 0:    /* Line */
	      if (!title) return 0;
		return check_handled_section(handled_sections, title);
	}
		
	return 1;
}

static short write_preference_set(FILE *fo, PREFSET_INFO *pi)
{	short count, quantum, first;
	u_char *plist;


	count = pi->playlist_size;
	if (count <= 0) return 1;
	fprintf (fo, "PREFSET %s\n", pi->name);
        fprintf (fo, "\tOWNER: %s\n", pi->owner);

	plist = pi->playlist;
	first = 1;
	while (count > 0)
	{  quantum  = count;
	   if (quantum > 12) quantum = 12;
	   count -= quantum;
	   if (first) first = 0; else  fprintf (fo, "\n");
	   fprintf (fo, "\tLIST:  ");
	   while (--quantum >= 0)	
	       fprintf(fo, " %4u", (u_int)*plist++);
	}

	if (!first) fprintf (fo, "\n");

	plist = pi->skiplist;
	count = pi->skiplist_size;
	first = 1;
	while (count > 0)
	{  quantum  = count;
	   if (quantum > 12) quantum = 12;
	   count -= quantum;
	   if (first) first=0; else fprintf (fo, "\n");
	   fprintf(fo, "\tSKIP:  ");
	   while (--quantum >= 0)	
		fprintf(fo, " %4u", (u_int)*plist++);
	}

	if (!first) fprintf (fo, "\n");
	fprintf (fo, "\n");
	pi->written = 1;
	return 1;

}

static short write_prefset_section(FILE *fo, short type, const char *title, char *options)
{
	PREFSET_INFO  *pi;
	int   count;

	static const char *handled_sections[] =
               {"owner", "list", "skip", NULL};

	switch(type)
	{  case 1:    /* Header */
		if (options == NULL) break;
		pi = cd_prefsets;
		count = cd_prefset_count;
		trim(options);
		while (--count >= 0)
		{  if (strcasecmp(pi->name, options) == 0) break;
		   pi++;
		} 
		if (count >= 0)
		   return write_preference_set(fo, pi);
		break;

	   case 0:    /* Line */
	        if (!title || strlen(title)==0) return 1;
		return check_handled_section(handled_sections, title);
	}
		
	return 1;
}

/* ================================================================= */
/*                       WRITE FRONT END                             */
/* ================================================================= */

static short process_output_header(FILE *fo, char *lineptr)
{	char header[MAX_HDR_LENGTH];
	int sec_count;
	struct section_struct *secptr;
	short ret;

	if (current_wtr &&
             (*current_wtr)(fo, -1, NULL, NULL) < 0) return -1;

	if (lineptr == NULL) return 1;

	strncpy(header, lineptr + strspn(lineptr, " \t"), MAX_HDR_LENGTH);	
	strtokm(header, ": \t");	

	sec_count = di_section_count;
	secptr = di_sections;
	while (--sec_count >=0)
	{  if (strcasecmp(header, secptr->name)==0) break;
	   secptr++;
	}

	if (sec_count >= 0) 
	{   secptr->written = 1;
	    current_wtr = secptr->wtr;
	}
	else
	    current_wtr = NULL;

	if (current_wtr)
	{  char *options;
	   ret = strlen(header);
	   if (ret >= strlen(lineptr))
		options = NULL;
	   else	
	   {  options = lineptr+ret+1;
		options += strspn(options, " \t");
	   }
	
	   ret = (*current_wtr)(fo, 1, header, options);
	   if (ret < 0) return 0;
	   if (ret > 0) return 1;
	}

	fprintf(fo, "%s\n", lineptr);
	return 1;
}	

static short process_output_line(FILE *fo, char *lineptr)
{	char header[MAX_HDR_LENGTH];
	short ret;

	strncpy(header, lineptr + strspn(lineptr, " \t"), MAX_HDR_LENGTH);	
	strtokm(header, ": \t");

	if (current_wtr)
	{  char *options;

	   ret = strlen(header);

	   if (ret >= strlen(lineptr))
		options = NULL;
	   else	
	   {  options = lineptr+ret+1;
		options += strspn(options, " \t");
	   }	

	   ret = (*current_wtr)(fo, 0, header, options);
	   if (ret < 0) return 0;
	   if (ret > 0) return 1;
	}

	fprintf(fo, "%s\n", lineptr);
	return 1;
}	

short scdb_write(void)
{
	FILE   *fo = NULL;
	FILE   *fi = NULL;
	PREFSET_INFO *pi;
	char *tmp_name=NULL;
	char *out_name=NULL;
	char frec[MAX_REC_LENGTH];
	struct section_struct *secptr;
	section_writer wtr;
	short res, ret;
	int   count;


	ret = 0;
	if (di_newfile == 0) 
	{  if (cur_fname == NULL) return ret;
	   fi = scddb_open_input_file(cur_dir, cur_fname, cur_zip); 
	}

	tmp_name = make_full_path(cur_dir, TEMP_SHORT_NAME);
	fo = fopen(tmp_name, "w");
	if (!fo) goto OutOfHere;

	mixer_written = 0;
	count = di_section_count;
	secptr = di_sections;
	while (--count >= 0)
	{  secptr->written = 0;
	   secptr++;
	}

	pi = cd_prefsets;
	if (pi!=NULL) 
	{	count = cd_prefset_count;
		while (--count >= 0)  pi++;
	}

	if (fi)
	{    current_wtr = NULL; res = 0;
	     while (fgets(frec, MAX_REC_LENGTH, fi))
	     {  strtokm(frec, "\r\n");
		   /*
		       res values:
			 1 - header/line OK
			 0 - unknown header/line
			-1 - wrong header/line
		   */	
		   if (isalpha(*frec))
		       res = process_output_header(fo, frec); 
		   else	
		       res = process_output_line(fo, frec);
		
		   if (res < 0) break; 
		}

		if (res >=0 )
		   res = process_output_header(fo, NULL); 

		scddb_close_input_file(fi, cur_zip);
		if (res < 0) goto OutOfHere;
	}
	else
		write_starting_comment(fo, 0);

	
	count = di_section_count;
	secptr = di_sections;
	while (--count >= 0)
	{  if (secptr->written == 0)
	   {  wtr = secptr->wtr;
		if (wtr && (*wtr)(fo, 1, NULL, NULL) < 0)
		      goto OutOfHere;
		fprintf(fo, "\n");
	   }
	   secptr++;
	}

	if (!mixer_written && SaveMixer)
	   write_mixer_section(fo, 1, NULL, NULL);

	pi = cd_prefsets;
	if (pi != NULL)
	{
		count = cd_prefset_count;

		while (--count >= 0)
		{  if (pi->written == 0)
		   {	if (write_preference_set(fo, pi) <= 0)
		    	  goto OutOfHere;
//			fprintf(fo, "\n");
		   }
		   pi++;
		}
	}

	fclose(fo); fo = NULL;

      out_name = make_full_path(cur_dir, cur_fname);
	if (!out_name) goto OutOfHere;
	if (cur_zip < 0)
	    rename (tmp_name, out_name);
	else
	{
	    const char *app = zip_apps[cur_zip];	
	    char  *buf;	

	    buf = malloc(strlen(app) + strlen(tmp_name)
                       + strlen(out_name) +  9);
	    if (buf == NULL) goto OutOfHere;

	    sprintf(buf, "%s \"%s\" > \"%s\"", app, tmp_name, out_name);

	    remove(out_name);
	    system(buf);
	    free(buf);
	}		
		   		
	ret = 1;
	new_cd_in_db = 0;
        di_newfile = 0;
	DataChanged = 0;

OutOfHere:
	if (tmp_name)
	{    remove(tmp_name);	
	     free(tmp_name);
	}
	if (out_name) free(out_name);
	if (fo) fclose(fo);
	return ret;
}


/* ============================================================ */
/*                   BACKWARD COMPATIBILITY                     */
/* ============================================================ */
#ifdef DEFAULT_WDB_NAME

PREFSET_INFO *process_plist(FILE *f, char *name)
{	u_int  w;
	int  pl_size;
	PREFSET_INFO  *pi;	
	u_char  *plist;	


	fscanf(f, "%d", &pl_size);
	if (pl_size == 0) return NULL;


	pi = cd_create_prefset(name);
	if (!pi) return NULL;

	plist = malloc(pl_size * sizeof(u_char));
	if (!plist) return NULL;
	pi->playlist = plist;
	pi->playlist_size = pl_size;	


	while (--pl_size >= 0)
	{    fscanf(f, "%u", &w);
	     *plist++ = w;
	}
	
	return pi;

}

static void get_text_fld2(char *buf, char *target1, int len1,
                                              char *target2, int len2)
{	char *ptr;

	strtokm(buf, "\n");

	ptr = strstr(buf, "//");   /* Workman style */
	if (ptr)
	{  *ptr = '\0';
	   ptr += 2;
	}
	else
	{ ptr = strchr(buf, '<');   /* Old sadp style */	
	  if (ptr)
	  { *ptr++ = '\0'; strtokm(ptr, ">"); }
	}
		
	trim(buf);     	
	if (target1) strncpy(target1, buf, len1);

	if (target2)
	{ 
	   if (ptr == NULL) ptr = ""; else trim(ptr);
     	   strncpy(target2, ptr, len2);
	}
}

static void get_text_fldf(FILE *f, char *buf, char *target, int length)
{	
	fgets(buf, 80, f); buf[80] = 0;
	strtok(buf, "\n");
	trim(buf);     	
	if (target) strncpy(target, buf, length);
}

static char *adjust_target(char *target, int *size)
{	int size1;

	size1 = strlen(target);
	if (*size <= (size1 + 1))
	{  *size = 0;
	   return NULL;
	}
	
	target += size1;
	*size -= size1;
	if (size1 > 0)
	{  *target++ = ' ';
         (*size)--;
	}

	return target;
}

int check_track_info(FILE *f)
{
	TRACK_INFO *ti;
	u_long start;
	char buf[81];
	int i;

	/* Comparing number of tracks */
	if (fscanf(f, "%d", &i) == 0) return 0;
	if (dtrack_last != i) { fflush(f);  return 0; }

	/* Comparing track data */
	ti = trk_info;
	i = dtrack_last-dtrack_first+1;
	while (--i >= 0)
	{  if (fscanf(f, "%lu", &start) == 0 || ti->start != start)
                                      break;
	   ti++;
	}

	if (i < 0)
	{  fgets(buf, 12, f);
	   if (sscanf(buf, "%lu",  &start) > 0
                    && ti->start/CD_FRAMES == start)
				return 1;
	}
	else  fflush(f);	

	return 0;
}

static int wcdb_find(void)
{
	FILE *f;
	char buf[81], *buf1;
	TRACK_INFO *ti, *ti1;
	PREFSET_INFO *pi = NULL;
	char  *ptr2;
	u_char *ptr1;
	int    size2, trkno;
	int    rc = 0;

	f = fopen(WDBName, "r");
	if (!f) return 0; 	              /* That's fine ! */


	if (cd_playlist) free(cd_playlist);
	cd_playlist_size = 0;

	while (!feof(f))
	{
		/* Looking for 'tracks' keyword */
		fgets(buf, 8, f);

		if (strcmp(buf, "tracks ") == 0)
		{
		  if(check_track_info(f)) goto Trovano;
		}
		else
		{
		  if (strchr(buf, '\n') == 0) fflush(f);
		}
	}

	goto Malfortuna;

Trovano:
	fflush(f);

	/* We found it! Unbelieva-b-b-b-le
	   Parse other relevant lines now.
	*/

	cd_multi_artist = 0;
	new_cd_in_db = 0;
	ti = trk_info;
	while (!feof(f) && fscanf(f, "%s", buf) > 0)
	{
		if (!strcmp(buf, "tracks")) break;

		if (!strcmp(buf, "cdname"))
		{
			get_text_fldf(f, buf, disc_name, 60);
			DataChanged = 1;   /* Forcing to save in new format */
			rc = 1;
		}			

		else
		if (!strcmp(buf, "artist"))
		{
			fgets(buf, 80, f);  buf[80] = 0;
			get_text_fld2(buf, disc_artist, DATA_NAME_SIZE-1,
                                        disc_extra, DATA_NAME_SIZE-1);
			if (*disc_artist == '\0')
 				               cd_multi_artist = 1;
		} 
		else
		if (!strcmp(buf, "track"))
		{
			fgets(buf, 80, f);  buf[80] = 0;
			trim(buf);
			size2 = DATA_NAME_SIZE-1;

			if (ti>trk_info && *buf=='+')
			{  ti1 = ti-1;
			   buf1 = buf+1;
			   if (memcmp(buf1, "//", 2)==0) buf1 += 2;	
			   ptr2 = (cd_multi_artist && strlen(ti1->artist)==0) ? ti1->artist : ti1->extra;
			   if (cd_multi_artist && strlen(ti1->artist)==0)
                     {  ptr2 = ti1->artist;
				size2 = DATA_NAME_SIZE;
			   }
			   else 	
			   {  ptr2 = ti1->extra;
				size2 = EXTRA_NAME_SIZE;
			   }	
			   ptr2 = adjust_target(ptr2, &size2);	
			   strtokm(buf1, "\n");
			   strncpy(ptr2, buf1, size2);
			}
			else
			{  /* ptr1 = ti->name; */
			   if (cd_multi_artist)
			   {  ptr2 = ti->artist;
				size2 = DATA_NAME_SIZE;
			   }
			   else
			   {  ptr2 = ti->extra;
				size2 = EXTRA_NAME_SIZE;
			   }
			   get_text_fld2(buf, ti->name, size2, ptr2, size2);
			   ti++;
			}
		} 
		else
		if (!strcmp(buf, "category"))
			get_text_fldf(f, buf, DiscCategory, 15);
		else
		if (!strcmp(buf, "revno"))
		{	get_text_fldf(f, buf, NULL, 11);
			sscanf(buf, "%lu", &disc_revision);
		}
		else
		if (!strcmp(buf, "plist"))
			pi = process_plist(f, NULL);
	}

	fclose(f);
	f = NULL;
	
	ptr2 = getenv("WORKMANRC");
	if (ptr2==NULL) ptr2 = "~/.workmanrc";
	ptr2 = get_full_fname(ptr2);	

	f = fopen(ptr2, "r");
	if (f == NULL) goto Malfortuna;
	
	
	while (!feof(f))
	{
		fgets(buf, 8, f);

		if (strcmp(buf, "tracks ") == 0)
		{
		  if(check_track_info(f)) goto PrefTrovano;
		}
		else
		{
		  if (strchr(buf, '\n') == 0) fflush(f);
		}
	}

	goto Malfortuna;
	
PrefTrovano:

	/* Processing preferences */
	fflush(f);

	while (!feof(f) && fscanf(f, "%s", buf) > 0)
	{
		if (!strcmp(buf, "tracks")) break;

		if (!strcmp(buf, "playlist"))
		{
		      fscanf(f, "%s", buf);

		      ptr2 = buf;
		      while ((ptr2 = strchr(ptr2, '_')) != NULL)
		            *ptr2++ = ' '; 
			    
		      pi = process_plist(f, buf);
		}			

		else
		if (!strcmp(buf, "dontplay") && pi != NULL)
		{
		      fscanf (f, "%d", &trkno);
		      if (trkno < track_first || trkno > track_last)
		                              continue;
					      
		      size2 = pi->skiplist_size + 1;
		      ptr1 = realloc(pi->skiplist, size2);
		      if (ptr1 == NULL) continue;
		      
		      pi->skiplist_size = size2;
		      pi->skiplist = ptr1;
		      ptr1 += (size2-1);
    		      fscanf(f, "%d", &size2);
		      *ptr1 = (u_char) trkno;
		} 
	}

	goto Malfortuna;


Malfortuna:
	if (f!=NULL) fclose(f);
	return rc;
}

#endif

/* ============================================================ */
/*                      XMCD COMPATIBILITY                      */
/* ============================================================ */
#ifdef DEFAULT_XMCD_PATH
static int xmcd_find(void)
{
	FILE *f = NULL;
	DIR 	*dir_stream;
	struct dirent *dir_entry;
	char   *fname, *category = NULL;
	char buf[257];

      cddb_disc_id = get_cddb_discid();
	
	dir_stream = opendir(XMCDPath);
	if (!dir_stream) return 0;

	fname = malloc(strlen(XMCDPath)+51);
	if (!fname)  return 0;

	while ((dir_entry = readdir(dir_stream)) != NULL)
	{
		category = dir_entry->d_name;
		if (*category == '.') continue;

		sprintf (fname, "%s/%s/%08lx", XMCDPath,
				 category, cddb_disc_id);
		f = fopen(fname, "r");
		if (f) break;
	}	

	free(fname);		
	if (!f) return 0;

	// Found file - get data
	
	strncpy(DiscCategory, category, 15);
	
	while (fgets(buf, 121, f))
	   process_cddb_line(buf, 0);

	fclose(f);
	DataChanged = 1;   /* Forcing to save in new format */
	return 1;
}
#endif
