/*
    This file is part of the 'ears' package.
    Copyright (C) 1994,1995,1996  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#pragma implementation 
#include <unistd.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stream.h>
#include "messages.h"
#include "mygetopt.h"
#include "modules/recognizer.h"
#include "config.h"
#include "pr_config.h"
#include "cfg_file.h"
#include "others/mystring.h"

//------------------------------PUBLICS---------------------------------
ConfigProtocol::ConfigProtocol (int i, char** p, bool b=false) 
: argc_(i), argv_(p), strict_(b), home_(getenv("HOME")) 
{
}

void ConfigProtocol::start()
{
  new config;     // Who owns it?

  check_libs (argv_[0]);
  
  if (!strict_)
    set_default();
  
  rc_file();
  strict_? tinyoptions() : options();
  
  globals();
  
  if (!strict_)
    make_paths();
}

//----------------------------PRIVATES----------------------------------
void ConfigProtocol::check_libs (string progname)
{
  string s = "ldd ";
  s += progname;
  s += ">ldd.output";
  system ( s.c_str() );
  
  ifstream lf ("ldd.output");
  string lib, dummy, ncurses="";
  while (lf)
  {
    lf >> dummy >> ws >> dummy >> ws >> lib;
    int pos = lib.find("stdc++.so.",0);
    if (pos>0)
    {
      int n1, n2, n3;
      get_major_minor(lib.substr(pos+10),n1,n2,n3);
      if ((n1==2 && (n2<7 || (n2==7 && n3<2))) ||
         (n1==27 && (n2<1 || (n2==1 && n3<4))))
      { 
        string s = "ldd shows that you have libstdc++ Version ";
        s = s + dec(n1) + "." + dec(n2) + "." + dec(n3) + "\n";
        s += "This program is known to dump core with versions < 27.1.4\n";
        s += "Please upgrade.\n";
        throw(fatal_exception(s));
      }
    }
    pos = lib.find("ncurses.so.",0);
    if (pos>0) ncurses=lib;
  }
  
  char buf[256];
  int n = readlink (ncurses.c_str(),buf,255);
  if (n>0) { buf[n]=0; ncurses=buf; }
  int pos = ncurses.find("ncurses.so.",0);
  if (pos>0)
  {
    int n1, n2, n3;
    get_major_minor(ncurses.substr(pos+11),n1,n2,n3);
    if (n1==1 && (n2<9 || (n2==9 && n3<8)))
    { 
      string s = "ldd shows that you have libncurses Version ";
      s = s + dec(n1) + "." + dec(n2) + "." + dec(n3) + "\n";
      s += "This program is known to dump core with versions < 1.9.8\n";
      s += "Please upgrade.\n";
      throw(fatal_exception(s));
    }
  }
  lf.close();
  remove("ldd.output");
}

void ConfigProtocol::get_major_minor (string s, int& n1, int& n2, int& n3)
{
  string nr1 = s; 
  string nr2 = nr1.substr(nr1.find(".")+1);
  string nr3 = nr2.substr(nr2.find(".")+1);
  nr1 = nr1.substr(0,nr1.find("."));
  nr2 = nr2.substr(0,nr2.find("."));
  n1 = atoi(nr1.c_str());
  n2 = atoi(nr2.c_str());
  n3 = atoi(nr3.c_str());
}

// First set the defaults.
void ConfigProtocol::set_default ()
{
  config& cfg = config::InstanceRef();
  cfg.set ("EARS_PATH",     home_+"/.ears");
  cfg.set ("KEEP_SAMPLES",  false);
  cfg.set ("FEATURE",       "MRASTA");
  cfg.set ("RECOGNIZER",    "NDTW");
  cfg.set ("MIN_NUM_WORDS", "3");
  cfg.set ("BASENAME",      "default");
  cfg.set ("SOUND_BITS",    "8");
  cfg.set ("SOUND_SPEED",   "8000");
  cfg.set ("DEBUG",         false);
  cfg.set ("NEWLINE",       false);
  cfg.set ("TEST",          false);
  cfg.set ("NOCURSES",	false);
  cfg.set ("EXECUTE",	false);
}

// Now read the ConfigProtocol file if it exists, overwriting the defaults.
void ConfigProtocol::rc_file ()
{
  string name = home_ + "/.earsrc";
  cfg_file cf(name);
  if (strict_ && !cf.exists())
    throw (fatal_exception("Missing .earsrc file.  Please run train_ears"));

  if (cf.exists())
  {
    config& cfg = config::InstanceRef();
    string first,second;
    while (cf.getpair(first,second))
    {
      string t = upcase(first);
      cfg.set(t,second);
    }
  }
  
  upcase_entries();
}

// print the available parameters
void ConfigProtocol::usage(char *progname)
{
  cerr << progname 
    << " [-k <keep_samples=yes>]"
    << " [-p ears_path]" << endl
    << " [-f feature]"
    << " [-r recognizer]"
    << " [-m min_num_words]"
    << " [-b basename]" << endl
    << " [-B sound_bits]"
    << " [-S sound_speed]" 
    << " [-d <debug=yes>]" 
    << " [-c <nocurses=yes>]" << endl
    << " [-x <execute=yes>]" << endl;
    
  throw (fatal_exception("Nothing to be done"));
}


// Then get the command line options, also overwriting.
void ConfigProtocol::options()
{
  GetOpt opt(argc_,argv_,"-p:kFf:r:m:b:B:S:dch");
  int ch;
  
  config& cfg = config::InstanceRef();
  while ((ch=opt())!=EOF)
    switch (ch)
    {
      case 1:
      case 'b': cfg.set ("BASENAME",      opt.optarg); break;
      case 'p': cfg.set ("EARS_PATH",     opt.optarg); break;
      case 'k': cfg.set ("KEEP_SAMPLES",        true); break;
      case 'f': cfg.set ("FEATURE",       opt.optarg); break;
      case 'r': cfg.set ("RECOGNIZER",    opt.optarg); break;
      case 'm': cfg.set ("MIN_NUM_WORDS", opt.optarg); break;
      case 'd': cfg.set ("DEBUG",               true); break;
      case 'c': cfg.set ("NOCURSES",            true); break;
      case 'x': cfg.set ("EXECUTE",             true); break;
      case '?': 
      case 'h': usage(argv_[0]);
      case 'B': {
                  int b = atoi(opt.optarg);
                  if (b!=8 && b!=16)
                  { cerr << "Invalid bits option: " << opt.optarg << endl;
                    usage(argv_[0]); }
                }
                cfg.set ("SOUND_BITS",    opt.optarg); break;
      case 'S': {
                  int s = atoi(opt.optarg);
                  if (s<8000 || s>44100)
                  { cerr << "Invalid rate option: " << opt.optarg << endl;
                    usage(argv_[0]); }
                }
                cfg.set ("SOUND_SPEED",   opt.optarg); break;
    }

  upcase_entries();    
}

void ConfigProtocol::tinyusage(char *progname)
{
  cerr << progname 
    << " [-p ears_path]" << " [-b basename]" << " [-d <debug=yes>]" 
    << " [-n <newline=yes>]" << " [-t <test=yes>]" << " [-c <nocurses=yes>]"
    << " [-x <execute=yes>]" << endl;

  throw (fatal_exception("Nothing to be done"));
}

void ConfigProtocol::tinyoptions()
{
  GetOpt opt(argc_,argv_,"p:b:dntch");
  int ch;
  
  config& cfg = config::InstanceRef();
  while ((ch=opt())!=EOF)
    switch (ch)
    {
      case 1:
      case 'b': cfg.set ("BASENAME",      opt.optarg); break;
      case 'p': cfg.set ("EARS_PATH",     opt.optarg); break;
      case 'd': cfg.set ("DEBUG",               true); break;
      case 'n': cfg.set ("NEWLINE",             true); break;
      case 't': cfg.set ("TEST",                true); break;
      case 'c': cfg.set ("NOCURSES",            true); break;
      case 'x': cfg.set ("EXECUTE",             true); break;
      case '?': 
      case 'h': tinyusage(argv_[0]);
    }
}

// Create necessary directories.
void ConfigProtocol::make_paths()            // Looks ugly
{
  config& cfg = config::InstanceRef();
  const char* path = (cfg["EARS_PATH"]).c_str();
  char* wd = getcwd(0,0);

  messages& msg = messages::InstanceRef();
  if (chdir(path)<0 && msg.yesno(M_Create_Dirs,YES,true)<=0) 
    throw(fatal_exception());
  if (chdir(path)<0 && mkdir(path,S_IRUSR|S_IWUSR|S_IXUSR)<0)
    throw(fatal_exception("Could not mkdir(ears_path)"));

  string raw = string(path) + "/raw";
  if (chdir(raw.c_str())<0 && mkdir(raw.c_str(),S_IRUSR|S_IWUSR|S_IXUSR)<0)
    throw((fatal_exception("Could not mkdir(ears_path/raw)")));

  if (chdir(cfg.pattern_path().c_str())<0 
      && mkdir(cfg.pattern_path().c_str(),S_IRUSR|S_IWUSR|S_IXUSR)<0)
    throw(fatal_exception("Could not mkdir(pattern_path)"));
    
  chdir(wd);
  cfg.save_rc(false);
}

// Switches some of the entries to UPPERCASE
void ConfigProtocol::upcase_entries()
{
  string entry;
  config& cfg = config::InstanceRef();

  entry = cfg["FEATURE"];
  cfg.set("FEATURE",upcase(entry));
  entry = cfg["RECOGNIZER"];
  cfg.set("RECOGNIZER",upcase(entry));
  entry = cfg["KEEP_SAMPLES"];
  cfg.set("KEEP_SAMPLES",upcase(entry));
  entry = cfg["DEBUG"];
  cfg.set("DEBUG",upcase(entry));
}

void ConfigProtocol::globals()
{
  config& cfg = config::InstanceRef();
  recognizer r (cfg["RECOGNIZER"],0);
  cfg.set_pattype (r.pat_type());
  string f = cfg["FEATURE"];
  string s = cfg["EARS_PATH"] + "/" + f + "-" + cfg.pat_type();
  cfg.set_patpath (s);
  cfg.train_ = r.training_needed();
  s = home_ + "/.earsrc";
  cfg.set_rcpath (s);

//  fvec::init(iget("SOUND_SPEED"));  // Beware, what if device says no?
}
