/*
    This file is part of the 'ears' package.
    Copyright (C) 1994,1995  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#pragma implementation 
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include "myrandom.h"
#include "words.h"
#include "exception.h"

// This is the interface to the words file.  Until EARS is capable
// of recognizing more than a few hundred words, they are kept in a
// linked list.

words* words::instance_ = 0;  // is a singleton
//-------------------------------PUBLICS----------------------------------
// Returns a random word out of those words that need learning.
// (That is, it is recorded less than MIN_NUM_WORDS times.)

string words::rnd_missing ()
{
  if (missing_==0) return "";
  
  RandomInteger Rnd (1, missing_);
  int r = Rnd();
  
  for (it_ = list_.begin(); it_!=list_.end(); it_++)
  {
    r -= (*it_)->missing;
    if (r<=0)
    {
      --((*it_)->missing);
      --missing_;
      return (*it_)->label;
    }
  }
  return "";
}

void words::try_to_load (const string& basename, const string& path)
// if file exists, load it and set wfn_ and cfp_
// if file doesn't exist, set wfn_ and load_err
{
  load_err_=false;
  wfn_ = path + "/" + basename + ".words";
  cfp_ = new cfg_file(wfn_);
  
  if (!cfp_->exists())
  {
    delete cfp_; cfp_=0;
    string wfn1 = string("./") + basename + ".words";
    cfg_file* cfp1 = new cfg_file(wfn1);
    if (!cfp1->exists()) 
      { delete cfp1; load_err_=true; return; }
    else 
      { wfn_=wfn1; cfp_=cfp1; }
  }

  string first,second;
  while (cfp_->getpair(first,second))
    list_.push_back(new Word (first,second,0));
  delete cfp_;
  cfp_=0;

  if (list_.size() < 2)
    throw(fatal_exception("Not enough words (min. 2)"));
}

static struct def_words { char *l,*a; } defaults[] =
{
  {"Zero","0"},{"One" ,"1"},{"Two" ,"2"},
  {"Three" ,"3"},{"Four" ,"4"},{"Five" ,"5"},{"Six" ,"6"},
  {"Seven" ,"7"},{"Eight" ,"8"},{"Nine" ,"9"}
};

void words::make_default_list()
{
  list_.erase (list_.begin(), list_.end());
  cfp_ = new cfg_file (wfn_);
  for (unsigned k=0; k<sizeof(defaults)/sizeof(def_words); k++)
  {
    Word* p = new Word;
    p->label = defaults[k].l;
    p->action = defaults[k].a;
    p->missing = 0;
    list_.push_back (p);
    cfp_->write (p->label, p->action);
  }
  delete cfp_;
  cfp_=0;
}

// Looks if all words are spoken a sufficient number of times.
// If not, counts and marks all that miss, for later recording.

void words::mark_missing (const string& path, int num)
{
  DIR* dp = opendir(path.c_str());
  struct dirent *p;
  while ((p=readdir(dp))!=0)
  {
    string fn = p->d_name;
    if (fn=="." || fn=="..") continue;
    fn = fn.substr(0,fn.find("."));
    for (it_ = list_.begin(); it_!=list_.end(); it_++)
      if ((*it_)->label==fn)
        ++((*it_)->missing);
  }
  closedir(dp);

  for (it_ = list_.begin(); it_!=list_.end(); it_++)
  {
    if ((*it_)->missing >= num)
      (*it_)->missing=0;
    else
    {
      int t = num - (*it_)->missing;
      (*it_)->missing = t;
      missing_ += t;
    }
  }
}
