/*
    This file is part of the 'ears' package.
    Copyright (C) 1994,1995,1996  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
// Thanks to Michael Beck for his vplay code and Hannu Savolainen for
// his great driver and programming documentation.

#pragma implementation 
#include <stdlib.h>
#include <values.h>
#include <fstream.h>
#include "pattern.h"
#include "recognizer.h"
#include "ears/fmat.h"
#include "ears/exception.h"

//--------------------------WORD_PATTERN-------------------------------

pattern::pattern (const feature& f, const string& r) : 
                  F_trace(20), B_quanta(6)
{ 
  recognizer rec(r,0);
  
  if (rec.pat_type()=="V")
    p = new var_pattern (f);
  else 
  if (rec.pat_type()=="F")
    p = new fixed_pattern (f,F_trace);
  else 
  { 
    string t = "Pattern.h: Unknown pat_type: ";
    t += rec.pat_type();
    throw(fatal_exception(t));
  }
}


// For every kind of pattern, we have two constructors --- creating a
// pattern from a feature, or from a file.
//----------------------------VAR_PATTERN---------------------------------

var_pattern::var_pattern (const feature &f)
{
  length = f.length();
  coeff = f.coeff();
  buf = new float* [length];
  for (int k=0; k<length; k++) 
  {
    buf[k] = new float [coeff];
    memcpy(buf[k],f.buffer()[k],sizeof(float)*coeff);
  }
}

var_pattern::var_pattern (const string& fn) 
{
  ifstream i(fn.c_str());
  i >> length >> coeff;
  buf = new float* [length];
  for (int k=0; k<length; k++) 
  {
    buf[k] = new float [coeff];
    for (int l=0; l<coeff; l++)
      i >> buf[k][l];
  }
}

PatternImplementation* var_pattern::copy() const
{
  var_pattern* p = new var_pattern;
  p->length = length;
  p->coeff = coeff;
  p->buf = new float* [length];
  for (int k=0; k<length; k++) 
  {
    p->buf[k] = new float [coeff];
    memcpy(p->buf[k],buf[k],sizeof(float)*coeff);
  }
  return p;
}

void var_pattern::write (ostream& o)
{
  o << length << " " << coeff << " ";
  for (int k=0; k<length; k++)
    for (int l=0; l<coeff; l++)
      o << buf[k][l] << " ";
}

void var_pattern::read (istream& i)
{
  bool ok = i >> length >> coeff;
  if (!ok || length<10 || coeff!=8) { err=true; return; }
  
  if (!buf)
  {
    buf = new float* [length];
    for (int k=0; k<length; k++) 
    {
      buf[k] = new float [coeff];
    }
    if (!i) { err=true; return; }
  }
  for (int k=0; k<length; k++)
    for (int l=0; l<coeff; l++)
      i >> buf[k][l];
      
  if (!i) err=true;
}

//----------------------------FIXED_PATTERN-------------------------------

fixed_pattern::fixed_pattern (const feature &f, int t) 
{
  bits=0;
  buf = new float* [trace=t];
  for (int k=0; k<trace; k++)
    buf[k] = new float [coeff=f.coeff()];

  if (f.length() < trace)  
    err=true;
  else  
    trace_segmentation (f.buffer().buf(), f.length()-1);
  
// Normalizing is not necessary.
}

fixed_pattern::fixed_pattern (const string& fn) 
{
  bits=0;
  ifstream i(fn.c_str());
  i >> trace >> coeff;
  
  buf = new float* [trace];
  for (int k=0; k<trace; k++)
    buf[k] = new float [coeff];
  
  for (int k=0; k<trace; k++)
    for (int l=0; l<coeff; l++)
      i >> buf[k][l];
}

PatternImplementation* fixed_pattern::copy() const
{
  fixed_pattern* p = new fixed_pattern;
  p->trace = trace;
  p->coeff = coeff;
  p->buf = new float* [trace];
  for (int k=0; k<trace; k++) 
  {
    p->buf[k] = new float [coeff];
    memcpy(p->buf[k],buf[k],sizeof(float)*coeff);
  }
  return p;
}


// Stolen from librecog.  Thanks to Tilo Schuerer

void fixed_pattern::trace_segmentation (float **fbuf, int diffsize)
{
  float dist = 0.0;
  float diff [diffsize];
  for (int i=0; i < diffsize; i++)
    diff[i] = 0.0;                   // initialize all distances to zero 

  for (int i=0; i < diffsize; i++)
  {
    for (int j=0; j < coeff; j++)
    {
      float abs_dist = fbuf[i+1][j] - fbuf[i][j];
      diff[i] += (abs_dist > 0) ? abs_dist : -abs_dist;
    }
    dist += diff[i];
  }
	
  float average_dist = dist/trace;     // average distance 

  int k=0;
  dist = 0.0;
  for (int i=0; i < trace; i++)
  {
    while((dist < average_dist)&&(k<diffsize))
      dist += diff[k++];	
	
    for(int j=0; j < coeff; j++)
      buf[i][j] = fbuf[k-1][j];
	
    dist -= average_dist;         // rest remaining for next loop 
  }
}

void fixed_pattern::write (ostream& o)
{
  o << trace << " " << coeff << " ";
  for (int k=0; k<trace; k++)
    for (int l=0; l<coeff; l++)
      o << buf[k][l] << " ";
}

void fixed_pattern::read (istream& i)
{
  i >> trace >> coeff;
  for (int k=0; k<trace; k++)
    for (int l=0; l<coeff; l++)
      i >> buf[k][l];
}


char* fixed_pattern::quantize (int q)
{
  // Note: This is local quantization, maybe we should compute
  // min,max of all patterns?

  // We determine min and max for each coeff over all frames and
  // divide the interval evenly in QUANTA sub-intervals.  Then
  // we set the pat array according to which sub-interval the
  // value falls in.  Then we fill the c array.  Big question:
  // What order do we use --- time-conserving or frame-conserving?

  bits = new char[trace*coeff*(q-1)+1];

  int k,l,m;
  float min[coeff],max[coeff];
  for (k=0; k<coeff; k++)
  {
    min[k]=MAXFLOAT;
    max[k]=-MAXFLOAT;
  }                               // a faster way to do this?
  
  for (k=0; k<trace; k++)
    for (l=0; l<trace; l++)
    {
      if (buf[k][l] < min[l])  min[l]=buf[k][l];
      if (buf[k][l] > max[l])  max[l]=buf[k][l];
    }
  
  char pat[trace][coeff];
  for (k=0; k<coeff; k++)
  {
    float size=(max[k]-min[k])/float(q);
if (size==0.0) { cerr<<"Warning: size=0\n"; size=1; }
    size *= 1.0001;
    for (l=0; l<trace; l++) 
      pat[l][k] = int(((buf[l][k]-min[k])/size));     // yields 0,...,Q-1
  }

  int count=0;  
  for (k=0; k<trace; k++)
    for (l=0; l<coeff; l++,count+=q-1)
      for (m=0; m<q-1; m++)
        bits[count+m] = (pat[k][l]>m)? 1:0;                // thermometer
        
  bits[trace*coeff*(q-1)]=0;
  return bits;
}



//----------------------------BIT_PATTERN---------------------------------
// For now, we compute the actual bit pattern when reading the fixed_pattern
// file, but this is only for being able to easily calibrate the 
// quantization parameters during testing!

bit_pattern::bit_pattern (const feature &f, int, int trace) : 
    fixed_pattern (f, trace)
{
}

void write (ostream&)
{
}
 
void read  (istream&)
{
}

PatternImplementation::~PatternImplementation() {}   // necessary
