#pragma interface
#ifndef Pattern_h
#define Pattern_h

#include <string>
#include "ears/feature.h"

class PatternImplementation  // This ABC and pattern form a bridge
{
public:
  PatternImplementation()           { err=false; };
  virtual ~PatternImplementation() =0;
  virtual PatternImplementation* copy() const =0;
  virtual float** buff() =0;
  virtual int len() const =0;
  virtual int co() const =0;
  virtual void write (ostream&) =0;
  virtual void read  (istream&) =0;
  bool bad() const    { return err; }

protected:
  bool err;
};

class var_pattern : public PatternImplementation   // variable sized pattern
{                                    // mostly identical to class feature
public:
  var_pattern() : buf(0) {}
  var_pattern (const feature &f);
  var_pattern (const string& fn);
  virtual ~var_pattern() 
  { 
    if (buf) 
    { 
      for (int k=0; k<length; k++) 
        delete [] buf[k];
      delete [] buf; 
    }
  }
  virtual PatternImplementation* copy() const;
  virtual float** buff()  { return buf; } 
  virtual int len() const { return length; }
  virtual int co() const  { return coeff; }
  virtual void write (ostream& o);
  virtual void read  (istream& i);
  int size()              { return length*coeff; }

private:
  float **buf;
  int coeff, length;
};

class fixed_pattern : public PatternImplementation  // fixed sized pattern, fitting
{                                      // for non-recurrent NNs
public:
  fixed_pattern() : buf(0) {}
  fixed_pattern (const feature &f, int trace);
  fixed_pattern (const string& fn);
  virtual ~fixed_pattern() 
  { 
    if (buf)
    {
      for (int k=0; k<trace; k++) 
        delete [] buf[k];
      delete [] buf; 
    }
    if (bits) delete [] bits; 
  }
  virtual PatternImplementation* copy() const;
  virtual float** buff()  { return buf; } 
  virtual int len() const { return trace; }
  virtual int co() const  { return coeff; }
  virtual void write (ostream& o);
  virtual void read  (istream& i);
  int size() const        { return trace*coeff; }
  char* quantize (int q);  // this won't stay here

private:
  float **buf;
  int trace, coeff;
  char* bits;
  void trace_segmentation (float**, int);
};

class bit_pattern : public fixed_pattern   // empty for testing purposes
{
public:
  bit_pattern (const feature& f, int q, int trace);
protected:
  virtual ~bit_pattern()          {}
  float** buff()                  { return 0; } 
  virtual void write (ostream& o) { fixed_pattern::write (o); }
  virtual void read  (istream& i) { fixed_pattern::read (i);  }
};

class pattern    // The class through which the user accesses patterns
{
public:
  const F_trace, B_quanta;

  pattern() : p(0) { }                  
  pattern (const feature&, const string&);
  pattern (const pattern& w) { p = (w.p)? (w.p)->copy() : 0; }
  ~pattern()                 { if (p) delete p; }
  
  float** buf() const   { return p->buff(); } 
  int length() const    { return p->len(); }
  int coeff() const     { return p->co(); }
  bool bad() const      { return p->bad(); }
  friend ostream& operator<< (ostream& o, const pattern& p)
	                { (p.p)->write(o); return o; }
  friend istream& operator>> (istream& i, const pattern& p)
	                { (p.p)->read(i); return i; }
private:
  PatternImplementation* p;
};


#endif
