/*
    This file is part of the 'ears' package.
    Copyright (C) 1996  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#pragma implementation
#ifdef OSS

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <fstream.h>
#include "soundcard.h"
#include "others/sndblock.h"
#include "sample.h"
#include "sound_oss.h"

//#define DEBUG_SOUND(x,y) cerr<<(x)<<(y)<<endl;
#define DEBUG_SOUND(x,y) /*nodebug*/

OssSound::OssSound (int r, int b) : 
  mixer_loudness(100), rate(r), bits(b), md(-1), sd(-1), 
  name_("OssSound"), error_(false), opened_(false)
{ 
  full_mic ();
  deaf_mic ();
}

OssSound::~OssSound() 
{ 
  if (!opened_) return;

  if (md>=0)
  {
    if (devices & SOUND_MASK_MIC)
      ioctl(md,MIXER_WRITE(SOUND_MIXER_MIC),&old_mic);    
    if (devices & SOUND_MASK_PCM)
      ioctl(md,MIXER_WRITE(SOUND_MIXER_PCM),&old_volume); 
    ::close(md); 
  }
  ::close(sd);
  opened_ = false;  
}

void OssSound::full_mic()
{
  if (opened_) return;

  sd = ::open ("/dev/dsp", O_RDONLY);

  if (sd<0) 
  { 
    errmsg_ = "Could not open /dev/dsp";
    error_ = true;
    return;
  }

  int arg = 0xffff0004;
  ioctl (sd, SNDCTL_DSP_SETFRAGMENT, &arg);
  
  md = ::open ("/dev/mixer", O_RDWR);
  if (md < 0)
  {
    errmsg_ = "Could not open /dev/mixer!";
    error_ = true;
    ::close(sd);
    return;
  }
    
  int r = ioctl(md,SOUND_MIXER_READ_DEVMASK,&devices);
  if (r==-1)
  {
    errmsg_ = "Soundcard doesn't have any mixer!";
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }
  DEBUG_SOUND("The following mixer devs are available: ", devices)
  
  int recsrc;
  ioctl(md,SOUND_MIXER_READ_RECMASK,&recsrc);
  if ((devices & SOUND_MASK_MIC)==0 || (recsrc & SOUND_MASK_MIC)==0)
  {
    errmsg_ = "Soundcard doesn't have mic recording capability!";
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }

  if (devices & SOUND_MASK_PCM)
  {
    ioctl(md,MIXER_READ(SOUND_MIXER_PCM),&old_volume); 
    int volume = 0;
    ioctl(md,MIXER_WRITE(SOUND_MIXER_PCM),&volume); 
  }

  int set_mic=0;
  ioctl (md, SOUND_MIXER_WRITE_RECSRC, &set_mic);
    
  int t = bits;
  int fmt = t==8? AFMT_U8 : AFMT_S16_LE;
  int myfmt = fmt;
  ioctl(sd, SNDCTL_DSP_SETFMT, &fmt);
  if (fmt != myfmt) 
  { 
    errmsg_ = "Can't set BITS to ";
    errmsg_ += dec(t);
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }
  DEBUG_SOUND("SOUND_BITS = ", t)
  
  int stereo=0;
  ioctl (sd, SNDCTL_DSP_STEREO, &stereo);

  t = rate;
  if (ioctl(sd, SNDCTL_DSP_SPEED, &t) == -1)
  { 
    errmsg_ = "Can't set SPEED to ";
    errmsg_ += dec(t);
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }
  DEBUG_SOUND("actual SOUND_SPEED = ", t)

  if (md>=0) 
  {
    int left=mixer_loudness, right=mixer_loudness;
    int mic = (left & 0xff) | ((right & 0xff) << 8);
    if (devices & SOUND_MASK_MIC)
      ioctl(md,MIXER_WRITE(SOUND_MIXER_MIC),&mic);
  }
  
  opened_ = true;
}


void OssSound::deaf_mic()
{
  if (!opened_) return;

  if (md>=0) 
  {
    int left=0, right=0;
    int mic = (left & 0xff) | ((right & 0xff) << 8);
    if (devices & SOUND_MASK_MIC)
      ioctl(md,MIXER_WRITE(SOUND_MIXER_MIC),&mic);
    ::close (md);
  }
  ::close(sd);
  opened_ = false;
}

void OssSound::empty_buffer() const
{
//  ioctl(sd,SNDCTL_DSP_RESET,0); buggy in OSS for 2.0.29
}

//-------------------------------------------------------------------------
void OssSound::fill_block (sndblock& s) const
{
  unsigned char uc;
  short last_us=0;
  long sum=0;
  for (int k=0; k<s.size_; k++)
  {
    short us;
    if (s.bits_==8) { read (sd,&uc,sizeof(uc)); us=uc; }
    else            { read (sd,&us,sizeof(us)); }
    s.buf_[k] = us;
    short diff = (last_us>us)?last_us-us:us-last_us;
    if (s.bits_==16) diff=int(diff*float(diff)/100.0);
    sum += diff;
    last_us=us;
  }
  s.energy_ = sum / s.size_;
}

void OssSound::save_sample (const class sample& s, const string& f) const
{
  string fn = f + ".wav";
  ofstream sfile (fn.c_str());
  sfile << sample_format::wav << s;
}

#endif