/*
    This file is part of the 'ears' package.
    Copyright (C) 1994,1995,1996  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
// Thanks to Michael Beck for his vplay code and Hannu Savolainen for
// his great driver and programming documentation.
// Thanks to Niels Thorwirth for porting to AF platforms.

#pragma implementation
#ifdef VOXWARE

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <fstream.h>
#include <sys/soundcard.h>
#include "others/sndblock.h"
#include "sample.h"
#include "sound_voxware.h"

//#define DEBUG_SOUND(x,y) cerr<<(x)<<(y)<<endl;
#define DEBUG_SOUND(x,y) /*nodebug*/

VoxwareSound::VoxwareSound (int r, int b) : 
  mixer_loudness(100), rate(r), bits(b), sd(-1), md(-1), 
  name_("VoxwareSound"), error_(false), opened_(false)
{ 
  full_mic ();
  deaf_mic ();
}

VoxwareSound::~VoxwareSound() 
{ 
  if (!opened_) return;
  
  if (md>=0)
  {
    if (devices & SOUND_MASK_MIC)
      ioctl(md,SOUND_MIXER_WRITE_MIC,&old_mic);    
    if (devices & SOUND_MASK_VOLUME)
      ioctl(md,SOUND_MIXER_WRITE_VOLUME,&old_volume); 
    ::close(md); 
  }

  ::close(sd);
  opened_ = false;
}

void VoxwareSound::deaf_mic()
{
  if (!opened_) return;

  if (md>=0) 
  {
    int left=0, right=0;
    int mic = (left & 0xff) | ((right & 0xff) << 8);
    if (devices & SOUND_MASK_MIC)
      ioctl(md,SOUND_MIXER_WRITE_MIC,&mic);
    ::close(md);
  }
  ::close(sd);
  opened_ = false;
}

void VoxwareSound::full_mic()
{
  if (opened_) return;

  sd = ::open ("/dev/dsp", O_RDONLY);

  if (sd<0) 
  { 
    errmsg_ = "Could not open /dev/dsp";
    error_ = true;
    return;
  }

  md = ::open ("/dev/mixer", O_RDWR);
  if (md < 0)
  {
    errmsg_ = "Could not open /dev/mixer!";
    error_ = true;
    ::close(sd);
    return;
  }
    
  if (md>=0) 
  {
    ioctl(md,SOUND_MIXER_READ_DEVMASK,&devices);
    DEBUG_SOUND("The following mixer devs are available: ", devices)
  
    if (devices & SOUND_MASK_VOLUME)
    {
      ioctl(md,SOUND_MIXER_READ_VOLUME,&old_volume); 
      int volume = 0;
      ioctl(md,SOUND_MIXER_WRITE_VOLUME,&volume); 
    }
  }
    
  int t = bits;
  ioctl(sd, SNDCTL_DSP_SAMPLESIZE, &t);
  if (t != bits) 
  { 
    cerr<<"SNDCTL_DSP_SAMPLESIZE\n"; 
    errmsg_ = "Can't set BITS to ";
    errmsg_ += dec(t);
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }
  DEBUG_SOUND("SOUND_BITS = ", t)

  t = rate;
  if (ioctl(sd, SNDCTL_DSP_SPEED, &t) == -1)
  { 
    cerr<<"SNDCTL_DSP_SPEED\n"; 
    errmsg_ = "Can't set SPEED to ";
    errmsg_ += dec(t);
    error_ = true;
    ::close(md);
    ::close(sd);
    return;
  }
  DEBUG_SOUND("actual SOUND_SPEED = ", t)

  if (md>=0) 
  {
    int left=mixer_loudness, right=mixer_loudness;
    int mic = (left & 0xff) | ((right & 0xff) << 8);
    if (devices & SOUND_MASK_MIC)
      ioctl(md,SOUND_MIXER_WRITE_MIC,&mic);
  }
  
  opened_ = true;
}

void VoxwareSound::empty_buffer() const
{
//  ioctl(sd,SOUND_PCM_RESET,0); buggy in OSS for 2.0.29
}

//-------------------------------------------------------------------------
void VoxwareSound::fill_block (sndblock& s) const
{
  unsigned char uc;
  short last_us=0;
  long sum=0;
  for (int k=0; k<s.size_; k++)
  {
    short us;
    if (s.bits_==8) { read (sd,&uc,sizeof(uc)); us=uc; }
    else            { read (sd,&us,sizeof(us)); }
    s.buf_[k] = us;
    short diff = (last_us>us)?last_us-us:us-last_us;
    if (s.bits_==16) diff=int(diff*float(diff)/100.0);
    sum += diff;
    last_us=us;
  }
  s.energy_ = sum / s.size_;
}

void VoxwareSound::save_sample (const class sample& s, const string& f) const
{
  string fn = f + ".wav";
  ofstream sfile (fn.c_str());
  sfile << sample_format::wav << s;
}

#endif