#ifndef Mymath_h
#define Mymath_h

#if !defined(__i386__) || defined(DEBUG_MATH)
#include <math.h>
#else

/* John Bowman's inline-math follows here. */

/* Copyright (C) 1995 Free Software Foundation, Inc.
This file is part of the GNU C Library.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#ifndef __MATH_H
#define __MATH_H

#ifdef	__GNUC__
#ifndef __NO_MATH_INLINES

#ifdef __cplusplus
#define	__MATH_INLINE __inline
#else
#define	__MATH_INLINE extern __inline
#endif

__MATH_INLINE double cos(const double);
__MATH_INLINE double sin(const double);

__MATH_INLINE double __expm1(const double x)
{
	register double value, exponent, __temp, arg = x;
	__asm__ __volatile__
		("fldl2e				# e^x - 1 = 2^(x * log2(e)) - 1\n\t"
		 "fmul	%%st(1)			# x * log2(e)\n\t"
		 "fstl	%%st(1)\n\t"
		 "frndint				# int(x * log2(e))\n\t"
		 "fxch\n\t"
		 "fsub	%%st(1)			# fract(x * log2(e))\n\t"
		 "f2xm1				# 2^(fract(x * log2(e))) - 1\n\t"
		 "fscale				# 2^(x * log2(e)) - 2^(int(x * log2(e)))\n\t"
		 : "=t" (value), "=u" (exponent) : "0" (arg));
	__asm__ __volatile__
		("fscale				# 2^int(x * log2(e))\n\t"
		 : "=t" (__temp) : "0" (1.0), "u" (exponent));
	__temp -= 1.0;
	return __temp+value;
}

__MATH_INLINE double __sgn1(const double x)
{
	return (x >= 0.0 ? 1.0 : -1.0);
}

__MATH_INLINE double sqrt(const double x)
{
	register double value, arg = x;
	__asm__ __volatile__
		("fsqrt" : "=t" (value): "0" (arg));

	return value;
}

__MATH_INLINE double fabs(const double x)
{
	register double value, arg = x;
	__asm__ __volatile__
		("fabs" : "=t" (value): "0" (arg));

	return value;
}

__MATH_INLINE double sin(const double x)
{
	register double value, arg = x;
	__asm__ __volatile__
		("fsin" : "=t" (value): "0" (arg));

	return value;
}

__MATH_INLINE double cos(const double x)
{
	register double value, arg = x;
	__asm__ __volatile__
		("fcos" : "=t" (value): "0" (arg));

	return value;
}

__MATH_INLINE double tan(const double x)
{
	register double value, value2, arg = x;
	__asm__ __volatile__
		("fptan" : "=t" (value2), "=u" (value): "0" (arg));

	return value;
}

__MATH_INLINE double atan2(const double y, const double x)
{
	register double value, argy = y, argx = x;
	__asm__ __volatile__
		("fpatan\n\t"
		 "fldl %%st(0)"
		 : "=t" (value): "0" (argx), "u" (argy));

	return value;
}

__MATH_INLINE double asin(const double x)
{
	register double arg = x;

	return atan2(arg,sqrt(1.0-arg*arg));
}

__MATH_INLINE double acos(const double x)
{
	register double arg = x;

	return atan2(sqrt(1.0-arg*arg),arg);
}

__MATH_INLINE double atan(const double x)
{
	register double value, arg = x;
	__asm__ __volatile__
		("fld1\n\t"
		 "fpatan"
		 : "=t" (value): "0" (arg));

	return value;
}

__MATH_INLINE double exp(const double x)
{
	register double value, arg = x, exponent;
	__asm__ __volatile__
		("fldl2e				# e^x = 2^(x * log2(e))\n\t"
		 "fmul	%%st(1)			# x * log2(e)\n\t"
		 "fstl	%%st(1)\n\t"
		 "frndint				# int(x * log2(e))\n\t"
		 "fxch\n\t"
		 "fsub	%%st(1)			# fract(x * log2(e))\n\t"
		 "f2xm1				# 2^(fract(x * log2(e))) - 1\n\t"
		 : "=t" (value), "=u" (exponent) : "0" (arg));
	value += 1.0;
	__asm__ __volatile__
		("fscale" : "=t" (value): "0" (value), "u" (exponent));
	return value;
}

__MATH_INLINE double sinh(const double x)
{
	register double arg = x, exm1 = __expm1(fabs(arg));

	return 0.5*(exm1/(exm1+1.0)+exm1)*__sgn1(arg);
}

__MATH_INLINE double cosh(const double x)
{
	register double ex = exp(x);

	return 0.5*(ex+1.0/ex);
}

__MATH_INLINE double tanh(const double x)
{
	register double arg = x, exm1 = __expm1(-fabs(arg+arg));

	return exm1/(exm1+2.0)*__sgn1(-arg);
}

__MATH_INLINE double log(const double x)
{
	register double value, arg = x ;
	__asm__ __volatile__
		("fldln2\n\t"
		 "fxch\n\t"
		 "fyl2x"
		 : "=t" (value) : "0" (arg));

 	return value;
}

__MATH_INLINE double log10(const double x)
{
	register double value, arg = x ;
	__asm__ __volatile__
		("fldlg2\n\t"
		 "fxch\n\t"
		 "fyl2x"
		 : "=t" (value) : "0" (arg));

	return value;
}

__MATH_INLINE double __log2(const double x)
{
	register double value, arg = x ;
	__asm__ __volatile__
		("fld1\n\t"
		 "fxch\n\t"
		 "fyl2x"
		 : "=t" (value) : "0" (arg));

 	return value;
}

__MATH_INLINE double fmod(const double x, const double y)
{
	register double value, value2, argx = x, argy = y;
	__asm__ __volatile__
		("1:	fprem\n\t"
		 "fstsw	%%ax\n\t"
		 "sahf\n\t"
		 "jp	1b"
		 : "=t" (value), "=u" (value2)
		 : "0" (argx), "1" (argy) : "ax", "cc");

	return value;
}

__MATH_INLINE double ldexp(const double x, const int y)
{
	register double value, argx = x, argy = (double)y;
	__asm__ __volatile__
		("fscale" : "=t" (value): "0" (argx), "u" (argy));

	return value;
}

__MATH_INLINE double pow(double x, const double y)
{
	register double value, argy = y, exponent;

	long p=(long) y;
	if(x == 0.0 && y > 0.0) return 0.0;
	if(y == (double) p) {
		double r=1.0;
		if (p == 0) return 1.0;
		if (p < 0) {p = -p; x = 1.0/x;}
		for(;;) {
			if (p & 1) r *= x;
			if ((p >>= 1) == 0) return r;
			x *= x;
		}
 	}
	__asm__ __volatile__
		("fmul	%%st(1)			# y * log2(x)\n\t"
		 "fstl	%%st(1)\n\t"
		 "frndint				# int(y * log2(x))\n\t"
		 "fxch\n\t"
		 "fsub	%%st(1)			# fract(y * log2(x))\n\t"
		 "f2xm1				# 2^(fract(y * log2(x))) - 1\n\t"
		 : "=t" (value), "=u" (exponent) :  "0" (__log2(x)), "1" (argy));
	value += 1.0;
	__asm__ __volatile__
		("fscale" : "=t" (value): "0" (value), "u" (exponent));
	return value;
}

__MATH_INLINE double floor(const double x)
{
	register double value, arg = x;
	volatile short cw, cwtmp;

	__asm__ volatile ("fnstcw %0" : "=m" (cw) : );
	cwtmp = (cw & 0xf3ff) | 0x0400; /* rounding down */
	__asm__ volatile ("fldcw %0" : : "m" (cwtmp));
	__asm__ volatile ("frndint" : "=t" (value) : "0" (arg));
	__asm__ volatile ("fldcw %0" : : "m" (cw));
	return value;
}

__MATH_INLINE double ceil(const double x)
{
	register double value, arg = x;
	volatile short cw, cwtmp;

	__asm__ volatile ("fnstcw %0" : "=m" (cw) : );
	cwtmp = (cw & 0xf3ff) | 0x0800; /* rounding up */
	__asm__ volatile ("fldcw %0" : : "m" (cwtmp));
	__asm__ volatile ("frndint" : "=t" (value) : "0" (arg));
	__asm__ volatile ("fldcw %0" : : "m" (cw));
	return value;
	
}

#ifdef __MATH_EXTENSIONS
#include <i386/extensions.h>
#endif

#endif /* __NO_MATH_INLINES */
#endif	/* __GNUC__	 */

#endif /* __MATH_H */

#ifndef M_PI
#define M_PI        3.14159265358979323846	/* pi */
#endif


#endif
#endif