#ifdef UI_NCURSES
/*
 *  cursesd.cc (C) 1995 Ralf W. Stephan <ralf@ark.franken.de>
 *  cursesd is derived from the dialog package
 *  Original AUTHOR of dialog : Savio Lam (lam836@cs.cuhk.hk)
 *     Stuart Herbert - S.Herbert@sheffield.ac.uk: radiolist extension
 *     Alessandro Rubini - rubini@ipvvis.unipv.it: merged checklist/radio
 *     Ralf W. Stephan <ralf@ark.franken.de>     : merged checklist/menubox
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
//----------------------------------------------------------------------------
// NOTE:  The cursesd source doesn't use any of the menu- or form-functions
// available with ncurses-1.9.4 and above.  It better should, as these
// libraries are very powerful and flexible.  So PLEASE see cursesd as
// an example what is possible without them, although a soon obsolete one. 
//
// Also, there is a problem with the textbox/inputbox combo when having
// input a search_string in textbox which I can't figure out.  This
// should easily disappear when we use forms for the inputbox.  RWS
//----------------------------------------------------------------------------
#pragma implementation
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include "cursesd.h"

/* 
 * Attribute values, default is for mono display
 */
static chtype attributes[] =
{
    A_NORMAL,			/* screen_attr */
    A_NORMAL,			/* shadow_attr */
    A_REVERSE,			/* dialog_attr */
    A_REVERSE,			/* title_attr */
    A_REVERSE,			/* border_attr */
    A_BOLD,			/* button_active_attr */
    A_DIM,			/* button_inactive_attr */
    A_UNDERLINE,		/* button_key_active_attr */
    A_UNDERLINE,		/* button_key_inactive_attr */
    A_NORMAL,			/* button_label_active_attr */
    A_NORMAL,			/* button_label_inactive_attr */
    A_REVERSE,			/* inputbox_attr */
    A_REVERSE,			/* inputbox_border_attr */
    A_REVERSE,			/* searchbox_attr */
    A_REVERSE,			/* searchbox_title_attr */
    A_REVERSE,			/* searchbox_border_attr */
    A_REVERSE,			/* position_indicator_attr */
    A_REVERSE,			/* menubox_attr */
    A_REVERSE,			/* menubox_border_attr */
    A_REVERSE,			/* item_attr */
    A_NORMAL,			/* item_selected_attr */
    A_REVERSE,			/* tag_attr */
    A_REVERSE,			/* tag_selected_attr */
    A_NORMAL,			/* tag_key_attr */
    A_BOLD,			/* tag_key_selected_attr */
    A_REVERSE,			/* check_attr */
    A_REVERSE,			/* check_selected_attr */
    A_REVERSE,			/* uarrow_attr */
    A_REVERSE			/* darrow_attr */
};

/*
 * Table of color values
 */
static int color_table[][3] =
{
  {6, 4, 1},  {0, 0, 1},  {0, 7, 0},  {3, 7, 1},  {7, 7, 1},  {7, 4, 1},
  {0, 7, 0},  {7, 4, 1},  {1, 7, 0},  {3, 4, 1},  {0, 7, 1},  {0, 7, 0},
  {0, 7, 0},  {0, 7, 0},  {3, 7, 1},  {7, 7, 1},  {3, 7, 1},  {0, 7, 0},
  {7, 7, 1},  {0, 7, 0},  {7, 4, 1},  {3, 7, 1},  {3, 4, 1},  {1, 7, 1},
  {1, 4, 1},  {0, 7, 0},  {7, 4, 1},  {2, 7, 1},  {2, 7, 1},
};

enum NCD_attr
{ 
  screen_attr=0, shadow_attr, dialog_attr, title_attr, border_attr, 
  button_active_attr, button_inactive_attr, button_key_active_attr, 
  button_key_inactive_attr, button_label_active_attr, 
  button_label_inactive_attr, inputbox_attr, inputbox_border_attr, 
  searchbox_attr, searchbox_title_attr, searchbox_border_attr, 
  position_indicator_attr, menubox_attr, menubox_border_attr, item_attr, 
  item_selected_attr, tag_attr, tag_selected_attr, tag_key_attr, 
  tag_key_selected_attr, check_attr, check_selected_attr, uarrow_attr, 
  darrow_attr, 
  attribute_count 
};

//----------------------------NCursesDialog------------------------------

NCursesDialog::NCursesDialog (int x, int y, int h, int w, const char* t)
: NCursesPanel(h,w,y,x),
  height_(h), width_(w), str_max_len_(2048), title_(t)
{
  keypad (TRUE);
  cbreak ();
  noecho ();
  
  color_setup ();
  draw_box (0,0,width_,height_,dialog_attr,border_attr,title_);
  refresh ();
}

NCursesDialog::~NCursesDialog()
{
  nocbreak();
  leaveok(TRUE);
  echo();
}

//-----------------------------private------------------------------------
static inline chtype a (NCD_attr att) { return attributes[att]; }

/*
 * Setup for color display
 */
void NCursesDialog::color_setup ()
{
  if (!has_colors ()) return;
  
  start_color ();

  /* Initialize color pairs */
  for (int i = 0; i < attribute_count; i++)
    init_pair (i + 1, color_table[i][0], color_table[i][1]);

  /* Setup color attributes */
  for (int i = 0; i < attribute_count; i++)
    attributes[i] = (color_table[i][2]? A_BOLD : 0) | COLOR_PAIR(i + 1);
}

//----------------------------protected------------------------------------
/*
 * Draw a rectangular box with line drawing characters
 */
// Could be optimized by first clear()ing then walking around the box.  RWS
//
void NCursesDialog::draw_box (int x, int y, int w, int h, int i, int o,
                              const char* title=0)
{
  const chtype border = a(NCD_attr(o));
  const chtype box = a(NCD_attr(i));
  
  attrset (0);
  for (int i = 0; i < h; i++) {
    move (i+y,x);
    for (int j = 0; j < w; j++)
      if (!i && !j)
        addch (border | ACS_ULCORNER);
      else if (i == h - 1 && !j)
        addch (border | ACS_LLCORNER);
      else if (!i && j == w - 1)
        addch (box | ACS_URCORNER);
      else if (i == h - 1 && j == w - 1)
        addch (box | ACS_LRCORNER);
      else if (!i)
        addch (border | ACS_HLINE);
      else if (i == h - 1)
        addch (box | ACS_HLINE);
      else if (!j)
        addch (border | ACS_VLINE);
      else if (j == w - 1)
        addch (box | ACS_VLINE);
      else
        addch (box | ' ');
    }

  if (title)
  {
    attrset (a(title_attr));
    move (0,(w-strlen(title_))/2-1);
    addstr (title);
  }
}

/*
 * Print a string of text in a window, automatically wrap around to the
 * next line if the string is too long to fit on one line. Note that the
 * string may contain "\n" to represent a newline character or the real
 * newline '\n', but in that case, auto wrap around will be disabled.
 */
void NCursesDialog::print_autowrap 
(const char *prompt, int width, int y, int x)
{
  if (strlen(prompt) >= str_max_len_)
  { cerr << "NCursesDialog::str_max_len_ exceeded.\n"; exit(1); } // throw
  char tempstr [str_max_len_ + 1];
  strcpy (tempstr, prompt);

  if (strstr (tempstr, "\\n") || strchr (tempstr, '\n')) 
  {                                       // Prompt contains "\n" or '\n' 
    char* word = tempstr;
    int cur_y = y;
    move (cur_y, x);
    
    while (1) 
    {
      char* bnl_pos = strstr (word, "\\n");
      char* nl_pos = strchr (word, '\n');
      
      if (!bnl_pos && !nl_pos)
	break;
      else if (!bnl_pos) 
      {		                          // No more "\n" 
	bnl_pos = nl_pos;
	*bnl_pos = '\0';
      } 
      else if (!nl_pos) 
	*bnl_pos++ = '\0';                // No more '\n' 
      else 
      {                                   // Prompt contains both "\n" and '\n'
        if (strlen(bnl_pos) <= strlen(nl_pos)) 
        {
	  bnl_pos = nl_pos;
	  *bnl_pos = '\0';
	} 
	else 
	  *bnl_pos++ = '\0';
      }

      addstr (word);
      word = bnl_pos + 1;
      move (++cur_y, x);
    }
    
    addstr (word);
    } 
    else if (strlen (tempstr) <= width - x * 2) 
    {	                                    // If prompt is short 
      move (y, (width - strlen (tempstr)) / 2);
      addstr (tempstr);
    } 
    else 
    {
      int cur_x = x, cur_y = y;             // Print prompt word by word, wrap 
      bool first = true;                    // around if necessary 
      char *word;
      while ((word = strtok (first ? tempstr : 0, " ")) != 0) 
      {
        if (first)		            // First iteration 
          first = 0;
        if (cur_x + strlen (word) > width) {
	cur_y++;	                    // wrap to next line 
	cur_x = x;
      }
      move (cur_y, cur_x);
      addstr (word);
      getyx (cur_y, cur_x);
      cur_x++;
    }
  }
}

void NCursesDialog::button_bar()
{
  attrset (a(border_attr));
  move (height_ - 3, 0);
  addch (ACS_LTEE);
  for (int i = 0; i < width_ - 2; i++)
    addch (ACS_HLINE);
  attrset (a(dialog_attr));
  addch (ACS_RTEE);
  move (height_ - 2, 1);
  for (int i = 0; i < width_ - 2; i++)
    addch (' ');
}

/*
 * Print a button
 */
void NCursesDialog::print_button 
(const char *label, int y, int x, bool selected)
{
  move (y, x);
  attrset (selected ? a(button_active_attr) : a(button_inactive_attr));
  addstr ("<");
  int temp = strspn (label, " ");
  label += temp;
  attrset (selected ? a(button_label_active_attr) : a(button_label_inactive_attr));
  for (int i = 0; i < temp; i++)
    addch (' ');
  attrset (selected ? a(button_key_active_attr) : a(button_key_inactive_attr));
  addch (label[0]);
  attrset (selected ? a(button_label_active_attr) : a(button_label_inactive_attr));
  addstr (label + 1);
  attrset (selected ? a(button_active_attr) : a(button_inactive_attr));
  addstr (">");
  move (y, x + temp + 1);
}

//---------------------------NCmsgbox-------------------------------
/*
 * Display a message box. Program will pause and display an "OK" button
 * if the parameter 'pause' is non-zero.
 */

NCmsgbox::NCmsgbox (const char *title, const char *prompt, int x, int y,
                    int height, int width, bool pause=true)
: NCursesDialog (x,y,height,width,title)
{
  keypad (TRUE);
  attrset(a(dialog_attr));
  print_autowrap (prompt, width - 2, 1, 2);

  if (pause)
  {
    button_bar();
    print_button ("  OK  ", height - 2, width / 2 - 4, true);
  } 

  refresh();
}

NCmsgbox::~NCmsgbox() {}

bool NCmsgbox::getkey()
{
  int key=0;
  while (key != 0x1b && key != 0x0d && key != ' ')
    key = getch ();
  return key != 0x1b;
}

//-----------------------------NCyesno---------------------------------
/*
 * Display a dialog box with two buttons - Yes and No
 */
NCyesno::NCyesno (const char *title, const char *prompt, int x, int y,
                    int height, int width)
: NCursesDialog (x,y,height,width,title)
{
  keypad (TRUE);
  button_bar();
  attrset(a(dialog_attr));
  print_autowrap (prompt, width - 2, 1, 3);

  int x = width / 2 - 10;
  int y = height - 2;
  print_button ("  No  ", y, x + 13, FALSE);
  print_button (" Yes ", y, x, TRUE);
  refresh ();

  int button = 0;
  while (1) 
  {
    key_ = getch ();
    switch (key_) 
    {
      case 'Y':
      case 'y': key_=0; return;
      case 'N':
      case 'n': key_=1; return;
      case 0x09:
      case KEY_UP:
      case KEY_DOWN:
      case KEY_LEFT:
      case KEY_RIGHT:
        if (!button) 
        {
	  button = 1;	/* "No" button selected */
	  print_button (" Yes ", y, x, FALSE);
	  print_button ("  No  ", y, x + 13, TRUE);
        } 
        else 
        {
	  button = 0;	/* "Yes" button selected */
	  print_button ("  No  ", y, x + 13, FALSE);
	  print_button (" Yes ", y, x, TRUE);
        }
        refresh ();
        break;
      case ' ':
      case 0x0a:
      case 0x0d: key_=button; return;
      case 0x1b:  key_=-1; return;
    }
  }
}

NCyesno::~NCyesno() {}



//==========================NCchecklist======================================

NCchecklist::NCchecklist 
(const char *title, const char *prompt, int x, int y, int height, int width,
 int list_height, int item_no, const char * const * items, 
 bool radio_flag=false, bool menu_flag=false)
 : NCursesDialog (x,y,height,width,title), list_(0)
{
  int key = 0, button = 0, choice = -1, scroll = 0;

  int status [item_no];
  choices_ = new bool[item_no];
  for (int i = 0; i < item_no; i++)
  {
    bool b = !strcasecmp (items[i * 3 + 2], "on");
    choices_[i] = b;
    status[i] = b;
    if (choices_[i] && choice<0)
      choice_ = choice = i;
  }
  if (!radio_flag && !menu_flag)
    choice_ = choice = 0;

  int max_choice = list_height <? item_no;

  keypad (TRUE);
  button_bar();
  attrset (a(dialog_attr));
  print_autowrap (prompt, width - 2, 1, 3);

  int list_width = width - 6;
  int cur_x, cur_y;
  getyx (cur_y, cur_x);
  int box_y = cur_y + 1;
  int box_x = (width - list_width) / 2 - 1;

  /* create new window for the list */
  list_ = new NCursesWindow (*this, list_height, list_width,
                                           y + box_y + 1, x + box_x + 1);
  list_->keypad (TRUE);

  /* draw a box around the list items */
  draw_box (box_x, box_y, list_width + 2, list_height + 2,
      menubox_border_attr, menubox_attr);

  check_x = 0;
  item_x = 0;
  /* Find length of longest item in order to center checklist */
  for (int i = 0; i < item_no; i++) 
  {
    check_x = (check_x >? strlen (items[i * 3]) 
              + strlen (items[i * 3 + 1]) + 6);
    item_x = (item_x >? strlen (items[i * 3]));
  }
  check_x = (list_width - check_x) / 2;
  item_x = check_x + item_x + 6;

  /* Print the list */
  for (int i = 0; i < max_choice; i++)
    print_item (list_, items[i * 3], items[i * 3 + 1],
                status[i], i, i == choice, radio_flag, menu_flag);
  list_->noutrefresh ();

  if (list_height < item_no) 
  {
    attrset (a(darrow_attr));
    move (box_y + list_height + 1, box_x + check_x + 5);
    addch (ACS_DARROW);
    move (box_y + list_height + 1, box_x + check_x + 6);
    addstr ("(+)");
  }
  x = width / 2 - 11;
  y = height - 2;
  print_button ("Cancel", y, x + 14, FALSE);
  print_button ("  OK  ", y, x, TRUE);
  refresh ();

  while (key != 0x1b) 
  {
    key = getch ();
    /* Check if key pressed matches first character of
       any item tag in list */
    int tmp=max_choice;
    for (int i = 0; i < max_choice; i++)
      if (toupper (key) == toupper (items[(scroll + i) * 3][0]))
        { tmp=i; break; }

    if (tmp < max_choice ||
      (key >= '1' && key <= ('9' <? ('0' + max_choice))) ||
      key == KEY_UP || key == KEY_DOWN || key == ' ' ||
      key == '+' || key == '-' )
    {
      if (key >= '1' && key <= ('9' <? ('0' + max_choice)))
        tmp = key - '1';
      else if (key == KEY_UP || key == '-') 
      {
        if (!choice) 
        {
          if (!scroll)
            continue;
            
          /* Scroll list down */
          getyx (cur_y, cur_x);
          
          if (list_height > 1) 
          {
          /* De-highlight current first item */
            print_item (list_, items[scroll * 3],
              items[scroll * 3 + 1], status[scroll],
              0, FALSE, radio_flag, menu_flag);
            list_->scrollok (TRUE);
            list_->scrl (-1);
            list_->scrollok (FALSE);
          }
          scroll--;
          print_item (list_, items[scroll * 3],
            items[scroll * 3 + 1],
            status[scroll], 0, TRUE, radio_flag, menu_flag);
          list_->noutrefresh ();

          /* print the up/down arrows */
          move (box_y, box_x + check_x + 5);
          attrset (scroll ? a(uarrow_attr) : a(menubox_attr));
          addch (scroll ? ACS_UARROW : ACS_HLINE);
          move (box_y, box_x + check_x + 6);
          addch (scroll ? '(' : ACS_HLINE);
          move (box_y, box_x + check_x + 7);
          addch (scroll ? '-' : ACS_HLINE);
          move (box_y, box_x + check_x + 8);
          addch (scroll ? ')' : ACS_HLINE);
          attrset (a(darrow_attr));
          move (box_y + list_height + 1, box_x + check_x + 5);
          addch (ACS_DARROW);
          move (box_y + list_height + 1, box_x + check_x + 6);
          addch ('(');
          move (box_y + list_height + 1, box_x + check_x + 7);
          addch ('+');
          move (box_y + list_height + 1, box_x + check_x + 8);
          addch (')');
          move (cur_y, cur_x);
          refresh ();
          continue;  /* wait for another key press */
        } 
        else
          tmp = choice - 1;
      } 
      else if (key == KEY_DOWN || key == '+') 
      {
        if (choice == max_choice - 1) 
        {
          if (scroll + choice >= item_no - 1)
            continue;
          /* Scroll list up */
          getyx (cur_y, cur_x);
          if (list_height > 1) 
          {
          /* De-highlight current last item before scrolling up */
            print_item (list_, items[(scroll + max_choice - 1) * 3],
              items[(scroll + max_choice - 1) * 3 + 1],
              status[scroll + max_choice - 1], max_choice - 1, FALSE, 
              radio_flag, menu_flag);
            list_->scrollok (TRUE);
            list_->scroll ();
            list_->scrollok (FALSE);
          }
          scroll++;
          print_item (list_, items[(scroll + max_choice - 1) * 3],
              items[(scroll + max_choice - 1) * 3 + 1],
              status[scroll + max_choice - 1], max_choice - 1, TRUE, 
              radio_flag, menu_flag);
          list_->noutrefresh ();

          /* print the up/down arrows */
          attrset (a(uarrow_attr));
          move (box_y, box_x + check_x + 5);
          addch (ACS_UARROW);
          move (box_y, box_x + check_x + 6);
          addstr ("(-)");
          move (box_y + list_height + 1, box_x + check_x + 5);
          attrset (scroll + choice < item_no - 1 ?
              a(darrow_attr) : a(menubox_border_attr));
          addch (scroll + choice < item_no - 1 ? ACS_DARROW : ACS_HLINE);
          move (box_y + list_height + 1, box_x + check_x + 6);
          addch (scroll + choice < item_no - 1 ? '(' : ACS_HLINE);
          move (box_y + list_height + 1, box_x + check_x + 7);
          addch (scroll + choice < item_no - 1 ? '+' : ACS_HLINE);
          move (box_y + list_height + 1, box_x + check_x + 8);
          addch (scroll + choice < item_no - 1 ? ')' : ACS_HLINE);
          move (cur_y, cur_x);
          refresh ();
          continue;  /* wait for another key press */
        } 
        else
          tmp = choice + 1;
      } 
      else if (key == ' ') 
      {  /* Toggle item status */
        if (menu_flag) continue;
        if (!radio_flag) 
        {
          status[scroll + choice] = !status[scroll + choice];
          getyx (cur_y, cur_x);
          list_->move (choice, check_x);
          list_->attrset (a(check_selected_attr));
          list_->printw ("[%c]", status[scroll + choice] ? 'X' : ' ');
        } 
        else 
        {
          if (!status[scroll + choice]) 
          {
            for (int i = 0; i < item_no; i++)
              status[i] = 0;
            status[scroll + choice] = 1;
            getyx (cur_y, cur_x);
            for (int i = 0; i < max_choice; i++)
              print_item (list_, items[(scroll + i) * 3],
                items[(scroll + i) * 3 + 1],
                status[scroll + i], i, i == choice, radio_flag, menu_flag);
          }
        }
        list_->noutrefresh ();
        move (cur_y, cur_x);
        refresh ();
        continue;  /* wait for another key press */
      }
      if (tmp != choice) 
      {
        /* De-highlight current item */
        getyx (cur_y, cur_x);
        print_item (list_, items[(scroll + choice) * 3],
            items[(scroll + choice) * 3 + 1],
            status[scroll + choice], choice, FALSE, radio_flag, menu_flag);
        /* Highlight new item */
        choice = tmp;
        print_item (list_, items[(scroll + choice) * 3],
            items[(scroll + choice) * 3 + 1],
            status[scroll + choice], choice, TRUE, radio_flag, menu_flag);
        list_->noutrefresh ();
        move (cur_y, cur_x);
        refresh ();
      }
      continue;    /* wait for another key press */
    }
    switch (key) 
    {
    case 'C':
    case 'c':
      key_=0;
      return;
    case 0x09:
    case KEY_LEFT:
    case KEY_RIGHT:
      if (!button) 
      {
        button = 1;  /* "Cancel" button selected */
        print_button ("  OK  ", y, x, FALSE);
        print_button ("Cancel", y, x + 14, TRUE);
      } 
      else 
      {
        button = 0;  /* "OK" button selected */
        print_button ("Cancel", y, x + 14, FALSE);
        print_button ("  OK  ", y, x, TRUE);
      }
      refresh ();
      break;
    case 'O':
    case 'o':
    case 0x0d:
      for (int i = 0; i < item_no; i++)
      {
        if (status[i]) 
          choice_=i;
        choices_[i] = (status[i]!=0);
      }
      if (menu_flag) choice_=choice+scroll;
      key_=!button;
      return;
    case 0x1b:
      key_=-1;
      return;
    }
  }

  key_=-1;
  return;      /* ESC pressed */
}

//-----------------------------privates------------------------------
/*
 * Print menu item
 */
void NCchecklist::print_item 
(NCursesWindow * win, const char *tag, const char *item, int status,
 int choice, bool selected, bool radio_flag, bool menu_flag)
{
  win->attrset (a(menubox_attr));
  win->move (choice, 0);
  for (int i = 0; i <= win->maxx(); i++)
    win->addch (' ');
  win->move (choice, check_x);
  if (!menu_flag)
  {
    win->attrset (selected ? a(check_selected_attr) : a(check_attr));
    if (!radio_flag)
      win->printw ("[%c]", status ? 'X' : ' ');
    else
      win->printw ("(%c)", status ? 'X' : ' ');
    win->attrset (a(menubox_attr));
    win->addch (' ');
  }
  win->attrset (selected ? a(tag_key_selected_attr) : a(tag_key_attr));
  win->addch (tag[0]);
  win->attrset (selected ? a(tag_selected_attr) : a(tag_attr));
  win->addstr (tag + 1);
  win->move (choice, item_x);
  win->attrset (selected ? a(item_selected_attr) : a(item_attr));
  win->addstr (item);
}

//============================NCradiolist================================
NCradiolist::~NCradiolist() {}

//============================NCmenubox==================================
NCmenubox::~NCmenubox() {}

//============================NCinputbox================================
NCinputbox::NCinputbox (const char *title, const char *prompt, int x, int y, 
                        int height, int width, const char *init)
: NCursesDialog(x,y,height,width,title), str_max_len_(2047)
{
  key_ = 0;
  str_ = new char[str_max_len_+1];
  char *instr = str_;
  int button=-1, scroll=0;

  keypad (TRUE);
  leaveok(FALSE);
  button_bar();
  attrset (a(dialog_attr));
  print_autowrap (prompt, width - 2, 1, 3);

  /* Draw the input field box */
  int box_width = width - 6;
  int x,y;
  getyx (y, x);
  int box_y = y + 2;
  int box_x = (width - box_width) / 2;
  draw_box (box_x - 1, y+1, box_width + 2, 3, border_attr, dialog_attr);

  x = width / 2 - 11;
  y = height - 2;
  print_button ("Cancel", y, x + 14, FALSE);
  print_button ("  OK  ", y, x, TRUE);

  /* Set up the initial value */
  move (box_y, box_x);
  attrset (a(inputbox_attr));
  if (!init)
    instr[0] = '\0';
  else
    strcpy (instr, init);
  int input_x = strlen (instr);
  if (input_x >= box_width) 
  {
    int scroll = input_x - box_width + 1;
    input_x = box_width - 1;
    for (int i = 0; i < box_width - 1; i++)
      addch (instr[scroll + i]);
  } 
  else
    addstr (instr);
  move (box_y, box_x + input_x);
  refresh ();

  int key=0;
  while (key != 0x1b) 
  {
    key = getch ();

    if (button == -1) 
    {	/* Input box selected */
      switch (key) 
      {
      case 0x09:
      case KEY_UP:
      case KEY_DOWN:
        break;
      case KEY_LEFT:
      case KEY_RIGHT:
        continue;
      case KEY_BACKSPACE:
      case 127:
        if (input_x || scroll) 
        {
          attrset (a(inputbox_attr));
          if (!input_x) 
          {
            scroll = scroll < box_width - 1 ? 0 : scroll - (box_width - 1);
            move (box_y, box_x);
            for (int i = 0; i < box_width; i++)
            addch (instr[scroll + input_x + i] ?
                   instr[scroll + input_x + i] : ' ');
            input_x = strlen (instr) - scroll;
          } 
          else
            input_x--;
          
          instr[scroll + input_x] = '\0';
          move (box_y, input_x + box_x);
          addch (' ');
          move (box_y, input_x + box_x);
          refresh ();
        }
        continue;
      default:
        if (key < 0x100 && isprint (key) && key!=0x0d) 
        {
          if (scroll + input_x < str_max_len_) 
          {
            attrset (a(inputbox_attr));
            instr[scroll + input_x] = key;
            instr[scroll + input_x + 1] = '\0';
            if (input_x == box_width - 1) 
            {
              scroll++;
              move (box_y, box_x);
              for (int i = 0; i < box_width - 1; i++)
                addch (instr[scroll + i]);
            } 
            else 
            {
              move (box_y, input_x++ + box_x);
              addch (key);
            }
            refresh ();
          } 
          else
            flash ();  /* Alarm user about overflow */
          continue;
        }
      }
    }

    switch (key) 
    {
    case 'O':
    case 'o':
      key_=1;
      return;
    case 'C':
    case 'c':
      key_=0;
      return;
    case KEY_UP:
    case KEY_LEFT:
      switch (button) 
      {
      case -1:
        button = 1;  /* Indicates "Cancel" button is selected */
        print_button ("  OK  ", y, x, FALSE);
        print_button ("Cancel", y, x + 14, TRUE);
        refresh ();
        break;
      case 0:
        button = -1;  /* Indicates input box is selected */
        print_button ("Cancel", y, x + 14, FALSE);
        print_button ("  OK  ", y, x, TRUE);
        move (box_y, box_x + input_x);
        refresh ();
        break;
      case 1:
        button = 0;  /* Indicates "OK" button is selected */
        print_button ("Cancel", y, x + 14, FALSE);
        print_button ("  OK  ", y, x, TRUE);
        refresh ();
        break;
      }
      break;
    case 0x09:
    case KEY_DOWN:
    case KEY_RIGHT:
      switch (button) 
      {
      case -1:
        button = 0;  /* Indicates "OK" button is selected */
        print_button ("Cancel", y, x + 14, FALSE);
        print_button ("  OK  ", y, x, TRUE);
        refresh ();
        break;
      case 0:
        button = 1;  /* Indicates "Cancel" button is selected */
        print_button ("  OK  ", y, x, FALSE);
        print_button ("Cancel", y, x + 14, TRUE);
        refresh ();
        break;
      case 1:
        button = -1;  /* Indicates input box is selected */
        print_button ("Cancel", y, x + 14, FALSE);
        print_button ("  OK  ", y, x, TRUE);
        move (box_y, box_x + input_x);
        refresh ();
        break;
      }
      break;
    case ' ':
    case 0x0d:
      key_= (button == -1 ? 1 : button);
      return;
    case 0x1b:
      break;
    }
  }

  key_=-1;
  return;      /* ESC pressed */
}


//============================NCtextbox==================================
NCtextbox::NCtextbox (const char* title, const char* file, int x, int y, 
                      int height, int width)
: NCursesDialog (x,y,height,width,title), text_(0), str_max_len_(2047),
  buf_size_(10*1024), hscroll_(0), 
  page_(0), buf_(0), line_(0), begin_reached_(true), end_reached_(false)
{
  /* Open input file for reading */
  if ((fd_ = open (file, O_RDONLY)) == -1) 
  {
    endwin ();
    fprintf (stderr,
     "\nCan't open input file %s in dialog_textbox().\n",file);
    exit (-1);
  }
  /* Get file size. Actually, 'file_size' is the real file size - 1,
     since it's only the last byte offset from the beginning */
  if ((file_size_ = lseek (fd_, 0, SEEK_END)) == -1) 
  {
    endwin ();
    fprintf (stderr, "\nError getting file size in dialog_textbox().\n");
    exit (-1);
  }
  /* Restore file pointer to beginning of file after getting file size */
  if (lseek (fd_, 0, SEEK_SET) == -1) 
  {
    endwin ();
    fprintf (stderr, "\nError moving file pointer in dialog_textbox().\n");
    exit (-1);
  }
  /* Allocate space for read buffer */
  buf_ = new char[buf_size_];
  line_ = new char[str_max_len_+1];
  if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
  {
    endwin ();
    fprintf (stderr, "\nError reading file in dialog_textbox().\n");
    exit (-1);
  }
  buf_[bytes_read_] = '\0';  /* mark end of valid data */
  page_ = buf_;    /* page is pointer to start of page to be displayed */
  char search_term[str_max_len_ + 1];
  search_term[0] = '\0';  /* no search term entered yet */

  keypad (TRUE);

  /* Create window for text region, used for scrolling text */
  text_ = new NCursesWindow (*this, height - 4, width - 2, y + 1, x + 1);
  text_->keypad (TRUE);

  button_bar();
  print_button (" EXIT ", height - 2, width / 2 - 4, TRUE);
  noutrefresh ();
  int cur_x, cur_y;
  getyx (cur_y, cur_x);  /* Save cursor position */

  /* Print first page of text */
//  text_->attr_clear (height - 4, width - 2, a(dialog_attr));
  text_->attrset (a(dialog_attr));
  print_page (height - 4, width - 2);
  print_position (height, width);
  move (cur_y, cur_x);  /* Restore cursor position */
  refresh ();

  int key=0, fpos;
  while ((key != 0x1b) && (key != 0x0d)) 
  {
    key = getch ();
    switch (key) 
    {
      case 'E':  /* Exit */
      case 'e':
      case 0x0d:
        close (fd_);
        key_=0;
        return;
      case 'g':  /* First page */
      case KEY_HOME:
        if (!begin_reached_) 
        {
          begin_reached_ = true;
          /* First page not in buffer? */
          if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
          {
            endwin ();
            fprintf (stderr,
              "\nError moving file pointer in dialog_textbox().\n");
            exit (-1);
          }
          if (fpos > bytes_read_)
          {
            if (lseek (fd_, 0, SEEK_SET) == -1) 
            { 
              endwin ();
              fprintf (stderr, "\nError moving file pointer in "
               "dialog_textbox().\n");
              exit (-1);
            }
            if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
            {
              endwin ();
              fprintf (stderr,
                 "\nError reading file in dialog_textbox().\n");
              exit (-1);
            }
            buf_[bytes_read_] = '\0';
          }
          page_ = buf_;
          print_page (height - 4, width - 2);
          print_position (height, width);
          move (cur_y, cur_x);  /* Restore cursor position */
          refresh ();
        }
        break;
      case 'G':  /* Last page */
      case KEY_END:
        end_reached_ = true;
        /* Last page not in buffer? */
        if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) {
          endwin ();
          fprintf (stderr,
              "\nError moving file pointer in dialog_textbox().\n");
          exit (-1);
        }
        if (fpos < file_size_) {  /* Yes, we have to read it in */
          if (lseek (fd_, -buf_size_, SEEK_END) == -1) 
          {
            endwin ();
            fprintf (stderr,
              "\nError moving file pointer in dialog_textbox().\n");
            exit (-1);
          }
          if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
          {
            endwin ();
            fprintf (stderr,
               "\nError reading file in dialog_textbox().\n");
            exit (-1);
          }
          buf_[bytes_read_] = '\0';
        }
        page_ = buf_ + bytes_read_;
        back_lines (height - 4);
        print_page (height - 4, width - 2);
        print_position (height, width);
        move (cur_y, cur_x);  /* Restore cursor position */
        refresh ();
        break;
      case 'K':  /* Previous line */
      case 'k':
      case KEY_UP:
        if (!begin_reached_) 
        {
          back_lines (page_length_ + 1);
          /* We don't call print_page() here but use scrolling to ensure
             faster screen update. However, 'end_reached_' and
             'page_length_' should still be updated, and 'page' should
             point to start of next page. This is done by calling
             get_line() in the following 'for' loop. */
          text_->scrollok (TRUE);
          text_->scrl (-1);  /* Scroll text region down one line */
          text_->scrollok (FALSE);
          page_length_ = 0;
          bool passed_end = false;
          for (int i = 0; i < height - 4; i++) 
          {
            if (!i) 
            {
              /* print first line of page */
              print_line (0, width - 2);
              text_->noutrefresh ();
            } 
            else
            /* Called to update 'end_reached_' and 'page' */
              get_line ();
            if (!passed_end)
              page_length_++;
            if (end_reached_ && !passed_end)
              passed_end = true;
          }
          print_position (height, width);
          move (cur_y, cur_x);  /* Restore cursor position */
          refresh ();
        }
        break;
      case 'B':  /* Previous page */
      case 'b':
      case KEY_PPAGE:
        if (begin_reached_)
      break;
        back_lines (page_length_ + height - 4);
        print_page (height - 4, width - 2);
        print_position (height, width);
        move (cur_y, cur_x);
        refresh ();
        break;
      case 'J':  /* Next line */
      case 'j':
      case KEY_DOWN:
        if (!end_reached_) 
        {
          begin_reached_ = false;
          text_->scrollok (TRUE);
          text_->scroll ();  /* Scroll text region up one line */
          text_->scrollok (FALSE);
          print_line (height - 5, width - 2);
          text_->move (height - 5, 0);
          text_->addch (' ');
          text_->move (height - 5, width - 3);
          text_->addch (' ');
          text_->noutrefresh ();
          print_position (height, width);
          move (cur_y, cur_x);  /* Restore cursor position */
          refresh ();
        }
        break;
      case ' ':  /* Next page */
      case KEY_NPAGE:
        if (end_reached_)
          break;
        begin_reached_ = false;
        print_page (height - 4, width - 2);
        print_position (height, width);
        move (cur_y, cur_x);
        refresh ();
        break;
      case '0':  /* Beginning of line */
      case 'H':  /* Scroll left */
      case 'h':
      case KEY_LEFT:
        if (hscroll_ <= 0)
          break;
        if (key == '0')
          hscroll_ = 0;
        else
          hscroll_--;
        /* Reprint current page to scroll horizontally */
        back_lines (page_length_);
        print_page (height - 4, width - 2);
        move (cur_y, cur_x);
        refresh ();
        break;
      case 'L':  /* Scroll right */
      case 'l':
      case KEY_RIGHT:
        if (hscroll_ >= str_max_len_)
          break;
        hscroll_++;
        /* Reprint current page to scroll horizontally */
        back_lines (page_length_);
        print_page (height - 4, width - 2);
        move (cur_y, cur_x);
        refresh ();
        break;
      case '/':  /* Forward search */
      case 'n':  /* Repeat forward search */
      case '?':  /* Backward search */
      case 'N':  /* Repeat backward search */
        /* set search direction */
        bool dir = (key == '/' || key == 'n');
        if (dir ? !end_reached_ : !begin_reached_) 
        {
          if (key == 'n' || key == 'N') 
          {
            if (search_term[0] == '\0') 
            {  /* No search term yet */
              fprintf (stderr, "\a");  /* beep */
              break;
            }
          } 
          else
          /* Get search term from user */
          if (get_search_term (search_term, height - 4,
                               width - 2) == -1) 
               break; 

          /* Save variables for restoring in case search term
             can't be found */
          char* tempptr = page_;
          bool temp = begin_reached_;
          bool temp1 = end_reached_;
          if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
          {
            endwin ();
            fprintf (stderr, "\nError moving file pointer in "
               "dialog_textbox().\n");
            exit (-1);
          }
          fpos -= bytes_read_;
          /* update 'page' to point to next (previous) line before
             forward (backward) searching */
          back_lines (dir ? page_length_ - 1 : page_length_ + 1);
          char* found = NULL;
          if (dir)  /* Forward search */
            while ((found = strstr (get_line (), search_term)) == NULL) 
            {
              if (end_reached_)
                break;
            } 
          else  /* Backward search */
            while ((found = strstr (get_line (), search_term)) == NULL) 
            {
              if (begin_reached_)
                break;
              back_lines (2);
            }
          if (found == NULL) 
          {  /* not found */
            fprintf (stderr, "\a");  /* beep */
            /* Restore program state to that before searching */
            if (lseek (fd_, fpos, SEEK_SET) == -1) 
            {
              endwin ();
              fprintf (stderr, "\nError moving file pointer in "
               "dialog_textbox().\n");
              exit (-1);
            }
            if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
            {
              endwin ();
              fprintf (stderr, "\nError reading file in "
               "dialog_textbox().\n");
              exit (-1);
            }
            buf_[bytes_read_] = '\0';
            page_ = tempptr;
            begin_reached_ = temp;
            end_reached_ = temp1;
            /* move 'page' to point to start of current page in order to
               re-print current page. Note that 'page' always points to
               start of next page, so this is necessary */
            back_lines (page_length_);
          } 
          else  /* Search term found */
            back_lines (1);
          /* Reprint page */
          text_->attrset (a(dialog_attr));
          print_page (height - 4, width - 2);
          if (found != NULL)
            print_position (height, width);
          move (cur_y, cur_x);  /* Restore cursor position */
          refresh ();
        } 
        else  /* no need to find */
          fprintf (stderr, "\a");  /* beep */
        break;
//      case 0x1b:
//        break;
    }
  }

  close (fd_);
  key_=-1;
  return;    /* ESC pressed */
}

//--------------------------------privates----------------------------------

void NCtextbox::print_page (int height, int width)
{
    bool passed_end = false;
    page_length_ = 0;
    for (int i = 0; i < height; i++) {
	print_line (i, width);
	if (!passed_end)
	    page_length_++;
	if (end_reached_ && !passed_end)
	    passed_end = true;
    }
    text_->noutrefresh();
}

void NCtextbox::print_line (int row, int width)
{
    char* line = get_line ();
    line += (strlen (line) <? hscroll_);	/* Scroll horizontally */
    text_->move (row, 0);	/* move cursor to correct line */
    text_->addch (' ');
    text_->addnstr (line, (strlen (line) <? (width - 2)));
    
    int x,y;
    text_->getyx (y, x);
    /* Clear 'residue' of previous line */
    for (int i = 0; i < width - x; i++)
	text_->addch (' ');
}

/*
 * Return current line of text. Called by dialog_textbox() and print_line().
 * 'page' should point to start of current line before calling, and will be
 * updated to point to start of next line.
 */
char * NCtextbox::get_line (void)
{
  end_reached_ = false;
  int i=0;
  while (*page_ != '\n') 
  {
    if (*page_ == '\0') 
    {
    /* Either end of file or end of buffer reached_ */
      int fpos;
      if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
      {
        endwin ();
	fprintf (stderr, "\nError moving file pointer in "
			 "get_line().\n");
	exit (-1);
      }
      if (fpos < file_size_) 
      {	/* Not end of file yet */
  	/* We've reached_ end of buffer, but not end of file yet,
  	   so read next part of file into buffer */
        if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
        {
          endwin ();
          fprintf (stderr, "\nError reading file in get_line().\n");
          exit (-1);
        }
        buf_[bytes_read_] = '\0';
        page_ = buf_;
      } 
      else 
      {
        end_reached_ = true;
        break;
      }
    } 
    else if (i < str_max_len_)
      line_[i++] = *(page_++);
    else 
    {
      /* Truncate lines longer than MAX_LEN characters */
      if (i == str_max_len_)
        line_[i++] = '\0';
      page_++;
    }
  }

  if (i <= str_max_len_)
    line_[i] = '\0';
  if (!end_reached_)
    page_++;			/* move pass '\n' */

  return line_;
}

/*
 * Display a dialog box and get the search term from user
 */
int NCtextbox::get_search_term (char *search_term, int height, int width)
{
  int box_height = 7, box_width = 30;
  int x = (width - box_width) / 2;
  int y = (height - box_height) / 2;

  int k;
  {
    NCinputbox box("X","Input search string:",x,y,box_height,box_width,search_term);
    strcpy(search_term, box.getstr());
    k = box.getkey();
  }
//  refresh();
//  text_->refresh();
  keypad(TRUE);
  return k;
}

void NCtextbox::print_position (int height, int width)
{
    int fpos;
    if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
    {
	endwin ();
	fprintf (stderr, "\nError moving file pointer in print_position().\n");
	exit (-1);
    }
    
    int percent = !file_size_ ?
	100 : ((fpos - bytes_read_ + page_ - buf_) * 100) / file_size_;
    attrset (a(position_indicator_attr));
    move (height - 3, width - 9);
    printw ("(%3d%%)", percent);
}

void NCtextbox::back_lines (int n)
{
  begin_reached_ = false;
  /* We have to distinguish between end_reached and !end_reached
     since at end of file, the line is not ended by a '\n'.
     The code inside 'if' basically does a '--page' to move one
     character backward so as to skip '\n' of the previous line */
  if (!end_reached_) 
  {
  /* Either beginning of buffer or beginning of file reached? */
    if (page_ == buf_) 
    {
      int fpos;
      if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
      {
	endwin ();
	fprintf (stderr, "\nError moving file pointer in "
		 "back_lines().\n");
	exit (-1);
      }
      if (fpos > bytes_read_) 
      {	/* Not beginning of file yet */
	/* We've reached beginning of buffer, but not beginning of
	   file yet, so read previous part of file into buffer.
	   Note that we only move backward for BUF_SIZE/2 bytes,
	   but not BUF_SIZE bytes to avoid re-reading again in
	   print_page() later */

	/* Really possible to move backward BUF_SIZE/2 bytes? */
	if (fpos < buf_size_ / 2 + bytes_read_) 
	{
          /* No, move less then */
          if (lseek (fd_, 0, SEEK_SET) == -1) 
          {
            endwin ();
            fprintf (stderr, "\nError moving file pointer in "
				 "back_lines().\n");
            exit (-1);
          }
          page_ = buf_ + fpos - bytes_read_;
        } 
        else 
        { /* Move backward BUF_SIZE/2 bytes */
          if (lseek (fd_, -(buf_size_ / 2 + bytes_read_), SEEK_CUR) == -1) 
          {
            endwin ();
            fprintf (stderr, "\nError moving file pointer in back_lines().\n");
            exit (-1);
          }
          page_ = buf_ + buf_size_ / 2;
        }
        
        if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
        {
          endwin ();
          fprintf (stderr, "\nError reading file in back_lines().\n");
          exit (-1);
        }
        
        buf_[bytes_read_] = '\0';
      } 
      else 
      {	/* Beginning of file reached */
        begin_reached_ = true;
        return;
      }
    }
    if (*(--page_) != '\n') 
    {	/* '--page' here */
      /* Something's wrong... */
      endwin ();
      fprintf (stderr, "\nInternal error in back_lines().\n");
      exit (-1);
    }
  }
  
  /* Go back 'n' lines */
  for (int i = 0; i < n; i++)
    do 
    {
      if (page_ == buf_) 
      {
        int fpos;
        if ((fpos = lseek (fd_, 0, SEEK_CUR)) == -1) 
        {
          endwin ();
          fprintf (stderr,
		  "\nError moving file pointer in back_lines().\n");
	  exit (-1);
	}
        if (fpos > bytes_read_) 
        {
          /* Really possible to move backward BUF_SIZE/2 bytes? */
          if (fpos < buf_size_ / 2 + bytes_read_) 
          {
            /* No, move less then */
            if (lseek (fd_, 0, SEEK_SET) == -1) 
            {
              endwin ();
              fprintf (stderr, "\nError moving file pointer "
			     "in back_lines().\n");
              exit (-1);
            }
            page_ = buf_ + fpos - bytes_read_;
          } 
          else 
          {	/* Move backward BUF_SIZE/2 bytes */
            if (lseek (fd_, -(buf_size_ / 2 + bytes_read_), SEEK_CUR) == -1) 
            {
              endwin ();
              fprintf (stderr, "\nError moving file pointer"
			     " in back_lines().\n");
              exit (-1);
            }
            page_ = buf_ + buf_size_ / 2;
          }
          if ((bytes_read_ = read (fd_, buf_, buf_size_)) == -1) 
          {
            endwin ();
            fprintf (stderr, "\nError reading file in "
			 "back_lines().\n");
            exit (-1);
          }
          buf_[bytes_read_] = '\0';
        } 
        else 
        {	/* Beginning of file reached */
          begin_reached_ = true;
          return;
        }
      }
    } while (*(--page_) != '\n');
    page_++;
}
#endif