/*
    This file is part of the 'ears' package.
    Copyright (C) 1996  Ralf Stephan <ralf@ark.franken.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef UI_NCURSES
#pragma implementation
#include <string>
#include <unistd.h>
#include <string.h>
#include "ui_ncurses.h"
#include "others/time.h"
#include "others/mystring.h"

//-----------------------------PUBLICS----------------------------------
UIncurses::UIncurses (string title) : recw_(0)
{
  std_ = new NCursesPanel();
  std_->frame (title.c_str());
  std_->refresh();
  curs_set(0);
  midx_ = std_->cols()/2;
  midy_ = std_->lines()/2;
  win_reg_.init (std_->cols(),std_->lines());
}

UIncurses::~UIncurses()
{
  if (std_==0) return;            // this is not good
  recw_close();
  close_all_wins();

  erase();
  refresh();
  delete std_;
  std_=0;
  erase();
  refresh();
  endwin();
  curs_set(1);
} 

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

int UIncurses::getkey (char* keys) const
{
  int key;

  raw();
  meta(0,TRUE);
  std_->keypad(TRUE);
  noecho();
  while ((key=std_->getch())!=ERR)
  {
    if (!keys) { noraw(); return key; }
    char *p = keys;
    while (*p)
      if (*p++ == key)
        { noraw(); return key; }
    if (key==0x03 || key==0x1b) { noraw(); return key; }
    beep();
  }
  noraw();
  return key;
}

int UIncurses::getkey_if_available (char* keys) const
{
  int key;

  raw();
  std_->nodelay(true);
  key=std_->getch();
  if (key!=ERR)
  {
    if (!keys) { std_->nodelay(false); noraw(); return key; }
    char *p = keys;
    while (*p)
      if (*p++ == key)
        { std_->nodelay(false); noraw(); return key; }
    if (key==0x03 || key==0x1b) { std_->nodelay(false); noraw(); return key; }
    beep();
  }
  noraw();
  std_->nodelay(false); 
  return key;
}

int UIncurses::query (string text, char*) const
{
  int key;
  {
    NCyesno box("!",text.c_str(),15,5,9,50);
    key = box.getkey();
    if (key==0) key='y';
    else if (key==1) key='n';
    else if (key==-1) key=27;
  }
  std_->refresh();
    
  return key;
}

void UIncurses::note (string text) const
{
  NCmsgbox box ("Note",text.c_str(),15,5,9,50);
  box.getkey();
}

void UIncurses::show_file (string file) const
{
  char f[256];
  strcpy(f,file.c_str());
  NCtextbox (f,f,2,2,std_->lines()-4,std_->cols()-4);
}

int UIncurses::menubox (string t, string s, int w, int size, const SSMap& m) 
{ 
  char buf[256];
  strcpy (buf,s.c_str());
  int h = std_->lines()-4 <? size+7;
  int y = (std_->lines()-h)/2;
  int x = (std_->cols()-w)/2;
  Menuboxstrings strings (m);
  NCmenubox box (t.c_str(),buf,x,y,h,w,h-7,size,strings);
  choice_ = box.choice();
  return box.getkey();
}

int UIncurses::inputbox (string s, int h, int w, string v) 
{ 
  char buf[256];
  strcpy (buf,v.c_str());
  int y = (std_->lines()-h)/2;
  int x = (std_->cols()-w)/2;
  NCinputbox box (0,s.c_str(),x,y,h,w,buf);
  inputstr_ = box.getstr();
  return box.getkey();
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

void UIncurses::recw_open()
{
  if (recw_) return;
  
  recw_w_ = 40;
  recw_h_ = 7;
  recw_ = new NCursesPanel (recw_h_,recw_w_,24-recw_h_,(80-recw_w_)/2);
  recw_->timeout(0);
  raw();
  recw_->box();
  recw_->frame("Recording");

  recw_->move(5,10);
  recw_->addstr("---|");
  
  recw_->move(1,28);
  recw_->addstr("Esc to stop");
  recw_->move(2,28);
  recw_->addstr("+ for pause");

  recw_->refresh();
}

void UIncurses::recw_close()
{
  if (!recw_) return;
  recw_->timeout(-1);
  delete recw_;
  noraw();
  recw_=0;
}

void UIncurses::recw_speak (string s)
{
  recw_->move(1,1);
  recw_->addstr("Please speak:");

  string spaces = replicate(' ',recw_w_-2);
  recw_->move(3,1);
  recw_->addstr(spaces.c_str());
  recw_->move(3,(recw_w_-s.length())/2);
  recw_->addstr(s.c_str());
  
  recw_->refresh();
}

void UIncurses::recw_result (string s)
{
  string spaces = replicate(' ',recw_w_-2);
  recw_->move(4,1);
  recw_->addstr(spaces.c_str());
  recw_->move(4,2);
  string res;
  if (s[0]=='?') res = "No match.";
  else { res = "Best match: "; res+=s; }
  recw_->addstr (res.c_str());
  
  recw_->refresh();
}

void UIncurses::recw_ok ()
{
  recw_->move(1,1);
  recw_->addstr("OK!          ");
  recw_->refresh();
  Time sleep(500000);
}

void UIncurses::recw_gauge (int energy)
{
  static int old_val=0;
//  int val = percent/10+1;
  int val;
  if (energy<0) val=0;
  else if (energy<9) val=energy;
  else if (energy<10) val=9;
  else val=10;
  if (val>10) val=10;
  
  if (val==old_val) return;  //The simple solution
  recw_->move(5,14);
  for (int k=0; k<val; k++)
    recw_->addch(ACS_BULLET);
  for (int k=val; k<10; k++)
    recw_->addch(' ');
  
/*  int diff = val-old_val;  // a faster alternative?
  old_val=val;
  
  if (diff==0) return;
  if (diff<0)
  {
    recw_->move(5, 13+val);
    while (++diff<=0) recw_->addch(' ');
  }
  else
  {
    recw_->move(5, 14+old_val);
    while (--diff>=0) recw_->addch(ACS_BULLET);
  }
  old_val=val;
*/

  old_val=val;
  recw_->refresh();
}

void UIncurses::recw_pause (bool paused)
{
  recw_->move(5,38);
  recw_->addch(paused? 'P':' ');
  recw_->refresh();
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

win_hdl UIncurses::makeInfow (string title) 
{ 
  win_hdl hdl = win_reg_.register_new();
  win_reg_[hdl].wp = 0;  // window is NOT drawn
  strcpy (win_reg_[hdl].str, title.c_str());
  return hdl;
}

void UIncurses::infow_fill (win_hdl hdl, const SList& list, bool) 
{
  int maxw=0;
  for (SList::const_iterator it = list.begin(); it!=list.end(); it++)
  {
    const l = (*it).length();
    if (l > maxw) maxw = l;
  }
  int h = list.size();
  
  NCursesPanel *p = win_reg_[hdl].wp;
  if (p==0)
  {
    win_reg_[hdl].h = h+2;
    win_reg_[hdl].w = maxw+2;
    win_reg_.adjustxy (hdl);   // assumes h and w are set
    p = new NCursesPanel(win_reg_[hdl].h, win_reg_[hdl].w,
                         win_reg_[hdl].y, win_reg_[hdl].x);
    p->box();
    p->frame (win_reg_[hdl].str);
    win_reg_[hdl].wp = p;
  }

  int y=1;
  for (SList::const_iterator it = list.begin(); it!=list.end(); it++,y++)
  {
    string s(*it);
    int too_short_by = maxw - s.length();
    if (too_short_by>0)
      s += replicate(' ',too_short_by);
      
    p->move (y,1);
    p->addstr (s.c_str());
  }
  p->refresh();
}

void UIncurses::close_all_wins() 
{
  for (win_hdl h=win_reg_.first_hdl(); h!=0 ; h=win_reg_.next_hdl())
    infow_close (h);
}


//---------------------------------------------------------------------
//---------------------------------------------------------------------
//---------------------------------------------------------------------
UIncurses::Menuboxstrings::Menuboxstrings (const SSMap& m)
{
  size_ = m.size();
//  if (size_<=0) throw;  or: max(size_,1)
  strings_ = new char*[size_*3];

  SSMap::const_iterator cit_ = m.begin();
  for (int k=0; cit_ != m.end(); cit_++,k++)
  {
    const string& s = (*cit_).first;
    int l = s.length();
    char* p = new char[l+1];
    strcpy (p,s.c_str());
    strings_[k*3] = p;

    const string& s1 = (*cit_).second;
    l = s1.length();
    p = new char[l+1];
    strcpy (p,s1.c_str());
    strings_[k*3+1] = p;
    strings_[k*3+2] = "";
  }
  strings_[2]="on";
}

UIncurses::Menuboxstrings::~Menuboxstrings()
{
  for (int k=0; k<size_; k++)
    delete [] strings_[k*3];
  delete [] strings_;
}


#endif