/*********************************************************************************
 *
 * $Id: speakermodel.cpp,v 1.14 1998/06/14 12:54:02 daniel Exp $
 *
 * $Log: speakermodel.cpp,v $
 * Revision 1.14  1998/06/14 12:54:02  daniel
 * switch off detect_mode before calling calibrate_micro
 *
 * Revision 1.13  1998/06/02 10:33:21  daniel
 * microphone calibration added
 *
 * Revision 1.12  1998/05/05 11:16:51  daniel
 * <import speakerfile> added
 *
 * Revision 1.11  1998/05/01 21:42:39  daniel
 * removed debugging output
 *
 * Revision 1.10  1998/05/01 21:14:45  daniel
 * TrainDialog introduced ....
 * changed some recognizer default values
 * can now specify speakerfile at startup time .... enter hidden/recognition-mode then!
 *
 * Revision 1.9  1998/04/30 15:46:21  daniel
 * unimportant modifications ...
 *
 * Revision 1.8  1998/04/29 22:17:17  daniel
 * min_distance moved to score now (all scoring is done within score now, as it should be done)
 * now calling branch&bound scoring - MUCH faster!
 *
 * Revision 1.7  1998/04/29 11:00:15  daniel
 * nothing important
 *
 * Revision 1.6  1998/04/29 02:15:54  daniel
 * tiny_mode removed -> now using docking to panel!!!
 * DockWidget introduced that shows two leds on the Panel indicating the
 * recognizers state ...
 * LED lamps removed (moved onto docking_widget infact)
 * #include's moved from header-file to this file as far as possible ->
 *   in speakermodel.h just keep class definitions (e.g. class SoundBuffer;)
 * detect_mode_on/off are now only executed when really toggling mode!
 *  (not always)
 *
 * Revision 1.5  1998/04/28 11:20:13  daniel
 * score_threshold moved to class Score now -> online rejection!
 *
 * Revision 1.4  1998/04/27 23:58:58  daniel
 * led_recognition working correctly now!
 *
 * Revision 1.3  1998/04/27 20:01:53  daniel
 * led_record and led_recognition added
 * led_recognition not yet working correctly!
 * tiny mode added (can be used during recognition mode; just displays the two LEDs)
 *
 * Revision 1.2  1998/04/27 00:29:19  daniel
 * detect_mode_off works fine now
 * options dialog implemented
 * save_config methode introduced (not yet working properly!!)
 *
 * Revision 1.1  1998/04/26 15:57:19  daniel
 * Initial revision
 *
 *
 *********************************************************************************/

#include <qfiledlg.h>
#include <qpushbt.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qfile.h>
#include <qdstream.h>
#include <qdatetm.h> 
#include <qmsgbox.h>
#include <stdlib.h>

#include "speakermodel.moc"

#include<iostream.h>

#include "reference.h"
#include "reference_editor.h"
#include "preprocessing.h"
#include "buffer.h"
#include "score.h"
#include "options.h"
#include "train_dialog.h"

#include "docking.h"
#include "kvoicecontrol.h"
extern DockWidget *dock_widget;

SpeakerModel::SpeakerModel(KConfig *config, 
			   QWidget *parent, const char *name) : QWidget( parent, name )
{
  resize(300, 200);
  
  ref_listbox = new QListBox( this, "ref_listbox" );
  ref_listbox->setGeometry(0, 0, 200, 200);
  ref_list = new QList<Reference>();
  ref_list->setAutoDelete(true);
  
  new_reference_btn = new QPushButton( "New", this, "new_reference" );
  new_reference_btn->setGeometry(220, 35, 60, 30);
  connect(new_reference_btn,  SIGNAL(clicked()), this, SLOT(new_reference()));
  edit_reference_btn = new QPushButton( "Edit ...", this, "edit_reference" );
  edit_reference_btn->setGeometry(220, 90, 60, 30);
  connect(edit_reference_btn,  SIGNAL(clicked()), this, SLOT(edit_reference()));
  delete_reference_btn = new QPushButton( "Delete", this, "delete_reference" );
  delete_reference_btn->setGeometry(220, 145, 60, 30);
  connect(delete_reference_btn,  SIGNAL(clicked()), this, SLOT(delete_reference()));

  fdialog  = new QFileDialog(NULL, ".spk", NULL, "dialog", TRUE);

  preprocessing= new Preprocessing();
  score        = new Score();
  buffer       = new SoundBuffer();

  led_timer = new QTimer( this );
  connect( led_timer, SIGNAL(timeout()), dock_widget, SLOT(toggle_led_recognition()) );

  do_detect = false;
  changed = false;

  // ***** load configuration
  // ***** not working yet !?

  QString s; 
  QString oldgroup = config->group();
  config->setGroup("Sound");
  s = config->readEntry("RecLevelThreshold", "10");
  buffer->set_rec_level_threshold(s.toInt());
  s = config->readEntry("AcceptedSilenceFrames", "4");
  buffer->set_accept_silence(s.toInt());
  config->setGroup("Recognizer");
  s = config->readEntry("AdjustmentWindowWidth", "70");
  score->set_adjust_width(s.toInt());
  s = config->readEntry("ScoreThreshold", "15.0" );
  score->set_threshold(s.toFloat());
  s = config->readEntry("MinimumScoreDistance", "2.4" );
  score->set_min_distance(s.toFloat());
  config->setGroup( oldgroup );

  
  options_dlg = new Options();

  connect(options_dlg, SIGNAL(reject_thresh_changed(float)), 
	  score, SLOT(set_threshold(float)));
  connect(options_dlg, SIGNAL(min_dist_changed(float)), 
	  score, SLOT(set_min_distance(float)));
  connect(options_dlg, SIGNAL(adj_win_width_changed(int)), 
	  score, SLOT(set_adjust_width(int)));

  connect(options_dlg, SIGNAL(acc_sil_frames_changed(int)), 
	  buffer, SLOT(set_accept_silence(int)));
  connect(options_dlg, SIGNAL(rec_level_thresh_changed(int)), 
	  buffer, SLOT(set_rec_level_threshold(int)));

}


SpeakerModel::~SpeakerModel()
{
  delete filename;
  delete fdialog;

  delete preprocessing;
  delete score;
  delete buffer;

  delete ref_listbox;
  delete ref_list;
}

/* -------------------------------------------------------------------------------- */

bool SpeakerModel::reset()
{
  bool ok = true;

  if (has_changed())
    ok = ask_save_changes();
  else
    ok = true;

  if (ok)
  {
    detect_mode_off();
    
    filename = 0;
    ref_listbox->clear();
    ref_list->clear();
    
    changed = false;
    
    emit new_title(QString("[none]"));

    return true;
  }

  return false;
}


void SpeakerModel::import()
{
  load(NULL, false);
}


void SpeakerModel::load()
{
  load(NULL, true);
}


void SpeakerModel::load(char *f=0, bool reset_first)
{
  detect_mode_off();
  
  QString filename_tmp;

  if (f == 0)
    filename_tmp = fdialog->getOpenFileName(NULL, "*.spk", NULL, "fdialog");
  else
    filename_tmp = QString(f);

  //cerr << "F " << filename_tmp << endl;
  
  if (filename_tmp != NULL)
  {
    if (!reset_first || reset())
    {
      //filename = new QString(filename_tmp);
      QFile file(filename_tmp);
      file.open( IO_ReadOnly );   
      QDataStream s( &file );     
      uint count;
      s >> count;
      for (uint count_tmp=0; count_tmp < count; count_tmp++)
      {
	Reference *r = new Reference();
	s >> r;
	ref_listbox->insertItem(r->get_name());
	ref_listbox->setCurrentItem(ref_listbox->count()-1);
	ref_list->append(r);
      }
      file.close();               
      
      if (reset_first)
      {	
	filename = new QString(filename_tmp);
	emit new_title(filename->right(filename->length()-filename->findRev('/')-1));
      }
    }
  }
}


void SpeakerModel::save()
{
  detect_mode_off();

  if (NULL == filename)
    save_as();
  else
  {
    QFile file(*filename);

    file.open( IO_WriteOnly );   
    QDataStream s( &file );     

    s << ref_list->count();

    for (uint count=0; count < ref_list->count(); count++)
    {
      s << *(ref_list->at(count));
    }

    changed = false;

    file.close();               
  }
}

 
void SpeakerModel::save_as()
{
  detect_mode_off();

  QString filename_tmp = fdialog->getSaveFileName(NULL, "*.spk", NULL, "fdialog");

  if (filename_tmp != NULL)
  {
    filename = new QString(filename_tmp);
    if (filename->right(4) != ".spk")
      filename->append(".spk");
    
    save();
    emit new_title(filename->right(filename->length()-filename->findRev('/')-1));
  }
}


void SpeakerModel::detect_mode_on()
{
  if (!do_detect && check_references())  
  {
    if (!buffer->detect_speech(true))
      return;

    connect(buffer, SIGNAL(end_detected()), this, SLOT(test_utterance()));
    do_detect = true;
    emit detect_mode_changed(true);
    connect(buffer, SIGNAL(recording_active()), dock_widget, SLOT(toggle_led_record()));
    dock_widget->led_record_on();
    KApplication::flushX();
    kapp->processEvents();
  }
}


void SpeakerModel::detect_mode_off()
{
  if (do_detect)
  {
    do_detect = false;
    
    disconnect(buffer, SIGNAL(end_detected()), this, SLOT(test_utterance()));
    buffer->detect_speech(false);
    
    emit detect_mode_changed(false);
    disconnect(buffer, SIGNAL(recording_active()), dock_widget, SLOT(toggle_led_record()));
    dock_widget->led_record_off();
    KApplication::flushX();
    kapp->processEvents();
  }
};


void SpeakerModel::toggle_detect_mode()
{
  dock_widget->led_recognition_off();
  KApplication::flushX();
  kapp->processEvents();

  if (do_detect)
    detect_mode_off();
  else
    detect_mode_on();
}


void SpeakerModel::test_utterance()
{
  detect_mode_off();
  dock_widget->led_recognition_on();
  led_timer->start(150);
  KApplication::flushX();
  kapp->processEvents();

  Utterance *test_utterance = preprocessing->preprocess_utterance(buffer->get_data(), 
								  buffer->get_size());

  // ***** branch&bound-score is by factor HELL faster ! ;-)
  //QString *result = score->score(test_utterance, ref_list);
  //QDateTime dt = QDateTime::currentDateTime();
  //cerr << dt.toString() << endl;
  QString *result = score->branchNbound_score(test_utterance, ref_list);
  //dt = QDateTime::currentDateTime();
  //cerr << dt.toString() << endl;

  led_timer->stop();
  
  if (NULL == result)
  {
    dock_widget->led_recognition_off();
    detect_mode_on();
  }
  else
  {
    QString command = result->copy();
    delete result;

    dock_widget->led_recognition_success();

    KApplication::flushX();
    kapp->processEvents();

    if (command != "detect_mode_off")
    {
      command = command.stripWhiteSpace();
      if (command.right(1) != "&")
	command.append(" &");
      system(command);

      detect_mode_on();
    }
  }
}

/* -------------------------------------------------------------------------------- */

bool SpeakerModel::check_references()
{
  for (uint i = 0 ; i < ref_list->count() ; i++)
  {
    if (ref_list->at(i) == 0)
    {
      char text[256];
      sprintf(text, "Reference number %d is empty!", i+1);
      QMessageBox::information( this, "Oops", text );
      return false;
    }
    if (ref_list->at(i)->count() < 2)
    {
      QString *text = new QString(ref_list->at(i)->get_name().copy());
      text->prepend("\"");
      text->append("\"\nneeds at least two utterances!!");
      QMessageBox::information( this, "Oops", *text );
      delete text;
      return false;
    }
  }
  if (ref_list->count() == 0)
  {
    QMessageBox::information( this, "Oops", "No references !!" );
    return false;
  }
  
  return true;
}


void SpeakerModel::new_reference()
{
  // ***** edit a new reference

  if (!ref_list->contains(NULL))
  {
    ref_listbox->insertItem("untitled");
    ref_listbox->setCurrentItem(ref_listbox->count()-1);
    ref_list->append(NULL);

    changed = true;
  }
}

void SpeakerModel::edit_reference()
{
  // ***** edit actual list box entry if one selected

  if (-1 != ref_listbox->currentItem())
  {
    if (do_detect)
      detect_mode_off();

    if (NULL == ref_list->at(ref_listbox->currentItem())) 
    {
      // ***** new reference
      
      Reference *r = new Reference();

      ref_editor = new ReferenceEditor(r, preprocessing, buffer, this, "ref_editor");

      if (ref_editor->exec())
      {
	uint pos = ref_listbox->currentItem();
	
	ref_list->remove(pos);
	ref_list->insert(pos, r);
	ref_listbox->removeItem(pos);
	ref_listbox->insertItem(r->get_name(), pos);
	ref_listbox->setCurrentItem(pos);

	changed = true;
      }
      delete ref_editor;
    }
    else
    {
      // ***** existing reference

      ref_editor = new ReferenceEditor(ref_list->at(ref_listbox->currentItem()), 
				       preprocessing, 
				       buffer,
				       this, 
				       "ref_editor");
      
      if (ref_editor->exec())
      {
	uint pos = ref_listbox->currentItem();
	
	ref_listbox->removeItem(pos);
	ref_listbox->insertItem(ref_list->at(pos)->get_name(), pos);
	ref_listbox->setCurrentItem(pos);

	changed = true;
      }

      delete ref_editor;
    }
  }
}


void SpeakerModel::delete_reference()
{
  // ***** delete actual list box entry if one selected

  if (-1 != ref_listbox->currentItem())
  {
    uint idx = ref_listbox->currentItem();
    ref_listbox->removeItem(idx);
    ref_list->remove(idx);

    changed = true;
  }
}


void SpeakerModel::append_reference(Reference *r)
{
  // ***** add this reference (no copy, just the pointer!!)

  if (r != NULL)
  {
    ref_listbox->insertItem(r->get_name());
    ref_listbox->setCurrentItem(ref_listbox->count()-1);
    ref_list->append(r);

    changed = true;
  }
}


bool SpeakerModel::has_changed()
{
  return changed;
}


bool SpeakerModel::ask_save_changes()
{
  switch( QMessageBox::information( this, "KVoiceControl",
				    "The speakermodel has changed!\n"
				    "Do you want to save it before exiting?",
				    "&Save", "&Don't Save", "&Cancel",
				    0, 2 ) )
  {
  case 0: 
    save();
    return(true);
    break;
  case 1:
    return(true);
    break;
  case 2:
    return(false);
    break;
  }
  
  return(true);
}


void SpeakerModel::show_options()
{
  detect_mode_off();

  if (!options_dlg->isVisible())
  {
    // ***** fill in values

    options_dlg->set_rec_thresh(buffer->get_rec_level_threshold());
    options_dlg->set_acc_sil_frames(buffer->get_acc_sil_frames());

    options_dlg->set_adj_win_width(score->get_adjust_win_width());
    options_dlg->set_reject_thresh(score->get_threshold());
    options_dlg->set_min_dist(score->get_min_distance());

    options_dlg->show();
  }
}


void SpeakerModel::train_references()
{
  detect_mode_off();

  QString refslist_file = fdialog->getOpenFileName(NULL, "*.txt", NULL, "fdialog");
  
  if (refslist_file != NULL)
  {
    TrainDialog *traindialog = new TrainDialog(refslist_file, this, preprocessing, 
					       buffer, this,  "train_dialog");
    //traindialog->exec();
    delete traindialog;
  }
}


void SpeakerModel::calibrate_micro()
{
  detect_mode_off();

  buffer->calibrate_micro();
}


void SpeakerModel::save_config(KConfig *config)
{
  // ***** save configuration
  // ***** not working yet !?

  QString oldgroup = config->group();
  config->setGroup("Sound");
  config->writeEntry("RecLevelThreshold", buffer->get_rec_level_threshold());
  config->writeEntry("AcceptedSilenceFrames", buffer->get_acc_sil_frames());
  config->setGroup("Recognizer");
  config->writeEntry("AdjustmentWindowWidth", score->get_adjust_win_width() );  
  QString s;
  s.sprintf("%f", score->get_threshold());
  config->writeEntry("ScoreThreshold", s );
  s.sprintf("%f", score->get_min_distance());
  config->writeEntry("MinimumScoreDistance", s );
  config->setGroup( oldgroup );
}

