/* 
 * fontManager.cc --
 *
 *      This file contains the definitions of the 'FontManager' class
 *      methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include "fontManager.h"



/*
 *----------------------------------------------------------------------
 *
 * FontManager --
 *
 *      This method is invoked every time a FontManager is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

FontManager::FontManager(Display *display) {

  int i;

  _disp = display;
  for(i=0; i<4; i++)
    Tcl_InitHashTable(&_tablePtr[i], TCL_STRING_KEYS);
}



/*
 *----------------------------------------------------------------------
 *
 * ~FontManager --
 *
 *      This method is invoked every time a FontManager is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontManager::~FontManager(void) {
}



/*
 *----------------------------------------------------------------------
 *
 * add_font --
 *
 *      This method is invoked to add a new font in the FontManager.
 *
 * Results:
 *      The created FontItem is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontFamily *
FontManager::add_font(char *family, FontStyle style, char *x11, char *ps) {

  int result;
  FontFamily *ffPtr;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_CreateHashEntry(&_tablePtr[(int)style], family, &result);
  if( result == 0 ) {
    fprintf(stderr, "FontManager::add_font: font '%s - %d' already exists\n",
	    family, (int)style);
    return NULL;
  }
  ffPtr = new FontFamily(family, style, x11, ps);
  Tcl_SetHashValue(entryPtr, ffPtr);
  return ffPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * del_font --
 *
 *      This method is invoked to delete a font from the FontManager.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
FontManager::del_font(char *family, FontStyle style) {

  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(&_tablePtr[(int)style], family);

  if( entryPtr == NULL )
    fprintf(stderr, "FontManager::del_font: font '%s-%d' doesn't exists\n",
	    family, (int)style);
  else
    Tcl_DeleteHashEntry(entryPtr);
}



/*
 *----------------------------------------------------------------------
 *
 * get_font --
 *
 *      This method is invoked to look for a font. But if the font
 *      familly and style are found, then the requested font is
 *      created.
 *
 * Results:
 *      Returns a FontItem if found (or created) and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontItem *
FontManager::get_font(char *family, FontStyle style, int size) {

  Tcl_HashEntry *entryPtr;
  FontFamily *ffPtr;
  FontItem *fiPtr;

  entryPtr = Tcl_FindHashEntry(&_tablePtr[(int)style], family);
  if( entryPtr == NULL )
    return NULL;

  ffPtr = (FontFamily *)Tcl_GetHashValue(entryPtr);

  fiPtr = ffPtr->query_item(size);
  if( fiPtr == NULL )
    fiPtr = ffPtr->add_item(_disp, size);
  
  return fiPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * query_font --
 *
 *      This method is invoked to look for a font.
 *
 * Results:
 *      Returns a FontItem if found and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontFamily *
FontManager::query_font(char *family, FontStyle style) {

  Tcl_HashEntry *entryPtr;
  FontFamily *ffPtr;

  entryPtr = Tcl_FindHashEntry(&_tablePtr[(int)style], family);

  if( entryPtr == NULL )
    ffPtr = NULL;
  else
    ffPtr = (FontFamily *)Tcl_GetHashValue(entryPtr);

  return ffPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * get_first --
 *
 *      This method gives the first font of a given style in
 *      the corresponding hash table.
 *
 * Results:
 *      Returns a FontItem if exists and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontFamily *
FontManager::get_first(FontStyle style) {

  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FirstHashEntry(&_tablePtr[(int)style], &_search);
  
  if( entryPtr != NULL ) 
    return (FontFamily *)Tcl_GetHashValue(entryPtr);
  else
    return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * get_next --
 *
 *      This method gives the next font of a given style in
 *      the corresponding hash table. A call to this method
 *      must be proceded by a call to get_first method.
 *
 * Results:
 *      Returns a FontItem if a such font exists and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontFamily *
FontManager::get_next(void) {

  Tcl_HashEntry *entryPtr;
  
  entryPtr = Tcl_NextHashEntry(&_search);

  if( entryPtr != NULL )
    return (FontFamily *)Tcl_GetHashValue(entryPtr);
  else
    return NULL;
}
