/* 
 * container.cc --
 *
 *      This file contains the definitions of the 'Container' class
 *      methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <stdio.h>
#include <malloc.h>
}

#include "container.h"



/*
 *----------------------------------------------------------------------
 *
 * Container --
 *
 *      This procedure is invoked every time a Container is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

Container::Container() {

  _children = NULL;
  _children_num = 0;
  _children_max = 0;

  _parent = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * ~Container --
 *
 *      This procedure is invoked every time a Container is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Every child of the container is deleted.
 *
 *----------------------------------------------------------------------
 */

Container::~Container() {

  int i;

  for(i=0; i<_children_num; i++)
    delete _children[i];
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      Container without known anything about it.
 *
 * Results:
 *      A new Container is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Container::new_of_same_type(void) {

  return new Container;
}



/*
 *----------------------------------------------------------------------
 *
 * alloc_offset --
 *
 *      This procedure is invoked everytime a child is inserted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Memory is reallocated if there's not enough space to insert
 *      'num' children.
 *
 *----------------------------------------------------------------------
 */

void
Container::alloc_offset(int num) {

  if( _children_max == 0 ) {
    _children_max = (num+1)*2;
    _children = (Container **)malloc(sizeof(Container *)*_children_max);
  }else
    if( _children_max <= num ) {
      
      _children_max = (num+1)*2;
      
      _children = (Container **)realloc(_children, sizeof(Container *)*_children_max);
      if( _children == NULL ) {
	fprintf(stderr, "Container::alloc_offset: Can't resize children array\n");
	exit(1);
      }
    }
}



/*
 *----------------------------------------------------------------------
 *
 * get_offset --
 *
 *      This procedure gives the Container offset.
 *
 * Results:
 *      Returns the Container offset.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Container::get_offset(void) const {

  int i;
  Container *parent;

  parent = get_parent();

  for(i=0; i<parent->get_children_num(); i++)
    if( parent->get_child(i) == this )
      return i;
  fprintf(stderr, "Container::get_offset: Child not found\n");
  exit(1);
}



/*
 *----------------------------------------------------------------------
 *
 * insert_children --
 *
 *      This procedure inserts 'howmany' children ('what') at 'where'
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if 'where' is not equal to '_children_num'.
 *
 *----------------------------------------------------------------------
 */

void
Container::insert_children(Container **what, int howmany, int where) {

  int i;

  alloc_offset( _children_num + howmany );

  for(i=_children_num-1; i>=where; i--)
    _children[i+howmany] = _children[i];

  for(i=0; i<howmany; i++) {
    _children[ i+where ] = what[i];
    _children[ i+where ]->_parent = this;
  }

  _children_num += howmany;
}



/*
 *----------------------------------------------------------------------
 *
 * delete_children --
 *
 *      This procedure deletes children from position 'where' to
 *      'where'+'howmany'.
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if "where" is not equal to "_children_num".
 *
 *----------------------------------------------------------------------
 */

void
Container::delete_children(int howmany, int where) {

  int i;

  for(i=where; i<_children_num-howmany; i++) {
    _children[ i ] = _children[ i+howmany ];
  }

  _children_num -= howmany;

  if( _children_num <= 0 ) {   // Tells its father to delete it
    _parent->delete_children(1, get_offset());
    delete this;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * move_children --
 *
 *      This function moves 'homany' children from position 'isrc'
 *      to container 'dst' at position 'idst'.
 *      The function returns if 'delete_children' was called (TRUE/FALSE),
 *      so we can call 'recompute_size' for example.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childrens are moved.
 *
 *----------------------------------------------------------------------
 */

void
Container::move_children(int howmany, int isrc, Container *dst, int idst) {
  
  dst->insert_children(&_children[isrc], howmany, idst);
  delete_children(howmany, isrc);
}



/*
 *----------------------------------------------------------------------
 *
 * merge_container --
 *
 *      This is a convenience function to concatenate two Containers.
 *      In fact, this function puts all the 'con' children at the end
 *      of the container 'this'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The result is : this + con -> this
 *
 *----------------------------------------------------------------------
 */

void
Container::merge_container(Container *con) {

  con->move_children(con->get_children_num(), 0, this, get_children_num());
}



/*
 *----------------------------------------------------------------------
 *
 * split_container --
 *
 *      This function splits the Container, at 'offset' position.
 *
 * Results:
 *      The new Container created (second part of the current).
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Container::split_container(int offset) {

  Container *bros;

  if( offset == 0  ||  offset >= _children_num )
    return this;

  bros = new_of_same_type();
  bros->_parent = _parent;

  _parent->insert_children(&bros, 1, get_offset()+1);
  move_children(_children_num-offset, offset, bros, 0);

  return bros;
}



/*
 *----------------------------------------------------------------------
 *
 * get_next_same_container --
 *
 *      This function gives the next same container after this.
 *
 * Results:
 *      The next same container.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Container::get_next_same_container(Container *con, int depth, int direc) {

  int offset;

  if( con == NULL )  {     // First call with the defaults args (depth, direc)
    if( _parent == NULL )
      return NULL;
    return _parent->get_next_same_container(this, 0, UP_DIRECTION);
  }
  
  depth += direc;
  
  if( depth == 0 )                          // the next same container is found
    return this;
  
  if( direc == UP_DIRECTION ) {
    offset = get_child_offset(con);

    if( offset != get_children_num() - 1 )   // 'con' is not the last child of 'this'
      return _children[ offset + 1 ]->get_next_same_container(this, depth, DOWN_DIRECTION);
    
    if( _parent == NULL )
      return NULL;

    return _parent->get_next_same_container(this, depth, UP_DIRECTION);
  }
  
  return _children[ 0 ]->get_next_same_container(this, depth, DOWN_DIRECTION);
}



/*
 *----------------------------------------------------------------------
 *
 * get_previous_same_container --
 *
 *      This function gives the previous same container after this.
 *
 * Results:
 *      The previous same container.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */


Container *
Container::get_previous_same_container(Container *con, int depth, int direc) {

  int offset;

  if( con == NULL ) {     // First to call
    if( _parent == NULL )
      return NULL;
    return _parent->get_previous_same_container(this, 0, UP_DIRECTION);
  }
  
  depth += direc;
  
  if( depth == 0 )                       // the previous same container is found
    return this;
  
  if( direc == UP_DIRECTION ) {
    offset = get_child_offset(con);

    if( offset != 0  )                    // 'con' is not the first child of 'this'
      return _children[offset-1]->get_previous_same_container(this, depth, DOWN_DIRECTION);
    if( _parent == NULL )
      return NULL;

    return _parent->get_previous_same_container(this, depth, UP_DIRECTION);
  }
  
  return _children[ _children_num-1 ]->get_previous_same_container(this,depth, DOWN_DIRECTION);
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the Container.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Container::debug(int depth) {

  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr, "%s (%d)\n", type(), _children_num);

  for(i=0; i<_children_num; i++)
    _children[ i ]->debug( depth+1 );
}


