/* 
 * util.cc --
 *
 *      This file contains utility procedures that are used by commands,
 *      to turn easy the manipulation of shapes and paragraphs.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 * Copyright (c) 1996 Universite de Paris 8, departement MIME.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <string.h>
#include <time.h>
}

#include "util.h"
#include "isfuncs.h"
#include "wordSegment.h"
#include "papyrus.h"




/*
 *----------------------------------------------------------------------
 *
 * MergeParagraphs --
 *
 *      This function merges two Paragraphs. A Line and a Paragraph
 *      are given in argument. So at the end of the function, the 
 *      lines which were in the same Paragraph than 'LinePtr' are
 *      now in the 'paraPtr' Paragraph.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      WARNING: This function does nothing in the document tree. So
 *      the call to this function may be followed by some code which
 *      concatenates respectively the last Line and the first Line 
 *      of the two Paragraph parts.
 *
 *----------------------------------------------------------------------
 */

void
MergeParagraphs(Line *linePtr, Paragraph *paraPtr) {
  
  Paragraph *old_paraPtr;

  old_paraPtr = linePtr->get_para();
    
  PositionSave();
  PositionSet(current.doc, (Shape *)linePtr->get_child(0)->get_child(0), 0);

  PositionRestore();

  while( linePtr != NULL && linePtr->get_para() == old_paraPtr ) {
    linePtr->set_para( paraPtr );
    linePtr = (Line *)linePtr->get_next_same_container();
  }

  paraPtr->recompute(current);  
  delete old_paraPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * First_of_Para_In_Page --
 *
 *      This funciton looks for the first line of the current paragraph
 *      which is in the current page
 *
 * Results:
 *      The founded Line.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Line *
First_of_Para_In_Page(Paragraph *para, Page *page) {
  
  Line *line;
  int i;

  for(i=0; i<page->get_children_num(); i++) {
    line = (Line *)page->get_child(i);
    if( line->get_para() == para )
      return line;
  }

  fprintf(stderr, "First_ot_Para_In_Page: no line found\n");
  exit(1);
}



/*
 *----------------------------------------------------------------------
 *
 * First_of_Para --
 *
 *      This function finds the first line of the current paragraph
 *      given by 'line'.
 *
 * Results:
 *      The founded Line.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Line *
First_of_Para(Line *line) {

  Line *first;

  first = line;
  line = (Line *)line->get_previous_same_container();

  while( first->is_in_same_para(line) == TRUE ) {
    first = line;
    line = (Line *)line->get_previous_same_container();
  }
  return first;
}



/*
 *----------------------------------------------------------------------
 *
 * Last_of_Para --
 *
 *      This function finds the last line of the current paragraph
 *      given by 'line'.
 *
 * Results:
 *      The founded Line.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Line *
Last_of_Para(Line *line) {

  Line *last;

  last = line;
  line = (Line *)line->get_next_same_container();

  while( last->is_in_same_para(line) == TRUE ) {
    last = line;
    line = (Line *)line->get_next_same_container();
  }
  return last;
}



/*
 *----------------------------------------------------------------------
 *
 * Alignment_to_String --
 *
 *      Given a StyleAlignType this function returns the corresponding
 *      string.
 *
 * Results:
 *      A string.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char *
Alignment_to_String(StyleAlignType a) {

  switch( a ) {
  case STYLE_ALIGN_LEFT:   return "left";   break;
  case STYLE_ALIGN_RIGHT:  return "right";  break;
  case STYLE_ALIGN_CENTER: return "center"; break;
  case STYLE_ALIGN_FULL:   return "full";   break;
  }

  /*
   * Just for compilation warning.
   */
  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * String_to_Alignment --
 *
 *      Given a string this function returns the corresponding
 *      StyleAlignType.
 *
 * Results:
 *      A StyleAlignType.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
String_to_Alignment(char *s) {

  int a;
  int length;
  char c;

  length = strlen(s);
  c = s[0];
  
  if( (c == 'l') && (strncmp(s, "left", length) == 0) )
    a = (int)STYLE_ALIGN_LEFT;
  else if( (c == 'f') && (strncmp(s, "full", length) == 0) )
    a = (int)STYLE_ALIGN_FULL;
  else if( (c == 'c') && (strncmp(s, "center", length) == 0) )
    a = (int)STYLE_ALIGN_CENTER;
  else if( (c == 'r') && (strncmp(s, "right", length) == 0) )
    a = (int)STYLE_ALIGN_RIGHT;
  else
    a = -1;
  
  return a;
}



/*
 *----------------------------------------------------------------------
 *
 * FontStyle_to_String --
 *
 *      Given a FontStyle this function returns the corresponding
 *      string.
 *
 * Results:
 *      A string.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char *
FontStyle_to_String(FontStyle s) {

  switch( s ) {
  case NORMAL:       return "normal";      break;
  case BOLD:         return "bold";        break;
  case ITALIC:       return "italic";      break;
  case BOLD_ITALIC:  return "bold_italic"; break;
  }

  /*
   * Just for compilation warning.
   */
  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * String_to_FontStyle --
 *
 *      Given a string this function returns the corresponding
 *      FontStyle.
 *
 * Results:
 *      A FontStyle.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
String_to_FontStyle(char *s) {

  int sl;
  int length;
  char c;

  length = strlen(s);
  c = s[0];
  
  if( (c == 'n') && (strncmp(s, "normal", length) == 0) )
    sl = (int)NORMAL;
  else if( (c == 'b') && (strncmp(s, "bold", length) == 0) )
    sl = (int)BOLD;
  else if( (c == 'i') && (strncmp(s, "italic", length) == 0) )
    sl = (int)ITALIC;
  else if( (c == 'b') && (strncmp(s, "bold_italic", length) == 0) )
    sl = (int)BOLD_ITALIC;
  else
    sl = -1;
  
  return sl;
}



/*
 *----------------------------------------------------------------------
 *
 * Update_Shape_Attributes --
 *
 *      This function moves attributes from 'cur.attr' to 'cur.shape'.
 *      The function is invoked principally in 'Insert_String_Cmd' when
 *      inserting string with positionned attributes.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      'cur' is moved.
 *
 *----------------------------------------------------------------------
 */

void
Update_Shape_Attributes(ThePosition &cur) {

  /*
   * Before moving attributes, we have to check if 'cur.pos' is not
   * at the end of a word, else we insert an empty WordSegment.
   */
  if( is_last_of_word(cur) == FALSE || is_first_of_word(cur) == FALSE ) {
    Container *parent;
    WordSegment *ws;
    int offset;
    
    ws = new WordSegment;
    
    offset = cur.shape->get_offset() + ((is_last_of_word(cur) == TRUE) ? 1 : 0);
    parent = cur.shape->get_parent();
    cur.shape = (Shape *)cur.shape->split_container(cur.pos);
    parent->insert_children((Container **)&ws, 1, offset);
    cur.shape = ws;
    cur.pos = 0;
  }
  
  /*
   * Check for shape word (cur.pos == 0 == last_of_word)
   */
  
  if( cur.pos != 0 ) {
    Add_Empty_WordSegment();
    cur.shape->can_fit(cur);
    cur.attr->delete_attributes();
  } else
    cur.shape->can_fit(cur);
}



/*
 *----------------------------------------------------------------------
 *
 * Add_Empty_WordSegment --
 *
 *      This function inserts a Word width an empty WordSegment at
 *      the current position. The function is invoked we trying to
 *      insert different kinds of incompatible objects like an image
 *      and wordSegments.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The current position (ThePosition) is updated, so now the
 *      current position is the beggining of the empty WordSegment.
 *
 *----------------------------------------------------------------------
 */

void
Add_Empty_WordSegment(void) {

  WordSegment *ws;
  Container *parent;
  int offset;

  ws = new WordSegment;

  if( current.shape->has_attributes() ) {
    if( current.attr->has_mark() == 0 )
      current.attr->copy_attributes(current.shape->get_attributes());
  }

  parent = current.shape->get_parent();
  offset = current.shape->get_offset();
  if( current.pos == current.shape->get_children_num() )
    offset++;

  parent->insert_children((Container **)&ws, 1, offset);
  current.shape = ws;

  if( current.pos == 0 )
    offset++;

  current.pos = 0;
  parent->split_container(offset);
}



/*
 *----------------------------------------------------------------------
 *
 * Sort_Attributes --
 *
 *      This function removes Shape attributes which are the same
 *      than the Paragraph Attributes.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The equal attributes are removed.
 *
 *----------------------------------------------------------------------
 */

void
Sort_Attributes(Paragraph *para, Attributes *attr) {

  void *a, *p;
  AttrType type;
  int i, result;

  if( attr->has_mark() == 0 )
    return;

  for(i=0; i<8; i++) {
    if( attr->has_mark(1<<i) ) {
      type = (AttrType)(1<<i);
      a = attr->get_attr(type);

      switch(type) {
      case FONT_STYLE_ATTR:
	p = (void *)((FontItem *)para->query(STYLE_FONT))->get_style();	
	result = ((FontStyle)p == (FontStyle)a);
	break;

      case FONT_SIZE_ATTR:
	p = (void *)((FontItem *)para->query(STYLE_FONT))->get_size();
	result = ((int)p == (int)a);
	break;

      case FONT_FAMILY_ATTR:
	p = (void *)((FontItem *)para->query(STYLE_FONT))->get_family();
	result = (strcmp((char *)p, (char *)a) == 0 ) ? 1 : 0;
	break;

      default:
	fprintf(stderr, "WordSegment::get_abolute_attr: attribute %d not yet available", type);
	result = 0;
      }
      if( result == 1 )
	  attr->del_attr(type);
    }
  }
}



/*
 *----------------------------------------------------------------------
 *
 * Time_to_String --
 *
 *      This function converts the current time into a string of
 *      the form :
 *                   "Wed Jun 30 21:49:08 1993\n"
 * Results:
 *      The String.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char *
Time_to_String(void) {
  struct tm tm;
  long time_value;
  
  time(&time_value);
  tm = *localtime(&time_value);
  return asctime(&tm);
}



/*
 *----------------------------------------------------------------------
 *
 * Check_Font_Attributes --
 *
 *      This function checks if the current attributes are
 *      correct so they are added to the 'Shape' attributes.
 *      The function is invoked when there's a family, size or
 *      style change (WordConfigure), to know if the combination
 *      of differents attributes is allowed.
 *      For example, the 'Symbol' font doesn't exists in bold, or
 *      italic style.
 * Results:
 *      A boolean to tell if the Font_Attributes are correct.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
Check_Font_Attributes(AttrType type, void *value) {
  
  WordSegment *ws;
  FontItem *fi;
  char *fa;
  FontStyle st;
  int sz;
  Attribute *attr;

  ws = (WordSegment *)current.shape;
  fi = ws->get_ws_font();

  fa = (char *)fi->get_family();
  st = (FontStyle)fi->get_style();
  sz = (int)fi->get_size();

  attr = current.attr->get_attributes();

  while( attr != NULL ) {
    switch( attr->type & FONT_ATTR ) {

    case FONT_FAMILY_ATTR:
      fa = (char *)attr->value;
      break;

    case FONT_STYLE_ATTR:
      st = (FontStyle)attr->value;
      break;

    case FONT_SIZE_ATTR:
      sz = (int)attr->value;
      break;

    default:
      break;
    }
    attr = attr->next;
  }

  switch( type ) {
  case FONT_FAMILY_ATTR:
    fa = (char *)value;
    break;
    
  case FONT_STYLE_ATTR:
    st = (FontStyle)value;
    break;
    
  case FONT_SIZE_ATTR:
    sz = (int)value;
    break;
    
  default:
    break;
  }

  if( papyrus->get_font(fa, st, sz) != NULL )
    return TRUE;
  else
    return FALSE;
}
