/*
    Copyright (C) 2001 by Andrew Zabolotny

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef __IMAGE_H__
#define __IMAGE_H__

#include "rgbpixel.h"

/**
 * Output format flags for Image::Save ()
 */
/// A mask which isolates image type bits
#define IMAGE_TYPE_MASK         0x0000000f
/// Automatically determine the best output format
#define IMAGE_TYPE_AUTO         0x00000000
/// Save the image as a 24-bit image
#define IMAGE_TYPE_TRUECOLOR    0x00000001
/// Save the image as 8-bit indexed
#define IMAGE_TYPE_PALETTED8    0x00000002
/// Save the image as 4-bit indexed
#define IMAGE_TYPE_PALETTED4    0x00000003
/// Save the image as 8-bit grayscale
#define IMAGE_TYPE_GRAY8        0x00000004
/// Save the image as 4-bit grayscale
#define IMAGE_TYPE_GRAY4        0x00000005
/// Save the image as 1-bit grayscale (monochrome)
#define IMAGE_TYPE_MONO         0x00000006
/// Set this bit to save the alpha channel
#define IMAGE_TYPE_ALPHA        0x00000010
/// Write the transparent color chunk
#define IMAGE_TYPE_TRANSP       0x00000020
/// Dither image if quantizing
#define IMAGE_TYPE_DITHER	0x00000040

/**
 * This class represents an image object.
 * It provides loading/saving from a PNG file functionality (through libPNG)
 * as well as a limited amount of editing functions. Also it is able to save
 * GIF files through libungif (with latest libungif - uncompressed, to avoid
 * patent issues).
 */
class Image
{
  FILE *file;
  int filesize;

public:
  /// The actual image data
  RGBpixel *image;
  /// Image size
  unsigned width, height;
  /// The transparent color
  RGBpixel transp;

  /// Initialize the image object
  Image ();
  /// Destroy the image object
  ~Image ();
  /// Open the file and prepare to read from it
  bool Open (const char *fName);
  /// Close the file
  void Close ();
  /// Free the image
  void Free ();
  /// Load next image from file: this completely discards previously loaded one
  bool LoadPNG ();
  /// Save the image into a PNG file in given format
  bool SavePNG (const char *fName, unsigned format);
  /// Save the image into a GIF file in given format
  bool SaveGIF (const char *fName, unsigned format);
  /// Check if file is at EOF
  bool AtEOF ()
  { return ftell (file) >= filesize; }
  /// Crop the image by removing outer lines/columns containing just `transp'
  bool AutoCrop ();
  /// Crop the image
  bool Crop (unsigned x, unsigned y, unsigned w, unsigned h);
  /// Scale down the image twice with antialiasing
  void Scale2X ();
  /// Mark the `transp' color as having alpha=0
  void MarkTransparent ();
  /// Combine transparent pixels with background
  void Combine ();
};

#endif // __IMAGE_H__
