/*
    System-dependent routines
    Copyright (C) 2000 by Andrew Zabolotny

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef __SYSLIB_H__
#define __SYSLIB_H__

#include <string.h>
#include <sys/types.h>

/**
 * These constants can be used to identify one of the
 * three standard streams of a process.
 */
enum
{
  PROCESS_STDIN = 0,
  PROCESS_STDOUT = 1,
  PROCESS_STDERR = 2
};

/**
 * This is a process. The process is spawned in background,
 * and we have full access to its stdin/stdout/stderr.
 * The process name can have no path in its pathname,
 * in which case it will be found along the PATH.
 */
class Process
{
  /// Process PID
  pid_t pid;
  /// Our ends of stdio/stdout/stderr
  int handles [3];
  /// Read buffers for stdout/stderr
  char *readcache [2];
  /// How much data is contained in read cache
  int cachesize [2];

public:
  /// Spawn given program with given arguments (last argv[] should be NULL)
  Process (const char *program, char **argv);
  /// Kill the process (if it is still running)
  ~Process ();

  /// Check if process is still ok
  bool IsOk ()
  { return (pid != -1); }
  /// Get process ID
  pid_t GetPID ()
  { return pid; }
  /// Check if process haven't died yet
  bool IsDead ();
  /// Wait until process finishes and return its exit code (-1 = error)
  int Wait ();

  /// Check if we have anything buffered yet in stdout or stderr
  bool IsEmpty (int handle);

  /// Read from process's stdout or stderr to buffer
  size_t Read (int handle, void *buff, size_t buffsize);
  /// Write a buffer to process's stdin
  size_t Write (void *buff, size_t buffsize);

  /// Read a line from process's stdout or stderr
  bool GetLine (int handle, char *buff, size_t buffsize);
  /// Write a string to process's stdin
  bool PutLine (char *buff)
  { return !!Write (buff, strlen (buff)); }
};

// Sleep for given number of seconds
extern void Sleep (int Seconds);
// Get current process ID
extern pid_t GetPID ();
// Erase a file
extern bool EraseFile (char *FileName);

#endif // __SYSLIB_H__
