/*
    Miscelaneous image manipulations
    Copyright (C) 2001 by Andrew Zabolotny

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

extern "C" {
#include <zlib.h>
#include <png.h>
#include <gif_lib.h>
#include <malloc.h>
#include <unistd.h>
}

#include "sysdefs.h"
#include "image.h"
#include "quantize.h"

Image::Image () : file (NULL), image (NULL)
{
}

Image::~Image ()
{
  Close ();
  Free ();
}

bool Image::Open (const char *fName)
{
  file = fopen (fName, "rb");
  if (!file)
    return false;

  fseek (file, 0, SEEK_END);
  filesize = ftell (file);
  fseek (file, 0, SEEK_SET);

  return true;
}

void Image::Close ()
{
  if (file)
  {
    fclose (file);
    file = NULL;
  }
}

void Image::Free ()
{
  if (image)
  {
    delete [] image;
    image = NULL;
  }
}

bool Image::LoadPNG ()
{
  png_structp png;
  png_infop info;

  size_t rowbytes, exp_rowbytes;
  png_bytep *row_pointers;

  Free ();

  png = png_create_read_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png)
    return false;

  info = png_create_info_struct (png);
  if (!info)
  {
    png_destroy_read_struct (&png, (png_infopp) NULL, (png_infopp) NULL);
    png = NULL;
    return false;
  }

  png_init_io (png, file);

  if (setjmp (png->jmpbuf))
    // If we get here, we had a problem reading the file
    goto nomem;

  png_read_info (png, info);

  // Get picture info
  png_uint_32 Width, Height;
  int bit_depth, color_type;

  png_get_IHDR (png, info, &Width, &Height, &bit_depth, &color_type,
    NULL, NULL, NULL);

  if (bit_depth > 8)
    // tell libpng to strip 16 bit/color files down to 8 bits/color
    png_set_strip_16 (png);
  else if (bit_depth < 8)
    // Expand pictures with less than 8bpp to 8bpp
    png_set_packing (png);

  switch (color_type)
  {
    case PNG_COLOR_TYPE_GRAY:
    case PNG_COLOR_TYPE_GRAY_ALPHA:
      png_set_gray_to_rgb (png);
      break;
    case PNG_COLOR_TYPE_PALETTE:
      png_set_palette_to_rgb (png);
      break;
    case PNG_COLOR_TYPE_RGB:
    case PNG_COLOR_TYPE_RGB_ALPHA:
      break;
    default:
      goto nomem;
  }

  // If there is no alpha information, fill with 0xff
  if (!(color_type & PNG_COLOR_MASK_ALPHA))
  {
    // Expand paletted or RGB images with transparency to full alpha
    // channels so the data will be available as RGBA quartets.
    if (png_get_valid (png, info, PNG_INFO_tRNS))
      png_set_tRNS_to_alpha (png);
    else
      png_set_filler (png, 0xff, PNG_FILLER_AFTER);
  }

  // Update structure with the above settings
  png_read_update_info (png, info);

  // Allocate the memory to hold the image
  image = new RGBpixel [(width = Width) * (height = Height)];
  if (!image)
    goto nomem;
  exp_rowbytes = Width * sizeof (RGBpixel);

  rowbytes = png_get_rowbytes (png, info);
  if (rowbytes != exp_rowbytes)
    goto nomem;                         // Yuck! Something went wrong!

  row_pointers = new png_bytep [Height];

  if (!row_pointers
   || setjmp (png->jmpbuf))             // Set a new exception handler
  {
    delete [] row_pointers;
nomem:
    png_destroy_read_struct (&png, &info, (png_infopp) NULL);
    Free ();
    return false;
  }

  for (png_uint_32 row = 0; row < Height; row++)
    row_pointers [row] = ((png_bytep)image) + row * rowbytes;

  // Read image data
  png_read_image (png, row_pointers);

  // read rest of file, and get additional chunks in info_ptr
  png_read_end (png, (png_infop)NULL);

  // Free the row pointers array that is not needed anymore
  delete [] row_pointers;

  png_destroy_read_struct (&png, &info, (png_infopp) NULL);

  // Initialize transparent pixel with top-left corner pixel value
  transp.Set (image [0]);

  return true;
}

static inline int isqr (int x)
{ return x * x; }

// Compress 8-bit values into 4+4
static void compress_8l_to_44 (uint8 *image8, unsigned w, unsigned h)
{
  uint8 *src = image8;
  uint8 *dst = image8;
  unsigned x, y;
  for (y = h; y > 0; y--)
  {
    for (x = w; x > 1; x -= 2, src += 2)
      *dst++ = (src [0] << 4) | (src [1] & 0x0f);
    if (x == 1)
    {
      *dst++ = (src [0] << 4);
      src++;
    }
  }
}

// Compress 8-bit values into 4+4
static void compress_8h_to_44 (uint8 *image8, unsigned w, unsigned h)
{
  uint8 *src = image8;
  uint8 *dst = image8;
  unsigned x, y;
  for (y = h; y > 0; y--)
  {
    for (x = w; x > 1; x -= 2, src += 2)
      *dst++ = (src [0] & 0xf0) | (src [1] >> 4);
    if (x > 0)
    {
      *dst++ = (src [0] & 0xf0);
      src++;
    }
  }
}

// Compress 8-bit values into 1-bit
static void compress_8h_to_1 (uint8 *image8, unsigned w, unsigned h)
{
  uint8 *src = image8;
  uint8 *dst = image8;
  unsigned x, y;
  for (y = h; y > 0; y--)
  {
    for (x = w; x > 7; x -= 8, src += 8)
      *dst++ = ((src [0] & 0x80)     ) | ((src [1] & 0x80) >> 1) |
               ((src [2] & 0x80) >> 2) | ((src [3] & 0x80) >> 3) |
               ((src [4] & 0x80) >> 4) | ((src [5] & 0x80) >> 5) |
               ((src [6] & 0x80) >> 6) | ((src [7]       ) >> 7);
    if (x > 0)
    {
      *dst = 0;
      for (int shift = 0; x > 0; shift++, x--, src++)
        *dst |= (*src & 0x80) >> shift;
      dst++;
    }
  }
}

// Convert RGB image to grayscale
static void rgb_to_gray8 (RGBpixel *image, uint8 *&image8, unsigned pixels,
  bool alpha)
{
  image8 = new uint8 [pixels * (alpha ? 2 : 1)];
  RGBpixel *src = image;
  uint8 *dst = image8;
  if (alpha)
    for (; pixels; pixels--, src++)
    {
      *dst++ = src->Intensity ();
      *dst++ = src->alpha;
    }
  else
    for (; pixels; pixels--, src++)
      *dst++ = src->Intensity ();
}

static void gray8_to_mono (uint8 *image8, unsigned pixels)
{
  while (pixels--)
    *image8++ = !!*image8;
}

bool Image::SavePNG (const char *fName, unsigned format)
{
  /* Remove the file in the case it exists and it is a link */
  unlink (fName);
  /* open the file */
  FILE *fp = fopen (fName, "wb");
  if (fp == NULL)
    return false;

  png_structp png = png_create_write_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

  if (!png)
  {
error1:
    fclose (fp);
    return false;
  }

  /* Allocate/initialize the image information data. */
  png_infop info = png_create_info_struct (png);
  if (info == NULL)
  {
error2:
    png_destroy_write_struct (&png, (png_infopp)NULL);
    goto error1;
  }

  /* Catch processing errors */
  if (setjmp(png->jmpbuf))
    /* If we get here, we had a problem writing the file */
    goto error2;

  /* Set up the output function */
  png_init_io (png, fp);

  // Quantize the image, if required
  uint8 *image8 = NULL;
  RGBpixel *palette8 = NULL;
  int palette8size;
  if (((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_AUTO)
   || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED8)
   || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED4))
  {
    if ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED4)
      palette8size = 16;
    else
      palette8size = 256;
    QuantizeRGB (image, width * height, width,
      image8, palette8, palette8size, format & IMAGE_TYPE_DITHER,
      (format & IMAGE_TYPE_TRANSP) ? &transp : NULL);

    if ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_AUTO)
    {
      if (palette8size > 255)
        format = (format & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_TRUECOLOR;
      else if (palette8size > 16)
      {
        bool gray = true;
        for (int i = 0; i < palette8size; i++)
        {
          int y = palette8 [i].Intensity ();
          if ((isqr (palette8 [i].red - y) +
               isqr (palette8 [i].green - y) +
               isqr (palette8 [i].blue - y)) > (3*3)*3)
          {
            gray = false;
            break;
          }
        }
        format = (format & ~IMAGE_TYPE_MASK) |
          (gray ? IMAGE_TYPE_GRAY8 : IMAGE_TYPE_PALETTED8);
      }
      else if (palette8size > 2)
      {
        bool gray = true;
        for (int i = 0; i < palette8size; i++)
        {
          int y = (palette8 [i].Intensity ()) & 0xf0;
          y |= (y >> 4);
          if ((isqr (palette8 [i].red - y) +
               isqr (palette8 [i].green - y) +
               isqr (palette8 [i].blue - y)) > (15*15)*3)
          {
            gray = false;
            break;
          }
        }
        format = (format & ~IMAGE_TYPE_MASK) |
          (gray ? IMAGE_TYPE_GRAY4 : IMAGE_TYPE_PALETTED4);
      }
      else
        format = (format & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_MONO;

      if (((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED8)
       || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED4))
      {
        if (format & IMAGE_TYPE_ALPHA)
          // Paletted images do not support alpha channel
          format = (format & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_TRUECOLOR;
      }
      if (((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_GRAY4)
       || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_MONO))
      {
        if (format & IMAGE_TYPE_ALPHA)
          // Only 8-bit grayscale images support alpha channel
          format = (format & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_GRAY8;
      }

      if (((format & IMAGE_TYPE_MASK) != IMAGE_TYPE_PALETTED8)
       && ((format & IMAGE_TYPE_MASK) != IMAGE_TYPE_PALETTED4))
      {
        delete [] image8; image8 = NULL;
        delete [] palette8; palette8 = NULL;
      }
    }
  }

  /* Set the image information here.  Width and height are up to 2^31,
   * bit_depth is one of 1, 2, 4, 8, or 16, but valid values also depend on
   * the color_type selected. color_type is one of PNG_COLOR_TYPE_GRAY,
   * PNG_COLOR_TYPE_GRAY_ALPHA, PNG_COLOR_TYPE_PALETTE, PNG_COLOR_TYPE_RGB,
   * or PNG_COLOR_TYPE_RGB_ALPHA.  interlace is either PNG_INTERLACE_NONE or
   * PNG_INTERLACE_ADAM7, and the compression_type and filter_type MUST
   * currently be PNG_COMPRESSION_TYPE_BASE and PNG_FILTER_TYPE_BASE. REQUIRED
   */
  int colortype, rowlen, bits;
  // Transparent color in image-dependent format
  png_color_16 tc;
  tc.red = transp.red;
  tc.green = transp.green;
  tc.blue = transp.blue;
  tc.gray = transp.Intensity ();
  if (palette8)
    tc.index = 0;

  switch (format & IMAGE_TYPE_MASK)
  {
    case IMAGE_TYPE_PALETTED8:
      colortype = PNG_COLOR_TYPE_PALETTE;
      rowlen = width;
      bits = 8;
      break;
    case IMAGE_TYPE_PALETTED4:
      colortype = PNG_COLOR_TYPE_PALETTE;
      rowlen = (width + 1) / 2;
      bits = 4;
      compress_8l_to_44 (image8, width, height);
      break;
    case IMAGE_TYPE_GRAY8:
      colortype = (format & IMAGE_TYPE_ALPHA) ? PNG_COLOR_TYPE_GRAY_ALPHA : PNG_COLOR_TYPE_GRAY;
      rowlen = (format & IMAGE_TYPE_ALPHA) ? width * 2 : width;
      bits = 8;
      rgb_to_gray8 (image, image8, width * height, format & IMAGE_TYPE_ALPHA);
      break;
    case IMAGE_TYPE_GRAY4:
      colortype = PNG_COLOR_TYPE_GRAY;
      rowlen = (width + 1) / 2;
      bits = 4;
      rgb_to_gray8 (image, image8, width * height, false);
      compress_8h_to_44 (image8, width, height);
      tc.gray >>= 4;
      break;
    case IMAGE_TYPE_MONO:
      colortype = PNG_COLOR_TYPE_GRAY;
      rowlen = (width + 7) / 8;
      bits = 1;
      rgb_to_gray8 (image, image8, width * height, false);
      compress_8h_to_1 (image8, width, height);
      tc.gray >>= 7;
      break;
    case IMAGE_TYPE_TRUECOLOR:
      colortype = (format & IMAGE_TYPE_ALPHA) ? PNG_COLOR_TYPE_RGB_ALPHA : PNG_COLOR_TYPE_RGB;
      rowlen = width * sizeof (RGBpixel);
      bits = 8;
      break;
    default:
      // unknown format
      goto error2;
  } /* endswitch */
  png_set_IHDR (png, info, width, height, bits, colortype,
    PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);

  /* set the palette if there is one. */
  if (colortype & PNG_COLOR_MASK_PALETTE)
  {
    png_colorp palette = (png_colorp)malloc (palette8size * sizeof (png_color));
    for (int i = 0; i < palette8size; i++)
    {
      palette [i].red   = palette8 [i].red;
      palette [i].green = palette8 [i].green;
      palette [i].blue  = palette8 [i].blue;
    } /* endfor */
    png_set_PLTE (png, info, palette, palette8size);
  } /* endif */

  /* otherwise, if we are dealing with a color image then */
  png_color_8 sig_bit;
  memset (&sig_bit, 0, sizeof (sig_bit));
  if (((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_GRAY8)
   || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_GRAY4)
   || ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_MONO))
    sig_bit.gray = bits;
  else
  {
    sig_bit.red = 8;
    sig_bit.green = 8;
    sig_bit.blue = 8;
  }
  /* if the image has an alpha channel then */
  if (colortype & PNG_COLOR_MASK_ALPHA)
    sig_bit.alpha = bits;
  png_set_sBIT (png, info, &sig_bit);

  if (format & IMAGE_TYPE_TRANSP)
    png_set_tRNS (png, info, &tc.index, 1, &tc);

  /* Write the file header information. */
  png_write_info (png, info);

  if ((format & IMAGE_TYPE_MASK) == IMAGE_TYPE_TRUECOLOR)
    /* Get rid of filler (OR ALPHA) bytes, pack XRGB/RGBX/ARGB/RGBA into
     * RGB (4 channels -> 3 channels). The second parameter is not used.
     */
    if (!(colortype & PNG_COLOR_MASK_ALPHA))
      png_set_filler (png, 0, PNG_FILLER_AFTER);

  /* The easiest way to write the image (you may have a different memory
   * layout, however, so choose what fits your needs best).  You need to
   * use the first method if you aren't handling interlacing yourself.
   */
  png_bytep *row_pointers = new png_bytep [height];
  uint8 *ImageData = image8 ? image8 : (uint8 *)image;
  for (unsigned i = 0; i < height; i++)
    row_pointers [i] = ImageData + i * rowlen;

  /* One of the following output methods is REQUIRED */
  png_write_image (png, row_pointers);

  /* It is REQUIRED to call this to finish writing the rest of the file */
  png_write_end (png, info);

  /* if you malloced the palette, free it here */
  if (info->palette)
    free (info->palette);

  /* clean up after the write, and free any memory allocated */
  png_destroy_write_struct (&png, (png_infopp)NULL);

  /* Free the row pointers */
  delete [] row_pointers;

  /* Free the quantized image, if any */
  if (image8)
    delete [] image8;
  if (palette8)
    delete [] palette8;

  /* close the file */
  fclose (fp);

  /* that's it */
  return true;
}

bool Image::SaveGIF (const char *fName, unsigned format)
{
  /* Remove the file in the case it exists and it is a link */
  unlink (fName);
  /* open the file */
  GifFileType *gif = EGifOpenFileName (fName, 0);
  if (!gif)
    return false;

  // Quantize the image
  uint8 *image8 = NULL;
  RGBpixel *palette8 = NULL;
  int palette8size;
  switch (format & IMAGE_TYPE_MASK)
  {
    case IMAGE_TYPE_AUTO:
    case IMAGE_TYPE_PALETTED8:
      palette8size = 256;
      break;
    case IMAGE_TYPE_PALETTED4:
      palette8size = 16;
      break;
    case IMAGE_TYPE_MONO:
      palette8size = 2;
      break;
    default:
      // Output format not supported
      return false;
  }
  if (palette8size > 2)
    QuantizeRGB (image, width * height, width,
      image8, palette8, palette8size, format & IMAGE_TYPE_DITHER,
      (format & IMAGE_TYPE_TRANSP) ? &transp : NULL);
  else
  {
    rgb_to_gray8 (image, image8, width * height, false);
    gray8_to_mono (image8, width * height);
    palette8 = new RGBpixel [2];
    palette8 [0].Set (0, 0, 0);
    palette8 [1].Set (255, 255, 255);
    palette8size = 2;
  }

  /* Prepare the palette */
  GifColorType ColorMap [256];
  for (int color = 0; color < palette8size; color++)
  {
    ColorMap [color].Red = palette8 [color].red;
    ColorMap [color].Green = palette8 [color].green;
    ColorMap [color].Blue = palette8 [color].blue;
  }

  gif->SWidth = width;
  gif->SHeight = height;
  gif->SColorResolution = BitSize (palette8size);
  gif->SColorMap = MakeMapObject (1 << gif->SColorResolution, ColorMap);

  SavedImage img;
  memset (&img, 0, sizeof (img));
  img.ImageDesc.Width = width;
  img.ImageDesc.Height = height;
  img.RasterBits = (char *)image8;
  MakeSavedImage (gif, &img);

  if (format & IMAGE_TYPE_TRANSP)
  {
    char ExtData [4];
    ExtData [0] = 1;
    ExtData [1] = ExtData [2] = 0;
    ExtData [3] = 0;//find_closest_index (palette8, palette8size, transp);
    MakeExtension (gif->SavedImages, GRAPHICS_EXT_FUNC_CODE);
    AddExtensionBlock (gif->SavedImages, sizeof (ExtData), ExtData);
  }

  /* Write out the GIF */
  EGifSpew (gif);

  delete [] image8;
  delete [] palette8;

  /* that's it */
  return true;
}

bool Image::AutoCrop ()
{
  if (!image)
    return false;

  unsigned top_y, bot_y, left_x, right_x;

  for (top_y = 0; top_y < height; top_y++)
  {
    RGBpixel *cur = image + top_y * width;
    for (unsigned count = width; count > 0; count--, cur++)
      if (*cur != transp)
        goto top_stop;
  }
top_stop:

  for (bot_y = height - 1; bot_y >= top_y; bot_y--)
  {
    RGBpixel *cur = image + bot_y * width;
    for (unsigned count = width; count > 0; count--, cur++)
      if (*cur != transp)
        goto bot_stop;
  }
bot_stop:

  for (left_x = 0; left_x < width; left_x++)
  {
    RGBpixel *cur = image + top_y * width + left_x;
    for (unsigned count = top_y; count <= bot_y; count++, cur += width)
      if (*cur != transp)
        goto left_stop;
  }
left_stop:

  for (right_x = width - 1; right_x >= left_x; right_x--)
  {
    RGBpixel *cur = image + top_y * width + right_x;
    for (unsigned count = top_y; count <= bot_y; count++, cur += width)
      if (*cur != transp)
        goto right_stop;
  }
right_stop:

  // Don't allow image to be smaller than one pixel
  if (right_x <= left_x)
    left_x = right_x = 0;
  if (bot_y <= top_y)
    top_y = bot_y = 0;

  return Crop (left_x, top_y, right_x - left_x + 1, bot_y - top_y + 1);
}

bool Image::Crop (unsigned x, unsigned y, unsigned w, unsigned h)
{
  // Check if we should crop the image at all
  if ((x == 0) && (y == 0) && (w == width) && (h == height))
    return true;

  // Allright, now we allocate a new image and copy the cropped image there
  RGBpixel *newimage = new RGBpixel [w * h];
  if (!newimage)
    return false;

  for (unsigned dy = 0; dy < h; dy++)
  {
    RGBpixel *src = image + (y + dy) * width + x;
    RGBpixel *dst = newimage + dy * w;
    memcpy (dst, src, w * sizeof (RGBpixel));
  }

  Free ();
  image = newimage;
  width = w;
  height = h;
  return true;
}

void Image::Scale2X ()
{
  int nw = (width + 1) / 2;
  int nh = (height + 1) / 2;
  RGBpixel *newimg = new RGBpixel [nw * nh];

  RGBpixel *src = image;
  RGBpixel *dst = newimg;

  for (int h = height; h > 1; h -= 2)
  {
    for (int w = width; w > 1; w -= 2, dst++, src += 2)
    {
      RGBpixel p00 = src [0];
      RGBpixel p01 = src [1];
      RGBpixel p10 = src [width];
      RGBpixel p11 = src [width + 1];
      int a = p00.alpha + p01.alpha + p10.alpha + p11.alpha;
      if (a)
      {
        dst->red   = (p00.red   * p00.alpha + p01.red   * p01.alpha +
                      p10.red   * p10.alpha + p11.red   * p11.alpha) / a;
        dst->green = (p00.green * p00.alpha + p01.green * p01.alpha +
                      p10.green * p10.alpha + p11.green * p11.alpha) / a;
        dst->blue  = (p00.blue  * p00.alpha + p01.blue  * p01.alpha +
                      p10.blue  * p10.alpha + p11.blue  * p11.alpha) / a;
        dst->alpha = a >> 2;
      }
      else
        dst->Set (0, 0, 0, 0);
    }

    if (width & 1)
    {
      RGBpixel p00 = src [0];
      RGBpixel p10 = src [width];
      int a = p00.alpha + p10.alpha;
      if (a)
      {
        dst->red   = (p00.red   * p00.alpha + p10.red   * p10.alpha) / a;
        dst->green = (p00.green * p00.alpha + p10.green * p10.alpha) / a;
        dst->blue  = (p00.blue  * p00.alpha + p10.blue  * p10.alpha) / a;
        dst->alpha = a >> 1;
      }
      else
        dst->Set (0, 0, 0, 0);
      dst++; src++;
    }

    src += width;
  }

  if (height & 1)
  {
    for (int w = width; w > 1; w -= 2, dst++, src += 2)
    {
      RGBpixel p00 = src [0];
      RGBpixel p01 = src [1];
      int a = p00.alpha + p01.alpha;
      if (a)
      {
        dst->red   = (p00.red   * p00.alpha + p01.red   * p01.alpha) / a;
        dst->green = (p00.green * p00.alpha + p01.green * p01.alpha) / a;
        dst->blue  = (p00.blue  * p00.alpha + p01.blue  * p01.alpha) / a;
        dst->alpha = a >> 1;
      }
      else
        dst->Set (0, 0, 0, 0);
    }

    if (width & 1)
      *dst++ = *src++;
  }

  Free ();
  image = newimg;
  width = nw;
  height = nh;
}

void Image::MarkTransparent ()
{
  RGBpixel *src = image;
  for (int pixels = width * height; pixels > 0; pixels--, src++)
    if (src->eq (transp))
      src->alpha = 0;
}

void Image::Combine ()
{
  RGBpixel *src = image;
  for (int pixels = width * height; pixels > 0; pixels--, src++)
  {
    int a = src->alpha;
    if (a < 255)
    {
      src->red   = ((src->red   * a) + transp.red   * (255 - a)) / 255;
      src->green = ((src->green * a) + transp.green * (255 - a)) / 255;
      src->blue  = ((src->blue  * a) + transp.blue  * (255 - a)) / 255;
      src->alpha = 255;
    }
  }
}
