/*
    System-dependent routines
    Copyright (C) 2001 by Andrew Zabolotny

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "sysdefs.h"
#include "syslib.h"

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>

// Note there is a known bug in this routine: if fork succeeds but
// the executable won't be found along the PATH, it will return a pid
// anyway, although execvp itself will fail in the second copy.
// We don't care much about it anyway since if executable is missing,
// this will result in other errors which will be caught.
Process::Process (const char *program, char **argv)
{
  pid = -1;
  memset (&readcache, 0, sizeof (readcache));
  memset (&cachesize, 0, sizeof (cachesize));
  memset (&handles, 0, sizeof (handles));

  // Create three pipes, for stdin, stdout and stderr
  int _i [2], _o [2], _e [2];
  if (pipe (_i))
    return;
  if (pipe (_o))
  {
err1:
    close (_i [0]);
    close (_i [1]);
    return;
  }
  if (pipe (_e))
  {
err2:
    close (_o [0]);
    close (_o [1]);
    goto err1;
  }

  // Save our stdin/stdout/stderr
  int old_stdin = dup (0);
  int old_stdout = dup (1);
  int old_stderr = dup (2);
  if ((old_stdin < 0)
   || (old_stdout < 0)
   || (old_stderr < 0))
  {
err3:
    close (old_stdin);
    close (old_stdout);
    close (old_stderr);
    close (_e [0]);
    close (_e [1]);
    goto err2;
  }

  // Put the ends of pipes as if we were the child process
  if ((dup2 (_i [0], 0) < 0)
   || (dup2 (_o [1], 1) < 0)
   || (dup2 (_e [1], 2) < 0))
  {
    dup2 (old_stdin, 0);
    dup2 (old_stdout, 1);
    dup2 (old_stderr, 2);
    goto err3;
  }

  // Allright, now launch the child process
  pid = fork ();
  if (!pid)
    // in child
    exit (execvp (program, argv));

  // Restore stdin/stdout/stderr
  dup2 (old_stdin, 0);
  dup2 (old_stdout, 1);
  dup2 (old_stderr, 2);
  close (old_stdin);
  close (old_stdout);
  close (old_stderr);

  // Close the ends of pipes we don't need anymore
  close (_i [0]);
  close (_o [1]);
  close (_e [1]);

  // Remember our ends of pipes
  handles [0] = _i [1];
  handles [1] = _o [0];
  handles [2] = _e [0];

  // Set the pipes we're going to read into nonblocking mode
  fcntl (handles [1], F_SETFL, O_NONBLOCK);
  fcntl (handles [2], F_SETFL, O_NONBLOCK);
}

Process::~Process ()
{
  if (pid != -1)
    kill (pid, SIGTERM);

  // Close the pipes
  close (handles [0]);
  close (handles [1]);
  close (handles [2]);
}

bool Process::IsDead ()
{
  if (pid == -1)
    return true;

  int status;
  pid_t res = waitpid (pid, &status, WNOHANG);
  if ((res == pid)
   || (errno == ECHILD))
  {
    pid = -1;
    return true;
  }

  return false;
}

int Process::Wait ()
{
  if (pid == -1)
    return -1;

  int status;
  if ((waitpid (pid, &status, 0) != pid)
   || !WIFEXITED (status))
    return -1;
  pid = -1;
  return WEXITSTATUS (status);
}

bool Process::GetLine (int handle, char *buff, size_t buffsize)
{
  if (handle < 1 || handle > 2)
    return false;

  char *&c = readcache [handle - 1];
  int &cs = cachesize [handle - 1];

  // If we have something in cache, move to buffer
  if (cs)
  {
    // Check if buffer contains a newline
    size_t i, min = cs;
    if (min > buffsize)
      min = buffsize;
    for (i = 0; i < min; i++)
    {
      buff [i] = c [i];
      if (buff [i] == '\n')
      {
        buff [i] = 0;
        cs -= i + 1;
        memmove (c, c + i + 1, cs);
        return true;
      }
    }
    buff += i;
    buffsize -= i;
    // If process is already dead, return what is available
    if (pid == -1)
    {
      buff [0] = 0;
      free (c);
      c = NULL;
      cs = 0;
      return true;
    }
  }

  size_t r = (buffsize > 0) ? read (handles [handle], buff, buffsize - 1) : 0;

  bool rc = false;
  if (r && r != size_t (-1))
  {
    if (r >= buffsize - 1)
    {
      rc = true;
      buff [r] = 0;
    }

    size_t i;
    for (i = 0; i < r; i++)
      if (buff [i] == '\n')
        break;

    // If we found a newline, move the rest of line to cache
    // Otherwise move entire buffer to the cache
    if (i < r)
    {
      rc = true;
      buff [i] = 0;
      buff += i + 1;
      r -= i + 1;
    }
    else
    {
      buff -= cs;
      r += cs;
    }

    c = (char *)realloc (c, cs = r);
    memcpy (c, buff, r);
  }

  return rc;
}

bool Process::IsEmpty (int handle)
{
  if (handle < 1 || handle > 2)
    return true;
  if (cachesize [handle - 1])
    return false;

  // Try to read something from the handle
  char buff [100];
  size_t r = read (handles [handle], buff, sizeof (buff));
  if (r)
  {
    readcache [handle - 1] = (char *)realloc (readcache [handle - 1],
      cachesize [handle - 1] = r);
    memcpy (readcache [handle - 1], buff, r);
    return false;
  }
  return true;
}

size_t Process::Read (int handle, void *buff, size_t buffsize)
{
  if (handle < 1 || handle > 2)
    return 0;
  return read (handles [handle], buff, buffsize);
}

size_t Process::Write (void *buff, size_t buffsize)
{
  return write (handles [0], buff, buffsize);
}

//------------------------------------------------------------------------------

void Sleep (int Seconds)
{
  sleep (Seconds);
}

pid_t GetPID ()
{
  return getpid ();
}

bool EraseFile (char *FileName)
{
  return (unlink (FileName) == 0);
}
