/*
    PostScript to PNG/GIF conversion utility
    Copyright (C) 2001 Andrew Zabolotny

    This program will take a PostScript file on input and will create a lot
    of PNG/GIF files (one per page of the original PS file). The program can
    render high-quality output by specifying a larger size for GhostScript
    output, and then by scaling down the result with antialiasing (or by
    using alpha-transparency).

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>

#include "rgbpixel.h"
#include "image.h"
#include "syslib.h"

enum file_format_t
{
  ffGIF, ffPNG
};

static struct
{
  int Density;
  char *Paper;
  char *OutputTemplate;
  bool Verbose;
  file_format_t OutputFileFormat;
  unsigned OutputFormat;
  int Antialias;
  RGBpixel Background;
  bool UseBackground;
  bool Dither;
} opt = 
{
  72,
  NULL,
  "img%d.%s",
  false,
  ffPNG,
  IMAGE_TYPE_AUTO,
  1,
  RGBpixel (0, 0, 0),
  false,
  false
};

static char *programname;

static void display_version ()
{
  printf ("ps2png   Version 0.1.0  Copyright (C) 2001 Andrew Zabolotny\n"); 
}

static void display_help ()
{
  display_version ();
  printf ("\nUsage: %s {option/s} [input.ps{ ...}]\n", programname);
  printf ("  -a# --antialias=#  Antialias images by creating them # times larger\n");
  printf ("                     and then scaling them down (# - a power of two)\n");
  printf ("  -b# --background=# Define the background color\n");
  printf ("  -d# --dither=#     Do dithering when quantizing colors (default: %s)\n", opt.Dither ? "yes" : "no");
  printf ("                     Dithering has effect only when saving paletted images\n");
  printf ("  -f# --format=#     Output image format:\n");
  printf ("                     p[t|a][c|p8|p4|g8|g4|m] - output PNG file\n");
  printf ("                     g[t][p8|p4|m] - output GIF file\n");
  printf ("                       t - use top-left pixel as transparency\n");
  printf ("                       a - use per-pixel alpha instead of transparency\n");
  printf ("                       c - output truecolor (24 bit) image\n");
  printf ("                       p4,p8 - paletted 4- and 8-bits-per-pixel format\n");
  printf ("                       g4,g8 - grayscale 4- and 8-bits-per-pixel format\n");
  printf ("                       m - monochrome output image format\n");
  printf ("  -o# --output=#     Create files using template (default:%s)\n", opt.OutputTemplate);
  printf ("  -p# --papersize=#  Use the specific paper size\n");
  printf ("  -r# --density=#    Output density in pixels per inch (default: %d)\n", opt.Density);
  printf ("  -h  --help         Display usage help\n");
  printf ("  -v  --verbose      Verbose operation\n");
  printf ("  -V  --version      Show program version\n");
  printf ("\n");
  printf ("If output format is PNG with alpha channel, and an antialiasing factor is\n");
  printf ("selected, then antialiasing with transparency will be done.\n");
  printf ("\n");
  printf ("If output format doesn't support alpha transparency, or it has not been\n");
  printf ("enabled, or the -b option has been specified the `background color' will\n");
  printf ("be blended into the final image to get a completely opaque image.\n");
  printf ("\n");
  printf ("Examples:\n");
  printf ("  ps2img tiger.ps\n");
  printf ("  ps2img images.ps -r144 -a2 -b#30e0ff\n");
  printf ("  ps2img tiger.ps -fpp8t -a2 -d1\n");
  printf ("\n");
  exit (-1);
}

static char *strnew (char *s)
{
  if (!s)
    return NULL;
  size_t sl = strlen (s) + 1;
  char *ret = new char [sl];
  memcpy (ret, s, sl);
  return ret;
}

static int progress_x = 0;
static bool progerss_first = true;

static void progress_init (const char *msg)
{
  printf (msg); fflush (stdout);
  progress_x = strlen (msg);
  progerss_first = true;
}

static void progress (char *page)
{
  size_t sl = 1 + strlen (page) + 2;
  if (progress_x + sl > 79)
  {
    printf ("\n");
    progress_x = 0;
  }
  printf (" [%s]", page);
  fflush (stdout);
  progress_x += sl;
}

static void prepare_gs_argv (int &argc, char **argv)
{
  argv [argc++] = "gs";
  argv [argc++] = "-q";
  argv [argc++] = "-dNOPAUSE";
  argv [argc++] = "-dNO_PAUSE";
  argv [argc++] = "-dBATCH";
  
  static char _paper [30];
  if (opt.Paper)
  {
    snprintf (_paper, sizeof (_paper), "-sPAPERSIZE=%s", opt.Paper);
    argv [argc++] = _paper;
  }

  // For some reason bounding box device SIGSEGV's when -r switch
  // is given on command line. For this reason we don't put it into
  // argv here, but only before doing actual rendering.
}

static void display_argv (char **argv)
{
  for (int i = 0; argv [i]; i++)
    printf ("%s%s", i ? " " : "", argv [i]);
  printf ("\n");
}

static bool compute_bounding_box (char *fname,
  int &xmin, int &ymin, int &xmax, int &ymax)
{
  char *argv [20];
  int argc = 0;

  prepare_gs_argv (argc, argv);
  argv [argc++] = "-sDEVICE=bbox";
  argv [argc++] = fname;
  argv [argc] = NULL;

  printf ("Running GhostScript to compute bounding boxes...\n");

  if (opt.Verbose)
    display_argv (argv);

  Process gs ("gs", argv);
  if (!gs.IsOk ())
  {
    fprintf (stderr, "%s: error running GhostScript\n", programname);
    return false;
  }

  xmin = 99999; ymin = 99999; xmax = 0; ymax = 0;
  char line [200];
  while (!gs.IsDead ()
      || !gs.IsEmpty (PROCESS_STDERR))
    // Try to read a line of GhostScript output
    if (gs.GetLine (PROCESS_STDERR, line, sizeof (line)))
    {
      if (!strncmp (line, "%%BoundingBox:", 14))
      {
        int coords [4];
        int count = 0;
        char *cur = line + 15;
        while (count < 4)
        {
          coords [count++] = atoi (cur);
          cur = strchr (cur, ' ');
          if (!cur) break;
          cur += strspn (cur, " \t");
        }

        // If we get strange numbers, ignore them
	// GhostScript has a bug which leads to very
	// strange `bounds' for empty pages.
	if (count == 4
	 && ((coords [0] < 0 && coords [1] < 0)
          || (coords [2] > 5000 && coords [3] > 5000)))
	  count = 0;

        if (count == 4)
        {
          if (coords [0] < xmin) xmin = coords [0];
          if (coords [1] < ymin) ymin = coords [1];
          if (coords [2] > xmax) xmax = coords [2];
          if (coords [3] > ymax) ymax = coords [3];
        }
      }
    }
    else
      Sleep (1);

  if (xmin > 0) xmin--;
  if (ymin > 0) ymin--;
  xmax++, ymax++;

  return true;
}

static char *ps_to_png (char *fname, int xmin, int ymin, int xmax, int ymax)
{
  char *argv [20];
  int argc = 0;

  prepare_gs_argv (argc, argv);
  argv [argc++] = "-sDEVICE=png16m";

  char _density [20];
  snprintf (_density, sizeof (_density), "-r%d", opt.Density);
  argv [argc++] = _density;

  char _outputfile [40];
  snprintf (_outputfile, sizeof (_outputfile), "/tmp/ps2png.%d.png", GetPID ());

  char _output [40];
  snprintf (_output, sizeof (_output), "-sOutputFile=%s", _outputfile);
  argv [argc++] = _output;

  argv [argc++] = "-";
  argv [argc] = NULL;

  FILE *inf = fopen (fname, "rb");
  if (!inf)
  {
    fprintf (stderr, "%s: cannot open file `%s' for reading\n",
      programname, fname);
    return NULL;
  }

  printf ("Running GhostScript for rendering...\n");

  if (opt.Verbose)
    display_argv (argv);

  Process gs ("gs", argv);
  if (!gs.IsOk ())
  {
    fprintf (stderr, "%s: error running GhostScript\n", programname);
    return NULL;
  }

  if (opt.Verbose)
    printf ("Common bounding box: %d,%d - %d,%d\n", xmin, ymin, xmax, ymax);
  progress_init ("Rendering:");

  // Feed the PostScript file to GhostScript, making on the fly some
  // dirty substitutions to get output of needed size. This includes
  // replacing "%%BoundingBox ..." with some PostScript commands
  // (boundingbox is ignored by GS output driver) and removing
  // paper size setting commands.

  enum { omThrough, omPaperSize } mode = omThrough;
  while (!feof (inf))
  {
    char line [1000];
    if (!fgets (line, sizeof (line), inf))
      break;
    if (!strncmp (line, "%%BoundingBox:", 14))
      sprintf (line, "<< /PageOffset [%d %d] /PageSize [%d %d] >> setpagedevice\n",
        -xmin, ymin, xmax - xmin, ymax - ymin);
    else if (!strncmp (line, "%%BeginPaperSize", 16))
      mode = omPaperSize;
    else if (!strncmp (line, "%%Page:", 7))
    {
      char *page = line + 7;
      page += strspn (page, " \t");
      char *eol = strchr (page, ' ');
      if (eol)
      {
        *eol = 0;
	progress (page);
	*eol = ' ';
      }
    }

    switch (mode)
    {
      case omThrough:
        gs.PutLine (line);
        break;
      case omPaperSize:
        if (!strncmp (line, "%%EndPaperSize", 14))
	  mode = omThrough;
        break;
    }
  }
  fclose (inf);
  printf (" done\n");

  // Tell GhostScript to quit after processing
  gs.PutLine ("quit\n");

  // Wait until process finishes
  gs.Wait ();

  return strnew (_outputfile);
}

static void convert_file (char *fname)
{
  int xmin, ymin, xmax, ymax;

  if (!compute_bounding_box (fname, xmin, ymin, xmax, ymax))
    return;

  if ((xmin >= xmax) || (ymin >= ymax))
  {
    printf ("%s: empty bounding box detected!\n", programname);
    return;
  }

  char *outputfile = ps_to_png (fname, xmin, ymin, xmax, ymax);
  if (!outputfile)
    return;

  Image img;
  if (!img.Open (outputfile))
  {
    fprintf (stderr, "%s: cannot open PNG image file %s\n", programname, outputfile);
    return;
  }

  progress_init ("Saving:");

  int page = 1;
  while (!img.AtEOF ())
  {
    if (!img.LoadPNG ())
    {
      fprintf (stderr, "\n%s: error reading next page from file %s\n",
        programname, outputfile);
      return;
    }
    char tmp [20];
    sprintf (tmp, "%d", page);
    progress (tmp);

    // Crop the image
    img.AutoCrop ();

    // Mark white color as transparent
    img.MarkTransparent ();

    // Antialias the image
    for (int a = opt.Antialias; a > 1; a >>= 1)
      img.Scale2X ();

    // Combine with background color if required
    if (opt.UseBackground)
      img.transp.Set (opt.Background);

    if (!(opt.OutputFormat & IMAGE_TYPE_ALPHA)
     || opt.UseBackground)
      img.Combine ();

    // Save the page
    char _outname [200];
    snprintf (_outname, sizeof (_outname), opt.OutputTemplate, page,
      opt.OutputFileFormat == ffPNG ? "png" : "gif");

    bool succ = false;
    switch (opt.OutputFileFormat)
    {
      case ffPNG:
        succ = img.SavePNG (_outname, opt.OutputFormat);
        break;
      case ffGIF:
        succ = img.SaveGIF (_outname, opt.OutputFormat);
        break;
    }

    if (!succ)
    {
      fprintf (stderr, "\n%s: cannot write output file %s\n",
        programname, _outname);
      return;
    }

    page++;
  }
  puts (" done");
  // Remove temporary PNG file
  EraseFile (outputfile);
  delete [] outputfile;
}

static void ParseOutputFormat (const char *arg)
{
  switch (*arg)
  {
    case 'p':
      opt.OutputFileFormat = ffPNG;
      break;
    case 'g':
      opt.OutputFileFormat = ffGIF;
      break;
    default:
      fprintf (stderr, "%s: unknown output file format (%c) requested\n",
        programname, *arg);
      exit (-1);
  }
  arg++;
  opt.OutputFormat = IMAGE_TYPE_AUTO;
  while (*arg)
  {
    if (*arg == 'p')
    {
      arg++;
      switch (*arg)
      {
        case '4':
          opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_PALETTED4;
          break;
        case '8':
          opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_PALETTED8;
          break;
        default:
          fprintf (stderr, "%s: unknown paletted bits-per-pixel (%c) requested\n",
            programname, *arg);
          exit (-1);
      }
    }
    else if (*arg == 'g')
    {
      if (opt.OutputFileFormat == ffGIF)
      {
        fprintf (stderr, "%s: GIF format does not support grayscale images\n",
          programname);
        exit (-1);
      }
      arg++;
      switch (*arg)
      {
        case '4':
          opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_GRAY4;
          break;
        case '8':
          opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_GRAY8;
          break;
        default:
          fprintf (stderr, "%s: unknown grayscale bits-per-pixel (%c) requested\n",
            programname, *arg);
          exit (-1);
      }
    }
    else if (*arg == 'm')
      opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_MONO;
    else if (*arg == 'c')
      opt.OutputFormat = (opt.OutputFormat & ~IMAGE_TYPE_MASK) | IMAGE_TYPE_TRUECOLOR;
    else if (*arg == 'a')
      opt.OutputFormat |= IMAGE_TYPE_ALPHA;
    else if (*arg == 't')
      opt.OutputFormat |= IMAGE_TYPE_TRANSP;
    else
    {
      fprintf (stderr, "%s: unknown image format option (%c)\n",
        programname, *arg);
      exit (-1);
    }
    arg++;
  }

  // Check for format (in)compatibilities
  switch (opt.OutputFileFormat)
  {
    case ffGIF:
      if (opt.OutputFormat & IMAGE_TYPE_ALPHA)
      {
        fprintf (stderr, "%s: per-pixel alpha not supported for GIF images\n",
          programname);
        exit (-1);
      }
      break;
    case ffPNG:
      if (opt.OutputFormat & IMAGE_TYPE_ALPHA)
        if (((opt.OutputFormat & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED4)
         || ((opt.OutputFormat & IMAGE_TYPE_MASK) == IMAGE_TYPE_PALETTED8))
        {
          fprintf (stderr, "%s: per-pixel alpha not supported for paletted PNG images\n",
            programname);
          exit (-1);
        }
        else if (((opt.OutputFormat & IMAGE_TYPE_MASK) == IMAGE_TYPE_GRAY4)
              || ((opt.OutputFormat & IMAGE_TYPE_MASK) == IMAGE_TYPE_MONO))
        {
          fprintf (stderr, "%s: alpha supported only for grayscale PNG images with 8 bpp\n",
	    programname);
          exit (-1);
        }
      if ((opt.OutputFormat & (IMAGE_TYPE_ALPHA | IMAGE_TYPE_TRANSP)) ==
          (IMAGE_TYPE_ALPHA | IMAGE_TYPE_TRANSP))
      {
        fprintf (stderr, "%s: simultaneous alpha and transparency not supported for PNG images!\n",
          programname);
        exit (-1);
      }
      break;
  }
}

static void ParseColor (RGBpixel &color, char *arg)
{
  int r, g, b;
  bool succ;

  if (arg [0] == '#')
    succ = (sscanf (arg + 1, "%02x%02x%02x", &r, &g, &b) == 3);
  else
    succ = (sscanf (arg, "%d,%d,%d", &r, &g, &b) == 3);

  if (!succ)
  {
    fprintf (stderr, "%s: invalid color specified (%s): 'r,g,b' or `#rrggbb' expected\n",
      programname, arg);
    exit (-1);
  }

  color.red = r;
  color.green = g;
  color.blue = b;
}

int main (int argc, char **argv)
{
  static struct option long_options[] =
  {
    {"format", required_argument, 0, 'f'},
    {"density", required_argument, 0, 'r'},
    {"papersize", required_argument, 0, 'p'},
    {"output", required_argument, 0, 'o'},
    {"antialias", required_argument, 0, 'a'},
    {"background", required_argument, 0, 'b'},
    {"dither", required_argument, 0, 'd'},
    {"verbose", no_argument, 0, 'v'},
    {"help", no_argument, 0, 'h'},
    {"version", no_argument, 0, 'V'},
    {0, 0, 0, 0}
  };

  programname = argv [0];

  int c;
  while ((c = getopt_long (argc, argv, "a:b:f:r:p:o:d:hv", long_options, 0)) != EOF)
    switch (c)
    {
      case '?':
	// unknown option
	return -1;
      case 'f':
        ParseOutputFormat (optarg);
        break;
      case 'r':
	opt.Density = atoi (optarg);
        if ((opt.Density < 72) || (opt.Density > 1400))
        {
          fprintf (stderr, "%s: invalid resolution given (%s): valid range 72..1400\n",
            programname, optarg);
          exit (-1);
        }
	break;
      case 'p':
	opt.Paper = optarg;
	break;
      case 'o':
	opt.OutputTemplate = optarg;
	break;
      case 'd':
        if (!strcasecmp (optarg, "yes")
	 || !strcasecmp (optarg, "true")
	 || !strcmp (optarg, "1"))
          opt.Dither = true;
	else if (!strcasecmp (optarg, "no")
	 || !strcasecmp (optarg, "false")
	 || !strcmp (optarg, "0"))
          opt.Dither = false;
	else
	{
	  fprintf (stderr, "%s: use 1/0 or yes/no for dithering flags (got `%s')\n",
	    programname, optarg);
	  exit (-1);
	}
	break;
      case 'a':
        opt.Antialias = atoi (optarg);
        if ((opt.Antialias < 1) || (opt.Antialias > 16)
         || (opt.Antialias & (opt.Antialias - 1)))
        {
          fprintf (stderr, "%s: invalid antialiasing scale factor (%s), allowed: 2,4,8,16\n",
            programname, optarg);
          exit (-1);
        }
        break;
      case 'b':
        ParseColor (opt.Background, optarg);
        opt.UseBackground = true;
        break;
      case 'v':
	opt.Verbose = true;
	break;
      case 'h':
	display_help ();
	return -1;
      case 'V':
	display_version ();
	return -1;
      default:
	// oops!
	abort ();
    }

  // Multiply density the number of times we need to scale image down
  opt.Density *= opt.Antialias;
  // If dithering is enabled, set respective bit
  if (opt.Dither)
    opt.OutputFormat |= IMAGE_TYPE_DITHER;

  // If no files on command-line, display help
  if (optind >= argc)
    display_help ();

  // Interpret the non-option arguments as file names
  for (; optind < argc; ++optind)
    convert_file (argv [optind]);

  return 0;
}
