/*
 *
 *  (c) COPYRIGHT INRIA, 1996-2001.
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */
 
/*----------------------------------------------------------------------
   
   Application Program Interface                     
   Language management                                          
  ----------------------------------------------------------------------*/
#include "thot_sys.h"
#include "constmedia.h"
#include "typemedia.h"
#include "libmsg.h"
#include "message.h"
#include "language.h"
#include "application.h"
#include "dictionary.h"

struct Langue_Ctl   LangTable[MAX_LANGUAGES];
static char         Langbuffer[2 * MAX_NAME_LENGTH];
static char         Langbuffer1[2 * MAX_NAME_LENGTH];
static char         Langbuffer2[2 * MAX_NAME_LENGTH];
static char         CodeBuffer[2 * MAX_NAME_LENGTH];
static int          breakPoints[MAX_POINT_COUP];
static char         StandardLANG[3];
static char        *dictPath;	/* Environment variable DICOPAR */
int                 FreeEntry;
int                 FirstUserLang;

#include "thotmsg_f.h"

/*	ISO 639 CODES ALPHABETIC BY LANGUAGE NAME (ENGLISH SPELLING) */
typedef unsigned char aLangName[MAX_LENGTH];
typedef struct _ISO639entry
  {
    aLangName	fullName;
    aLangName	code;
  }
ISO639entry;

static ISO639entry	ISO639table[] =
{
	{"ABKHAZIAN",      "AB"},
	{"AFAN (OROMO)",   "OM"},
	{"AFAR",           "AA"},
	{"AFRIKAANS",      "AF"},
	{"ALBANIAN",       "SQ"},
	{"AMHARIC",        "AM"},
	{"ARABIC",         "AR"},
	{"ARMENIAN",       "HY"},
	{"ASSAMESE",       "AS"},
	{"AYMARA",         "AY"},
	{"AZERBAIJANI",    "AZ"},
	{"BASHKIR",        "BA"},
	{"BASQUE",         "EU"},
	{"BENGALI",        "BN"},
	{"BANGLA",         "BN"},
	{"BHUTANI",        "DZ"},
	{"BIHARI",         "BH"},
	{"BISLAMA",        "BI"},
	{"BRETON",         "BR"},
	{"BULGARIAN",      "BG"},
	{"BURMESE",        "MY"},
	{"BYELORUSSIAN",   "BE"},
	{"CAMBODIAN",      "KM"},
	{"CATALAN",        "CA"},
	{"CHINESE",        "ZH"},
	{"CORSICAN",       "CO"},
	{"CROATIAN",       "HR"},
	{"Czech",          "CS"},
	{"DANISH",         "DA"},
	{"Dutch",          "NL"},
	{"English",        "EN"},
	{"ESPERANTO",      "EO"},
	{"ESTONIAN",       "ET"},
	{"FAROESE",        "FO"},
	{"FIJI",           "FJ"},
	{"Finnish",        "FI"},
	{"French",         "FR"},
	{"FRISIAN",        "FY"},
	{"GALICIAN",       "GL"},
	{"GEORGIAN",       "KA"},
	{"German",         "DE"},
	{"Greek",          "EL"},
	{"GREENLANDIC",    "KL"},
	{"GUARANI",        "GN"},
	{"GUJARATI",       "GU"},
	{"HAUSA",          "HA"},
	{"HEBREW",         "HE"},
	{"HINDI",          "HI"},
	{"HUNGARIAN",      "HU"},
	{"ICELANDIC",      "IS"},
	{"INDONESIAN",     "ID"},
	{"INTERLINGUA",    "IA"},
	{"INTERLINGUE",    "IE"},
	{"INUKTITUT",      "IU"},
	{"INUPIAK",        "IK"},
	{"IRISH",          "GA"},
	{"Italian",        "IT"},
	{"JAPANESE",       "JA"},
	{"JAVANESE",       "JV"},
	{"KANNADA",        "KN"},
	{"KASHMIRI",       "KS"},
	{"KAZAKH",         "KK"},
	{"KINYARWANDA",    "RW"},
	{"KIRGHIZ",        "KY"},
	{"KURUNDI",        "RN"},
	{"KOREAN",         "KO"},
	{"KURDISH",        "KU"},
	{"LAOTHIAN",       "LO"},
	{"LATIN",          "LA"},
	{"LATVIAN",        "LV"},
	{"LETTISH",        "LV"},
	{"LINGALA",        "LN"},
	{"LITHUANIAN",     "LT"},
	{"MACEDONIAN",     "MK"},
	{"MALAGASY",       "MG"},
	{"MALAY",          "MS"},
	{"MALAYALAM",      "ML"},
	{"MALTESE",        "MT"},
	{"MAORI",          "MI"},
	{"MARATHI",        "MR"},
	{"MOLDAVIAN",      "MO"},
	{"MONGOLIAN",      "MN"},
	{"NAURU",          "NA"},
	{"NEPALI",         "NE"},
	{"NORWEGIAN",      "NO"},
	{"OCCITAN",        "OC"},
	{"ORIYA",          "OR"},
	{"PASHTO",         "PS"},
	{"PUSHTO",         "PS"},
	{"PERSIAN",        "FA"},
	{"Polish",         "PL"},
	{"Portuguese",     "PT"},
	{"PUNJABI",        "PA"},
	{"QUECHUA",        "QU"},
	{"RHAETO-ROMANCE", "RM"},
	{"ROMANIAN",       "RO"},
	{"RUSSIAN",        "RU"},
	{"SAMOAN",         "SM"},
	{"SANGHO",         "SG"},
	{"SANSKRIT",       "SA"},
	{"SCOTS GAELIC",   "GD"},
	{"SERBIAN",        "SR"},
	{"SERBO-CROATIAN", "SH"},
	{"SESOTHO",        "ST"},
	{"SETSWANA",       "TN"},
	{"SHONA",          "SN"},
	{"SINDHI",         "SD"},
	{"SINGHALESE",     "SI"},
	{"SISWATI",        "SS"},
	{"SLOVAK",         "SK"},
	{"SLOVENIAN",      "SL"},
	{"SOMALI",         "SO"},
	{"Spanish",        "ES"},
	{"SUNDANESE",      "SU"},
	{"SWAHILI",        "SW"},
	{"Swedish",        "SV"},
	{"TAGALOG",        "TL"},
	{"TAJIK",          "TG"},
	{"TAMIL",          "TA"},
	{"TATAR",          "TT"},
	{"TELUGU",         "TE"},
	{"THAI",           "TH"},
	{"TIBETAN",        "BO"},
	{"TIGRINYA",       "TI"},
	{"TONGA",          "TO"},
	{"TSONGA",         "TS"},
	{"TURKISH",        "TR"},
	{"TURKMEN",        "TK"},
	{"TWI",            "TW"},
	{"UIGUR",          "UG"},
	{"UKRAINIAN",      "UK"},
	{"URDU",           "UR"},
	{"UZBEK",          "UZ"},
	{"VIETNAMESE",     "VI"},
	{"VOLAPUK",        "VO"},
	{"WELSH",          "CY"},
	{"WOLOF",          "WO"},
	{"XHOSA",          "XH"},
	{"YIDDISH",        "YI"},
	{"YORUBA",         "YO"},
	{"ZHUANG",         "ZA"},
	{"ZULU",           "ZU"},
	{"",               ""}
};

/* this table associates the ancient language names used in Thot documents
   with their standard code */
static ISO639entry	OldLangTable[] =
{
	{"American",    "EN-US"},
	{"Deutsch",     "DE"},
	{"Espa\361ol",  "ES"},
	{"Fran\347ais", "FR"},
	{"ISO_latin_1", "x-Latin1"},
	{"ISO_latin_2", "x-Latin2"},
	{"Italiano",    "IT"},
	{"Symbol",      "x-Symbol"},
	{"",            ""}
};


/*----------------------------------------------------------------------
   TtaGetLanguageNameFromCode
   Returns the full name of a language whose RFC-1766 code is known
  ----------------------------------------------------------------------*/
char *TtaGetLanguageNameFromCode (char *code)
{
  int                 i;

  Langbuffer[0] = EOS;
  for (i = 0; Langbuffer[0] == EOS && ISO639table[i].code[0] != EOS; i++)
    {
      if (!strcasecmp (code, ISO639table[i].code))
	strcpy (Langbuffer, ISO639table[i].fullName);
    }
  if (Langbuffer[0] == EOS)
    for (i = 0; Langbuffer[0] == EOS && OldLangTable[i].code != EOS; i++)
      {
	if (!strcasecmp (code, OldLangTable[i].code))
	  strcpy (Langbuffer, OldLangTable[i].fullName);
      }
  return Langbuffer;
}


/*----------------------------------------------------------------------
   TtaGetLanguageCodeFromName
   Returns the RFC-1766 code for a language whose name is known
  ----------------------------------------------------------------------*/
char *TtaGetLanguageCodeFromName (char *name)
{
  int                 i;

  CodeBuffer[0] = EOS;
  for (i = 0; CodeBuffer[0] == EOS && ISO639table[i].fullName[0] != EOS; i++)
    {
      if (!strcasecmp (name, ISO639table[i].fullName))
	strcpy (CodeBuffer, ISO639table[i].code);
    }
  if (CodeBuffer[0] == EOS)
    for (i = 0; CodeBuffer[0] == EOS && OldLangTable[i].fullName[0] != EOS; i++)
      {
	if (!strcasecmp (name, OldLangTable[i].fullName))
	  strcpy (CodeBuffer, OldLangTable[i].code);
      }
  return CodeBuffer;
}


/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
void InitLanguage ()
{
   int                 i, j;

   /* Initialization of remaining entries of the language table */
   for (i = 0; i < MAX_LANGUAGES; i++)
     {
	LangTable[i].LangName[0] = EOS;
	LangTable[i].LangCode[0] = EOS;
	LangTable[i].LangAlphabet = 'L';
	for (j = 0; j < MAX_DICTS; j++)
	   LangTable[i].LangDict[j] = NULL;
	LangTable[i].LangPattern[0] = EOS;
	LangTable[i].LangTabPattern.Charge = 0;
     }
   /* Loading the default system languages */
   i = 0;
   strcpy (LangTable[i].LangName, "ISO_latin_1");
   strcpy (LangTable[i].LangCode, "la");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Usigle");
   strcpy (LangTable[i].LangSecondary, "Uname");

   i = 1;
   strcpy (LangTable[i].LangName, "ISO_latin_2");
   strcpy (LangTable[i].LangCode, "x-Latin2");
   LangTable[i].LangAlphabet = '2';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   i = 2;
   strcpy (LangTable[i].LangName, "ISO_latin_9");
   strcpy (LangTable[i].LangCode, "x-Latin9");
   LangTable[i].LangAlphabet = '9';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   i = 3;
   strcpy (LangTable[i].LangName, "Symbol");
   strcpy (LangTable[i].LangCode, "x-Symbol");
   LangTable[i].LangAlphabet = 'G';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   /* Loading the default user languages */
   FirstUserLang = 4;
   i = 4;
   strcpy (LangTable[i].LangName, "French");
   strcpy (LangTable[i].LangCode, "fr");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Fprinc");
   strcpy (LangTable[i].LangSecondary, "Fperso");
   strcpy (LangTable[i].LangPattern, "francais.ptn");

   i = 5;
   strcpy (LangTable[i].LangName, "English");
   strcpy (LangTable[i].LangCode, "en");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Eprinc");
   strcpy (LangTable[i].LangSecondary, "Eperso");
   strcpy (LangTable[i].LangPattern, "english.ptn");

   i = 6;
   strcpy (LangTable[i].LangName, "American");
   strcpy (LangTable[i].LangCode, "en-US");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Eprinc");
   strcpy (LangTable[i].LangSecondary, "Eperso");
   strcpy (LangTable[i].LangPattern, "american.ptn");

   i = 7;
   strcpy (LangTable[i].LangName, "German");
   strcpy (LangTable[i].LangCode, "de");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Gprinc");
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "deutsch.ptn");

   i = 8;
   strcpy (LangTable[i].LangName, "Italian");
   strcpy (LangTable[i].LangCode, "it");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Iprinc");
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "italiano.ptn");

   i = 9;
   strcpy (LangTable[i].LangName, "Spanish");
   strcpy (LangTable[i].LangCode, "es");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Sprinc");
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "espanol.ptn");

   i = 10;
   strcpy (LangTable[i].LangName, "Portuguese");
   strcpy (LangTable[i].LangCode, "pt");
   LangTable[i].LangAlphabet = 'L';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "portug.ptn");

   i = 11;
   strcpy (LangTable[i].LangName, "Dutch");
   strcpy (LangTable[i].LangCode, "nl");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Nprinc");
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "nederl.ptn");

   i = 12;
   strcpy (LangTable[i].LangName, "Swedish");
   strcpy (LangTable[i].LangCode, "sv");
   LangTable[i].LangAlphabet = 'L';
   strcpy (LangTable[i].LangPrincipal, "Wprinc");
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "swedish.ptn");

   i = 13;
   strcpy (LangTable[i].LangName, "Finnish");
   strcpy (LangTable[i].LangCode, "fi");
   LangTable[i].LangAlphabet = 'L';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;
   strcpy (LangTable[i].LangPattern, "finish.ptn");

   i = 14;
   strcpy (LangTable[i].LangName, "Greek");
   strcpy (LangTable[i].LangCode, "el");
   LangTable[i].LangAlphabet = 'G';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   i = 15;
   strcpy (LangTable[i].LangName, "Czech");
   strcpy (LangTable[i].LangCode, "cs");
   LangTable[i].LangAlphabet = '2';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   i = 17;
   strcpy (LangTable[i].LangName, "Polish");
   strcpy (LangTable[i].LangCode, "pl");
   LangTable[i].LangAlphabet = '2';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;

   i = 18;
   strcpy (LangTable[i].LangName, "Turkish");
   strcpy (LangTable[i].LangCode, "tr");
   LangTable[i].LangAlphabet = '9';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;
   FreeEntry = 19;

   i = 19;
   strcpy (LangTable[i].LangName, "Icelandic");
   strcpy (LangTable[i].LangCode, "is");
   LangTable[i].LangAlphabet = 'L';
   LangTable[i].LangPrincipal[0] = EOS;
   LangTable[i].LangSecondary[0] = EOS;
   FreeEntry = 19;
}


/*----------------------------------------------------------------------
   TtaNewLanguage

   Not available for TYPO languages.
   Declares a new language, its alphabet and optionally the names of the
   principal ans secondary dictionaries. All languages used in a Thot
   document must be explicitely declared, except for predefined languages.
   Thot registers all declared languages and allocates a different
   identifier to each of them.
   This function does not load dictionaries but loads their name.
   If principalDictionary and/or secondDictionary are not NULL values,
   dictionaries are loaded when necessary from files
   $DICOPAR/principalDictionary and/or $DICOPAR/secondDictionary.
   If an application redeclares an existing language, this new declaration
   has no effect.
   Parameters:
   languageName: name of the language according RFC 1766.
   languageAlphabet: alphabet to be used for writing that language:
   `L' for ISO-Latin-1, `G' for Symbol (Greek).
   principalDictionary: name of the principal dictionary or NULL.
   secondDictionary: name of the secondary dictionary or NULL.
   Return value:
   identifier of the new language or 0 if the language cannot be added.
  ----------------------------------------------------------------------*/
Language TtaNewLanguage (char *languageName, char languageAlphabet,
			 char *principalDictionary, char *secondDictionary)
{
   int                 i;

   i = 0;
   /* Avoids error cases */
   if (languageName == NULL)
      TtaError (ERR_invalid_parameter);
   else if (languageName[0] == EOS)
      TtaError (ERR_invalid_parameter);
   else if (strlen (languageName) >= MAX_NAME_LENGTH)
      TtaError (ERR_string_too_long);
   else if (FreeEntry == MAX_LANGUAGES)
      TtaError (ERR_too_many_languages);
   else
     {
        if (strlen (languageName) == 2 ||
	   languageName[1] == '-' || languageName[2] == '-')
	   /* it's an ISO-639 code */
	  {
	  for (i = 0; i < FreeEntry; i++)
	    if (!strcasecmp (languageName, LangTable[i].LangCode))
	      /* The language is already defined */
	      return i;
	  strcpy (LangTable[FreeEntry].LangName, languageName);
	  strcpy (LangTable[FreeEntry].LangCode, TtaGetLanguageCodeFromName (languageName));
	  }
	else
	  {
	  /* Consults the languages table to see if the language exists. */
	  for (i = 0; i < FreeEntry; i++)
	    if (!strcasecmp (languageName, LangTable[i].LangName))
	      /* The language is already defined */
	      return i;
	  strcpy (LangTable[FreeEntry].LangName, languageName);
	  strcpy (LangTable[FreeEntry].LangCode, TtaGetLanguageCodeFromName (languageName));
	  }

	/* Saves the new language */
	i = FreeEntry;
	LangTable[i].LangAlphabet = languageAlphabet;
	if (principalDictionary != NULL)
	  {
	     strncpy (LangTable[i].LangPrincipal, principalDictionary, MAX_NAME_LENGTH);
	     LangTable[i].LangPrincipal[MAX_NAME_LENGTH - 1] = EOS;
	  }
	else
	   LangTable[i].LangPrincipal[0] = EOS;
	if (secondDictionary != NULL)
	  {
	     strncpy (LangTable[i].LangSecondary, secondDictionary, MAX_NAME_LENGTH);
	     LangTable[i].LangSecondary[MAX_NAME_LENGTH - 1] = EOS;
	  }
	else
	   LangTable[i].LangSecondary[0] = EOS;
	FreeEntry++;
     }
   return i;
}


/*----------------------------------------------------------------------
   TtaRemoveLanguage

   Remove a language from the Thot language table.
   Parameters:
       language: the language to be removed.
  ----------------------------------------------------------------------*/
void TtaRemoveLanguage (Language language)
{
   int                 i;

   i = (int) language;
   if (i >= FreeEntry || i < 0)
      TtaError (ERR_language_not_found);
   else
      {
      LangTable[i].LangName[0] = EOS;
      LangTable[i].LangCode[0] = EOS;
      /* don't erase LangAlphabet */
      LangTable[i].LangPrincipal[0] = EOS;
      LangTable[i].LangSecondary[0] = EOS;
      LangTable[i].LangPattern[0]   = EOS;
      }
}

/*----------------------------------------------------------------------
   TtaGetLanguageIdFromName

   Available for TYPO languages.
   Returns the identifier of a language that matches a language name.
   Parameters:
   name: name of the language.
   Return value:
   identifier of that language or 0 if the language is unknown.
  ----------------------------------------------------------------------*/
Language TtaGetLanguageIdFromName (char *name)
{ 
  char                code[MAX_LENGTH];
  int                 i;
  ThotBool            again;

  i = 0;
  /* Avoids error cases */
  if (name == NULL)
    TtaError (ERR_invalid_parameter);
  else if (name[0] == EOS)
    TtaError (ERR_invalid_parameter);
  else if (strlen (name) >= MAX_NAME_LENGTH)
    TtaError (ERR_string_too_long);
  else
    {
      /* Consults the languages table to see if the language exists. */
      again = TRUE;
      while (again && i < FreeEntry)
	{
	  if (!strcasecmp (name, LangTable[i].LangCode))
	    /* The language is already defined */
	    again = FALSE;
	  else if (!strcasecmp (name, LangTable[i].LangName))
	    again = FALSE;
	  else
	    i++;
	}
      if (again)
	{
	  strcpy (code, TtaGetLanguageCodeFromName (name));
	  if (code[0] != EOS)
	    {
	      i = 0;
	      while (again && i < FreeEntry)
		if (!strcasecmp (code, LangTable[i].LangCode))
                  /* The language is already defined */
		  again = FALSE;
		else
		  i++;
	    }
	}
      if (again)
	{
	  /* The language does not exist */
	  i = 0;
	  TtaError (ERR_language_not_found);
	}
    }
  /* returned value */
  return (Language) i;
}

/*----------------------------------------------------------------------
   TtaGetVarLANG

   Returns the 2 first chars of environment variable LANG or 'fr'.
   Return value:
   a string of 2 chars.
  ----------------------------------------------------------------------*/
char *TtaGetVarLANG ()
{ 
   char   *langEVar = TtaGetEnvString ("LANG");

   if (langEVar == NULL)
     strcpy (StandardLANG, "en");
   else if (!strcmp (langEVar, "C") ||
	    !strcasecmp (langEVar, "iso_8859_1"))
     strcpy (StandardLANG, "fr");
   else
     {
       strncpy (StandardLANG, langEVar, 2);
       StandardLANG[2] = EOS;
     }
   return (StandardLANG);
}

/*----------------------------------------------------------------------
   TtaGetDefaultLanguage

   Returns the identifier of the default language.
   Return value:
   identifier of the default language.
  ----------------------------------------------------------------------*/
Language TtaGetDefaultLanguage ()
{
   return TtaGetLanguageIdFromName (TtaGetVarLANG ());
}

/*----------------------------------------------------------------------
   TtaGetLanguageIdFromAlphabet

   Returns the identifier of the first language that uses a given alphabet
   `L' -> ISO_latin_1, `G' -> Symbol.
   Parameters:
   languageAlphabet: the alphabet of interest.
   Return value:
   identifier of that language or 0 if the language is unknown.
  ----------------------------------------------------------------------*/
Language TtaGetLanguageIdFromAlphabet (char languageAlphabet)
{
   int                 i;
   ThotBool            again;

   i = 0;
   /* Consults the languages table to see if the language exists. */
   again = TRUE;
   while (again && i < FreeEntry)
     {
       if (languageAlphabet == LangTable[i].LangAlphabet)
	 /* The language is already defined */
	 again = FALSE;
       else
	 i++;
     }
   
   if (again)
     {
       /* The language does not exist */
       i = 0;
       TtaError (ERR_language_not_found);
     }
   return (Language) i;
}


/*----------------------------------------------------------------------
   TtaGetAlphabet

   Not available for TYPO languages. Returns the alphabet of a language.
   Parameters:
   languageId: name of the language.
   Return value:
   a character that identifies the alphabet ('L' = latin, 'G' = greek).
  ----------------------------------------------------------------------*/
char TtaGetAlphabet (Language languageId)
{
   int                 i;

   i = (int) languageId;
   /* Verification of the parameter */
   if (i >= FreeEntry || i < 0)
     {
	TtaError (ERR_language_not_found);
	return (EOS);
     }
   return LangTable[i].LangAlphabet;
}


/*----------------------------------------------------------------------
   TtaGetLanguageName

   Not available for TYPO languages. Returns the name of a given language.
   Parameters:
   languageId: identifier of the language.
   Return value:
   the name of the language.
  ----------------------------------------------------------------------*/
char *TtaGetLanguageName (Language languageId)
{
   int                 i;

   i = (int) languageId;
   if (i >= FreeEntry || i < 0)
     {
	TtaError (ERR_language_not_found);
	Langbuffer1[0] = EOS;
     }
   else
      strcpy (Langbuffer1, LangTable[i].LangName);
   return Langbuffer1;
}


/*----------------------------------------------------------------------
   TtaGetLanguageCode

   Not available for TYPO languages.
   Returns the RFC-1766 code of a given language.
   Parameters:
   languageId: identifier of the language.
   Return value:
   the code of the language.
  ----------------------------------------------------------------------*/
char *TtaGetLanguageCode (Language languageId)
{
  int                 i;

  i = (int) languageId;
  if (i >= FreeEntry || i < 0)
    {
      TtaError (ERR_language_not_found);
      Langbuffer2[0] = EOS;
    }
  else
    strcpy (Langbuffer2, LangTable[i].LangCode);
  return Langbuffer2;
}


/*----------------------------------------------------------------------
   TtaGetNumberOfLanguages

   Not available for TYPO languages.
   Returns the current number of languages known by Thot.
   Return value:
   the current number of languages.
  ----------------------------------------------------------------------*/
int TtaGetNumberOfLanguages ()
{
  return FreeEntry;
}


/*----------------------------------------------------------------------
   TtaGetFirstUserLanguage

   Not available for TYPO languages.
   Returns the first user language known by Thot.
   Return value:
   the first user language number.
  ----------------------------------------------------------------------*/
int TtaGetFirstUserLanguage ()
{
  return FirstUserLang;
}


/*----------------------------------------------------------------------
  GetPatternList: Read the pattern file of the language and creates the
  appropriate structure.
  Return 0 if problem else returns 1 
  ----------------------------------------------------------------------*/
ThotBool GetPatternList (Language langageId)
{
   FILE               *in;
   unsigned char       patternGot[MAX_LET_PATTERN];
   char                weightGot[MAX_LET_PATTERN + 1];
   char                patternFileName[THOT_MAX_CHAR];
   char               *ptPattern;
   int                 lang;
   int                 currentIndex;
   int                 previousLength;
   int                 i, lg;

   dictPath = TtaGetEnvString ("DICOPAR");
   if (dictPath == NULL)
     /* The environment variable DICOPAR does not exist */
     return (FALSE);

   strcpy (patternFileName, dictPath);
   strcat (patternFileName, DIR_STR);
   lang = (int) langageId;
   ptPattern = LangTable[lang].LangPattern;
   strcat (patternFileName, ptPattern);
   if ((in = fopen (patternFileName, "r")) == NULL)
     return (FALSE);

   currentIndex = 0;
   previousLength = 0;
   i = 0;
   while ((fscanf (in, "%s %s", patternGot, weightGot)) != EOF)
     {
	i++;
	lg = strlen (patternGot);
	if (lg != previousLength)
	  {
	     previousLength = lg;
	     currentIndex++;
	     LangTable[lang].LangTabPattern.ind_pattern[previousLength] = i;
	  }
	strcpy (LangTable[lang].LangTabPattern.liste_pattern[i].CarPattern, patternGot);
	strcpy (LangTable[lang].LangTabPattern.liste_pattern[i].PoidsPattern, weightGot);
     }
   LangTable[lang].LangTabPattern.NbPatt = i;
   LangTable[lang].LangTabPattern.Charge = 1;
   fclose (in);
   return (TRUE);
}


/*----------------------------------------------------------------------
  FoundPatternInList verifies if a string belongs to the pattern list.
  if true, it returns 1 else 0 
  ----------------------------------------------------------------------*/
static char *FoundPatternInList (Language langageId,
				 unsigned char substring[MAX_LET_PATTERN])
{
   int                 language;
   int                 lgstring;
   int                 i, max;
   struct PatternList *ptrTabPattern;

   language = (int) langageId;
   lgstring = strlen (substring);
   if (lgstring >= MAX_LET_PATTERN)
     return (NULL);
   else
     {
       ptrTabPattern = &LangTable[language].LangTabPattern;
       i = ptrTabPattern->ind_pattern[lgstring];
       if (i == 0)
	 return (NULL);
       else
	 {
	   /* search last index */
	   max = lgstring + 1;
	   if (max == MAX_LET_PATTERN)
	     max = ptrTabPattern->NbPatt;
	   else
	     {
	       max = ptrTabPattern->ind_pattern[max];
	       if (max == 0)
		 max = ptrTabPattern->NbPatt;
	     }
	   while (i < max)
	     {
	       if (!strcmp (ptrTabPattern->liste_pattern[i].CarPattern, substring))
		 return (ptrTabPattern->liste_pattern[i].PoidsPattern);
	       i++;
	     }
	   return (NULL);
	 }
     }
}


/*----------------------------------------------------------------------
   * FoundHyphenPoints: apply Liang algo. on a word and returns the 
   * hypen points.
  ----------------------------------------------------------------------*/
static void FoundHyphenPoints (Language langageId, char wordToCut[THOT_MAX_CHAR])
{
   int                 lang;
   unsigned char       wordToTreat[THOT_MAX_CHAR];	/* "." + wordToCut + "." */
   unsigned char       subword[THOT_MAX_CHAR];
   char               *weight_subword;
   int                 tab_weight[THOT_MAX_CHAR];
   int                 wordLength;
   int                 size_subword;
   int                 currentPosition;
   int                 i, j;

   lang = (int) langageId;
   wordLength = strlen (wordToCut) + 2;
   if (wordLength > THOT_MAX_CHAR)
     return;

   for (i = 0; i < THOT_MAX_CHAR; i++)
      tab_weight[i] = 0;
   strcpy (wordToTreat, ".");
   strcat (wordToTreat, wordToCut);
   strcat (wordToTreat, ".");
   size_subword = 1;
   while ((size_subword <= wordLength) && (size_subword <= MAX_LET_PATTERN))
     {
	currentPosition = 0;
	while ((currentPosition + size_subword) <= wordLength)
	  {
	     j = 0;
	     for (i = currentPosition; j < size_subword; i++)
		subword[j++] = wordToTreat[i];
	     subword[j] = 0;
	     if ((weight_subword = FoundPatternInList (lang, subword)))
	       {
		  for (j = 0; j <= size_subword; j++)
		     if (weight_subword[j] > tab_weight[currentPosition + j])
			tab_weight[currentPosition + j] = weight_subword[j];
	       }
	     currentPosition++;
	  }
	size_subword++;
     }
   j = 0;
   for (i = 3; i < (wordLength - 2); i++)
      if (ISHYPHENABLE (tab_weight[i]))
	 breakPoints[j++] = i - 1;
   breakPoints[j] = 0;
}


/*----------------------------------------------------------------------
 TtaGetPatternHyphenList 
   returns a pointer on the list of values representing the hyphen points
   or NULL 
  ----------------------------------------------------------------------*/
int *TtaGetPatternHyphenList (char word[THOT_MAX_CHAR], Language languageId)
{
   int                 language;
   int                 i;

   language = (int) languageId;
   if (word[0] == EOS)
      return (NULL);
   if (strlen (word) < 2)
      return (NULL);
   if (LangTable[language].LangPattern[0] == EOS)
      /* Language without a pattern */
      return (NULL);

   if (!LangTable[language].LangTabPattern.Charge)
      /* patterns not loaded by the language */
      if (!GetPatternList (languageId))
	 return (NULL);

   for (i = 0; i < MAX_POINT_COUP; i++)
      breakPoints[i] = 0;

   FoundHyphenPoints (languageId, word);
   return (breakPoints);

}

/*----------------------------------------------------------------------
 * TtaExistPatternList verifies if a list of patterns is defined
 * for a given language
  ----------------------------------------------------------------------*/
ThotBool            TtaExistPatternList (Language languageId)
{
   int                 language;

   language = (int) languageId;
   if (LangTable[language].LangPattern[0] != EOS)
      return TRUE;
   else
      return FALSE;
}
