// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qmsgbox.h>
#include "Cache.h"
#include "FileConn.h"
#include "HttpConn.h"
#include "Request.h"

Cache::Cache()
{
//    _data.setAutoDelete( TRUE );
}

Cache::~Cache()
{
    while ( _data.first() ) {
	delete _data.first();
	_data.remove();
    }
    while ( _active.first() ) {
	delete _active.first();
	_active.remove();
    }
    while ( _pending.first() ) {
	delete _pending.first();
	_pending.remove();
    }
    while ( _reqs.first() ) {
	delete _reqs.first();
	_reqs.remove();
    }
}

bool Cache::getUrl( Request* request, const Url& url, bool reload )
{
    CacheData* d;

    // Search for complete data in cache.
    if ( ( d = findData( url ) ) ) {
        if ( reload ) {
            // Destroy cached data for reload.
            _data.removeRef( d );
            delete d;
        } else {
            d->refcount++;
            Req* r = new Req;
            r->req  = request;
            r->data = d;
            _pending.append( r );
            startTimer( 0 );
            connect( r->req, SIGNAL( done( Request* ) ), this, SLOT( requestDone( Request* ) ) );
            return TRUE;
        }
    }

    // Searh for an active connection (partial data may be available).
    if ( ( d = findActive( url ) ) ) {
        d->refcount++;
        Req* r = new Req;
        r->req  = request;
        r->data = d;
        _pending.append( r );
        startTimer( 0 );
        connect( r->req, SIGNAL( done( Request* ) ), this, SLOT( requestDone( Request* ) ) );
        connect( r->data->conn, SIGNAL( status( QString ) ), r->req, SLOT( fwdStatus( QString ) ) );
        connect( r->data->conn, SIGNAL( urlChanged( const Url& ) ), r->req, SLOT( fwdUrlChanged( const Url& ) ) );
        return TRUE;
    }

    // Create a new connection.
    d = new CacheData( url );
    d->complete = FALSE;
    if ( url.method() == "file" ) {
        d->conn = new FileConn( url );
    } else if ( url.method() == "http" ) {
        d->conn = new HttpConn( url );
    } else {
        QString error;
        error.sprintf( "Unrecognized method '%s'", url.method().data() );
        QMessageBox::message( "QWeb: Error", error );
        delete d;
        return FALSE;
    }

    connect( d->conn, SIGNAL( data( const char*, int ) )            , this, SLOT( data( const char*, int ) ) );
    connect( d->conn, SIGNAL( endOfData() )                         , this, SLOT( endOfData() ) );
    connect( d->conn, SIGNAL( startOfData( QString, QString, int ) ), this, SLOT( startOfData( QString, QString, int ) ) );
    connect( d->conn, SIGNAL( destroyed() )                         , this, SLOT( connDestroyed() ) );

    _active.append( d );

    d->refcount++;
    Req* r = new Req;
    r->req  = request;
    r->data = d;
    _pending.append( r );
    startTimer( 0 );
    connect( r->req, SIGNAL( done( Request* ) ), this, SLOT( requestDone( Request* ) ) );
    connect( r->data->conn, SIGNAL( status( QString ) ), r->req, SLOT( fwdStatus( QString ) ) );
    connect( r->data->conn, SIGNAL( urlChanged( const Url& ) ), r->req, SLOT( fwdUrlChanged( const Url& ) ) );

    if ( d->conn->open() ) {
        return TRUE;
    } else {
        _pending.removeRef( r );
//        delete d->conn;
        return FALSE;
    }
}

void Cache::requestDone( Request* req )
{
    Req* r;
    
    for ( r = _reqs.first(); r; r = _reqs.next() ) {
        if ( r->req == req ) {
            if ( r->data ) {
                r->data->refcount--;
                if ( r->data->refcount <= 0 ) {
                    delete r->data->conn;
                }
            }
            _reqs.remove();
            return;
        }
    }
}

void Cache::clear()
{
    _data.clear();
}

void Cache::connDestroyed()
{
    Connection* conn = (Connection*)sender();
    CacheData* d = findActive( conn );
    if ( d ) {
        _active.removeRef( d );
        delete d;
    }
}

void Cache::startOfData( QString mediaType, QString mediaSubtype, int totalSize )
{
    Connection* conn = (Connection*)sender();
    CacheData* d = findActive( conn );
    if ( d ) {
        if ( mediaType.isNull() ) {
            d->mediaType = "";
        } else {
            d->mediaType = mediaType.copy();
        }

        if ( mediaSubtype.isNull() ) {
            d->mediaSubtype = "";
        } else {
            d->mediaSubtype = mediaSubtype.copy();
        }

        d->totalSize = totalSize;
    }
}

void Cache::data( const char* bytes, int length )
{
    Connection* conn = (Connection*)sender();
    CacheData* d = findActive( conn );
    if ( d ) {
        uint oldSize = d->data.size();
        d->data.resize( oldSize + length );
        for ( int i = 0; i < length; i++ ) {
            d->data[oldSize+i] = bytes[i];
        }
    }
}

void Cache::endOfData()
{
    Connection* conn = (Connection*)sender();
    CacheData* d = findActive( conn );
    if ( d ) {
        d->conn     = 0;
        d->complete = TRUE;
        _active.removeRef( d );

        // Remove all requests associated with this CacheData.
        Req* r;
        for ( r = _reqs.first(); r; r = _reqs.next() ) {
            if ( r->data == d ) {
                r->data = 0;
            }
        }

        // We don't cache data for "file" method URLs.
//        if ( d->url.method() == "file" ) {
//            delete d;
//        } else {
            _data.append( d );
//        }
    }
}

CacheData* Cache::findData( const Url& url )
{
    CacheData* d;
    for ( d = _data.first(); d; d = _data.next() ) {
        if ( ( d->url.method()     == url.method()     ) &&
             ( d->url.user()       == url.user()       ) &&
             ( d->url.password()   == url.password()   ) &&
             ( d->url.hostname()   == url.hostname()   ) &&
             ( d->url.path()       == url.path()       ) &&
             ( d->url.query()      == url.query()      ) &&
             ( d->url.parameters() == url.parameters() ) &&
             ( d->url.port()       == url.port()       ) ) {
            return d;
        }
    }

    return 0;
}

CacheData* Cache::findActive( const Url& url )
{
    CacheData* d;
    for ( d = _active.first(); d; d = _active.next() ) {
        if ( ( d->url.method()     == url.method()     ) &&
             ( d->url.user()       == url.user()       ) &&
             ( d->url.password()   == url.password()   ) &&
             ( d->url.hostname()   == url.hostname()   ) &&
             ( d->url.path()       == url.path()       ) &&
             ( d->url.query()      == url.query()      ) &&
             ( d->url.parameters() == url.parameters() ) &&
             ( d->url.port()       == url.port()       ) ) {
            return d;
        }
    }

    return 0;
}

CacheData* Cache::findActive( Connection* conn )
{
    CacheData* d;
    for ( d = _active.first(); d; d = _active.next() ) {
        if ( d->conn == conn ) {
            return d;
        }
    }

    return 0;
}

void Cache::timerEvent( QTimerEvent* )
{
    killTimers();

    // Copy outstanding reqs to a separate list before emitting signals!
    Req*       r;
    QList<Req> reqs;
    for ( r = _pending.first(); r; r = _pending.first() ) {
        _pending.remove();
        reqs.append( r );
    }

    // Now process the pending requests.
    for ( r = reqs.first(); r; r = reqs.first() ) {
        // Send the start of data signal.
        if ( ( !r->data->mediaType.isNull() ) && ( !r->data->mediaSubtype.isNull() ) ) {
            r->req->fwdStartOfData( r->data->mediaType, r->data->mediaSubtype, r->data->totalSize );
        }

        // Send the data signal.
        uint len = r->data->data.size();
        uint pos;
        for ( pos = 0; pos < len; pos += 1024 ) {
            r->req->fwdData( r->data->data.data() + pos, pos + 1024 > len ? len - pos : 1024 );
        }

        // Send endOfData signal if complete, else connect the request to the active connection.
        if ( r->data->complete ) {
            r->req->fwdEndOfData();
        } else {
            // Active connection, connect it to the request.
            if ( r->data->conn ) {
                connect( r->data->conn, SIGNAL( data( const char*, int ) )            , r->req, SLOT( fwdData( const char*, int ) ) );
                connect( r->data->conn, SIGNAL( endOfData() )                         , r->req, SLOT( fwdEndOfData() ) );
                connect( r->data->conn, SIGNAL( startOfData( QString, QString, int ) ), r->req, SLOT( fwdStartOfData( QString, QString, int ) ) );
                connect( r->data->conn, SIGNAL( status( QString ) )                   , r->req, SLOT( fwdStatus( QString ) ) );
                connect( r->data->conn, SIGNAL( urlChanged( const Url& ) )            , r->req, SLOT( fwdUrlChanged( const Url& ) ) );
                _reqs.append( r );
            }
        }
        reqs.remove();
    }
}
