/* $Id: suidcgi.c,v 1.3 1998/07/05 17:12:55 sverrehu Exp $ */
/**************************************************************************
 *
 *  FILE            suidcgi.c
 *  MODULE OF       suidcgi - a set UID wrapper for CGI scripts.
 *
 *  DESCRIPTION     This is a setuid wrapper, intended to run CGI scripts
 *                  accessed from the World Wide Web. The script will be
 *                  run as the user owning it, giving access to files
 *                  without making them readable and writable to all.
 *
 *  WRITTEN BY      Sverre H. Huseby <sverrehu@online.no>
 *
 **************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

extern char **environ;

#ifndef PATH_MAX
#define PATH_MAX 4095
#endif

#include "config.h"

/**************************************************************************
 *                                                                        *
 *                       P R I V A T E    D A T A                         *
 *                                                                        *
 **************************************************************************/

static char executable[PATH_MAX];  /* the program to start */
static char wwwRootDir[PATH_MAX];  /* root of user's WWW-hierarchy */
static char *realName;             /* name of real user (HTTP) */
static char *effName;              /* name of effective user (program owner) */
static char *effHome;              /* home of effective user (program owner) */
static char **newEnviron = { NULL };/* cleaned environment variables */
static int  numNewEnviron = 0;     /* number of cleaned variables */



/**************************************************************************
 *                                                                        *
 *                   P R I V A T E    F U N C T I O N S                   *
 *                                                                        *
 **************************************************************************/

static void
fatal(const char *format, ...)
{
    va_list ap;

    fflush(stdout);
    fprintf(stderr, "suidcgi: ");
    va_start(ap, format);
    vfprintf(stderr, format, ap);
#ifdef PASS_ERROR_TO_CLIENT
    printf("Content-Type: text/plain\n\nsuidcgi: ");
    vprintf(format, ap);
#endif
    va_end(ap);
    exit(1);
}

static void
getUserInfo(void)
{
    struct passwd *pw;

    if ((pw = getpwuid(getuid())) == NULL)
	fatal("unable to get password info for real user\n");
    realName = strdup(pw->pw_name);
    if ((pw = getpwuid(geteuid())) == NULL)
	fatal("unable to get password info for effective user\n");
    effName = strdup(pw->pw_name);
    effHome = strdup(pw->pw_dir);
    if (realName == NULL || effName == NULL || effHome == NULL)
	fatal("out of memory\n");
}

static void
locateWwwRootDirectory(void)
{
    int len;

    strcpy(wwwRootDir, WWW_ROOT_DIRECTORY);
    if (*wwwRootDir != '/') {
	strcpy(wwwRootDir, effHome);
	len = strlen(wwwRootDir);
	if (len && wwwRootDir[len - 1] == '/')
	    wwwRootDir[len - 1] = '\0';
	strcat(wwwRootDir, "/");
	strcat(wwwRootDir, WWW_ROOT_DIRECTORY);
    }
    strcat(wwwRootDir, "/");
}

#ifdef USE_SCRIPT_NAME
static void
locateExecutable(void)
#else
static void
locateExecutable(const char *suidcgi)
#endif
{
    int         q, found;
    char        *end, *append, strippedExt[100];
#ifdef USE_SCRIPT_NAME
    int         len;
    char        *scriptName;
#endif
    static char *ext[] = { PROGRAM_EXTENSIONS };

#ifdef USE_SCRIPT_NAME
    if ((scriptName = getenv("SCRIPT_NAME")) == NULL)
	fatal("no SCRIPT_NAME environment variable\n");
    len = strlen(effName);
    if (strlen(scriptName) < len + 3
	|| strncmp(scriptName + 2, effName, len) != 0
	|| scriptName[0] != '/' || scriptName[1] != '~'
	|| scriptName[len + 2] != '/')
	fatal("SCRIPT_NAME isn't user relative\n");
    strcpy(executable, wwwRootDir);
    strcat(executable, scriptName + len + 3);
    /* if just a path (no filename) is given, assume `index.cgi' */
    if (executable[strlen(executable) - 1] == '/')
	strcat(executable, "index.cgi");
#else
    if (*suidcgi != '/')
	fatal("`%s' isn't an absolute path\n", suidcgi);
    strcpy(executable, suidcgi);
#endif
    append = end = executable + strlen(executable);
    while (append > executable && *append != '.' && *append != '/')
	--append;
    if (*append != '.')
	append = end;
    strncpy(strippedExt, append, sizeof(strippedExt) - 1);
    found = 0;
    for (q = 0; q < sizeof(ext) / sizeof(char *); q++) {
	if (strcmp(ext[q], strippedExt) == 0)
	    continue;
	strcpy(append, ext[q]);
	if (access(executable, X_OK) == 0) {
	    found = 1;
	    break;
	}
    }
    if (!found) {
	*append = '\0';
	fatal("no executable found for `%s.*'.\n", executable);
    }
}

static void
verifyCaller(void)
{
    pid_t uid, euid;

    uid = getuid();
    euid = geteuid();
    if (uid == 0 || euid == 0)
	fatal("root doesn't want to do this. period.\n");
    if (uid == euid)
	return; /* testing */
    if (strcmp(realName, HTTP_USER) != 0)
	fatal("suidcgi can only be run by user `%s'\n", HTTP_USER);
}

static void
setNewEnv(const char *env)
{
    if (!numNewEnviron)
	newEnviron = (char **) malloc(2 * sizeof(char *));
    else
	newEnviron = (char **) realloc(newEnviron,
				       (numNewEnviron + 2) * sizeof(char *));
    if (newEnviron == NULL)
	fatal("out of memory\n");
    if ((newEnviron[numNewEnviron++] = strdup(env)) == NULL)
	fatal("out of memory\n");
    newEnviron[numNewEnviron] = NULL;
}

static void
setNewEnvNameValue(const char *name, const char *value)
{
    char *s;

    if ((s = (char *) malloc(strlen(name) + strlen(value) + 2)) == NULL)
	fatal("out of memory\n");
    strcpy(s, name);
    strcat(s, "=");
    strcat(s, value);
    setNewEnv(s);
}

static void
setupEnvironment(void)
{
    int         q, found;
    char        **env;
    static char *safeEnv[] = { SAFE_ENVIRONMENT_VARIABLES };

    /* a preset PATH may be a security threat. */
    setNewEnvNameValue("PATH", DEFAULT_PATH);
    /* the two next are for convenience. */
    setNewEnvNameValue("HOME", effHome);
    setNewEnvNameValue("USER", effName);

    /* now copy clean variables */
    env = environ;
    while (*env) {
	if (strncmp(*env, "HTTP_", 5) == 0)
	    setNewEnv(*env);
	else {
	    found = 0;
	    for (q = 0; q < sizeof(safeEnv) / sizeof(char *); q++)
		if (strncmp(*env, safeEnv[q], strlen(safeEnv[q])) == 0) {
		    found = 1;
		    break;
		}
	    if (found)
		setNewEnv(*env);
	}
	++env;
    }
}

static void
verifyExecutable(void)
{
    int         len;
    struct stat st;

    if (stat(executable, &st) < 0)
	fatal("cannot stat `%s'\n", executable);
    if (st.st_uid != geteuid())
	fatal("illegal owner of `%s'\n", executable);
    if (st.st_mode & 022)
	fatal("won't run `%s' -- it's writable by someone\n", executable);
    len = strlen(wwwRootDir);
    if (strncmp(wwwRootDir, executable, len) != 0)
	fatal("won't run `%s' -- not in `%s'\n", executable, wwwRootDir);
}

static void
execute(int argc, char *argv[])
{
    argv[0] = executable;
    execve(executable, argv, newEnviron);
    fatal("execution of `%s' failed.\n", executable);
}



/**************************************************************************
 *                                                                        *
 *                    P U B L I C    F U N C T I O N S                    *
 *                                                                        *
 **************************************************************************/

int
main(int argc, char *argv[])
{
    /* get info from password entries for real and effective users */
    getUserInfo();

    /* check that the caller is the user of the http-server */
    verifyCaller();

    /* fix environment variables */
    /* set up PATH etc. */
    setupEnvironment();

    /* find user's WWW root directory */
    locateWwwRootDirectory();

#ifdef USE_SCRIPT_NAME
    /* use the SCRIPT_NAME environment variable to locate the program
     * to execute. it is supposed to be found in the same directory as
     * suidcgi. */
    locateExecutable();
#else
    /* it is assumed that the full path to this program may be found in
     * argv[0], and that the program to start is in the same directory. */
    locateExecutable(argv[0]);
#endif

    /* check that the owner of the program to start matches the suid'ed user */
    verifyExecutable();

    /* start the program */
    execute(argc, argv);

    /* never get here, actually. execute doesn't return. */
    return 0;
}
