/*
 *  hashlib.c - enables hash
 *
 *	Copyright (c) 1997 Naoya Tozuka <naochan@naochan.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You cannot use this program and its sources for commercial purposes.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include "hashlib.h"

long    words;
NodePtr hashtable[HASHSIZE];
Node    white_node = { 
          (Node *)NULL, (Node *)NULL, (char *)NULL, (char *)NULL };

void memory_error( void )
{
  printf("Insufficient memory.\n");
  exit(EXIT_FAILURE);
}
char *save_string( char *string )
{
  char *new_string;

  new_string = (char *)malloc(strlen(string) + 1);
  if (new_string == NULL)
	memory_error();
  strcpy( new_string, string );

  return new_string;
}

void hash_add_peer( Node **node, char *key, char *value )
{
  int cmp_result;

  if ((*node) == NULL) {
	(*node) = (Node *)malloc(sizeof(Node));
	if ((*node) == NULL)
	  memory_error();
	(*node)->left = NULL;
	(*node)->right = NULL;
	(*node)->key = save_string( key );
	(*node)->value = save_string( value );
	words++;
	return;
  }

  cmp_result = strcmp((*node)->key, key);
  if (cmp_result == 0) {
	if (strcmp((*node)->value, value) != 0) {
	  free((*node)->value);
	  (*node)->value = save_string( value );
	}
	return;
  }
  if (cmp_result < 0)
	hash_add_peer( &(*node)->right, key, value );
  else
	hash_add_peer( &(*node)->left, key, value );
}

void hash_clear_peer( Node **node )
{
  if ((*node) == NULL) return;

  hash_clear_peer( &(*node)->right );

  (*node)->key[0] = (*node)->value[0] = 0;
  free( (*node)->key );
  free( (*node)->value );
  words--;

  hash_clear_peer( &(*node)->left );

  free( *node );

  *node = NULL;
}

void hash_list_peer( Node *node )
{
  if (node == NULL) return;

  hash_list_peer( node->left );
  printf("%s = %s\n", node->key, node->value);
  hash_list_peer( node->right );
}

char *hash_value_peer( Node *node, char *key )
{
  int cmp_result;

  if (node == NULL)
	return NULL;

  cmp_result = strcmp(node->key, key);
  if (cmp_result == 0)
	return node->value;

  if (cmp_result < 0)
	return hash_value_peer( node->right, key );
  else
	return hash_value_peer( node->left, key );
}


int hashlib_test_main( void )
{
  char  buf[MAXWORDLEN * 2 + 80 + 1], opcode[80], 
        key[MAXWORDLEN + 1], value[MAXWORDLEN + 1];

  words = 0;
  hash_init();

  while (fgets(buf, sizeof(buf), stdin) != NULL) {
	if (strlen(buf) < 2) continue;
	sscanf( buf, "%s %s %s", opcode, key, value );
	switch (opcode[0]) {
	case 'a': /* add */
	  printf("(add) %s := %s\n", key, value);
	  hash_add( key, value );
	  break;
	case 'c': /* clear */
	  printf("(clear)");
	  hash_clear();
	  printf(" done!\n");
	  break;
	case 'l': /* list */
	  printf("(list)\n");
	  hash_list();
	  printf("total %ld words.\n", words);
	  break;
	case 's': /* show */
	  printf("(show) %s = %s\n", key, hash_value( key ));
	  break;
	case 'q': /* quit */
	  printf("(quit)\nbye!\n");
	  return EXIT_SUCCESS;
	default:
	  printf("(???) unknown opcode '%s'.\n", opcode);
	  break;
	}
  }
  return EXIT_SUCCESS;
}


int hash( char *s )
{
  unsigned v;
 
  for (v=0; *s != '\0'; s++)
	v = ((v << CHAR_BIT) + *s) % HASHSIZE;
  return (int)v;
}

void hash_init( void )
{
  int  i;

  for (i=0; i<HASHSIZE; i++) 
	hashtable[i] = NULL;  /* initialize the hashtable[] */
}

void hash_add( char *key, char *value )
{
  char *p;

  /* strlwr( key ) */
  for (p=key; *p != '\0'; p++)
	*p = tolower(*p);

  hash_add_peer( &hashtable[hash(key)], key, value );
}

void hash_clear( void )
{
  int  i;

  for (i=0; i<HASHSIZE; i++)
	hash_clear_peer( &hashtable[i] );
}

void hash_list( void )
{
  int  i;

  for (i=0; i<HASHSIZE; i++)
	hash_list_peer( hashtable[i] );
}

char *hash_value( char *key )
{
  return hash_value_peer( hashtable[hash(key)], key );
}

