/*
 *  latex.c - barf a LaTeX code
 *
 *	Copyright (c) 1997 Naoya Tozuka <naochan@naochan.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You cannot use this program and its sources for commercial purposes.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include "latex.h"
#include "hashlib.h"
#include "http.h"

int  font_size = 3;
int  fontsize_pushbuf[256] = { 0 };
int  mode_pushbuf[256] = { 0 };
int  table_cols, table_rows,
     table_current_col, table_current_row;
int  table_border;
int  is_in_table = 0;
int  is_in_verbatim = 0;
int  is_document_started = 0;
int  list_indent = 0;
int  is_graphic_enable = 0;
int  is_japanese_mode = 0;
int  is_showlink_mode = 0;
char *something = NULL;

ModeType   mode;
TableCell *table[TABLE_ROWS_MAX][TABLE_COLUMNS_MAX];
char      *table_caption;

char  *str_fontsize[7 + 1] = { "", 
	"footnotesize", "small", "normalsize",
    "large", "Large", "LARGE", "huge" };
char *str_h_level[6 + 1] = { "",
   "chapter*", "section*", "subsection*", "subsubsection*", 
   "paragraph", "subparagraph" };

char *str_hichar[95] = {  /* 161 - 255 */
  "!`", "c\\llap/", "\\pounds{}", "o\\llap{x}", "Y\\llap=",   /* 161 - 165 */
  "\\verb+|+", "\\S{}", "\\\"{ }", "\\copyright{}", "$^a$",   /* 166 - 170 */
  "${\\footnotesize <}$", "$\\neg$", "-", "\\reg ", "\\={ }", /* 171 - 175 */
  "\\degree{}", "$\\pm$", "$^2$", "$^3$", "\\'{ }",           /* 176 - 180 */
  "$\\mu$", "\\P{}", "$\\bullet$", "\\c{ }", "$^1$",          /* 181 - 185 */
  "$^{\\tiny o}$", "${\\footnotesize >}$", 
             "$\\frac14$", "$\\frac12$", "$\\frac34$",        /* 186 - 190 */
  "?`", "\\`A{}", "\\'A{}", "\\^A{}", "\\~A{}",               /* 191 - 195 */
  "\\\"A", "\\AA{}", "\\AE{}", "\\c{C}", "\\`E",              /* 196 - 200 */
  "\\'E", "\\^E", "\\\"E", "\\`I", "\\'I",                    /* 201 - 205 */
  "\\^I", "\\\"I", "(Eth)", "\\~N", "\\`O",                   /* 206 - 210 */
  "\\'O", "\\^O", "\\~O", "\\\"O", "$\\times$",               /* 211 - 215 */
  "\\O{}", "\\`U", "\\'U", "\\^U", "\\\"U",                   /* 216 - 220 */
  "\\'Y", "(Thorn)", "\\ss{}", "\\`a", "\\'a",                /* 221 - 225 */
  "\\^a", "\\~a", "\\\"a", "\\aa{}", "\\ae{}",                /* 226 - 230 */
  "\\c{c}", "\\`e", "\\'e", "\\^e", "\\\"e",                  /* 231 - 235 */
  "\\`\\i", "\\'\\i", "\\^\\i", "\\\"\\i", "(eth)",           /* 236 - 240 */
  "\\~n", "\\`o", "\\'o", "\\^o", "\\~o",                     /* 241 - 245 */
  "\\\"o", "$\\div$", "\\o{}", "\\`u", "\\'u",                /* 246 - 250 */
  "\\^u", "\\\"u", "\\'y", "(thorn)", "\\\"y"                 /* 251 - 255 */
};

extern FILE *yyout;

void push_fontsize( int size )
{
  if (fontsize_pushbuf[0] < 255)
	fontsize_pushbuf[++fontsize_pushbuf[0]] = size;
}
int pop_fontsize( void )
{
  return (int)fontsize_pushbuf[fontsize_pushbuf[0]--];
}

void push_mode( ModeType mode_type )
{
  if (mode_pushbuf[0] < 255)
	mode_pushbuf[++mode_pushbuf[0]] = (int)mode_type;
}
ModeType pop_mode( void )
{
  return (ModeType)mode_pushbuf[mode_pushbuf[0]--];
}

void puts0( char *string )
{
  if (is_in_table > 0) {
	if (mode == M_CAPTION)
	  table_append( -1, -1, string );
	else
	  table_append( table_current_col, table_current_row, string );
  } else
	fputs( string, yyout );
}
void putc0( int ch )
{
  char  ch_temp[2];

  if (is_in_table > 0) {
	ch_temp[0] = ch;
	ch_temp[1] = '\0';
	if (mode == M_CAPTION)
	  table_append( -1, -1, ch_temp );
	else
	  table_append( table_current_col, table_current_row, ch_temp );
  }
  else
	fputc( ch, yyout );
}
void puti0( int num )
{
  char buf[18];
  sprintf( buf, "%d", num );

  if (is_in_table > 0) {
	if (mode == M_CAPTION)
	  table_append( -1, -1, buf );
	else
	  table_append( table_current_col, table_current_row, buf );
  } else
	fputs( buf, yyout );
}
void puti0_r( int num )
{
  char buf[10];
  char *roman[] = { "0",
    "i", "ii", "iii", "iv", "v", "vi", "vii", "viii", "ix", "x", "xi", "xii",
    "xiii", "xiv", "xv", "xvi", "xvii", "xviii", "xix", "xx" };

  if (num <= 0 || 20 < num)
	strcpy( buf, "0" );
  else
	strcpy( buf, roman[num] );

  if (is_in_table > 0) {
	if (mode == M_CAPTION)
	  table_append( -1, -1, buf );
	else
	  table_append( table_current_col, table_current_row, buf );
  } else
	fputs( buf, yyout );
}
int printf0( char *format, ... )
{
  int   ret_value;
  char  printf0_buf[2048];
  va_list arg_p;

  va_start( arg_p, format );
  ret_value = vsprintf( printf0_buf, format, arg_p );
  va_end( arg_p );

  puts0( printf0_buf );

  return ret_value;
}

void latex_begin( void )
{
  if (is_graphic_enable == 0) {
	if (is_japanese_mode == 1)
	  puts0("\\documentstyle[a4j,webtex]{jreport}\n");
	else
	  puts0("\\documentstyle[a4,webtex]{report}\n");
  } else {
	if (is_japanese_mode == 1)
	  puts0("\\documentstyle[a4j,epsbox,webtex]{jreport}\n");
	else
	  puts0("\\documentstyle[a4,epsbox,webtex]{report}\n");
  }
}
void latex_end( void )
{
  if (is_document_started == 0)
	puts0("\\begin{document}\n");

  puts0("\n\\end{document}\n");
}

void do_fragment( char *fragment_str )
{
  int   c;
  char *p;

  /* if (is_document_started == 0) {
	 puts0("\\begin{document}\n");
    is_document_started = 1;
  } */

  for (p=fragment_str; *p != '\0'; p++) {
	c = (unsigned char)*p;
	if (c >= 0x0080) {  /* EUC */
	  putc0(*p++);
	  putc0(*p);
	} else
	  do_char(c); 
  }
}
void do_space( void )
{
  putc0(' ');
}
void do_return( void )
{
  if ( is_in_verbatim )
	putc0('\n');
}
void do_char( char ch )
{
  if ( is_in_verbatim ) {
	switch( ch ) {
	case '\\': if (is_japanese_mode > 0)
	             puts0("Y\\llap=");     /* Japanese mode */
               else
                 puts0("{\\char92}");   /*  English mode */
               break;
	case '{':  puts0("\\{"); break;
	case '}':  puts0("\\}"); break;
	default:   putc0( ch ); break;	
	}
  } else {
	switch( ch ) {
	case '"':  puts0("''"); break;
	case '#':  puts0("\\#"); break;
	case '$':  puts0("\\$"); break;
	case '%':  puts0("\\%"); break;
	case '&':  puts0("\\&"); break;
	case '/':  puts0("\\slash "); break;
	case '<':  puts0("\\(<\\)"); break;
	case '>':  puts0("\\(>\\)"); break;
	case '_':  puts0("\\_"); break;
	case '~':  puts0("\\tt\\symbol{\"7e}"); break;
	case '^':  puts0("\\tt\\symbol{\"5e}"); break;
	case '\\': if (is_japanese_mode > 0)
         	     puts0("\\yen ");         /* Japanese mode */
	           else
	             puts0("$\\backslash$");  /*  English mode */
	           break;
	case '{':  puts0("\\{"); break;
	case '}':  puts0("\\}"); break;
	case '[':  puts0("{\\char91}"); break;
	case ']':  puts0("{\\char93}"); break;
	default:   putc0( ch ); break;	
	}
  }
}
void do_hichar( int ch )
{
  if (ch < 128)
	do_char( (char)ch );
  else if (161 <= ch && ch <= 255)
	puts0( str_hichar[ch - 161] );
}
void do_begin( void )
{
  putc0('{');
}
void do_end( void )
{
  puts0("}\n");
}

void do_html_begin( void )
{
}
void do_html_end( void )
{
}

void do_title_begin( void )
{
  /* puts0("\\title{"); */
  puts0("\\markright{");
}
void do_title_end( void )
{
  puts0("}\n");
  puts0("\\begin{document}\n");
  is_document_started = 1;
}

void do_body_begin( void )
{
  if (is_document_started == 0)
	puts0("\\begin{document}\n");
  is_document_started = 1;
}
void do_body_end( void )
{
}

void do_br( void )
{
  if (is_in_table) return;

  switch( mode ) {
  case M_BOLD:
  case M_ITALIC:
  case M_TYPEWRITER:
  case M_IMG:       break;
  case M_HR:        puts0("\\par\n"); break;
  case M_PRE:       putc0('\n'); break;
  case M_ANCHOR:
  case M_UNDERLINE: puts0("}\\par\n\\underline{"); break;
  case M_BODY:      puts0("\\par\n"); break;
  case M_Hn:        puts0("\\\\ "); break;
  case M_CENTER:
  case M_PARAGRAPH: puts0("\n\n"); break;
  case M_FONT:
  case M_UNKNOWN:
  default:          
	puts0("\\par\n");
	break;
  }
}
void do_wbr( void )
{
  puts0("\\linebreak[0]\n");
}

void do_hr( void )
{
  int   i, percent;
  char *v, *align, align_c, *size, temp[80];
  float size_f = 0.4;

  v = hash_value("width");
  percent = (v == NULL)? 100 : atoi(v);

  do_br();

  if (is_in_verbatim) {
	puts0("\\hrulefill\n");	
  } else {
	puts0("\\setlength\\hrWidth{\\textwidth}\n");
	printf0("\\divide\\hrWidth by 100\n");
	printf0("\\multiply\\hrWidth by %d\n", percent);

	align = hash_value("align");
	align_c = (align == NULL)? 'c' : tolower(align[0]);
	size = hash_value("size");
    size_f = ((size == NULL)? 0.4 : 0.4*atof(size));
	puts0("\\hbox to \\textwidth{");
	sprintf(temp, "\\vbox{\\hrule width \\hrWidth height %gpt}", size_f);

	switch( align_c ) {
	case 'l':
	  printf0("%s\\hfill", temp); break;
	case 'r':
	  printf0("\\hfill%s", temp); break;
	case 'c':
	default:
	  printf0("\\hfill%s\\hfill", temp);
	  break;
	}
	puts0("}\n");
  }
  do_br();
}

void do_img( void )
{
  char *image_src, *image_width, *image_height, *image_alt;
  char *img_file_name;
  int   is_size_set;

  image_src = hash_value("src");
  if (image_src == NULL) {
	puts0("[SRC=???]");
	return;
  }

  image_alt = hash_value("alt");
  if (image_alt == NULL) image_alt = "[IMAGE]";

  if ((is_graphic_enable == 0) || (is_in_verbatim))
	puts0( image_alt );
  else {
	img_file_name = getimage(image_src);
	if (img_file_name == NULL) {
	  puts0( image_alt );
	  return;
	}

	/* \def\alternateStr{....} */
	printf0("\\def\\alternateStr{%s}\\psbox[", image_alt);

	is_size_set = 0;
	image_width = hash_value("width");
	if (image_width != NULL) {
	  printf0("width=%dpt", atoi(image_width) );  /* not for % */
	  is_size_set++;
	}
	image_height = hash_value("height");
	if (image_height != NULL) {
	  if (is_size_set > 0)
		putc0(',');
	  printf0("height=%dpt", atoi(image_height) );  /* not for % */
	  is_size_set++;
	}
	if (is_size_set == 0)
	  puts0("scale=0.66666667");

	/* cut the suffix.epsf */
	img_file_name[ strlen(img_file_name) - 5 ] = '\0';
	printf0("]{%s}", img_file_name );
  }
}

void do_paragraph_begin( void )
{
  puts0("\n\n");
}
void do_paragraph_end( void )
{
  puts0("\n\n");
}

void do_h_begin( int h_level )
{
  char *align;

  align = hash_value("align");
  if (align != NULL) {
	switch( tolower(align[0]) ){
	case 'c': do_something_begin("center"); break;
	case 'l': do_something_begin("flushleft"); break;
	case 'r': do_something_begin("flushright"); break;
	default: break;
	}
  }

  if (h_level < 1 || 6 < h_level) {
	putc0('{');
	return;
  }

  if (is_in_table)
	printf0("{\\%s ", str_fontsize[7 - h_level] );
  else
	printf0("\n\n\\%s{", str_h_level[h_level] );
}
void do_h_end( void )
{
  if (is_in_table)
	puts0("}\\\\\n");
  else
	puts0("}\n");

  do_something_end();
}

void do_something_begin( char *sth )
{
  if (something != NULL)
	do_something_end();

  printf0("\n\\begin{%s}\n", something);

  something = save_string( sth );
}
void do_something_end( void )
{
  if (something != NULL) {
	printf0("\n\\end{%s}\n", something);
	free(something);
	something = NULL;
  }
}

void do_anchor_begin( void )
{
  char  *name, *href;

  name = hash_value("name");
  href = hash_value("href");

  if (href != NULL)  
	do_underline_begin();
  else 
	putc0('{');
}

void do_anchor_end( void )
{
  int   port;
  char  hostname[80], path[256], *link_url;

  /* putc0('}'); */
  do_underline_end();

  link_url = hash_value("href");
  if ((is_showlink_mode == 1) && (link_url != NULL)) {
	puts0( "\\footnote{{\\tt " );

	port = url_analyse( link_url, hostname, path );
	switch (port) {
	case 0:   puts0( "file://" );
	          do_fragment( path );
	          break;
	case 21:  printf0( "ftp://%s", hostname );
	          do_fragment( path );
	          break;
	case 23:  printf0( "telnet://%s/", hostname );
	          break;
	case 25:  puts0( "mailto:" );
	          do_fragment( path );
	          break;
	case 70:  printf0( "gopher://%s", hostname );
	          do_fragment( path );
	          break;
	case 80:
	default:  printf0( "http://%s", hostname );
	          do_fragment( path );
	          break;
	}

	puts0( "}}" );
  }
}

void do_center_begin( void )
{
  puts0("\n\\begin{center}\n");
}
void do_center_end( void )
{
  puts0("\n\\end{center}\n");
}

void do_font_begin( void )
{
  int   isize;
  char *size, *face;

  size = hash_value("size");
  face = hash_value("face");

  putc0('{');

  push_fontsize( font_size );
  isize = font_size;
  if (size != NULL) {
	switch( size[0] ){
	case '+': isize += atoi(&size[1]); break;
	case '-': isize -= atoi(&size[1]); break;
	default: 
	  isize = atoi(size);
	  break;
	}
	if (isize < 1 || 7 < isize) isize = 3;
	font_size = isize;

	printf0("\\%s ", str_fontsize[isize] );
  }
}
void do_font_end( void )
{
  putc0('}');
  font_size = pop_fontsize();
}
void do_big_begin( void )
{
  push_fontsize( font_size );
  if (font_size < 7)
	font_size++;

  printf0("{\\%s ", str_fontsize[font_size] );
}
void do_big_end( void )
{
  putc0('}');
  font_size = pop_fontsize();
}

void do_small_begin( void )
{
  push_fontsize( font_size );
  if (1 < font_size)
	font_size--;

  printf0("{\\%s ", str_fontsize[font_size] );
}
void do_small_end( void )
{
  putc0('}');
  font_size = pop_fontsize();
}

void do_script_begin( void )
{
}
void do_script_end( void )
{
}

void do_pre_begin( void )
{
// puts0("\\begin{verbatim}\n");
  puts0("\n\\begin{alltt}\n");
  is_in_verbatim = 1;
}
void do_pre_end( void )
{
// puts0("\\end{verbatim}\n");
  puts0("\n\\end{alltt}\n");
  is_in_verbatim = 0;
}


/*
 * table 
 */

void table_init( void )
{
  int  x, y;

  table_caption = NULL;

  for (y=0; y<TABLE_ROWS_MAX; y++)
	for (x=0; x<TABLE_COLUMNS_MAX; x++)
	  table[y][x] = NULL;

  table_cols = table_rows = 0;
  table_current_row = 0;
  table_current_col = -1;
  table_border = 0;
}

void table_start( void )  /* clear */
{
  int         x, y;
  char       *s_border;
  TableCell  *tcell;

  if (table_caption != NULL) {
	free( table_caption );
	table_caption = NULL;
  }

  for (y=0; y<table_rows; y++)
	for (x=0; x<table_cols; x++) {
      tcell = table[y][x];
	  if (tcell != NULL) {
		if ( tcell->content != NULL) {
		  free( tcell->content );
		  tcell->content = NULL;
		  tcell->colspan = tcell->rowspan = 0;
		}
		free( table[y][x] );
		table[y][x] = NULL;
	  }
	}

  table_cols = table_rows = 0;
  table_current_row = 0;
  table_current_col = -1;

  is_in_table = 1;

  s_border = hash_value("border");
  table_border = (s_border == NULL)? 0 : atoi( s_border );
}

void do_table_cell_begin( void )
{
  int   x, y, x_max, y_max;
  char *align, *valign, *colspan, *rowspan;
  TableCell  *pcell, *tcell;

  table_current_col++;

  for (y=0; y<=table_current_row; y++) {
	for (x=0; x<=table_current_col; x++) {
	  if ((y == table_current_row) && (x == table_current_col)) break;
	  pcell = table[y][x];
	  if (pcell == NULL) continue;

	  x_max = x + (pcell->colspan - 1);
	  y_max = y + (pcell->rowspan - 1);

	  if (table_current_col <= x_max && table_current_row <= y_max)
		table_current_col = x_max + 1;
	}
  }

  tcell = table[table_current_row][table_current_col] 
        = (TableCell *)malloc( sizeof(TableCell) );
  if (tcell == NULL)
	memory_error();
  tcell->content = NULL;

  align = hash_value("align");
  if (align == NULL) 
	tcell->align = 'c';
  else {
	switch( tolower(align[0]) ){
	case 'l':  tcell->align = 'l'; break;
	case 'r':  tcell->align = 'r'; break;
	case 'c':
    default:   tcell->align = 'c'; break;
	}
  }
  valign = hash_value("valign");
  if (valign == NULL) 
	tcell->valign = 'm';
  else {
	switch( tolower(align[0]) ){
	case 't':  tcell->valign = 't'; break;
	case 'b':  tcell->valign = 'b'; break;
	case 'm':
    default:   tcell->valign = 'm'; break;
	}
  }
  colspan = hash_value("colspan");
  tcell->colspan = (colspan == NULL)? 1 : atoi(colspan);
  rowspan = hash_value("rowspan");
  tcell->rowspan = (rowspan == NULL)? 1 : atoi(rowspan);

}
void do_table_cell_end( void )
{
  table_current_col += 
	(table[table_current_row][table_current_col]->colspan - 1);
}

void do_table_th_begin( void )
{
  do_table_cell_begin();
  puts0("{\\bf ");
}
void do_table_th_end( void )
{
  putc0('}');
  do_table_cell_end();
}

void do_table_td_begin( void )
{
  do_table_cell_begin();
  putc0('{');
}
void do_table_td_end( void )
{
  putc0('}');
  do_table_cell_end();
}

void table_newline( void )
{
  table_current_row++;

  if (table_cols == 0)
	table_cols = table_current_col + 1;

  table_current_col = -1;
}
void table_end( void )
{
  if (table_rows == 0)
	table_rows = table_current_row;
}

void table_append( int x, int y, char *str )
{
  int  len;
  char *t, *new_string;
  TableCell  *tcell;

  if (x == -1 && y == -1) {
	/* caption */
	if (table_caption == NULL)
	  table_caption = save_string( str );
	else {
	  len = strlen( table_caption ) + strlen( str ) + 1;
	  new_string = (char *)malloc( len );
	  if (new_string == NULL)
		memory_error();
	  sprintf( new_string, "%s%s", table_caption, str );
	  free( table_caption );
	  table_caption = new_string;
	}
	return;
  }

  if (x < 0 || TABLE_COLUMNS_MAX <= x) 
	return;
  if (y < 0 || TABLE_ROWS_MAX <= y) 
	return;

  if (1 <= y && table_cols <= x)
	return; /* do nothing */

  tcell = table[y][x];
  if (tcell->content == NULL)
	tcell->content = save_string( str );
  else {
	len = strlen( tcell->content ) + strlen( str ) + 1;
	new_string = (char *)malloc( len );
	if (new_string == NULL)
	  memory_error();
	sprintf( new_string, "%s%s", tcell->content, str );
	free( tcell->content );
	tcell->content = new_string;
  }
}

void table_show( void )
{
  int  x, y;
  char temp[1024];
  TableCell *tcell;
 
  is_in_table = 0;

  putc0('\n');

  if (table_caption != NULL)
	puts0("\\sbox{\\mytablebox}{");

  puts0("\\begin{tabular}{");
  if (table_border > 0) {
	for (x=0; x<table_cols; x++)
	  puts0("|c");
	puts0("|} \\hline\n");
  } else {
	for (x=0; x<table_cols; x++)
	  putc0('c');
	puts0("}\n");
  }

  for (y=0; y<table_rows; y++) {
	for (x=0; x<table_cols; x++) {
	  tcell = table[y][x];
	  if ( tcell != NULL) {
		if (table_border > 0)
		  printf0("\\multicolumn{%d}{|%c|}", tcell->colspan, tcell->align);
		else
		  printf0("\\multicolumn{%d}{%c}", tcell->colspan, tcell->align);

		if( tcell->rowspan == 1 )
		  printf0("{%s}", tcell->content);
		else
		  puts0("{}");

		if (tcell->colspan > 1)
		  x += (tcell->colspan - 1);
	  } else {
		int  ix, iy;
		int  left, right;
		left = right = ( (table_border > 0) ? 1 : 0 );

		for (iy=0; iy<=y; iy++) {
		  TableCell *pcell;
		  pcell = table[iy][x];
		  if ((pcell != NULL) && (y == iy + (pcell->rowspan / 2))) {
			if (table_border > 0)
			  printf0("\\multicolumn{%d}{|%c|}", pcell->colspan, pcell->align);
			else
			  printf0("\\multicolumn{%d}{%c}", pcell->colspan, pcell->align);

			printf0("{\\raisebox{%gex}[0pt]{%s}}", 
					1.5 * pcell->rowspan / 2, pcell->content);

			if (pcell->colspan > 1)
			  x += (pcell->colspan - 1);
			goto done;
		  }

		  for (ix=0; ix<=x; ix++) {
			int  ix_max, iy_max;
			pcell = table[iy][ix];
			if (pcell == NULL) continue;
			if (ix == x) {
			}

			ix_max = ix + (pcell->colspan - 1);
			iy_max = iy + (pcell->rowspan - 1);
			if (y <= iy_max) {
			  if (ix < x && x <= ix_max)
				left = 0;
			  if (x+1 <= ix_max)
				right = 0;
			}
		  }
		}
		printf0("\\multicolumn{1}{");
		if (left) putc0('|');
		putc0('c');
		if (right) putc0('|');
		puts0("}{}");
	  }
	  done:

	  if (x < (table_cols-1))
		puts0(" & ");
	}
	puts0(" \\\\");

	if (table_border > 0) {
	  for (x=0; x<table_cols; x++) {
		int  ix, iy, ix_max, iy_max;
        int  can_be_lined = 1;

		for (iy=0; iy<=y; iy++) {
		  for (ix=0; ix<=x; ix++) {
		    TableCell *pcell;

			pcell = table[iy][ix];
			if (pcell == NULL) continue;

			ix_max = ix + (pcell->colspan - 1);
			iy_max = iy + (pcell->rowspan - 1);

			if ((x <= ix_max) && (y+1 <= iy_max))
			  can_be_lined = 0;
		  }
		}
		if (can_be_lined)
		  printf0("\\cline{%d-%d}", 1+x, 1+x);
	  }
	}
	putc0('\n');
  }

  if (table_caption != NULL) {
	puts0("\\end{tabular}}\n");
	puts0("\\hbox to \\the\\wd\\mytablebox{\\hfill ");
	do_fragment( table_caption );
	puts0("\\hfill}\n");
	puts0("\\vspace{3mm}\n");
	puts0("\\usebox{\\mytablebox}\n");
  } else
	puts0("\\end{tabular}\n");
}


/*
 * listing
 */

void do_ulist_begin( void )
{
  char *label_type;

  list_indent++;

  label_type = hash_value("type");
  if (label_type != NULL) {
	puts0( "\\def\\labelitem" );
	puti0_r( list_indent );
	switch( tolower(label_type[0]) ) {
	case 'c':  puts0("{\\(\\circ\\)}"); break;
	case 's':  puts0("{\\(\\Box\\)}"); break;
	case 'd': 
	default:   puts0("{\\(\\bullet\\/)}"); break;
	}
  }

  puts0("\n\\begin{itemize}\n");
}
void do_ulist_end( void )
{
  puts0("\n\\end{itemize}\n");

  puts0( "\\def\\labelitem" );
  puti0_r( list_indent );
  puts0("{\\(\\bullet\\)}");

  list_indent--;
}

void do_olist_begin( void )
{
  char *label_type, *counter_start;

  list_indent++;

  label_type = hash_value("type");
  if (label_type != NULL) {
	puts0( "\\def\\theenum" );
	puti0_r( list_indent );
	switch( label_type[0] ) {
	case 'A':  puts0("{\\Alph{enum"); break;
	case 'a':  puts0("{\\alph{enum"); break;
	case 'I':  puts0("{\\Roman{enum"); break;
	case 'i':  puts0("{\\roman{enum"); break;
	case '1':
	default:   puts0("{\\arabic{enum"); break;
	}
	puti0_r( list_indent );
	puts0("}}");
  }

  puts0("\n\\begin{enumerate}\n");

  counter_start = hash_value("start");
  if (counter_start != NULL) {
	puts0("\\setcounter{enum");
	puti0_r( list_indent );
	printf0("}{%d}\n", atoi(counter_start) - 1);
  }
}
void do_olist_end( void )
{
  puts0("\n\\end{enumerate}\n");

  puts0( "\\def\\theenum" );
  puti0_r( list_indent );
  puts0(" {\\arabic{enum" );
  puti0_r( list_indent );
  puts0( "}}" );

  list_indent--;
}
void do_menu_begin( void )
{
  list_indent++;
  puts0("\n\\begin{list}{*}{}\n");
}
void do_menu_end( void )
{
  puts0("\n\\end{list}\n");
  list_indent--;
}
void do_dir_begin( void )
{
  list_indent++;
  puts0("\n\\begin{list}{}{}\n");
  puts0("{\\tt ");
}
void do_dir_end( void )
{
  puts0("}\n");
  puts0("\\end{list}\n");
  list_indent--;
}

void do_listitem( void )
{
  puts0("\n\\item ");
}

void do_dlist_begin( void )
{
  list_indent++;

  puts0("\n\\begin{description}\n");
}
void do_dlist_end( void )
{
  puts0("\n\\end{description}\n");

  list_indent--;
}
void do_dlist_term( void )
{
  puts0("\n\\item[");
}
void do_dlist_define( void )
{
  putc0(']');
  do_br();
}


/*
 * listing
 */

void do_bold_begin( void )
{
  puts0("{\\bf ");
}
void do_bold_end( void )
{
  putc0('}');
}

void do_italic_begin( void )
{
  puts0("{\\it ");
}
void do_italic_end( void )
{
  puts0("\\/}");
}

void do_typewriter_begin( void )
{
  puts0("{\\tt ");
}
void do_typewriter_end( void )
{
  putc0('}');
}

void do_underline_begin( void )
{
  puts0("\\underline{");
}
void do_underline_end( void )
{
  putc0('}');
}

void do_blockquote_begin( void )
{
  puts0("\n\\begin{quote}\n");
}
void do_blockquote_end( void )
{
  puts0("\n\\end{quote}\n");
}

void do_address_begin( void )
{
  puts0("{\\it ");
}
void do_address_end( void )
{
  puts0("\\/}");
}

void do_em_begin( void )
{
  puts0("{\\em ");
}
void do_em_end( void )
{
  putc0('}');
}

void do_strong_begin( void )
{
  puts0("{\\bf ");
}
void do_strong_end( void )
{
  putc0('}');
}

void do_code_begin( void )
{
  puts0("\\verb+");
}
void do_code_end( void )
{
  putc0('+');
}

void do_samp_begin( void )
{
  puts0("{\\tt ");
}
void do_samp_end( void )
{
  putc0('}');
}

void do_kbd_begin( void )
{
  puts0("{\\tt ");
}
void do_kbd_end( void )
{
  putc0('}');
}

void do_var_begin( void )
{
  puts0("{\\it ");
}
void do_var_end( void )
{
  puts0("\\/}");
}

void do_cite_begin( void )
{
  puts0("{\\it ");
}
void do_cite_end( void )
{
  puts0("\\/}");
}

void do_dfn_begin( void )
{
  puts0("{\\it ");
}
void do_dfn_end( void )
{
  puts0("\\/}");
}

void do_nobr_begin( void )
{
  puts0("\\mbox{");
}
void do_nobr_end( void )
{
  putc0('}');
}

void do_blink_begin( void )
{
  puts0("{");
}
void do_blink_end( void )
{
  putc0('}');
}


void do_ensp( void )
{
  puts0("\\hspace*{0.5em}");
}
void do_emsp( void )
{
  puts0("\\hspace*{1em}");
}
void do_nbsp( void )
{
  putc0('~');
}
void do_endash( void )
{
  puts0("--");
}
void do_emdash( void )
{
  puts0("---");
}

void do_term( char *term )
{
  int len;

  len = strlen(term);
  if (len < 3) return;

  switch( term[0] ) {
  case 'T':  /* TeX   */
	puts0("\\TeX{}");
	break;
  case 'L':  /* LaTeX, LaTeX2e */
	if (len > 5)
	  puts0("\\LaTeXe{}");
	else
	  puts0("\\LaTeX{}");
	break;
  case 'p':  /* pTeX, pLaTeX, pLaTeX2e */
	switch( term[1] ) {
	case 'L':  /* pLaTeX */
	  if (len > 6)
		puts0("\\pLaTeXe{}");
	  else
		puts0("\\pLaTeX{}");
	  break;
	case 'T':  /* pTeX   */
	  puts0("\\pTeX{}");
	  break;
	default:
	  break;
	}
	break;
  default:
	break;
  }
}

void do_form_begin( void )
{
}
void do_form_end( void )
{
}

void do_form_input( void )
{
  char  *type, *name, *size, *value, *checked;
  float  size_f = 20.0;
  int    i, len;

  type = hash_value("type");
  if (type == NULL) type = "text"; /* assume 'type="TEXT"' */

  name = hash_value("name");
  if (name == NULL) name = "";
  
  size = hash_value("size");
  checked = hash_value("checked");

  switch( tolower(type[0]) ){
  case 'h':  /* hidden */
	break;
  case 't':  /* text */
	if (size != NULL) size_f = atof(size);
	printf0("\\framebox[%gem][l]{%s}", size_f * 0.6, name);   /* 0.6em/char */
	break;
  case 'p':  /* password */
	if (size != NULL) size_f = atof(size);
	printf0("\\framebox[%gem][l]{", size_f * 0.6);
    len = strlen( name );
    for (i=0; i<len; i++)
	  puts0("\\(\\bullet\\)");
	putc0('}');
	break;
  case 'c':  /* checkbox */
	if (checked != NULL)
	  puts0("\\boxed{\\(\\surd\\)}");
	else
	  puts0("\\boxed{}");
	puts0("\\hspace{2mm}");
	break;
  case 'r':  /* radio */
	if (tolower(type[1]) == 'a') {
	  if (checked != NULL)
		puts0("\\circled{\\(\\bullet\\)}");
	  else
		puts0("\\circled{}");
	  puts0("\\hspace{2mm}");
	  break;
	}
	/* else fall down ... (reset) */
  case 's':  /* submit */
	puts0("\\keytop{\\hspace*{2mm}");
	value = hash_value("value");
	if (value != NULL)
	  do_fragment( value );
	puts0("\\hspace*{2mm}}");
	break;
  default:
	break;
  }

  
}

void do_select_begin( void )
{
  puts0("\\par\n");
}
void do_select_end( void )
{
  puts0("\n\\par\n");
}
void do_select_option_begin( void )
{
  char *selected;

  selected = hash_value("selected");
  if (selected != NULL)
	printf0("\\par\n\\verb|(*)| ");
  else
	printf0("\\par\n\\verb|( )| ");
}

void do_textarea_begin( void )
{
  float  width, height;
  char  *rows, *cols;

  rows = hash_value("rows");
  height = ((rows == NULL)? 10.0 : atof(rows)) * 2.0;

  cols = hash_value("cols");
  width = ((cols == NULL)? 10.0 : atof(cols)) * 0.6;

  puts0("\\par\n\\vspace{1mm}\n");
  printf0("\\frame{\\hbox to %gem{\\hfill", width+1 );
  printf0("\\vbox to %gex{\\hsize=%gem\\vfill", height+1, width );
  printf0("\\vbox to %gex{%%\n", height );
}
void do_textarea_end( void )
{
  puts0("\\vfill}\n\\vfill}\\hfill}}\\vspace{1mm}\n");
}


