#!/usr/bin/perl -w
##
## bigbrother
## http://snarfed.org/space/bigbrother
## Copyright 2003, 2004 Ryan Barrett <bigbrother@ryanb.org>
##
## File: bigbrother.pl
##
## This is a Perl plugin for Gaim 0.68+ that logs away messages to a remote (or
## local) computer running sshd. It registers a callback that is called when
## any active AIM account logs on, sets an away message, returns from away, or
## logs off. The callback writes the away message, or an appropriate message
## for the event, to the log file.
##
## If the away message is the empty string, this means that the account is
## returning from away. In this case, the string "Online" is written to the
## log.
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the Free
## Software Foundation; either version 2 of the License, or (at your option)
## any later version.
##
## This program is distributed in the hope that it will be useful, but WITHOUT
## ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
## FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
## more details.
##
## You should have received a copy of the GNU General Public License along with
## this program; if not, write to the Free Software Foundation, Inc., 59 Temple
## Place, Suite 330, Boston, MA 02111-1307 USA

require 5.004;
use Gaim;

# This is the screen name for which you want to log away messages. This is not
# required, but is highly recommended. If it is filled in, it prevents other
# Gaim accounts from interfering with the logging.
my $screenname  = "";

# If you want to log to a remote machine using ssh, enter the ssh command line
# here. A typical command line would be "ssh -i ~/.ssh/id_rsa my.server.net"
my $ssh_cmd     = "";

# This is the file away messages will be logged to.
my $logfile     = "~/away.log";

# This is the "away message" to use when you're online (ie not away :P).
my $online_msg  = "Online";

# The command used to get a timestamp. You should only need to change this if
# you're on a non-*nix system.
my $date_cmd    = 'date +"%F %H:%M:%S"';

# This is used internally; it stores the last away message we've seen. It's
# used to work around a bug where the "signed-on" signal is occasionally sent
# immediately after the "account-away" signal.
my $last_msg    = "";


%PLUGIN_INFO = (
    perl_api_version      => 2,
    name                  => "bigbrother",
    version               => "0.5.1",
    summary               => "Logs away messages to a file, local or remote.",
    description           =>
     "Logs away messages to a file, local or remote. Intended for use with " .
     "the bigbrother program. See the web site for more information.",
    author                => "Ryan Barrett <bigbrother\@ryanb.org>",
    url                   => "http://snarfed.org/space/bigbrother",
    load                  => "plugin_load",
    unload                => "plugin_unload"
);

sub plugin_init {
    return %PLUGIN_INFO;
}


sub plugin_load {
    my $plugin = shift;

    # register callbacks to log away messages, signons, and signoffs
    Gaim::signal_connect(Gaim::Accounts::handle,
                         "account-away", $plugin, \&away);
    Gaim::signal_connect(Gaim::Connections::handle,
                         "signed-on",  $plugin, \&signon);
    Gaim::signal_connect(Gaim::Connections::handle,
                         "signed-off", $plugin, \&signoff);
}

sub plugin_unload {
    # nothing to do here!
}


sub signon {
    my ($conn, $data) = @_;
    # only call away() if this is actually at signon. (ths works around a bug
    # in Gaim - or maybe it's an undesirable feature - that occasionally causes
    # the signon signal to be sent while you're already online, usually right
    # after setting an away message, the oscar prpl calls this a "NOP."
    if ($last_msg eq "") {
        away($conn->get_account(), "Signon", "");
    }
}


sub signoff {
    my ($conn, $data) = @_;
    $last_msg = "";
    away($conn->get_account(), "Signoff", "Offline");
}

sub away {
    my ($account, $state, $message) = @_;

    if ($account->get_username() eq $screenname) {
        if ($message eq "") {
            $message = $online_msg;    # returning from away
        }

        $message =~ tr/\"/\'/;  # get rid of double quotes

        # since we append to the file, touch it first to ensure it exists
        my $log_cmd =
            "touch $logfile; echo `$date_cmd` \"$message\" >> $logfile";

        if ($ssh_cmd ne "") {
            # sshing is slow; do it in the background, so the UI doesn't hang
            $log_cmd = "$ssh_cmd '$log_cmd' &";
        }

        system($log_cmd);
    }
}
