#!/usr/bin/python
#
# bigbrother
# http://snarfed.org/space/bigbrother
# Copyright 2003, 2004 Ryan Barrett <bigbrother@ryanb.org>
#
# File: bigbrother_unittest.py
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

"""
Unit tests for bigbrother.py.
"""

import sys
import StringIO
import deluxetest
import bigbrother
from bigbrother import *

LINES = [
  '2003-10-07 08:14:37 in the shower',
  '2003-10-07 08:24:48 Online',
  '2003-10-07 08:30:50 voting then train',
  '2003-10-07 10:58:14 at work',
  '2003-10-07 18:11:51 on the train',
  '2003-10-07 18:29:53 at work',
  '2003-10-07 18:30:14 on the train',
  '2003-10-07 20:49:34 Online',
  '2003-10-07 20:59:00 Online',
  '2003-10-07 23:33:02 doing nothing. i love my life!',
  '2003-10-07 23:41:53 Online',
  '2003-10-07 23:41:53 asleep',
  '2003-10-08 07:38:19 Online',
  '2003-10-08 07:38:20 in the shower',
  '2003-10-08 07:57:00 on the train',
  ]

UNSORTED_LINES = [
  '2003-10-07 23:41:53 Online',
  '2003-10-07 23:41:53 asleep',
  '2003-10-08 07:38:20 in the shower',
  '2003-10-08 07:38:19 Online',
]

TUESDAY_2300 = 1065592800
WEDNESDAY_0000 = 1065596400
WEDNESDAY_0700 = 1065621600

MSGS_BY_TIME = [(1065539677, 'in the shower'),
                (1065540288, 'Online'),
                (1065540650, 'voting then train'),
                (1065549494, 'at work'),
                (1065575511, 'on the train'),
                (1065576593, 'at work'),
                (1065576614, 'on the train'),
                (1065584974, 'Online'),
                (1065585540, 'Online'),
                (1065594782, 'doing nothing. i love my life!'),
                (1065595313, 'asleep'),
                (1065623899, 'Online'),
                (1065623900, 'in the shower')]

# the last away msg, "on the train" is at 07:57:00, or 1065625020
LOG_LENGTH                      = 1065625020 - MSGS_BY_TIME[0][0]
LOG_LENGTH_START_TUES           = 1065625020 - MSGS_BY_TIME[8][0]
LOG_LENGTH_END_TUES             = MSGS_BY_TIME[9][0] - MSGS_BY_TIME[0][0]
LOG_LENGTH_START_TUES_END_WED   = MSGS_BY_TIME[-2][0] - MSGS_BY_TIME[8][0]

TIMES_BY_MSG = {
  'Online': [(1065540288, 362), (1065584974, 566), (1065585540, 9242),
             (1065623899, 1)],
  'asleep': [(1065595313, 28586)],
  'at work': [(1065549494, 26017), (1065576593, 21)],
  'doing nothing. i love my life!': [(1065594782, 531)],
  'in the shower': [(1065539677, 611), (1065623900, 1120)],
  'on the train': [(1065575511, 1082), (1065576614, 8360)],
  'voting then train': [(1065540650, 8844)] }

FREQS = { 'Online': 4, 'at work': 2, 'doing nothing. i love my life!': 1,
          'in the shower': 2, 'on the train': 2, 'voting then train': 1,
          'asleep': 1 }

LENGTHS = { 'Online': 10171, 'at work': 26038, 'asleep': 28586,
            'doing nothing. i love my life!': 531, 'in the shower': 1731,
            'on the train': 9442, 'voting then train': 8844 }

AVG_LENGTHS = { 'Online': 10171.0 / 4, 'at work': 13019,
            'doing nothing. i love my life!': 531, 'in the shower': 865.5,
            'on the train': 4721, 'voting then train': 8844, 'asleep': 28586 }


ONLINE_DAY_DIST = {
  secs_since_midnight(8, 0, 0): 2,
  secs_since_midnight(8, 30, 0): 1,
  secs_since_midnight(9, 0, 0): 1,
  secs_since_midnight(20, 30, 0): 1,
  secs_since_midnight(21, 0, 0): 1,
  secs_since_midnight(21, 30, 0): 1,
  secs_since_midnight(22, 0, 0): 1,
  secs_since_midnight(22, 30, 0): 1,
  secs_since_midnight(23, 0, 0): 1,
  secs_since_midnight(23, 30, 0): 1,
  secs_since_midnight(0, 0, 0): 1,
  secs_since_midnight(7, 30, 0): 1 }

ONLINE_WED_DIST = {
  secs_since_midnight(0, 0, 0): 1,
  secs_since_midnight(7, 30, 0): 1,
  secs_since_midnight(8, 0, 0): 1 }

ASLEEP_TUE_DIST = {
  secs_since_midnight(23, 30, 0): 1 }

ASLEEP_WED_DIST = {
  secs_since_midnight(0, 0, 0): 1,
  secs_since_midnight(0, 30, 0): 1,
  secs_since_midnight(1, 0, 0): 1,
  secs_since_midnight(1, 30, 0): 1,
  secs_since_midnight(2, 0, 0): 1,
  secs_since_midnight(2, 30, 0): 1,
  secs_since_midnight(3, 0, 0): 1,
  secs_since_midnight(3, 30, 0): 1,
  secs_since_midnight(4, 0, 0): 1,
  secs_since_midnight(4, 30, 0): 1,
  secs_since_midnight(5, 0, 0): 1,
  secs_since_midnight(5, 30, 0): 1,
  secs_since_midnight(6, 0, 0): 1,
  secs_since_midnight(6, 30, 0): 1,
  secs_since_midnight(7, 0, 0): 1,
  secs_since_midnight(7, 30, 0): 1,
  secs_since_midnight(8, 0, 0): 1 }

ONLINE_WEEK_DIST = [0, 362 + 9808, 1, 0, 0, 0, 0]
ASLEEP_WEEK_DIST = [0, 1087, 28586-1087, 0, 0, 0, 0]



class bigbrotherTest(deluxetest.TestCase):

  def setUp(self):
    bigbrother.set_defaults()
    bigbrother.config.set('bigbrother', 'sample_interval', `30 * 60`)
    bigbrother.config.set('bigbrother', 'discard_blips', 'true')
    read_awaylog(LINES)


  def test_read_awaylog(self):
    # check basic functionality
    self.assertEqual(MSGS_BY_TIME, bigbrother.msgs_by_time)
    self.assertEqual(TIMES_BY_MSG, bigbrother.times_by_msg)

    copy_mbt = list(bigbrother.msgs_by_time)
    copy_mbt.sort()
    self.assertEqual(copy_mbt, bigbrother.msgs_by_time)


  def test_enforce_order(self):
    # if the option is true, it should check that the log is sorted
    bigbrother.config.set('bigbrother', 'enforce_order', 'true')
    try:
      read_awaylog(UNSORTED_LINES)
    except SystemExit, msg:
      snippet = ('Log file is not strictly increasing by time, line 4:\n' +
                  UNSORTED_LINES[3])
      self.assertEquals(snippet, msg.code)

    # if the option is false, it should not complain
    bigbrother.config.set('bigbrother', 'enforce_order', 'false')
    read_awaylog(UNSORTED_LINES)
    self.assertEqual(MSGS_BY_TIME[-3:-1], bigbrother.msgs_by_time)


  def test_ignore_day_of_week(self):
    # the day of the week in the log is redundant; worse, it's sometimes out of
    # sync with the date. so, it's deprecated. this tests that we ignore it.
    copy_lines = ['Thu ' + line for line in LINES]
    copy_lines[0] = 'Sun' + copy_lines[0][3:]
    copy_lines[1] = 'Sat' + copy_lines[1][3:]
    copy_lines[2] = 'Fri' + copy_lines[2][3:]
    copy_lines[3] = 'asdf' + copy_lines[3][3:]
    copy_lines[4] = 'qwert' + copy_lines[4][3:]

    # redirect stderr so we can catch the deprecation warning
    sys.stderr = StringIO.StringIO()
    read_awaylog(copy_lines)
    assert sys.stderr.getvalue().find('contains day of week tags') > -1
    sys.stderr = sys.__stderr__
    
    self.assertEqual(MSGS_BY_TIME, bigbrother.msgs_by_time)
    self.assertEqual(TIMES_BY_MSG, bigbrother.times_by_msg)


  def test_start_end_date(self):
    self.assertRaises(AssertionError, read_awaylog, LINES,
                      WEDNESDAY_0700, TUESDAY_2300)
    read_awaylog(LINES, start=TUESDAY_2300)
    self.assertEqual(MSGS_BY_TIME[-5:], bigbrother.msgs_by_time)
    read_awaylog(LINES, end=TUESDAY_2300)
    self.assertEqual(MSGS_BY_TIME[:-4], bigbrother.msgs_by_time)
    read_awaylog(LINES, TUESDAY_2300, WEDNESDAY_0700)
    self.assertEqual(MSGS_BY_TIME[-5:-2], bigbrother.msgs_by_time)

  def test_discard_blips(self):
    # check that discard_blips works
    bigbrother.config.set('bigbrother', 'discard_blips', 'false')
    read_awaylog(LINES)

    # copy so we don't trash MSGS_BY_TIME
    blipped_mbt = list(MSGS_BY_TIME)
    blipped_mbt.insert(10, (1065595313, 'Online'))
    self.assertEqual(bigbrother.msgs_by_time, blipped_mbt)

    # copy dict *and* 'Online' list so we don't trash TIMES_BY_MSG
    blipped_tbm = dict(TIMES_BY_MSG)
    blipped_tbm['Online'] = list(blipped_tbm['Online'])
    blipped_tbm['Online'].insert(3, (1065595313, 0))
    self.assertEqual(bigbrother.times_by_msg, blipped_tbm)


  def test_calc_freqs(self):
    self.assertEqual(calc_freqs(), FREQS)


  def test_calc_lengths(self):
    self.assertEqual(calc_lengths(), LENGTHS)


  def test_log_length(self):
    self.assertEqual(LOG_LENGTH, log_length())
    read_awaylog(LINES, start=TUESDAY_2300)
    self.assertEqual(LOG_LENGTH_START_TUES, log_length())
    read_awaylog(LINES, end=TUESDAY_2300)
    self.assertEqual(LOG_LENGTH_END_TUES, log_length())
    read_awaylog(LINES, TUESDAY_2300, WEDNESDAY_0700)
    self.assertEqual(LOG_LENGTH_START_TUES_END_WED, log_length())


  def test_calc_avgs(self):
    self.assertEqual(calc_avg_lengths(), AVG_LENGTHS)

  def test_tier(self):
    self.assertEqual([], get_tier({}, {}, 5))
    self.assertEqual([], get_tier(LENGTHS, FREQS, 0))

    bigbrother.config.set('bigbrother', 'minimum_tier_freq', '1')
    self.assertIsTier(get_tier(LENGTHS, FREQS, 1), 1)
    self.assertIsTier(get_tier(LENGTHS, FREQS, 2), 2)
    self.assertIsTier(get_tier(LENGTHS, FREQS, 7), 7)
    self.assertIsTier(get_tier(LENGTHS, FREQS, 20), 7)

    bigbrother.config.set('bigbrother', 'minimum_tier_freq', '2')
    self.assertIsTier(get_tier(LENGTHS, FREQS, 1), 1)
    self.assertIsTier(get_tier(LENGTHS, FREQS, 2), 2)
    self.assertIsTier(get_tier(LENGTHS, FREQS, 7), 4)


  def test_calc_day_distribution(self):
    online_all = calc_day_distribution('Online')
    self.assertEqual(online_all, ONLINE_DAY_DIST)
    online_wed = calc_day_distribution('Online', ['Wed'])
    self.assertEqual(online_wed, ONLINE_WED_DIST)
    asleep_tue = calc_day_distribution('asleep', ['Tue'])
    self.assertEqual(asleep_tue, ASLEEP_TUE_DIST)
    asleep_wed = calc_day_distribution('asleep', ['Wed'])
    self.assertEqual(asleep_wed, ASLEEP_WED_DIST)

    

  def test_calc_week_distribution(self):
    self.assertEqual(ONLINE_WEEK_DIST, calc_week_distribution('Online'))
    self.assertEqual(ASLEEP_WEEK_DIST, calc_week_distribution('asleep'))

  #
  # test the bigbrother utility fns
  #
  def test_config_getdate(self):
    self.assertRaises(ConfigParser.NoOptionError, config_getdate,
                      'bigbrother', 'nonexistent')

    bigbrother.config.set('bigbrother', 'not_date', 'asdf')
    self.assertRaises(ValueError, config_getdate,
                      'bigbrother', 'not_date')

    bigbrother.config.set('bigbrother', 'bad', '2002-58-05')
    self.assertRaises(ValueError, config_getdate,
                      'bigbrother', 'bad')

    bigbrother.config.set('bigbrother', 'good', '1970-01-02')
    time = (24 + 8) * 60 * 60  # add seven hours for PST (GMT-0800)
    self.assertEqual(time, config_getdate('bigbrother', 'good'))

    bigbrother.config.set('bigbrother', 'empty', '')
    self.assertEqual(None, config_getdate('bigbrother', 'empty'))
    

  def test_prettyprint(self):
    self.assertEqual(prettyprint([]), '')
    self.assertEqual(prettyprint([[], [], []]), '\n\n')
    self.assertEqual(prettyprint([['asdf']]), '"asdf"')
    self.assertEqual(prettyprint([[1]]), '1')
    self.assertEqual(prettyprint([[1.234], [5.6789]]), '1.23\n5.68')
    self.assertEqual(prettyprint([[1], [2], [3]]), '1\n2\n3')
    self.assertEqual(prettyprint([['a'], [3], [1.2]]), '"a" \n3   \n1.20')
    self.assertEqual(prettyprint([['abc'], ['d'], ['ef']]),
                     '"abc"\n"d"  \n"ef" ')
    self.assertEqual(
      prettyprint([['abc', 1, 234], [1, 'abcd', 234], [1, 234, 'abcde']]),
'''"abc" 1      234    
1     "abcd" 234    
1     234    "abcde"''')

  def test_secs_since_midnight(self):
    self.assertEqual(secs_since_midnight(0, 0, 0), 0)
    self.assertEqual(secs_since_midnight(0, 0, 5), 5)
    self.assertEqual(secs_since_midnight(0, 0, 61), 61)
    self.assertEqual(secs_since_midnight(0, 5, 3), 5 * 60 + 3)
    self.assertEqual(secs_since_midnight(0, 65, 3), 65 * 60 + 3)
    self.assertEqual(secs_since_midnight(1, 3, 3), 3600 + 3 * 60 + 3)
    self.assertEqual(secs_since_midnight(25, 3, 3), 25 * 3600 + 3 * 60 + 3)

  def test_strip_time(self):
    self.assertEqual(WEDNESDAY_0000, strip_time(WEDNESDAY_0000))
    self.assertEqual(WEDNESDAY_0000, strip_time(WEDNESDAY_0000 + 1))
    self.assertEqual(WEDNESDAY_0000 - SECS_IN_DAY,
                     strip_time(WEDNESDAY_0000 - 1))

    self.assertEqual(WEDNESDAY_0000 - SECS_IN_DAY, strip_time(TUESDAY_2300))
    self.assertEqual(WEDNESDAY_0000, strip_time(WEDNESDAY_0700))

  def test_secs_to_time(self):
    self.assertEqual(secs_to_time(0), '00:00')
    self.assertEqual(secs_to_time(5), '00:00')
    self.assertEqual(secs_to_time(61), '00:01')
    self.assertEqual(secs_to_time(5 * 60 + 3), '00:05')
    self.assertEqual(secs_to_time(65 * 60 + 3), '01:05')
    self.assertEqual(secs_to_time(1 * 3600 + 3 * 60 + 3), '01:03')
    self.assertEqual(secs_to_time(13 * 3600 + 3 * 60 + 3), '13:03')

  def test_ceiling_normalize(self):
    # without ceiling
    self.assertAllClose(ceiling_normalize([]), [])
    self.assertAllClose(ceiling_normalize([0, 0, 0]), [0, 0, 0])
    self.assertAllClose(ceiling_normalize([0, 1, 0]), [0, 1, 0])
    self.assertAllClose(ceiling_normalize([3, 5, 4, 9, 1, 3]),
                        [.333, .555, .444, 1, .111, .333])
    self.assertAllClose(ceiling_normalize([1.9, 98.5, 30.4, 63.8, 5.4]),
                        [.019, 1, .309, .648, .055])

    # with ceiling
    self.assertAllClose(ceiling_normalize([], 3), [])
    self.assertAllClose(ceiling_normalize([0, 0, 0], 3), [0, 0, 0])
    self.assertAllClose(ceiling_normalize([0, 1, 0], 3), [0, 1, 0])
    self.assertAllClose(ceiling_normalize([2, 3, 4], 3), [.667, 1, 1])
    self.assertAllClose(ceiling_normalize([4, 5, 6], 3), [1, 1, 1])

  def test_reverse_sort_by_value(self):
    self.assertEqual(reverse_sort_by_value({}), [])
    self.assertEqual(reverse_sort_by_value({0: 0}), [(0, 0)])

    test = {'a': 1, 'b': 3, 'c': 0}
    golden = [('b', 3), ('a', 1), ('c', 0)]
    self.assertEqual(reverse_sort_by_value(test), golden)
    self.assertEqual(reverse_sort_by_value(test.items()), golden)

    test = {1: 'a', 3: 'b', 0: 'c'}
    golden = [(0, 'c'), (3, 'b'), (1, 'a')]
    self.assertEqual(reverse_sort_by_value(test), golden)
    self.assertEqual(reverse_sort_by_value(test.items()), golden)


  #
  # utility fns
  #
  def assertIsTier(self, tier, size):
    """ Raises an assertion if the parameter is not a tier of LENGTHS of the
    given size.
    """
    self.assertEqual(size, len(tier))

    # do the messages occur at least minimum_tier_freq times?
    min_freq = bigbrother.config.getint('bigbrother', 'minimum_tier_freq')
    for msg in tier:
      assert FREQS[msg] >= min_freq

    # does the tier start with the longest used msg?
    tier_lens = [LENGTHS[msg] for msg in LENGTHS if FREQS[msg] >= min_freq]
    maxlen = max(tier_lens)
    self.assertEqual(maxlen, LENGTHS[tier[0]])
    
    # is the rest of it a tier?
    tier_lens = [LENGTHS[msg] for msg in tier]
    sorted = tier_lens
    sorted.sort()
    sorted.reverse()
    self.assertEqual(sorted, tier_lens)


    
if __name__ == '__main__':
  deluxetest.main()
